/**
 * @file signals.c
 *
 * Code for the NJAMD signal handling she-hot.
 * 
 * Copyright (C) 2001 Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details. 
 */ 
#include <lib/njamd.h>
#include <lib/portability.h>
#include <lib/signals.h>
#include <lib/public.h>
#include <lib/util.h>
#include <stdlib.h>
#include <config.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

/* From public.c */
#ifdef HAVE_POSIX_SIGACTION
void __nj_public_signals_dispatch(int sig, siginfo_t *info, void *arg);
#else
void  __nj_public_signals_dispatch(int sig);
#endif

/**
 * Registers signal handlers.
 *
 * Register all the signals that would cause termination of the program, so we
 * can still spew some stats 
 *
 * Must be called AFTER libc is initted
 */
void __nj_signals_bootstrap_init(struct nj_signals *signals,
		struct nj_libc_syms *syms)
{
	int i;
#ifdef HAVE_POSIX_SIGACTION
	/* The reason why we can use sigaction is because if dlopen is broke,
	 * it won't be defined in public.c */
	if((signals->libc_sigaction = __nj_libc_syms_resolve_libc(syms, "sigaction")) == NULL)
		signals->libc_sigaction = sigaction;
#endif
	/* ditto */
	if((signals->libc_signal = __nj_libc_syms_resolve_libc(syms, "signal")) == NULL)
		signals->libc_signal = signal;
	
	signals->core_dump_type = NJ_DUMP_HARD_CORE;

	for(i = 0; i < sizeof(u_long)*8; i++)
	{
		signals->handler[i] = SIG_DFL;
#ifdef HAVE_POSIX_SIGACTION
		signals->sigaction[i] = NULL;
		memset(&signals->act[i], 0, sizeof(struct sigaction));
#endif
	}
}

/**
 * Init the signals 
 * 
 * @param signals The signal handlers
 * @param prefs The user prefs
 */
void __nj_signals_user_init(struct nj_signals *signals, struct nj_prefs *prefs)
{
#ifdef HAVE_POSIX_SIGACTION
	struct sigaction act;

	memset(&act, 0, sizeof(act));
	act.sa_sigaction = __nj_public_signals_dispatch;
	sigemptyset(&act.sa_mask);
	act.sa_flags = SA_RESTART | SA_SIGINFO;
#endif           
	signals->core_dump_type = prefs->stat.core_dump_type;

    /* MOVED FROM BOOTSTRAP_INIT.
     * Solaris has its head up its ass. It won't let you register signal
     * handlers before main when using pthreads. Go figgure.
     */
#ifdef HAVE_POSIX_SIGACTION
	signals->libc_sigaction(SIGQUIT, &act, NULL);
	signals->libc_sigaction(SIGABRT, &act, NULL);
	signals->libc_sigaction(SIGINT, &act, NULL);
	signals->libc_sigaction(SIGTERM, &act, NULL);
	signals->libc_sigaction(SIGUSR1, &act, NULL);

	/* If hardcore dumping, don't handle signals */
	if(signals->core_dump_type != NJ_DUMP_HARD_CORE)
	{
		signals->libc_sigaction(SIGILL, &act, NULL);
		signals->libc_sigaction(SIGFPE, &act, NULL);
		signals->libc_sigaction(SIGSEGV, &act, NULL);
		signals->libc_sigaction(SIGBUS, &act, NULL);
	}
#else
	signals->libc_signal(SIGQUIT, __nj_public_signals_dispatch);
	signals->libc_signal(SIGABRT, __nj_public_signals_dispatch);
	signals->libc_signal(SIGINT, __nj_public_signals_dispatch);
	signals->libc_signal(SIGTERM, __nj_public_signals_dispatch);
	signals->libc_signal(SIGUSR1, __nj_public_signals_dispatch);
	if(signals->core_dump_type != NJ_DUMP_HARD_CORE)
	{
		signals->libc_signal(SIGILL, __nj_public_signals_dispatch);
		signals->libc_signal(SIGFPE, __nj_public_signals_dispatch);
		signals->libc_signal(SIGBUS, __nj_public_signals_dispatch);
		signals->libc_signal(SIGSEGV, __nj_public_signals_dispatch);
	}
#endif
}

/**
 * Does nothing 
 */
void __nj_signals_fini(struct nj_signals *signals)
{
	return;
}


/**
 * Displays a string version of the sigaction si_code to the NJAMD error output.
 * 
 * @TODO Move me to output object.
 */
#ifdef HAVE_POSIX_SIGACTION
static void sigaction_perror(char *str, int sig, int si_code)
{
	switch(sig)
	{
		case SIGILL:	
# ifdef ILL_ILLOPC
			switch(si_code)
			{
				case ILL_ILLOPC: 
					__nj_eprintf("%sIllegal opcode\n", str); 
					break;
				case ILL_ILLOPN: 
					__nj_eprintf("%sIllegal operand\n", str); 
					break;
				case ILL_ILLADR: 
					__nj_eprintf("%sIllegal addressing mode\n", str); 
					break;
				case ILL_ILLTRP: 
					__nj_eprintf("%sIllegal trap\n", str); 
					break;
				case ILL_PRVOPC: 
					__nj_eprintf("%sPrivileged opcode\n", str); 
					break;
				case ILL_PRVREG: 
					__nj_eprintf("%sPrivileged register\n", str); 
					break;
				case ILL_COPROC: 
					__nj_eprintf("%sCopressesor error\n", str); 
					break;
				case ILL_BADSTK: 
					__nj_eprintf("%sInternal stack error\n", str); 
					break;
				default: 
					__nj_eprintf("%sUnknown code %d (sig %d). See sigaction(2)\n", str, si_code, sig);
			}
# endif
			break;
		case SIGFPE:
# ifdef FPE_INTDIV
			switch(si_code)
			{
				case FPE_INTDIV: 
					__nj_eprintf("%sInteger divide by zero\n", str); 
					break;
				case FPE_INTOVF: 
					__nj_eprintf("%sInteger overflow\n", str); 
					break;
				case FPE_FLTDIV: 
					__nj_eprintf("%sFloating point divide by zero\n", str); 
					break;
				case FPE_FLTOVF: 
					__nj_eprintf("%sFloating point overflow\n", str); 
					break;
				case FPE_FLTUND: 
					__nj_eprintf("%sFloating point underflow\n", str); 
					break;
				case FPE_FLTRES: 
					__nj_eprintf("%sFloating point inexact result\n", str); 
					break;
				case FPE_FLTINV: 
					__nj_eprintf("%sFloating point invalid operation\n", str); 
					break;
				case FPE_FLTSUB: 
					__nj_eprintf("%sSubscript out of range\n", str); 
					break;
				default: 
					__nj_eprintf("%sUnknown code %d (sig %d). See sigaction(2)\n", str, si_code, sig);
			}
# endif
			break;
		case SIGSEGV:
# ifdef SEGV_MAPERR
			switch(si_code)
			{
				case SEGV_MAPERR: 
					__nj_eprintf("%sAddress not mapped\n", str); 
					break;
				case SEGV_ACCERR: 
					__nj_eprintf("%sAccess to protected region\n", str); 
					break;
				default: 
					__nj_eprintf("%sUnknown code %d (sig %d). See sigaction(2)\n", str, si_code, sig);
			}
# endif
			break;

		case SIGBUS:
# ifdef BUS_ADRALN
			switch(si_code)
			{
				case BUS_ADRALN: 
					__nj_eprintf("%sInvalid address alignment\n", str); 
					break;
				case BUS_ADRERR: 
					__nj_eprintf("%sNon-existant physical address\n", str); 
					break;
				case BUS_OBJERR: 
					__nj_eprintf("%sObject specific hardware error\n", str); 
					break;
				default: 
					__nj_eprintf("%sUnknown code %d (sig %d). See sigaction(2)\n", str, si_code, sig); 
			}
# endif
			break;

		default: 
			__nj_eprintf("%sUnknown code %d (sig %d). See sigaction(2)\n", str, si_code, sig); 
			break;
	}
}
#endif /* POSIX_SIGACTION */

/**
 * Handles dispatching of various signals, deadly and nondeadly.
 *
 * NJAMD installs its own signal handlers for many signals. This function acts
 * as a relay point so that user's signal handlers still get called after the
 * necessary NJAMD action is taken. Its prototype is either that of a
 * sigaction handler or signal handler, depending on which is available.
 * 
 * @sideeffects Depending on the signal, either leaks are dumped (USR1), core
 * is dumped, and/or program termination
 *
 * @TODO Use the act field here to handle user flags on sigaction
 * @TODO Use the output object
 */
#ifdef HAVE_POSIX_SIGACTION
void __nj_signals_dispatch(struct nj_signals *signals, int sig, 
		siginfo_t *info, void *arg)
#else
void  __nj_signals_dispatch(struct nj_signals *signals, int sig)
#endif
{

	/* Print out messages */
	switch(sig)
	{
		case SIGSEGV:
			__nj_eprintf("\nSegmentation fault (caught by NJAMD)\n");
			break;
		case SIGBUS:
			__nj_eprintf("\nBus Error (caught by NJAMD)\n");
			break;
		case SIGFPE:
			__nj_eprintf("\nFloating Point (caught by NJAMD)\n");
			break;
		case SIGILL:
			__nj_eprintf("\nIllegal Instruction (caught by NJAMD)\n");
			break;
	}


	switch(sig)
	{
		case SIGUSR1:
			/** @TODO Use output object */
	//		__nj_dump_leaks(TRACE_DEPTH);
			if(signals->handler[sig] != SIG_DFL)
				signals->handler[sig](sig);
#if HAVE_POSIX_SIGACTION
			else if(signals->sigaction[sig] != NULL)
				signals->sigaction[sig](sig, info, arg);
#endif
			break;
		/* Handle all the death signals */
		case SIGTERM:
		case SIGINT:
		case SIGQUIT:
		case SIGABRT:
			if(signals->handler[sig] != SIG_DFL)
				signals->handler[sig](sig);
#if HAVE_POSIX_SIGACTION
			else if(signals->sigaction[sig] != NULL)
				signals->sigaction[sig](sig, info, arg);
#endif
			else
			{
				__nj_njamd_fini();

				/* If we're aborting, we want a core file */
				if(sig != SIGABRT)
					exit(sig);
			}
			break;
		case SIGILL:
		case SIGFPE:
		case SIGBUS:
		case SIGSEGV:
			__nj_callstack_dump();
#if HAVE_POSIX_SIGACTION
			/* If the cause of the fault was not a raise(), print the real
			 * cause */
			if(info->si_code != SI_USER)
			{
				sigaction_perror("\nCause of fault: ", sig, info->si_code);
				__nj_eprintf("Address of fault is 0x%x\n", info->si_addr);
			}
			/* TODO: if we ever allow the option of saving the state of all 
			 * allocs (CHK_FREE=infonly), then we will also wanna check the
			 * database if si_code = SEGV_MAPERR */
# ifdef SEGV_ACCERR
			if(info->si_code == SEGV_ACCERR)
# endif
				__nj_ptr_info((u_long)info->si_addr);
#endif

			if(signals->core_dump_type == NJ_DUMP_HARD_CORE)
			{
				__nj_eprintf("NJAMD: Error, signal still handled in hardcore mode. mail mikepery@fscked.org\n");
			}

			__nj_njamd_fini();
			if(signals->handler[sig] != SIG_DFL)
				signals->handler[sig](sig);
#if HAVE_POSIX_SIGACTION
			else if(signals->sigaction[sig] != NULL)
				signals->sigaction[sig](sig, info, arg);
#endif
			else
			{
				if(signals->core_dump_type == NJ_DUMP_SOFT_CORE)
					abort();
				else
					exit(sig);
			}
			break;
		default:
			__nj_eprintf("NJAMD: Error, catching sig %d is not our job\n", sig);
			return;
	}

#ifndef HAVE_POSIX_SIGACTION
	signal(sig, __nj_public_signals_dispatch);
#endif
}

/**
 * Register a signal for the user
 *
 * We need to wrap certain signals, so we will call the users handler for them 
 * when we handle a signal
 * @param signals The signals object
 * @param num the sig number
 * @param newhandler The new signal handler
 * @returns The old signal handler
 */
nj_sighandler_t __nj_signals_register_user_signal(struct nj_signals *signals, 
		int num, nj_sighandler_t newhandler)
{
	nj_sighandler_t ret;

	/* If we dump hardcore, we don't handle segv */
	if(signals->core_dump_type == NJ_DUMP_HARD_CORE)
	{
		switch(num)
		{
			case SIGQUIT:
			case SIGABRT:
			case SIGUSR1:
			case SIGINT:
			case SIGTERM:
				/* Look up ret in the handler array, save the new one, and
				 * declaire no sigaction for it. return ret */
				/** @TODO One day do EVERYTHING through the sigaction act
				 * field */
				ret = signals->handler[num];
				signals->handler[num] = newhandler;
				signals->sigaction[num] = NULL;
				return ret;
		}

	}
	else
	{
		switch(num)
		{
			case SIGQUIT:
			case SIGILL:
			case SIGABRT:
			case SIGFPE:
			case SIGUSR1:
			case SIGSEGV:
			case SIGBUS:
			case SIGINT:
			case SIGTERM:
				/* Look up ret in the handler array, save the new one, and
				 * declaire no sigaction for it. return ret */
				ret = signals->handler[num];
				signals->handler[num] = newhandler;
				signals->sigaction[num] = NULL;
				return ret;
		}
	}

	TR;

	/* If we are still here, we don't care about the signal. Just let the 
	 * kernel handle it */
	return signals->libc_signal(num, newhandler);
}

/**
 * Register a sigaction for the user
 *
 * We need to wrap certain signals, so we will call the users handler for them 
 * when we handle a signal
 * @param signals The signals object
 * @param num the sig number
 * @param new the new sigaction
 * @param old the old sigaction
 * @returns Error value
 */
int __nj_signals_register_user_sigaction(struct nj_signals *signals, 
		int num, const struct sigaction *new, struct sigaction *old)
{
	/* Handle deadly signals internally, so we always exit cleanly and with
	 * status messages, if the user wishes */
	if(signals->core_dump_type == NJ_DUMP_HARD_CORE)
	{
		switch(num)
		{
			case SIGUSR1:
			case SIGQUIT:
			case SIGABRT:
			case SIGINT:
			case SIGTERM:
				TR;
				/** @TODO Use the act field here */
				if(old)
				{
					old->sa_handler = signals->handler[num];
					old->sa_sigaction = signals->sigaction[num];	
				}
				if(new)
				{
					if(new->sa_flags & SA_SIGINFO)
					{
						signals->sigaction[num] = new->sa_sigaction;
						signals->handler[num] = SIG_DFL;
					}
					else
					{
						signals->handler[num] = new->sa_handler;
						signals->sigaction[num] = NULL;
					}
				}
				return 0;
		}
	}
	else
	{
		switch(num)
		{
			case SIGUSR1:
			case SIGILL:
			case SIGFPE:
			case SIGQUIT:
			case SIGABRT:
			case SIGSEGV:
			case SIGBUS:
			case SIGINT:
			case SIGTERM:
				TR;
				if(old)
				{
					old->sa_handler = signals->handler[num];
					old->sa_sigaction = signals->sigaction[num];
				}
				if(new)
				{
					if(new->sa_flags & SA_SIGINFO)
					{
						signals->sigaction[num] = new->sa_sigaction;
						signals->handler[num] = SIG_DFL;
					}
					else
					{
						signals->handler[num] = new->sa_handler;
						signals->sigaction[num] = NULL;
					}
				}
				return 0;
		}
	}
	TR;

	return signals->libc_sigaction(num, new, old);

}
// vim:ts=4
