/** 
 * @file public.c
 * 
 * Provides the libc interface into NJAMD.
 *
 * This is how we override malloc, realloc and calloc. The functions in NJAMD
 * take precedence over libc when preloaded.
 *
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 * 
 */

/* Avoid conflicting types */
#define pthread_create __nj_dummy1 
#define pthread_exit __nj_dummy2 
#define signal __nj_dummy4
#define exit __nj_dummy5
#include <lib/njamd.h>

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <string.h>

#undef exit
#undef signal
#undef pthread_exit
#undef pthread_create

/**
 * Conditional init function
 *
 * Checks the initial state of NJAMD, calling the proper init functions
 * if needed. It's called each time we enter public.c
 *
 * Also marks the start of public entrace to NJAMD, so DO NOT MOVE
 * (Used in callstack.c)
 */
void __nj_public_init()
{
	if(__NJAMD__.state < NJ_STATE_BOOTSTRAPPED)
		__nj_njamd_bootstrap_init(&__NJAMD__);

#ifndef HAVE_ATTRIB_CONSTRUCT
	/* If we haven't crossed main, but callstack says we have, run the user init */
	if(__NJAMD__.state < NJ_STATE_CROSSED_MAIN && __nj_callstack_crossed_main())
		 __NJAMD__.state = NJ_STATE_CROSSED_MAIN;
#endif

	if(__NJAMD__.state == NJ_STATE_CROSSED_MAIN) 
		__nj_njamd_user_init(&__NJAMD__);

	return;
}




#ifdef HAVE_POSIX_SIGACTION
void __nj_public_signals_dispatch(int sig, siginfo_t *info, void *arg)
{
	__nj_signals_dispatch(&__NJAMD__.signals, sig, info, arg);
}
#else
void  __nj_public_signals_dispatch(int sig)
{
	__nj_signals_dispatch(&__NJAMD__.signals, sig);
}
#endif

#ifdef HAVE_SANE_DLOPEN
/**
 * Wraps libc signal, makes sure the user doesn't overwrite singals NJAMD
 * needs to handle. 
 *
 * NJAMD needs to catch segv, bus error and other deadly signals to clean up
 * properly and display some info to the user, as well as USR1 to dump leaks. 
 * So we wrap signal to save the handler the user wants to execute, and then
 * when the signal actually occurs we just call their functions.
 *
 * @param num signal number
 * @param newhandler New signal handler function
 * @returns The previous signal handler the user may have registered.
 * @sideeffects Replaces the signal table entry at sig with the newhandler.
 *              More importantly, the #__nj_sig_actn action table entry is set
 *              to NULL for this signal.  Also runs the init functions if they
 *              haven't already.
 * @see __nj_sig_dispatch()
 */
nj_sighandler_t signal(int num, nj_sighandler_t newhandler)
{
	__nj_public_init();

	return __nj_signals_register_user_signal(&__NJAMD__.signals, num, newhandler);
}

# ifdef HAVE_POSIX_SIGACTION
/**
 * Wraps libc sigaction, makes sure the user doesn't overwrite singals NJAMD
 * needs to handle. 
 *
 * NJAMD needs to catch segv, bus error and other deadly signals to clean up
 * properly and display some info to the user, as well as USR1 to dump leaks. 
 * So we wrap signal to save the handler the user wants to execute, and then
 * when the signal actually occurs we just call their functions.
 *
 * @param num signal number
 * @param new New sigaction descriptor.
 * @param old Old sigaction descriptor is stored here if non-null
 * @returns -1 On error.
 * @sideeffects Replaces the signal table entry at sig with the new action.
 *              More importantly, the __nj_sig_hdlr table entry is set
 *              to NULL for this signal if we are using SA_SIGINFO.
 *              Also runs the init functions if they haven't already.
 * @see __nj_sig_dispatch()
 */
int sigaction(int num, const struct sigaction *new, struct sigaction *old)
{
	__nj_public_init();
	
	return __nj_signals_register_user_sigaction(&__NJAMD__.signals, num, new, old);
}
# endif /* SIGACTION */

# ifdef _THREAD_SAFE

/**   
 * We wrap pthread_create so we KNOW that __nj_primary_init() will run
 * before any threads exist. Also, in some threads implementations, the bottom
 * of the stack isn't well defined (ie, it's just some garbage pointer). So we
 * need to start all threads from a known function, and then stop backtrace at
 * that function (__nj_pthread_launch())
 * 
 * @par Paramaters: Those of pthread_create(3)
 * @sideeffects Calls __nj_primary_init() if not called already. This way we know
 * 				that all init functions run before we need mutexes to protect
 * 				against race conditions.
 */
int  pthread_create(pthread_t  *  thread, pthread_attr_t *attr,
		void * (*start_routine)(void *), void * arg)
{
	__nj_public_init();

	return __nj_threads_create_thread(&__NJAMD__.threads, thread, attr,
		start_routine, arg);
}


/** 
 * Food for the dynamic linker. Do NOT move. 
 * Used to determine the start of pthread_exit so that we can tell if a
 * descructor was called during exit of a thread. 
 */
void __nj_pthread_exit_start()
{
	return;
}

/** 
 * Simply wraps pthread_exit. We need to be able to find the end of the stack
 * if a destructor was called.
 */
void pthread_exit(void *retval)
{
	/* Pthread exit damn well better not be called before create */
	__NJAMD__.threads.libc_pthread_exit(retval);

}
# endif /* THREADSAFE */
#endif /* DLOPEN */

/** Do NOT move this from pthread_exit. Used to determine it's end value */
void __nj_pthread_exit_end(void)
{
	return;
}

#ifdef HAVE_SANE_DLOPEN

/** Food for the dynamic linker. Do NOT move. Same thing. Used to determine
 * the end of the stack for destructors */
void __nj_exit_start(void)
{
	return;
}

/**
 * Wrapped exit, used just to determine the end of stack when free is called
 * from a destructor. 
 */
void exit(int ret)
{
	__nj_public_init();

	__NJAMD__.libc_exit(ret);
}
#endif

/** Do NOT move this from exit() */
void __nj_exit_end(void)
{
	return;
}

/**@{ @name Public NJAMD intercace functions */

/**
 * For backwards compatibility. Calls __nj_eentry_pool_print_by_ptr 
 *
 * @param ptr The ptr to use
 */
void __nj_ptr_info(nj_addr_t ptr)
{
	__nj_entry_pool_print_by_addr(&__NJAMD__.allocator.entry_pool, ptr);
}

/** Prints all outstanding memory leaks to the screen */ 
void __nj_dump_leaks(void)
{
	__nj_entry_pool_dump_leaks(&__NJAMD__.allocator.entry_pool);
}

/**
 * Allows you to set all dynamic options at once.
 * 
 * @param alloc_type One of the NJ_PROT_XXXX's
 * @param free_type One of the NJ_CHK_FREE_XXXX's
 * @param info_on_free True or false
 * @param trace_libs True or false
 * @param align Alignment, must be a power of two.
 */
void __nj_set_options(int alloc_type, int free_type, int info_on_free, 
		int trace_libs, int align)
{
	struct nj_dynamic_prefs prefs;

	/* Yes, it IS faster to set these here and then pass them as a unit, 
	 * for those of you who nitpick... prefs is only a single int large */
	prefs.trace_libs = trace_libs;
	prefs.info_on_free = info_on_free;
	if(!__NJAMD__.prefs.stat.mutable_alloc && prefs.alloc_type != __NJAMD__.prefs.dyn.alloc_type)
		__nj_eprintf("NJAMD: You must set NJAMD_MUTABLE_ALLOC to change the protection type\n");
	else
#ifndef HAVE_SANE_DLOPEN
		/** @FIXME use output object */
		if(alloc_type == NJ_PROT_NONE)
			__nj_eprintf("NJAMD: You have no working dlopen on this sys, you can't use PROT_NONE\n");
		else	
			prefs.alloc_type = alloc_type;
#else
	prefs.alloc_type = alloc_type;
#endif

	if(NJ_IS_POWER_OF_TWO(align))
	{
		prefs.align = align;
	}
	else
		__nj_eprintf("NJAMD: Alignment must be a power of two!");

	prefs.free_type = free_type;

	__nj_prefs_set(&__NJAMD__.prefs, prefs);
}

/**
 * changes the prot type
 *
 * @param new_type the new protection type, in the same format as the environment variable
 */
void __nj_change_prot_type(char *new_type)
{
	struct nj_dynamic_prefs new_prefs = __nj_prefs_get(&__NJAMD__.prefs);

	if(!__NJAMD__.prefs.stat.mutable_alloc)
	{
		__nj_eprintf("You must set NJAMD_MUTABLE_ALLOC=1 to change the alloc type at runtime.\n");
		return;
	}
	
	if(strncmp(new_type, "over",4) == 0)
		new_prefs.alloc_type = NJ_PROT_OVER;
	else if(strncmp(new_type, "under", 5) == 0)
		new_prefs.alloc_type = NJ_PROT_UNDER;
	else if(strncmp(new_type, "strict", 6) == 0)
		new_prefs.alloc_type = NJ_PROT_SUNDER;
	else if(strncmp(new_type, "none", 4) == 0)
	{
#ifndef HAVE_SANE_DLOPEN
		__nj_eprintf("NJAMD: Error, you don't have a working dlopen on this sys.\nYou can't chose PROT=NONE\n");
		return;
#endif
		new_prefs.alloc_type = NJ_PROT_NONE;
	}
	else
		__nj_eprintf("NJAMD: Invalid malloc checking: %s\n", new_type);

	__nj_prefs_set(&__NJAMD__.prefs, new_prefs);
}

/**
 * changes the free checking
 *
 * @param new_type the new checking type, in the same format as the environment variable
 */
void __nj_change_chk_free_type(char *new_type)
{
	struct nj_dynamic_prefs new_prefs = __nj_prefs_get(&__NJAMD__.prefs);
	
	if(strcmp(new_type, "error") == 0)
		new_prefs.free_type = NJ_CHK_FREE_ERROR;
	else if(strcmp(new_type, "none") == 0)
		new_prefs.free_type = NJ_CHK_FREE_NONE;
	else if(strcmp(new_type, "segv") == 0) 
		new_prefs.free_type = NJ_CHK_FREE_SEGV;
	else if(strcmp(new_type, "nofree") == 0)
		new_prefs.free_type = NJ_CHK_FREE_NOFREE;
	else
		__nj_eprintf("NJAMD: Invalid free checking: %s\n", new_type);
	
	__nj_prefs_set(&__NJAMD__.prefs, new_prefs);
}

/**
 * Changes the default alignment
 *
 * @param new_align The new alignment
 */
void __nj_change_default_align(int new_align)
{
	struct nj_dynamic_prefs new_prefs = __nj_prefs_get(&__NJAMD__.prefs);
	
	if(NJ_IS_POWER_OF_TWO(new_align))
	{
		new_prefs.align = new_align;
	}
	else
		__nj_eprintf("NJAMD: Alignment must be a power of two!");
	
	__nj_prefs_set(&__NJAMD__.prefs, new_prefs);
	
}

/**
 * Changes the trace libs setting
 *
 * @param val True if you want to have library information in callstacks
 * 
 */
void __nj_change_trace_libs(int val)
{
	struct nj_dynamic_prefs new_prefs = __nj_prefs_get(&__NJAMD__.prefs);
	
	new_prefs.trace_libs = val;
	
	__nj_prefs_set(&__NJAMD__.prefs, new_prefs);
}

/**
 * Changes the free info setting
 *
 * @param val True if you want to keep information on freed segments
 * 
 */
void __nj_change_free_info(int val)
{
	struct nj_dynamic_prefs new_prefs = __nj_prefs_get(&__NJAMD__.prefs);
	
	if(__NJAMD__.prefs.stat.callstack_depth)
	{
		__nj_eprintf("NJAMD: Error, in order to set no_free_info you must chose a fixed callstack depth (NJ_CALLSTACK_DEPTH)\n");
		return;
	}
	else
		new_prefs.info_on_free = val;

	__nj_prefs_set(&__NJAMD__.prefs, new_prefs);
}
/*@}*/

/**
 * The main library entry point. It calls the
 * appropriate alloc via the #__nj_known_alloc pointer, if we have run
 * __nj_secondary_init(). If secondary init has NOT run, use the default preinit
 * allocator.
 *
 * @param size The size to allocate
 * @returns The allocated segment
 * @sideeffects Calls the init functions if they haven't been run, and it's
 * 				time (ie __nj_secondary_init()) is only called if main has
 * 				been entered).
 * @see __nj_none_alloc() __nj_overflow_alloc() __nj_sunderflow_alloc() __nj_underflow_alloc()
 */ 
void *malloc(size_t size)
{
	__nj_public_init();

	return (void *)__nj_allocator_request_user_chunk(&__NJAMD__.allocator, size,
			__nj_prefs_get(&__NJAMD__.prefs));
}

/**
 * Implements valloc, which simply calls malloc.
 *
 * @param size The size to allocate
 * @returns The allocated segment
 * @sideeffects Calls the init functions if they haven't been run, and it's
 * 				time (ie __nj_secondary_init() is only called if main has been entered).
 * @see __nj_none_alloc() __nj_overflow_alloc() __nj_sunderflow_alloc() __nj_underflow_alloc()
 */ 
void *valloc(size_t size)
{
	__nj_public_init();

	return (void *)__nj_allocator_request_user_chunk(&__NJAMD__.allocator, size,
			__nj_prefs_get(&__NJAMD__.prefs));
}

/**
 * Implements memalign, to returned aligned memory. It calls the
 * appropriate alloc via the __nj_known_alloc pointer, if we have run
 * __nj_secondary_init(). If secondary init has NOT run, use the default preinit
 * allocator. The only difference between this and malloc is that the
 * alignment is passed to the allocation function
 * 
 * @param alignment The requested alignment
 * @param size The size to allocate
 * @returns The allocated segment
 * @sideeffects Calls the init functions if they haven't been run, and it's
 * 				time (ie secondary_init is only called if main has been
 * 				entered).
 * @see __nj_none_alloc() __nj_overflow_alloc() __nj_sunderflow_alloc() __nj_underflow_alloc()
 */
void *memalign(size_t alignment, size_t size)
{
	struct nj_dynamic_prefs prefs;

	__nj_public_init();

	if(!NJ_IS_POWER_OF_TWO(alignment))
	{
		__nj_eprintf("NJAMD/memalign: Alignment %d is not a power of two!\n", alignment);
		__nj_callstack_dump();
		return NULL;
	}

	prefs = __nj_prefs_get(&__NJAMD__.prefs);
	prefs.align = alignment;

	return (void *)__nj_allocator_request_user_chunk(&__NJAMD__.allocator, size,
			prefs);
}

/**
 * Implements posix_memalign, to returned aligned memory. It calls the
 * appropriate alloc via the __nj_known_alloc pointer, if we have run
 * __nj_secondary_init(). If secondary init has NOT run, use the default preinit
 * allocator. The only difference between this and realloc is that the
 * alignment is passed to the allocation function.
 * 
 * @param alignment The requested alignment 
 * @param size The size to allocate
 * @returns I have no idea. Does anyone have a POSIX spec handy?
 * @sideeffects Calls the init functions if they haven't been run, and it's
 * 				time (ie secondary_init is only called if main has been
 * 				entered).
 * @see __nj_none_alloc() __nj_overflow_alloc() __nj_sunderflow_alloc() __nj_underflow_alloc()
 */
int posix_memalign(void **buf, size_t alignment, size_t size)
{
	struct nj_dynamic_prefs prefs;

	__nj_public_init();

	TR;	
	if(!buf)
	{
		__nj_eprintf("NJAMD/posix_memalign: NULL pointer passed\n");
		return -1;
	}
	
	/** @FIXME Use output object */
	if(!NJ_IS_POWER_OF_TWO(alignment))
	{
		__nj_eprintf("NJAMD/memalign: Alignment %d is not a power of two!\n", alignment);
		__nj_callstack_dump();
		return -1;
	}

	prefs = __nj_prefs_get(&__NJAMD__.prefs);
	prefs.align = alignment;

	if(*buf)
		*buf = (void *)__nj_allocator_resize_user_chunk(&__NJAMD__.allocator, 
				(nj_addr_t)*buf, size, prefs);
	else
		*buf =  (void *)__nj_allocator_request_user_chunk(&__NJAMD__.allocator, size,
				            prefs);
	

	/** @FIXME will the allocator only return on success? does it die alwyas on failure? */
	return 0;
}


/**
 * The main library entry point. It calls the
 * appropriate alloc via the #__nj_known_alloc pointer, if we have run
 * __nj_secondary_init(). If secondary init has NOT run, use the default preinit
 * allocator. The only difference between this and malloc is that zero_fill is
 * set to true.
 *
 * @param size The size to allocate
 * @returns The allocated segment
 * @sideeffects Calls the init functions if they haven't been run, and it's
 * 				time (ie secondary_init is only called if main has been entered).
 * @see __nj_none_alloc() __nj_overflow_alloc() __nj_sunderflow_alloc() __nj_underflow_alloc()
 */ 
void *calloc(size_t nmemb, size_t size)
{
	static char zeroes[256];
	int i;
	void *buf;

	__nj_public_init();

	/** FIXME Consider using libc_calloc for prot_none.. but calloc probably
	    also just does a memset.. so I'm saying screw it for now */
	buf = (void *)__nj_allocator_request_user_chunk(&__NJAMD__.allocator, nmemb*size,
			__nj_prefs_get(&__NJAMD__.prefs));

	for(i=0; i < nmemb*size / sizeof(zeroes); i++)
		memcpy(buf, zeroes, sizeof(zeroes));

	memcpy(buf, zeroes, nmemb*size % sizeof(zeroes));

	return buf;
}

/**
 * Implements realloc, calling free if size is 0. It calls the
 * appropriate alloc via the __nj_known_alloc pointer, if we have run
 * __nj_secondary_init(). If secondary init has NOT run, use the default preinit
 * allocator. The only difference between this and malloc is that the ptr is
 * passed to the allocation function
 * 
 * @param ptr the previously allocated function.
 * @param size The size to allocate
 * @returns The allocated segment
 * @sideeffects Calls the init functions if they haven't been run, and it's
 * 				time (ie secondary_init is only called if main has been
 * 				entered).
 * @see __nj_none_alloc() __nj_overflow_alloc() __nj_sunderflow_alloc() __nj_underflow_alloc()
 */
void *realloc(void *ptr, size_t size)
{
	__nj_public_init();	

	if(!ptr)
		return (void *)__nj_allocator_request_user_chunk(&__NJAMD__.allocator, size,
				            __nj_prefs_get(&__NJAMD__.prefs));

	if(!size)
	{
		if(!ptr)
		{
			/** FIXME Use legit error codes, etc */
			if(!__NJAMD__.prefs.stat.allow_free_0)
				__nj_output_user_warning(0,0,0,NULL, "NJAMD: Free of NULL!\n");
			return;
		}

		__nj_allocator_release_user_chunk(&__NJAMD__.allocator, (nj_addr_t)ptr, __nj_prefs_get(&__NJAMD__.prefs));
	}

	return (void *)__nj_allocator_resize_user_chunk(&__NJAMD__.allocator, 
			(nj_addr_t)ptr, size, __nj_prefs_get(&__NJAMD__.prefs));
}


/**
 * Implements libc free. We keep track of the region of memory allocated
 * with an unknown allocator by looking at the slab sections set up in
 * __nj_secondary_init().
 *
 * @param ptr The pointer to free.
 * @see __nj_none_free() __nj_overflow_free() __nj_sunderflow_free() __nj_underflow_free()
 */
void free(void *ptr)
{
	__nj_public_init();

	if(!ptr)
	{
		/** FIXME Use legit error codes, etc */
		if(!__NJAMD__.prefs.stat.allow_free_0)
			__nj_output_user_warning(0,0,0,NULL, "NJAMD: Free of NULL!\n");
		return;
	}

	__nj_allocator_release_user_chunk(&__NJAMD__.allocator, (nj_addr_t)ptr, __nj_prefs_get(&__NJAMD__.prefs));
}

/**
 * Marks the end of public entrace to NJAMD
 *
 * Used in callstack.c
 */
void __nj_public_end()
{
	return;
}

// vim:ts=4
