/** 
 * @file entry_pool.c
 * Contains heap entry allocation functions
 *
 * The entry pool is where heap entries for individual malloced segments come 
 * from. The storage is persistant in the heap file. When you request a heap 
 * entry, you actually get an index into a table of entires.
 *
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 */
#include <lib/entry_pool.h>
#include <lib/allocator.h>
#include <lib/portability.h>
#include <lib/output.h>
#include <lib/block.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

static void heap_entry_init(struct nj_heap_entry *, nj_addr_t, size_t, 
		struct nj_callstack, struct nj_dynamic_prefs);
static void heap_entry_init(struct nj_heap_entry *, nj_addr_t, size_t, 
		struct nj_callstack, struct nj_dynamic_prefs);
static int heap_entry_has_addr(struct nj_heap_entry *, nj_addr_t);

/**
 * Init the heap entry pool.
 *
 * @param entry_pool The pool to init
 * @param file The file behind the pool
 */ 
void __nj_entry_pool_bootstrap_init(struct nj_entry_pool *entry_pool)
{
	__nj_callstack_pool_bootstrap_init(&entry_pool->cs_pool);
	/* We need a NEW heapfd for this.. we can't just expand the old one
	 * Even though errno should be valid by now, we want to keep the format
	 * consistant */
#ifdef HAVE_WORKING_ERRNO 
	snprintf(entry_pool->file, sizeof(entry_pool->file), "./njamd-%d-heap", 
			getpid());
#else
	strncpy(entry_pool->file, "./njamd-heap", sizeof(entry_pool->file));
#endif

	if(__nj_table_bootstrap_init(&entry_pool->entry_table, entry_pool->file, 
				NJ_ENTRY_POOL_INIT_SIZE, 1, 1))
		__nj_critical_error(__FUNCTION__);
	__nj_stack_bootstrap_init(&entry_pool->free_list);
}

/**
 * Init the entry pool with user prefs
 *
 * @param entry_pool The entry pool
 * @param Prefs The initialized user prefs object.
 */
void __nj_entry_pool_user_init(struct nj_entry_pool *entry_pool,
		struct nj_prefs *prefs)
{
	entry_pool->dump_leaks = prefs->stat.dump_leaks;
	__nj_callstack_pool_user_init(&entry_pool->cs_pool, prefs);

	__nj_table_user_init(&entry_pool->entry_table, prefs);
	__nj_stack_user_init(&entry_pool->free_list);
}

/**
 * Cleans up the heap pool and writes it to disk.
 *
 * @param entry_pool
 */
void __nj_entry_pool_fini(struct nj_entry_pool *entry_pool)
{
	/** @FIXME Stats get their own file.. or do they? Either way, they're
	 * not gonna be here for the initial release, so screw it for now. */
#if 0
	struct nj_stats_light *stats = __nj_table_request_chunk(&entry_pool->entry_table, sizeof(struct nj_stats_light));
	*stats = (struct nj_stats_light *)stats;
#endif

	if(entry_pool->dump_leaks)
		__nj_entry_pool_dump_leaks(entry_pool);

	__nj_callstack_pool_fini(&entry_pool->cs_pool);
	__nj_table_fini(&entry_pool->entry_table);
	__nj_stack_fini(&entry_pool->free_list);
}

/**
 * Creates a new heap entry describing a memory segment.
 *
 * @param entry_pool The pool of heap entries
 * @param start The start of the region, as returned from mmap.
 * @param len The length of the user's portion of the segment
 * @param alloced A return address callstack recorded in malloc() and friends.
 * @returns A brand new heap entry index filled with the given info.
 * @sideeffects May switch __nj_heap_tbl if it is full.
 *
 * @see resize_heap() __nj_entry_pool_index_init()
 */
nj_entry_index_t __nj_entry_pool_request_index(struct nj_entry_pool *entry_pool)
{
	nj_entry_index_t idx;
	struct nj_stack_item *item;
	struct nj_heap_entry *entry;

	TR;
	/* Assume that it's the right length for now. We're only gonna worry 
	 * about fixed length callstacks if they wanna free them. */
	if((item = __nj_stack_pop(&entry_pool->free_list)))
	{
		idx = (nj_entry_index_t)item->data;
		/* Shortcut: The stack item is in fact the beginning of the heap_entry */
		entry = (struct nj_heap_entry *)item;
	}
	else
	{
		entry = (struct nj_heap_entry *)__nj_table_get_chunk(&entry_pool->entry_table, sizeof(struct nj_heap_entry));
		/* The table stores the index inside the chunk */
		idx = *(u_int *)entry/sizeof(struct nj_heap_entry);
	}
	
	entry->alloced.index = NJ_CALLSTACK_INDEX_NONE;
	entry->freed.index = NJ_CALLSTACK_INDEX_NOTFREE;
	
	TR;
	return idx;
}


/**
 * Return an index to the entry pool
 *
 * @param entry_pool The pool to return to
 * @param idx The index to return
 */ 
static void entry_pool_release_index(struct nj_entry_pool *entry_pool, 
		nj_entry_index_t idx)
{
	struct nj_heap_entry *entry = NJ_ENTRY_POOL_INDEX_TO_ENTRY(*entry_pool, idx);
	struct nj_stack_item *item;

	__nj_callstack_pool_release_index(&entry_pool->cs_pool, entry->alloced);
	__nj_callstack_pool_release_index(&entry_pool->cs_pool, entry->freed);
	
	memset(entry, 0, sizeof(*entry));
	
	/* Mark the entry as out of use */
	entry->alloced.index = NJ_CALLSTACK_INDEX_FALLOW;
	
	/* No need for a mutex here, because we won't ever race to release the 
	 * same segment unless it's a user error... hrmm */
	item = (struct nj_stack_item *)entry;

	item->data = (nj_generic_t)idx;

	__nj_stack_push(&entry_pool->free_list, item);
}

/**
 * Renew an entry pool index
 * 
 * Will either save the old heap index and return a new one, or reuse the old
 * one, depending on the info_on_free pref
 *
 * @param entry_pool The entry pool
 * @param idx The entry index
 * @param block The block
 * @param user_len The user requested allocation length
 * @param prefs The user prefrences 
 *
 * @returns The renewed index (may or may not be the same as idx)
 */
nj_entry_index_t __nj_entry_pool_renew_index(struct nj_entry_pool *entry_pool,
		nj_entry_index_t idx, nj_addr_t block, size_t user_len, struct nj_dynamic_prefs prefs)
{
	struct nj_heap_entry *entry;
	
	if(prefs.info_on_free)
	{
		/* We have to keep the indices. SO finish the old one.. */
		__nj_entry_pool_index_fini(entry_pool, idx, prefs);
		
		/* And make a new one */
		idx = __nj_entry_pool_request_index(entry_pool);
		__nj_entry_pool_index_init(entry_pool, idx, block, user_len, prefs);
	}
	else
	{
		/* Keep the old one */
		entry = NJ_ENTRY_POOL_INDEX_TO_ENTRY(*entry_pool, idx);
		__nj_callstack_pool_renew_index(&entry_pool->cs_pool, 
			entry->alloced, prefs);

		heap_entry_init(entry, block, user_len, entry->alloced, prefs);
	}
	return idx;
}

/**
 * Init an entry pool index
 *
 * @param entry_pool The entry pool
 * @param idx The index to initialize
 * @param block The start of the block from the memory pool
 * @param user_len The user's requested length
 * @param prefs The dynamic prefrences
 */
void __nj_entry_pool_index_init(struct nj_entry_pool *entry_pool,
		nj_entry_index_t idx, nj_addr_t block, size_t user_len, 
		struct nj_dynamic_prefs prefs)
{
	struct nj_heap_entry *entry = NJ_ENTRY_POOL_INDEX_TO_ENTRY(*entry_pool, idx);
	struct nj_callstack alloced;
	
	alloced = __nj_callstack_pool_request_index(&entry_pool->cs_pool, prefs);

	heap_entry_init(entry, block, user_len, alloced, prefs);
}

/**
 * 'Finish' an entry pool index
 *
 * This function will either release the index or associate it with a callstack
 * and save it, depending on user prefs
 * 
 * @param entry_pool The entry pool we are working with
 * @param idx Index to finish
 * @param prefs The user prefs
 */
void __nj_entry_pool_index_fini(struct nj_entry_pool *entry_pool,
		nj_entry_index_t idx, struct nj_dynamic_prefs prefs)
{
	struct nj_heap_entry *entry = NJ_ENTRY_POOL_INDEX_TO_ENTRY(*entry_pool, idx);

	if(prefs.info_on_free)
		entry->freed = __nj_callstack_pool_request_index(&entry_pool->cs_pool, prefs);
	else
		entry_pool_release_index(entry_pool, idx);
}

/**
 * Gets a valid entry
 *
 * Takes an index and makes sure it matches this block and user chunk.
 *
 * @param entry_pool The entry pool to check against
 * @param idx The index
 * @param block The block
 * @param user_chunk The user's chunk
 *
 * @returns A valid heap entry or NULL if idx did not match block
 */
struct nj_heap_entry *__nj_entry_pool_get_valid_entry(struct nj_entry_pool *entry_pool, 
		nj_entry_index_t idx, nj_addr_t block, nj_addr_t user_chunk)
{
	struct nj_heap_entry *entry;

	/* Check for double free.. Remember in the memory pool we map the 
	 * memory to the fencepost fd on error_chk_free? well here we are */
	if(idx == NJ_FENCEPOST)
		__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Double Free");
	
	if(NJ_ENTRY_POOL_VALID_INDEX(*entry_pool, idx))
	{
		entry = NJ_ENTRY_POOL_INDEX_TO_ENTRY(*entry_pool, idx);
		if((entry->block == block) || ((block == NJ_ALLOCATOR_BLOCK_UNKNOWN) 
				&& heap_entry_has_addr(entry, user_chunk)))
			return entry;
		else
			return NULL;
	}

	return NULL;
}

/**
 * Prints information contained in a heap entry.
 * 
 * @param entry_pool Entry pool the use
 * @param idx The entry to print
 */
void __nj_entry_pool_print_index(struct nj_entry_pool *entry_pool, nj_entry_index_t idx)
{
	struct nj_heap_entry *entry = NJ_ENTRY_POOL_INDEX_TO_ENTRY(*entry_pool, idx);
	/** FIXME: Make me use the output object */
	/** @TODO Perhaps it would be more useful to do the range as the USER's memory region */
	__nj_eprintf("\n0x%lx-0x%lx: Aligned len %d\n   Allocation callstack:\n", 
			entry->block, 
			entry->block + __nj_block_calc_size(entry->user_len, 1<<entry->align_shift, entry->alloc_type),
			entry->user_len);

	__nj_callstack_pool_print_index(&entry_pool->cs_pool, entry->alloced);
	
	if(!NJ_HEAP_ENTRY_FREED(entry))
		__nj_eprintf("   Not Freed\n");
	else
	{
		__nj_eprintf("   Freed callstack:\n");
		__nj_callstack_pool_print_index(&entry_pool->cs_pool, entry->freed);
	}
}

/**
 * A plugin function for for_all_entry_pools().
 *
 * This function takes the heap table give, and sees the addr passed in the
 * va_list is in that table. If So, return that heap entry.
 * 
 * @param entry The entry to check for the address
 * @param va The va_list that contains the address to look for
 * @returns The heap entry that describes a segment containing the address
 *
 * @see __nj_table_for_all_entries
 */ 
static int iterator_heap_entry_has_addr(struct nj_table *tbl, int idx, va_list ap)
{
	struct nj_heap_entry *entry = NJ_TABLE_INDEX_TO_PTR(*tbl, idx, struct nj_heap_entry);
	nj_addr_t addr = va_arg(ap, u_long);

	/* Make sure ulen is not zero */
	/* and if prot = none, then give the ulen, not the rounded amt */
	if(heap_entry_has_addr(entry, addr))
		return idx;
	else
		return -1;

}

/** 
 * Provides information on a malloced pointer.
 * 
 * External symbol that can be called from gdb even if you don't have
 * PERSISTANT_HEAP set 
 * 
 * @param entry_pool The entry pool to use
 * @param addr The pointer to look up and display information about.
 */
void __nj_entry_pool_print_by_addr(struct nj_entry_pool *entry_pool, nj_addr_t addr)
{
	nj_entry_index_t idx;
	
	if((idx = __nj_table_for_all_indicies(&entry_pool->entry_table, 0,
					sizeof(struct nj_heap_entry), iterator_heap_entry_has_addr, addr)) == -1)
	{
		__nj_eprintf("Address 0x%lx not found in heap\n", addr);
		return;
	}

	__nj_entry_pool_print_index(entry_pool, idx);
}

/**
 * A plugin function for for_all_entries_in_tbl()
 * 
 * @param ent The heap entry
 * @param va A va_list containing the depth of leaks to display
 * @returns NULL, since we want to iterate over all entries.
 * @see for_all_entry_pools() for_all_entries_in_tbl()
 */
static int dump_index_if_leaked(struct nj_table *tbl, int idx, va_list va)
{
	struct nj_heap_entry *entry = NJ_TABLE_INDEX_TO_PTR(*tbl, idx, struct nj_heap_entry);
	struct nj_entry_pool *entry_pool = va_arg(va, struct nj_entry_pool *);
	
	if(!NJ_HEAP_ENTRY_FREED((struct nj_heap_entry *)entry))
		__nj_entry_pool_print_index(entry_pool, idx);

	/* return null, so we don't stop */
	return -1; 
}

/**
 * Dumps all outstanding leaks to standard error
 *
 * @param entry_pool The pool to dump leaks from.
 */
void __nj_entry_pool_dump_leaks(struct nj_entry_pool *entry_pool)
{
	__nj_table_for_all_indicies(&entry_pool->entry_table, 0,
			sizeof(struct nj_heap_entry), dump_index_if_leaked, entry_pool);
}

/**@{ @name Private heap_entry functions */
/**
 * Initialize a new heap entry
 *
 * Sets all the fields in a new heap entry.
 * @param entry THe entry
 * @param block The start of the block that this entry is recording
 * @param user_len The length of the user's segment (Max 128Megs)
 * @param alloc_type The allocation mechanism used
 * @param alloced The allocation callstack
 * @param align The alignment (Max 128)
 *
 * @NOTES Severe hackery can increase the maximum alignment up to 512
 * 		  (Basically is we use the assumption that everything is at least 
 * 		  int aligned in NJAMD, we can give the user 4 times more...)
 * 		  Or we could provide global a base alignment, and then they have 
 * 		  freedom to have alignment up to 128*base align.
 */ 
static void heap_entry_init(struct nj_heap_entry *entry, nj_addr_t block, 
		size_t user_len, struct nj_callstack alloced, struct nj_dynamic_prefs prefs)
{
	entry->block = block;
	entry->user_len = user_len;
	entry->alloc_type = prefs.alloc_type;
	entry->freed.index = NJ_CALLSTACK_INDEX_NOTFREE;
	entry->alloced = alloced;
	entry->align_shift = __nj_log2(prefs.align);
}

/**
 * Checks to see if an addr is inside the region described by this heap_entry
 * 
 * @param addr The address
 * @returns True if inside
 */
static int heap_entry_has_addr(struct nj_heap_entry *entry, nj_addr_t addr)
{
		
	return (entry->user_len && entry->block <= addr && addr < entry->block 
			+ __nj_block_calc_size(entry->user_len, 1<<entry->align_shift, entry->alloc_type));
}
/*@}*/
// vim:ts=4
