/**
 * @file table.h
 *
 * Header file for table object
 *
 */
#ifndef __NJ_LIB_TABLE_H__
#define __NJ_LIB_TABLE_H__

#include <lib/prefs.h>
#include <sys/types.h>
#include <stdarg.h>
#include <config.h>

#ifdef _THREAD_SAFE
# include <pthread.h>
#endif


/** Table data structure */
struct nj_table
{
	void *data;							/**< Data itself */
	u_int top;								/**< top free byte */
	u_int resize   : 1;						/**< Resize if full */
	u_int has_file : 1;						/**< Has file backing? */
	u_int atomic   : 1;						/**< Protect with mutex? */
	u_int persist  : 1;						/**< Do we delete the table? */
	u_int size     : sizeof(u_int)*8 - 4;	/**< in bytes */
#ifdef _THREAD_SAFE
	pthread_mutex_t lock;					/**< Mutex */
#endif
	char *file;								/**< Backing store */
};

/** Tables that don't require pthread locking or file backing */
struct nj_table_light
{
	void *data;								/**< Data itself */
	u_int top;								/**< top free byte */
	u_int resize   : 1;						/**< Resize if full */
	u_int has_file : 1;						/**< Has file backing? */
	u_int atomic   : 1;						/**< Protect with mutex? */
	u_int persist  : 1;						/**< Do we delete the table? */
	u_int size     : sizeof(u_int)*8 - 4;	/**< in bytes */
};

/**@{ @name Table iterator types */
typedef void *nj_table_entry_iterator_t	(void *, va_list);
typedef int nj_table_index_iterator_t	(struct nj_table *tbl, int, va_list);
/*@}*/

/* Get an index in a table */
#define NJ_TABLE_INDEX(table, idx, dtype)			\
				    (((dtype *)(table).data)[idx])

/** Convert an index of a table to a pointer */
#define NJ_TABLE_INDEX_TO_PTR(table, idx, dtype)			\
				    (&NJ_TABLE_INDEX(table, idx, dtype))

int __nj_table_bootstrap_init(struct nj_table *, char *, size_t, int, int);
void __nj_table_user_init(struct nj_table *, struct nj_prefs *);
void __nj_table_fini(struct nj_table *);
void __nj_table_sync(struct nj_table *);
void __nj_table_trunc(struct nj_table *table);
u_int *__nj_table_request_top(struct nj_table *, size_t);
void __nj_table_release_top(struct nj_table *, size_t);
u_int *__nj_table_get_chunk(struct nj_table *, size_t);
void *__nj_table_for_all_entries(struct nj_table *, u_int *, size_t,
	nj_table_entry_iterator_t, ...);
int __nj_table_for_all_indicies(struct nj_table *, u_int *, size_t, 
	nj_table_index_iterator_t, ...);
void __nj_table_trim(struct nj_table *, size_t);

    

#endif /* __NJ_LIB_TABLE_H__ */
// vim:ts=4
