/** 
 * @file memory_pool.h
 * Header for memory pools
 *
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 * 
 */
#ifndef __NJ_LIB_MEMORY_POOL_H__
#define __NJ_LIB_MEMORY_POOL_H__

#include <lib/libc_syms.h>
#include <lib/stack.h>
#include <lib/table.h>
#include <config.h>

/** Number of blocks of cache */
#define NJ_MP_CACHE_BLOCKS		4
/** Number of caches (one for each prot type) */
#define NJ_MP_NUM_CACHES		3

/**@{ @name Methods of protecting freed memory */
#define NJ_CHK_FREE_ERROR		0
#define NJ_CHK_FREE_SEGV		1
#define NJ_CHK_FREE_NONE		2
#define NJ_CHK_FREE_NOFREE		3
/*@}*/

/**@{ @name Allocation methods */
#define NJ_PROT_OVER			0
#define NJ_PROT_UNDER			1
#define NJ_PROT_SUNDER			2
#define NJ_PROT_NONE			3
#define NJ_PROT_UNKNOWN			7	/* 111 */
/*@}*/

/** The memory pool object */
struct nj_memory_pool
{
	struct nj_table memory_tables; /**< The source of memory tables */
	struct nj_stack block_cache[NJ_MP_NUM_CACHES][NJ_MP_CACHE_BLOCKS]; /**< The cache for blocks */
	u_int first_table[NJ_MP_CACHE_BLOCKS];	/**< The first table with free blocks for each block size */
	int fencepost_fd;			/**< The fencepost file descriptor, for error check free */
	void (*libc_free)(void *);	/**< The libc free */
	void *(*libc_malloc)(size_t); /**< Libc malloc */
};

/** 
 * Size of a table of memory. 32 megs for 32 bit processor, 128 megs for 64
 * @FIXME: Change this as RAM gets cheaper/more abundant (it really should be
 * an exponential fit, like 32 and 512) 
 */
#define NJ_MEM_TABLE_MEGS			(sizeof(void *)*24 - 64)
#define NJ_MEM_TABLE_SIZE			(NJ_MEM_TABLE_MEGS << NJ_SHIFT_MEGABYTE)

/** Number of entries in the memory pool table (it's a table of tables) */
#define NJ_MP_TABLE_NENTRIES		NJ_ROUND_TO_PAGE(NJ_ADDRESS_SIZE_32 / NJ_MEM_TABLE_SIZE + 1)
#define NJ_MP_TABLE_SIZE			(NJ_MP_TABLE_NENTRIES*sizeof(struct nj_table_light))

nj_addr_t __nj_memory_pool_bootstrap_init(struct nj_memory_pool *memory_pool);
nj_addr_t __nj_memory_pool_user_init(struct nj_memory_pool *, struct nj_libc_syms *, struct nj_prefs *);
void __nj_memory_pool_fini(struct nj_memory_pool *);
void __nj_memory_pool_trim(struct nj_memory_pool *);
nj_addr_t __nj_memory_pool_request_block(struct nj_memory_pool *, size_t, struct nj_dynamic_prefs);
void __nj_memory_pool_release_block(struct nj_memory_pool *, nj_addr_t, size_t, int, int);

nj_addr_t __nj_memory_pool_resize_block(struct nj_memory_pool *, nj_addr_t, 
		size_t, size_t, int, int);

#endif /* memory_pool.h */
// vim:ts=4
