/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvXsectFrame.h"

// convienience method, (vk/1999-08-05)
// checks if x is between r1 and r2
bool isWithinRange( double r1, double r2, double x )
{
   if( r1 <= x && x <= r2 )
      return true;

   if( r2 <= x && x <= r1 )
      return true;

   return false;
}

// Deletes all info about a parameter, including all space used
// for the level data.
ParamInfo::~ParamInfo()
{
   LevelIterator ii;

   for ( ii = levels_.begin();ii != levels_.end(); ii++ )
   {
      if ((*ii).second )
         delete [] (*ii).second->xvalues_;
   }
}

bool ParamInfo::IsSurface()
{
   return (LevelType() == "sfc" || NrLevels() < 1) ? true : false;
}

string ParamInfo::ExpVerTitle()
{
   if( expver_ == "_" )      //-- missing ExpVer is stored as "_"
      return string("");
   else
      return string( "Expver " ) + expver_;
}

// Fill in data into a level in the LevelMap. Generate the level if needed,
// and delete any old values from the level.
void ParamInfo::Level(double* x,string lev, int n,int modlev)
{
   double flev = atof(lev.c_str() );

   //	Save date for one level
   LevelIterator ii = levels_.find(lev);
   if ( ii == levels_.end() )
   {
      //levels_.insert(pair<string,LevelInfo*>(lev,new LevelInfo(flev,modlev)));
      levels_[ lev ] = new LevelInfo( flev, modlev );
      ii = levels_.find(lev);
   }
   else
   {
      if ( (*ii).second->xvalues_ )
      {
         delete [] (*ii).second->xvalues_;
         (*ii).second->xvalues_ = 0;
      }
      if ( (*ii).second->yvalue_ >= DBL_MAX )
         (*ii).second->yvalue_ = modlev ? flev : flev * 100;
   }

   if (  (*ii).second->xvalues_ == 0 )
      (*ii).second->xvalues_ = new double[n];

   double *xx = (*ii).second->xvalues_;
   for (int i = 0; i < n; i++)
      xx[i] = x[i];
}

void ParamInfo::UpdateLevels(int modlev)
{
   LevelIterator ii;

   for ( ii = levels_.begin();ii != levels_.end(); ii++ )
   {
      double flev = atof((*ii).first.c_str());
      (*ii).second->UpdateLevel(flev,modlev);
   }
}

void ParamInfo::AddLevel(string lev)
{
   if ( levels_.find(lev) == levels_.end() )
      levels_[ lev ] = new LevelInfo;
}

MvDate ParamInfo::ReferenceDate()
{
   double tt = date_ + time_/2400.; //YYYYMMDD + HHMM
   return MvDate(tt);
}

MvDate ParamInfo::VerificationDate()
{
   double tt = date_ + time_/2400. + step_/24.; //YYYYMMDD + HHMM + HH
   return MvDate(tt);
}

/////////////////// ApplicationInfo /////////////////////

// Constructor
ApplicationInfo::ApplicationInfo() :
                    logax_(false),
                    x1_(0), x2_(0), y1_(0), y2_(0),
                    gridNS_(0), gridEW_(0),
                    PresTop_(0),PresBot_(0),
                    nrPoints_(0), nrLevels_(0), modlev_(XS_PL),
                    interpolate_(true), viaPole_(false), haveLNSP_(false)
{ }

// Update the levels according to the modlev parameter.
// This info is not known when the parameters and levels were added.
void ApplicationInfo::updateLevels(ParamMap &params,int modlev)
{
   ParamIterator ii;
   for ( ii = params.begin(); ii != params.end(); ii++)
      (*ii).second->UpdateLevels(modlev);
}

void ApplicationInfo::Grid( double ns, double ew )
{
   gridNS_ = ns;
   gridEW_ = ew;

   if ( ns == cValueNotGiven || ew == cValueNotGiven )
   {
      //-- has to be synchronised with averageAlong() (020731/vk)
      gridNS_ = gridEW_ = 1.0;
   }
}

// Fill lat/long values if this is an XSection
void ApplicationInfo::computeLine(double* lon, double* lat)
{
   double  dx = double( x2_ - x1_ ) / double(nrPoints_ -1);
   double  dy = double( y2_ - y1_ ) / double(nrPoints_ -1);

   lon[0] = x1_;
   lat[0] = y1_;

   if( viaPole() )
   {
      double lat_k_prev = y1_;            //-- xsect line over a pole
      bool overThePole  = false;

      for( int k = 1;  k< nrPoints_; k++ )
      {
         lon[k] = lon[k-1] + dx;
         double lat_k = lat_k_prev + dy;
         if( ( lat_k < -90.0 || lat_k > 90.0 ) && ! overThePole )
         {
            lat_k_prev += dy;           //-- compensate for mirroring
            dy = -dy;                   //-- start mirroring backwards
            if( lat_k < -90.0 )
               lat_k = -180.0 - lat_k;   //-- fix for South Pole
            else
               lat_k = 180.0 - lat_k;    //-- fix for North Pole

            if( dx == 0 )
            {
               lon[k] = x1_ + 180.0;   //-- straight over the pole
               if( lon[k] > 360.0 )    //-- ensure inside normal range
                  lon[k] = lon[k] - 360.0;
            }

            overThePole = true;
         }

         lat[k] = lat_k;
         lat_k_prev += dy;
      }
   }
   else
   {
      for (int k = 1;  k < nrPoints_; k++) //-- no pole
      {
         lon[k] = lon[k-1] + dx;
         lat[k] = lat[k-1] + dy;
      }
   }
}

// Generate data values.
// Only ever called if modlev_ != XS_ML_LNSP, ie if we use pressure levels
// or UKMO model levels
void ApplicationInfo::InterpolateVerticala( double **cp,ParamInfo *param )
{
   if ( ! interpolate_ )
   {
      this->getLevelValues(param,cp);
      return;
   }

   int i;
   double lpa,pk;
   LevelInfo *firstlev, *nextlev;
   for ( int k = 0; k < nrLevels_; k++ )
   {
      // Invert Model level data, except for UKMO
      //kmodel = (modlev_ == cML_UKMO_ND) ? k : (nrLevels_)-k-1;

      pk = GetInterpolatedYValue(k);

      if ( findPressure(param->Levels(),pk,firstlev, nextlev) )
      {
         if ( modlev_ )
            lpa = ( pk - firstlev->YValue()) / (nextlev->YValue() - firstlev->YValue());
         else
            if ( logax_ )
               lpa = ( pk - firstlev->YValue()) / (nextlev->YValue() - firstlev->YValue());
            else
               lpa = log( pk/firstlev->YValue()) / log(nextlev->YValue()/firstlev->YValue());

         double *xvals = firstlev->XValues();
         double *xnext = nextlev->XValues();

         for (i = 0; i < nrPoints_; i++ )
         {
            if ( xvals[i] < DBL_MAX && xnext[i] < DBL_MAX )
               cp[k][i] = xvals[i] + ( xnext[i] - xvals[i]) *lpa; //replace k with kmodel
            else
               cp[k][i] = XMISSING_VALUE; // replace k with kmodel
         }
      }
      else
         for (i = 0;i < nrPoints_; i++)
            cp[k][i] = XMISSING_VALUE;
   }
}

// Generate data values.
// Only ever called if modlev_ == XS_ML_LNSP ie if we use non-UKMO model levels
void ApplicationInfo::InterpolateVerticalb( MvField &field, double **cp, ParamInfo *param, double *splin)
{
   if ( ! interpolate_ )
   {
      this->getLevelValues(param,cp);
      return;
   }

   int i;
   double zdp = ( PresBot_ - PresTop_)/ ( nrLevels_ - 1 );
   double deltalog,lpa;
   double *sp = new double[nrPoints_];
   LevelInfo *firstlev,*nextlev;
   double  firstval, nextval;

   if ( logax_ )
      deltalog = (log10(PresBot_) - log10(PresTop_))/(nrLevels_ - 1);

   for (i = 0; i < nrPoints_; i++)
      sp[i] = exp(splin[i]);

   for ( int k = 0; k < nrLevels_; k++ )
   {
      double pk = logax_ ? exp((log10(PresTop_) + k * deltalog) *log(10.0)) : PresTop_ + k * zdp;

      // Invert Model level data
      //kmodel = nrLevels_-k-1;
      for ( i =0; i < nrPoints_;i++)
      {
         if ( sp[i] > DBL_MAX )
         {
            cp[k][i] = XMISSING_VALUE;
            continue;
         }

         if ( findModelPressure( param->Levels(), pk, sp[i], field, firstlev,nextlev, firstval, nextval) )
         {
            if ( firstlev->XValues()[i] < DBL_MAX && nextlev->XValues()[i] < DBL_MAX )
            {
               lpa = logax_ ? (pk - firstval)/(nextval - firstval) : log(pk/firstval)/log(nextval/firstval);

               cp[k][i] = firstlev->XValues()[i] + ( nextlev->XValues()[i] - firstlev->XValues()[i] ) * lpa;
            }
            else
               cp[k][i] = XMISSING_VALUE;
         }
         else
            cp[k][i] = XMISSING_VALUE;
      }
   }

   delete [] sp;
}


bool ApplicationInfo::findPressure( LevelMap &lmap, double value, LevelInfo *&l1, LevelInfo *&l2)
{
   LevelIterator ii = lmap.begin();
   LevelInfo *tmp1 = (*ii).second, *tmp2;
   ii++;

   for (; ii != lmap.end(); ii++ )
   {
      tmp2 = (*ii).second;
      if ( isWithinRange( tmp1->YValue(), tmp2->YValue(), value ) )
      {
         l1 = tmp1;
         l2 = tmp2;
         return true;
      }

      tmp1 = tmp2;
   }

   return false;
}      //kmodel = k;



bool ApplicationInfo::findModelPressure( LevelMap &lmap, double value, double splin,
                                         MvField &field, LevelInfo *&l1, LevelInfo *&l2,
                                         double &val1,double &val2 )
{
   LevelIterator ii = lmap.begin();
   double tmpval1, tmpval2;
   LevelInfo *tmpl1, *tmpl2;

   tmpl1 = (*ii).second;
   ii++;
   tmpval1 = field.meanML_to_Pressure_bySP(splin,(int)(tmpl1->YValue()));

   for (; ii != lmap.end(); ii++ )
   {
      tmpl2 = (*ii).second;
      tmpval2 = field.meanML_to_Pressure_bySP(splin,(int)(tmpl2->YValue()));

      if ( isWithinRange( tmpval1, tmpval2, value ) )
      {
         l1 = tmpl1; l2 = tmpl2;
         val1 = tmpval1; val2 = tmpval2;
         return true;
      }

      tmpl1 = tmpl2;
      tmpval1 = tmpval2;
   }

   l1 = tmpl1;
   return false;
}

void ApplicationInfo::scaleVelocity(ParamInfo *par)
{
   LevelMap lmap = par->Levels();
   LevelIterator ii = lmap.begin();

   // Compute distance
   double dellat = ABS(y2_-y1_), dellon = ABS(x2_-x1_);
   double latm = (y1_+y2_)/2.;
   double dellona = dellon*cos( cCDR*latm );
   double dist = 110442.3*sqrt (dellat*dellat + dellona*dellona);

   double hscale = 21./dist;
   double vscale = 12./(PresBot_ - PresTop_);

   if (modlev_ == XS_ML )
      vscale = 12./(100 * 1000);

   double factor = -vscale/hscale;
   double *values;
   for ( ; ii  != lmap.end(); ii++ )
   {
      values = (*ii).second->XValues();

      for (int i=0; i<nrPoints_; i++)
      {
         if (values[i] < DBL_MAX)
            values[i] *= factor;
         else
            values[i] = DBL_MAX;
      }
   }

   return;
}

void ApplicationInfo::setMinMaxLevels( double P1, double P2, int nlevel )
{
    PresBot_ = P1;
    PresTop_ = P2;

    // If interpolate=true, nrLevels_ is hardcoded
    nrLevels_ = (!nlevel || interpolate_) ? 85 : nlevel;
}

void ApplicationInfo::getMinMaxLevels( double& P1, double& P2, int& nlevel )
{
    P1 = PresBot_;
    P2 = PresTop_;
    nlevel = nrLevels_;
}

#if 0
void ApplicationInfo::setYValues(MvNcVar *yvar, ParamInfo *par)
{
   int nrY = NrYValues(par);
   double *yvalues = new double[nrY];

   // Retrieve y values
   if ( interpolate_ )
   {
      if ( modlev_ == XS_ML )
      {
         yvalues[0] = PresTop_;   //ncx2->setCurrent(ntime_);
   //ncx2->put(x_values,1,nrPoints);

         yvalues[1] = PresBot_;
      }
      else if ( modlev_ == cML_UKMO_ND )
      {
         yvalues[1] = PresTop_;
         yvalues[0] = PresBot_;
      }
      else
      {
         yvalues[0] = PresTop_/100;
         yvalues[1] = PresBot_/100;
      }
   }
   else
   {
      LevelMap lmap = par->Levels();
      int i = 0;   // the index of the first axis value to set
      int inc = 1; // the amount to increment the axis value index by

      if ( modlev_ == cML_UKMO_ND )  // UK Met Office model levels are 'upside-down'
      {
         i = nrY - 1;
         inc = -1;
      }   //ncx2->setCurrent(ntime_);
   //ncx2->put(x_values,1,nrPoints);


      for ( LevelIterator ii = lmap.begin(); ii != lmap.end(); ii++, i+=inc )
      {
         yvalues[i] = (*ii).second->YValue();
         if ( modlev_ == XS_PL )
            yvalues[i] /= 100;
      }
   }

   // Insert Y values as an attribute of the input variable
   yvar->addAttribute("_Y_VALUES",nrY,yvalues);
   //ncx2->setCurrent(ntime_);
   //ncx2->put(x_values,1,nrPoints);

   delete [] yvalues;
}
#endif

void ApplicationInfo::computeLevelInfo( ParamInfo* par, vector<double>& vals )
{
   // To convert ML&LNSP to PL there are 2 algorithms:
   // 1. find the correspondent PL for each ML value
   // 2. split the interval between top and bottom level in N intervals
   // Option 2 is currently implemented.

   // Compute Pl values, if ML+LNSP is given
   double factor = (modlev_ == XS_ML) ? 1. : 100.;
   vals.reserve(nrLevels_);
   if ( modlev_ == XS_ML_LNSP || interpolate_ )
   {
      double presTop = PresTop_/factor;
      double presBot = PresBot_/factor;
      double zdp = ( presBot - presTop ) / double( nrLevels_ - 1 );
      for ( int i = 0; i < nrLevels_; i++ )
         vals.push_back(presTop + i * zdp);

      return;
   }

   // Get levels
   LevelMap lmap = par->Levels();
   for ( LevelIterator jj = lmap.begin(); jj != lmap.end(); jj++ )
      vals.push_back( (*jj).second->YValue() / factor );

      //if ( modlev == XS_ML_LNSP)
      //   value = field.meanML_to_Pressure_byLNSP(*splin,(int)((*jj).second->YValue()) -1 )/100.;

   return;
}

void ApplicationInfo::getLevelValues( ParamInfo *par, double **cp )
{
   LevelMap lmap = par->Levels();
   LevelIterator ii;

#if 1
   int k = 0;
   for (ii = lmap.begin(); ii != lmap.end(); ii++, k++ )

#else
   int nrY = NrYValues(par);
   int k = nrY - 1;      // the index of the first level to set
   int kinc = -1;        // the amount to increment the level index by

   if ( modlev_ == cML_UKMO_ND )  // UK Met Office model levels are 'upside-down'
   {
      k = 0;
      kinc = 1;
   }

   for (ii = lmap.begin(); ii != lmap.end(); ii++, k += kinc )
#endif

   {
      LevelInfo *lInfo = (*ii).second;
      for (int i = 0; i < nrPoints_; i++ )
         cp[k][i] = lInfo->XValues()[i];
   }
}

double* ApplicationInfo::getOneLevelValues( ParamInfo* par, const string& clev )
{
   // Find the input level
   LevelMap levelMap = par->Levels();
   LevelIterator ll = levelMap.find(clev);
   if ( ll == levelMap.end() )
      return 0;

   return (*ll).second->XValues();
}

#if 0
double* ApplicationInfo::GetYValues(ParamInfo *param, double *topValue, double *bottomValue)
{
   // Interpolation
   if ( interpolate_ )
   {
      double *yvals = new double[nrLevels_];
      int i = nrLevels_ - 1; // should not do this

      // Retrieve y values
      for ( int k = 0; k < nrLevels_; k++ )
      {
         yvals[i] = GetInterpolatedYValue(k);
         if ( this->verticalLevelType() )
            yvals[i] /= 100;
         i--;
      }

      *topValue    = yvals[nrLevels_-1];
      *bottomValue = yvals[0];

      return yvals;
   }

   // No interpolation
   int nrY = NrYValues(param);
   double *yvals = new double[nrY];
   LevelMap lmap = param->Levels();
   int i = nrY - 1;   // the index of the first axis value to set
   int inc = -1; // the amount to increment the axis value index by
   if ( modlev_ == cML_UKMO_ND )  // UK Met Office model levels are 'upside-down'
   {
      i = 0;
      inc = 1;
   }

   for ( LevelIterator ii = lmap.begin(); ii != lmap.end(); ii++, i+=inc )
   {
      yvals[i] = (*ii).second->YValue();
      if ( modlev_ == XS_PL ) yvals[i] /= 100;
   }

   *topValue    = yvals[nrY - 1];
   *bottomValue = yvals[0];

   return yvals;
}
#endif

double ApplicationInfo::GetInterpolatedYValue(int index)
{
   double pk;

   // Interpolation
   if ( logax_ )
   {
      double deltalog = (log10(PresBot_) - log10(PresTop_))/(nrLevels_ - 1);
      pk = exp((log10(PresTop_) + (double)index*deltalog)*log(10.));
   }
   else
   {
      double zdp = ( PresBot_ - PresTop_)/ ( nrLevels_ - 1 );
      pk = modlev_ == cML_UKMO_ND ? PresTop_ + index : PresTop_+ index * zdp;
   }

   return pk;
}

void ApplicationInfo::setAreaLine( double L1, double L2, double R1, double R2 )
{
   // Initialize area or line coordinates
   x1_ = L1;
   x2_ = L2;
   y1_ = R1;
   y2_ = R2;

   // Check values
   if ( y1_ > 90. || y1_ < -90. || y2_ > 90. || y2_ < -90.)
      viaPole_ = true;

   return;
}

void ApplicationInfo::getAreaLine( double& L1, double& L2, double& R1, double& R2 )
{
   L1 = x1_;
   L2 = x2_;
   R1 = y1_;
   R2 = y2_;
}

void ApplicationInfo::getAreaLine( double* area )
{
   area[0] = y1_;
   area[1] = x1_;
   area[2] = y2_;
   area[3] = x2_;
}

bool ApplicationInfo::verticalLevelType ()
{
   if ( modlev_ == XS_PL || modlev_ == XS_ML_LNSP )
      return true;
   else
      return false;
}

double ApplicationInfo::Ancos()
{
      double angle = atan2 (y2_ - y1_,x2_ - x1_);
      return cos(angle);
}

double ApplicationInfo::Ansin()
{
      double angle = atan2 (y2_ - y1_,x2_ - x1_);
      return sin(angle);
}
