// Copyright (c) 2009, Jens Peter Secher <jpsecher@gmail.com>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

class FileUtil
{
	public static function directoryExists( path : String ) : Bool
	{
		return
		(
			sys.FileSystem.exists( path ) &&
			sys.FileSystem.isDirectory( path )
		);
	}

	public static function fileExists( path : String ) : Bool
	{
		return
		(
			sys.FileSystem.exists( path ) &&
			! sys.FileSystem.isDirectory( path )
		);
	}

	//
	// Recursively delete everything reachable from the path.
	// Silently ignores nonexisting paths.
	//
	public static function unlink( path : String ) : Void
	{
		if( sys.FileSystem.exists( path ) )
		{
			if( sys.FileSystem.isDirectory( path ) )
			{				
				for( entry in sys.FileSystem.readDirectory( path ) )
				{
					unlink( path + "/" + entry );
				}
				sys.FileSystem.deleteDirectory( path );
			}
			else
			{
				sys.FileSystem.deleteFile( path );
			}
		}
	}

	//
	// Recursively delete every empty directory reachable from the path.
	// Silently ignores nonexisting paths.
	//
	public static function prune( path : String ) : Void
	{
		if( sys.FileSystem.exists( path ) )
		{
			if( sys.FileSystem.isDirectory( path ) )
			{				
				for( entry in sys.FileSystem.readDirectory( path ) )
				{
					prune( path + "/" + entry );
				}
				for( entry in sys.FileSystem.readDirectory( path ) )
				{
					try
					{
						sys.FileSystem.deleteDirectory( path + "/" + entry );
					}
					catch( e : Dynamic )
					{
						// Not empty, ignore.
					}
				}
			}
		}
	}

	//
	// Recursively copy everything reachable from the path to the target
	// directory.  The target directory must exist.
	//
	public static function copy( from : String, to : String )
	{
		if( sys.FileSystem.exists( from ) )
		{
			var leaf = haxe.io.Path.withoutDirectory( from );
			if( sys.FileSystem.isDirectory( from ) )
			{				
				var targetDir = to + "/" + leaf;
				sys.FileSystem.createDirectory( targetDir );
				for( entry in sys.FileSystem.readDirectory( from ) )
				{
					copy( from + "/" + entry, targetDir );
				}
			}
			else
			{
				var target = to + "/" + leaf;
				sys.io.File.copy( from, target );
				var stat = sys.FileSystem.stat( from );
				var executable = (stat.mode & 0x40) == 0x40;
				if( executable )
				{
					Sys.command( "chmod", ["+x", target] );
				}
			}
		}
	}
}
