# Copyright (c) 2004 CentralNic Ltd. All rights reserved. This program is
# free software; you can redistribute it and/or modify it under the same
# terms as Perl itself.
# $Id: Cookbook.pm,v 1.19 2004/04/22 14:01:01 gavin Exp $

package WWW::CNic::Cookbook;
use vars qw($VERSION);

=pod

=head1 NAME

WWW::CNic::Cookbook - The WWW::CNic Cookbook

=head1 DESCRIPTION

This document provides a fairly complete explanation of how to implement basic Reseller-Registry functions with CentralNic using C<WWW::CNic>. Below you will find examples of how to use WWW::CNic to search for domains, get detailed information about a domain, register a domain, and then modify it.

This document is a work in progress, if you want to see something in it that isn't, or find an error, please let us know by e-mailing C<webmaster@centralnic.com>.

=head1 Test Mode

The Toolkit can be told to run in I<test mode>, that is, to use a non-live copy of the database so that any changes made don't affect the live domains.

To enable test mode, simply set the C<test> argument on the constructor, like so:

	my $query = WWW::CNic->new(test => 1, ... );

=head1 Getting a Suffix List

CentralNic's range of available domain names changes occasionally and you may want to periodically update the list of domains we support. You can use the C<suffixes> command to retrieve an array containing all the domain suffixes CentralNic supports.

	use WWW::CNic;

	# create a request object:
	my $query =	WWW::CNic->new(	command	=> 'suffixes',
					use_ssl	=> 0,
			);

	# execute the query to return a response object:
	my $response = $query->execute();

	# use the suffixes() method to get a list of suffixes:
	my @suffixes = $response->suffixes();

This can be shortened to:

	use WWW::CNic;

	my @suffixes = WWW::CNic->new(command=>'suffixes')->execute()->suffixes();

=head1 Doing a Domain Availability Search

The traditional method for checking the availability of a domain name is to query the registry's whois server, and do a pattern match against the response looking for indications that the domain is registered. This is not an optimal approach for several reasons - firstly, the whois protocol was never designed for it. Secondly, the lack of a whois record does not signify availablity. It also can't handle multiple lookups very well.

The C<search> function is a very powerful way to check on the availability of a domain name. It allows you to check the availability of a domain name across one, several or all of CentralNic's suffixes.

Here's how you might do a check against a particular domain name:

	use WWW::CNic;

	my $domain = 'test-domain';
	my $suffix = 'uk.com'; 

	my $query =	WWW::CNic->new(	command	=> 'search',
					use_ssl	=> 0,
					domain	=> $domain,
			);

	$query->set(suffixlist => [ $suffix ]);

	my $response = $query->execute();

	if ($response->is_registered($suffix)) {
		printf("domain is registered to %s\n", $response->registrant($suffix));
	} else {
		print "domain is available.\n";
	}

Notice the extra step (using the C<set()> method), where we set the 'suffixlist' parameter to be an anonymous array containing the single suffix we want to check. Omitting this step would make the query check against all CentralNic suffixes.

The response object returned has a method C<is_registered()> which returns true if the domain is already registered. Additionally, you can use the C<registrant($suffix)> and C<expiry($suffix)> methods to get the registrant name and a UNIX timestamp of the expiry date respectively.

=head1 Getting Detailed Information About a Domain

Prior to submitting a modification, you may wish to get detailed information about a domain name to present to your users. The C<whois> command allows the lookup of the same detailed information that a whois server provides.

For example:

	use WWW::CNic;

	my $domain = 'test-domain.uk.com';

	my $query =	WWW::CNic->new(	command	=> 'whois',
					domain	=> $domain,
					use_ssl	=> 0,
			);

	my $response = $query->execute();

	print "Domain: $domain\n";

	my %tech_handle = %{$response->response('thandle')};
	printf("Tech Handle: %s (%s)\n", $tech_handle{userid}, $tech_handle{name});

	my $dns = $response->response('dns');
	foreach my $server(@{$dns}) {
		if (ref($server) eq 'ARRAY') {
			my ($name, $ipv4) = @{$server};
			print "Server: $name ($ipv4)\n";
		} else {
			print "Server: $server\n";
		}
	}

The response object contains values for the following keys:

	registrant			the domain's registrant
	created				registration date
	expires				expiry date
	status				status (e.g. Live, On Hold, Pending Deletion etc)
	chandle				Client Handle
	bhandle				Technical Handle
	thandle				Billing Handle
	dns				DNS Servers

The C<registrant> and C<status> keys are just strings. The C<created> and C<expires> keys are UNIX timestamps. The C<chandle>, C<thandle> and C<bhandle> keys are all references to hashes with the following keys:

	userid
	name				
	company
	addr
	postcode
	country
	phone
	fax
	email

These values can be accessed in the following way:

	# get the tech handle:
	my %tech_handle = %{$response->response('thandle')};

	print $tech_handle{addr};

	# and so on for the other keys

The C<dns> key is an array. Elements in the array may be either a plain scalar containing the DNS hostname of the server, or an anonymous array containing the DNS hostname and IPV4 address in that order. Use the C<ref()> function to work out which.

=head1 Registering a Domain

You can register a domain using C<WWW::CNic> in real-time - no waiting for automaton responses to come through!

We recommend that you enable SSL when making registration and modification requests. This is to protect your password when it's sent. WWW::CNic supports SSL communications transparently, since it uses C<LWP> to do all HTTP communication. C<LWP> will handle SSL if the C<Crypt::SSLeay> or C<IO::Socket::SSL> modules have been installed.

	use WWW::CNic;

	my $handle = H12345;
	my $passwd = 'XXXXXXXXX';

	my $query =	WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'register',
					domain		=> 'test-domain.uk.com',
					username	=> $handle,
					password	=> $passwd,
			);

	$query->set(	registrant	=> 'Test Company',
			chandle		=> {	name	=> 'John Q. Test',
						company	=> 'Test Company',
						address	=> "163 New King's Road, Fulham, London",
						postcode=> 'SW6 4SN',
						country	=> 'UK',
						phone	=> '020 7751 9000',
						fax	=> '020 7751 0958',
						email	=> 'johnqt@test-domain.com', },
			thandle		=> $handle,
			dns		=> [ns0.test-domain.com test-domain.com],
			period		=> 4);

	my $response = $query->execute();

	if ($response->is_success()) {
		printf("Domain registered at price %01.2f\n", $response->amount());
	} else {
		printf("Error: %s\n", $response->error());
	}

IMPORTANT NOTE: the details you enter for the Client Handle (C<chandle>) should be the contact details of your customer.

In order to make a registration transaction, you need to supply the C<username> and C<password> parameters - these correspond to your Reseller Handle's ID and password. Your password is C<crypt()>ed before it is sent to provide a minimum of security if you can't use SSL.

You need to set a range of extra parameters to register a domain. These are explained below.

=over

=item 1

C<registrant> - the name of the domain's registrant

=item 2

C<chandle> - the client handle. This may take two values. It can either be a scalar containing the Handle ID of an existing handle, or a reference to a hash containing the following:

	name
	company
	address
	postcode
	country
	phone
	fax
	email

=item 3

C<thandle> - the technical handle. This may take I<three> values. It can be a scalar containing the Handle ID of an existing handle, or "C<chandle>" to set it to be whatever C<chandle> is, or another reference to a hash, formatted as above.

=item 4

C<dns> - the DNS servers for the domain. This can either be an anonymous array of DNS hostnames, or a reference to hash such as that below:

	my $dns = 	{	'ns0.centralic.net'	=> '195.82.125.70',
				'ns1.centralic.net'	=> '212.18.224.66',
			};

If you have specified default DNS servers using the Reseller Console you can set C<$dns> to be 'C<defaults>' and the system will use these.

=item 5

C<period> - the registration period for the domain name. CentralNic currently supports 2, 4, 6 and 8 year registrations. if this field is ommitted the domain will be registered for 2 years.

=back

The response object for this command has all the usual methods (as documented in L<WWW::CNic::Response>), plus the C<amount()> method, which returns the price in Sterling for the domain.

=head1 Creating a New Handle

If you are going to be registering multiple domains, you should create a new handle and use that to register the domains with, rather than supply new contact details for each registration, which will result in a new client handle being created each time.

To do so, use the C<create_handle> command:

	use WWW::CNic;

	my $handle = H12345;
	my $passwd = 'XXXXXXXXX';

	my $query =	WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'create_handle',
					username	=> $handle,
					password	=> $passwd,
			);

	$query->set(handle => $handle);

	my $response = $query->execute();

	if ($response->is_success()) {
		printf("New handle created with ID\n", $response->handle());
	} else {
		printf("Error: %s\n", $response->error());
	}

The format for C<$handle> is exactly the same as explained in the 'Registering a Domain' section.

You can use the C<handle()> method to return the ID of the created handle, for use when registering.

=head1 Modifying a Domain

You can use C<WWW::CNic> to do real-time modification of a domain. The procedure is somewhat similar to that of registration.

	use WWW::CNic;

	my $handle = H12345;
	my $passwd = 'XXXXXXXXX';

	my $query =	WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'modify',
					domain		=> 'test-domain.uk.com',
					username	=> $handle,
					password	=> $passwd,
			);

	$query->set(	thandle	=> $handle,
			dns	=> {	drop	=> ['ns0.centralnic.net', 'ns1.centralnic.net'],
					add	=> ['ns0.test-domain.com,', 'ns1.test-domain.com'] });

	my $response = $query->execute();

	if ($response->is_error()) {
		printf("Error, could not modify domain: %s\n", $response->error());
	} else {
		print "Domain modified OK.\n";
	}

The C<modify> command allows you to add and remove DNS servers and to change the Technical Handle. You can set two parameters for the transaction:

=over

=item 1

C<thandle> corresponds to a new Technical Handle. The allowed values are the same as those allowed for handle parameters in the C<register> command - either a scalar containing a Handle ID, or an anonymous hash containing the keys (as shown above).

=item 2

C<dns> must be an anonymous hash (or a reference to a hash) with two keys: C<add> and C<drop>. Their values are anonymous arrays of DNS hostnames.

=back

If you create a new Technical Handle, you can use the C<response()> method of the response object to find out its handle ID. For example:

	use WWW::CNic;

	my $handle = H12345;
	my $passwd = 'XXXXXXXXX';

	my $query =	WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'modify',
					domain		=> 'test-domain.uk.com',
					username	=> $handle,
					password	=> $passwd,
			);

	$query->set(	thandle	=> {

	my $response = $query->execute();

	if ($response->is_error()) {
		printf("Error, could not modify domain: %s\n", $response->error());
	} else {
		printf("Added technical handle %s.\n", $response->response('thandle'));
	}

=head1 Registration of an @GB domain name

@GB domain names are different from regular CentralNic domains in that they are offered as part of a package that includes a firstname.surname.gb.com third-level domain, a firstname.surname@gb.com e-mail address, and an online homepage building tool hosted by CentralNic.

	use WWW::CNic;

	my $handle = H12345;
	my $passwd = 'XXXXXXXXX';

	my $query =	WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'register_atgb',
					domain		=> 'bob.smith.gb.com',
					username	=> $handle,
					password	=> $passwd);

	$query->set(	handle		=> {	name	=> 'Bob Smith',
						company	=> 'SmithCo plc',
						address	=> "Smith Street",
						postcode=> 'SW6 4SN',
						country	=> 'UK',
						phone	=> '020 7751 9000',
						fax	=> '020 7751 0958',
						email	=> 'bob@smitcho.plc.uk', },
			user_password	=> 'ex@mpl3Passw0rd',
			send_email	=> 1);

	my $response = $query->execute();

	if ($response->is_error()) {
		printf("Error, could not register domain: %s\n", $response->error());
	} else {
		printf("Domain registered at price %01.2f, on proforma number %d\n", $response->amount(), $response->proforma());
	}

The C<send_mail> key controls whether or not CentralNic e-mails the user ID and password to the registration.

The response object for this command inherits from C<WWW::CNic::Response::Register>, and you are advised to study L<WWW::CNic::Response::Register> for more information. it also provides the following methods:

	$password = $response->password();

This returns the password used for the account created. If you leave the C<user_password> key undefined, a password will be generated by the server.

	$proforma_id = $response->proforma();

This returns the number, if any, of any pro forma invoice issued against the registration.

	$invoice_id = $response->invoice();

This returns the number, if any, of any invoice issued against the registration.

=head1 Getting a List of Upcoming Renewals

	use WWW::CNic;
	use POSIX qw(strftime);

	my $months = 3;

	my $query = WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'renewals',
					username	=> $handle,
					password	=> $passwd,
					months		=> $months);

	my $response = $query->execute();

	if ($response->is_error()) {
		printf("Error, couldn't get list of upcoming renewals: %s\n", $response->error());
	} else {
		foreach my $domain ($response->domains()) {
			printf(	"Domain %s expires on %s and will cost %01.2f",
				$domain,
				strftime('%d/%m/%Y', localtime($response->expiry($domain))),
				$response->amount($domain)
			);
		}
	}

This command lets you retrieve a list of domain names due for renewal in the last C<$months> months. You can use the C<amount> and C<expiry> methods to retrieve the renewal price and expiry date for each domain.

=head1 Getting a Domain List

	use WWW::CNic;
	use POSIX qw(strftime);

	my $query = WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'list_domains',
					username	=> $handle,
					password	=> $passwd,
				);

		$query->set(	offset	=> 5,
				length	=> 10
				orderby	=> 'name');

		my $response = $query->execute();

		if ($response->is_success()) {
			foreach my $domain ($response->domains()) {
				printf(	"%s: %s - %s (%s)\n",
					$domain,
					strftime('%d/%m/%Y', localtime($response->regdate($domain))),
					strftime('%d/%m/%Y', localtime($response->expirydate($domain))),
					$response->status($domain),
				);
			}
		} else {
			print "Error: ".$response->error()."\n";
		}

You can use this command to retrieve a list of domains against your handle. The response object returned has methods allowing the retrieval of the registration date and expiry date and status. The C<offset> and C<length> parameters work in the SQL-ish way you'd expect. The C<orderby> parameter can be C<name>, C<regdate> or C<expirydate>.

=head1 Issuing Advance Renewals

	use WWW::CNic;
	use strict;

	my $query = WWW::CNic->new(	use_ssl		=> 0,
					command		=> 'issue_renewals',
					username	=> $handle,
					password	=> $passwd);

	$query->set(domains => ['domain1.uk.com', 'domain2.uk.com']);

	my $response = $query->execute();

	if ($response->is_error()) {
		print "Error: ".$response->error()."\n";
	} else {
		printf(	"%s #%d issued at a value of %01.2f.\n",
			($response->invoice() > 0 ? 'Invoice' : 'Pro forma'),
			($response->invoice() > 0 ? $response->invoice() : $response->proforma()),
			$response->amount()
		);
	}

You can issue advance renewals for domains using this command. You simply set a C<domains> parameter to be an anonymous array of domain names. The server returns the invoice/pro forma ID and the total (in Sterling, excluding VAT) for the renewals.

=head1 Getting Pricing Information

	use WWW::CNic;
	use strict;
	
	my $query = WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'get_pricing',
					username	=> $handle,
					password	=> $password,
				);
	
	$query->set(type => 'renewal');
	
	my $response = $query->execute();
	
	if ($response->is_success()) {
		printf("the price we pay for renewals of uk.com domains is %.2f\n", $response->response('uk.com'));
	} else {
		print "Error: ".$response->error()."\n";
	}

This command allows you to retrieve pricing information for your reseller account. You must specify a C<type> parameter, which can be either 'C<registration>' (the default) or 'C<renewal>'.

=head1 Deleting a Domain Name

	my $query = WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'delete',
					domain		=> $domain,
					username	=> $handle,
					password	=> $password,
				);

	my $response = $query->execute();

	if ($response->is_success()) {
		print "Domain has been deleted.\n";
	} else {
		print "Error: ".$response->error()."\n";
	}

Using this service, you can delete an unwanted domain name. However, you must supply a reason code in order for the deletion to take place. The currently available codes are listed below:

	Code 	Meaning
	R1 	Payment not received
	R2 	Fraudulent Registration
	R3 	Domain no longer required by registrant
	R4 	Domain registered in error

In accordance with our policy, an e-mail will be sent to the domain's Client Handle informing them that the domain has been deleted.

=head1 Declining Renewal

	my $query = WWW::CNic->new(	use_ssl		=> 1,
					command		=> 'decline',
					domain		=> $domain,
					username	=> $handle,
					password	=> $password,
				);

	my $response = $query->execute();

	if ($response->is_success()) {
		print "Domain will not be renewed.\n";
	} else {
		print "Error: ".$response->error()."\n";
	}

I<Declining renewal> means that a domain name will not be renewed when it expires - it will simply be deleted upon expiry.

=head1 FEEDBACK

We're always keen to find out about how people are using our Toolkit system, and we're happy to accept any comments, suggestions or problems you might have. If you want to get in touch, please e-mail L<toolkit@centralnic.com>.

=head1 COPYRIGHT

This module is (c) 2004 CentralNic Ltd. All rights reserved. This module is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=head1 SEE ALSO

=over

=item *

L<http://toolkit.centralnic.com/>

=item *

L<WWW::CNic>

=item *

L<WWW::CNic::Simple>

=back

=cut

=cut
