/*
    Copyright (C) 2000 Steve Brown
	Copyright (C) 2000,2001 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: lib.c,v 1.30 2001/11/09 17:09:28 loic Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/lib.c,v $
    $Date: 2001/11/09 17:09:28 $
    $Author: loic $
*/


#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <mysql/mysql.h>
#include <errno.h>
#include <ctype.h>

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#include "lib.h"

/*EOP*/

/* _nss_mysql_isempty
 * checks if a string only contains spaces
 * Returns:
 * 0, string is not empty
 * 1, string is empty
 */

int _nss_mysql_isempty(char * str) {
	if (!str) return 1;
	while(*str != '\0') 
		if (!isspace(*(str++))) return 0;
	return 1;
}

/* _nss_mysql_strtol
 * nss-MySQL strtol version
 * Converts ascii into long
 * str: string to convert
 * fallback: fallback to this value if strtol is not happy
 * error: if (*error), an error has occured, we have fallback.
 */

long _nss_mysql_strtol(char * str, long fallback, int * error) {
	char * endptr;
	long toreturn;
	
	
	/* sanity checks */
	if (!str) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_strol: string pointer is NULL.");
		*error = 1;
		return fallback;
	}

	if (*str == '\0') {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_strtol: string is empty.");
		*error = 1;
		return fallback;
	}
	
	toreturn = strtol(str,&endptr,10);
	
	if (endptr == str) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_strtol: can't convert %s",str);
		*error = 1;
		return fallback;
	}

	if (*endptr != '\0') {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_strtol_: incomplete conversion of "
			               "%s to %ld. Falling back.",str,toreturn);
		*error = 1;
		return fallback;
	}

	if (errno != ERANGE) {
		*error = 0;
		return toreturn;
	}

	_nss_mysql_log(LOG_ERR,"_nss_mysql_strol: overflow when converting %s. "
			               "Fix your database.",str);
	*error = 1;
	return toreturn;
}

/* _nss_mysql_log
 * write in syslog 
 * arguments: error level, printf type format and args
*/
void _nss_mysql_log(int err, const char *format, ...) {
	static int openlog_ac = 0;
	va_list args;
	
	va_start(args, format);
	/* this is not thread safe, but it does not matter here */
	if (! openlog_ac) {
		++openlog_ac;
		openlog("nss-mysql", LOG_PID, LOG_AUTH);
	}
	vsyslog(err, format, args);
	va_end(args);

	/* according to its manpage calling closelog is optional */
#if DEBUG
	closelog(); 
#endif
}

/* Close the database connexion
 * Arguments: ptr to the connexion you want to close
*/

void _nss_mysql_db_close (MYSQL ** mysql_auth) {
	if (*mysql_auth == NULL) {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_db_close: called with a NULL pointer");
		return; /* closed already */
	}
	mysql_close(*mysql_auth);
	/* Trust no one */
	*mysql_auth = NULL;
}

/* MySQL access functions */

/* _nss_mysql_db_connect
 * opens a MySQL connection
 * arguments: mysql_auth of the new connection, database host, database user,
 *            database password, database name
 * returns 1 on success, and 0 on failure
 */

int _nss_mysql_db_connect (MYSQL ** mysql_auth,char * host,const char * dbuser,const char * dbpasswd,const char * database) {
	MYSQL * tmp;
	char * p;
	char * hostname = NULL;
	char * unix_socket = NULL;
	unsigned port = 3306; /* default mysql port */

	
	if (*mysql_auth != NULL) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_db_connect: called with a non NULL MySQL connexion");
		return 1;
	}
	
	tmp = mysql_init(NULL);
	if (tmp == NULL) {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_db_connect: not enough memory to allocate a mysql "
				                     "object");
		*mysql_auth = NULL;
		return 0;
	}

 
	if (strncmp(host, "unix:", 5) == 0) {
		/* we use an UNIX socket */
		unix_socket = host + 5;
	} else {
		/* inet type connection, allowed syntaxes :
		 * inet:host:port
		 * host:port
		 * host
		 */
		if (strncmp(host, "inet:", 5) == 0)
			host += 5;
		if ((p = strchr(host,':')) == 0 || *++p == '\0') {
			/* no port */
			if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_db_connect: using default port");
			hostname = strdup(host);
		} else {
			hostname = malloc((p - host) * sizeof(char));
			if (hostname == NULL) {
				if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_db_connect not enough memory to "
						                     "parse hostname");
				*mysql_auth = NULL;
				return 0;
			}               
			hostname[0] = '\0';
			strncat(hostname, host, p - host - 1);
			port = strtol(p, NULL, 10);
		}
	}

	if (DEBUG && unix_socket) 
		_nss_mysql_log(LOG_ERR,"_nss_mysql_db_connect: connection with user=%s,passwd=%s,database%s,unix_socket=%s",dbuser,dbpasswd,database,unix_socket);
	if (DEBUG && ! unix_socket) 
		_nss_mysql_log(LOG_ERR,"_nss_mysql_db_connect: connection with host=%s,user=%s,passwd=%s,database=%s,port=%u",hostname,dbuser,dbpasswd,database,port);

	*mysql_auth = mysql_real_connect (tmp,
					 hostname,
					 dbuser,
					 dbpasswd,
					 database, port, unix_socket, 0);

	
	if (hostname) free(hostname);
					 
	if (*mysql_auth == NULL) {
		/* connection has failed */
		_nss_mysql_log(LOG_INFO, "_nss_mysql_db_connect: connection failed: %s", mysql_error(tmp));
		mysql_close(tmp);
		return 0;
	}
	
	return 1;
}

/* _nss_mysql_sqlprintf: is called similar to printf. pointer to a malloc'ed string 
 * is returned. you have to free() it after use!
 */

#define BUFFER_SIZE 1024
char * _nss_mysql_sqlprintf(const char * format, ...) {
	va_list args;
	char * str = NULL;
	int len;
	int buffersize = BUFFER_SIZE;
	char * tmp;

	va_start(args, format);
	
	str = malloc(buffersize);
	if (!str) {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_sqlprintf: not enough memory");
		return NULL;
	}
	
	len = vsnprintf(str, buffersize, format, args);
	while (len < 0 || len >= buffersize) {
		/* output has been truncated, we will allocate a bigger space */
		buffersize += BUFFER_SIZE;
		tmp = realloc(str, buffersize);
		if (! tmp) {
			if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_sqlprintf: not enough memory");
			free(str);
			return NULL;
		} else {
			str = tmp;
		}
		
		len = vsnprintf(str, buffersize, format, args);
	}

	if (DEBUG) _nss_mysql_log(LOG_ERR, "_nss_mysql_sqlprintf(): buffersize=%d, len=%d", buffersize, len);
	va_end(args);

	return str;
}
