#include "../config.h"
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <usb.h>
#include "libnjb.h"
#include "njbusb.h"
#include "njb_error.h"
#include "defs.h"
#include "base.h"
#include "protocol.h"
#include "protocol3.h"
#include "usb_io.h"

int njb_debug_flags= 0;
int __sub_depth= 0;

/*
 * Search the USB bus for a Nomad JukeBox.  We can handle up to
 * NJB_MAX_DEVICES JukeBox's per USB simultaneously (a value arbitrarily
 * set in base.h, because it's easier to work with a fixed size array than
 * build dynamically growing code...if anyone _really_ plans on having
 * more than NJB_MAX_DEVICES jukeboxes on their system, they can recompile
 * with a bigger number).  The USB device starts at usb0 and goes to usbNNN
 * where NNN is USB_MAX_DEVICES.  Check each bus for a usb device.  Return
 * -1 if errors occurred...though this doesn't mean we failed to find
 * devices.  *count gives the number of NJB devices that were found.
 * Store the resulting NJB structues structure into the njbs[NJB_MAX_DEVICES]
 * array.
 */

#ifdef __NetBSD__
#define MAXDEVNAMES USB_MAX_DEVNAMES
#endif

int njb_discover (njb_t *njbs, int limit, int *count)
{
	__dsub= "njb_discover";
	struct usb_bus *busses;
	struct usb_bus *bus;
	struct usb_device *device;
	int retval= 0;

	__enter;
	*count= 0;

	usb_init();

	/* Try to locate busses and devices 
	 * used to be:
	 * if ( usb_find_busses() ) retval= -1; 
	 * if ( usb_find_devices() ) retval= -1;
	 * which was no good idea.
	 */
	usb_find_busses();
	usb_find_devices();
	busses = usb_get_busses();

	bus= busses;
	while ( bus != NULL ) {
		device= bus->devices;
		while ( device != NULL ) {
			if ( device->descriptor.idVendor == NJB1_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJB1_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJB1;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJB2_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJB2_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJB2;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJB3_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJB3_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJB3;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJBZEN_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZEN_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJBZEN;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJBZEN2_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZEN2_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJBZEN2;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJBZENNX_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZENNX_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJBZENNX;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJBZENXTRA_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZENXTRA_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_NJBZENXTRA;
				njbs++;
				(*count)++;
			}
			if ( device->descriptor.idVendor == NJBDELLDJ_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBDELLDJ_PRODUCT_ID ) {
				njbs->device= device;
				njbs->dev= NULL;
				njbs->device_type= NJB_DEVICE_DELLDJ;
				njbs++;
				(*count)++;
			}
			device= device->next;
		}
		bus= bus->next;
	}

	__leave;
	return retval;
}

/*
 * Open a specific njb for reading and writing.  
 *
 * NJB devices have only one USB configuration.
 */

void njb_close (njb_t *njb)
{
	__dsub= "njb_close";
	__enter;

        usb_release_interface(njb->dev, 0);

        /*
	 * Resetting the USB bus is not popular amongst
	 * NJB2/3/ZEN devices, and will just be made for
	 * NJB1.
         */
	if (njb->device_type == NJB_DEVICE_NJB1){
                usb_resetep(njb->dev, njb->bulkep);
                usb_reset(njb->dev);
	}

        usb_close(njb->dev);

	if ( njb->tmpdir ) free(njb->tmpdir);
	
	__leave;
}

int njb_open (njb_t *njb)
{
	__dsub= "njb_open";
	__enter;

	njb->tmpdir= strdup(TMPDIR);
	if ( njb->tmpdir == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}

	if ( (njb->dev= usb_open(njb->device)) == NULL ) {
		NJB_RERROR("usb_open", -1);
		__leave;
		return -1;
	}

	if ( usb_set_configuration(njb->dev, 1) ) {
		NJB_RERROR("usb_set_configuration", -1);
		__leave;
		return -1;
	}

	/* With several jukeboxes connected, a call will often fail
	 * here when you try to connect to the second jukebox after
	 * closing the first. Why? */
	if ( usb_claim_interface(njb->dev, 0) ) {
		NJB_RERROR("usb_claim_interface", -1);
		__leave;
		return -1;
	}

	if ( usb_set_altinterface(njb->dev, 0) ) {
		NJB_RERROR("usb_set_altinterface", -1);
		__leave;
		return -1;
	}

	njb->ctl = njb->dev;
	njb->bin = njb->dev;
	njb->bout = njb->dev;
	njb->iin = njb->dev;
	njb->iout = njb->dev;

	njb->xfersize = DEF_XFER_BLOCK_SIZE;
	njb->reset_get_track_tag = 1;
	njb->reset_get_playlist = 1;
	njb->reset_get_datafile_tag = 1;
	njb->get_extended_tag_info = 0;
	njb->extrarun_on_rename = 0;

	__leave;
	return 0;
}

void njb_set_debug (int flags)
{
	njb_debug_flags= flags;
}

int njb_debug (int flags)
{
	return njb_debug_flags & flags;
}

