#include "global.h"
#include "scr.h"
#include "kbd.h"
#include "keydefs.h"
#include "pallete.h"

char szDebugLogName[256] = "debug.log";
BOOL bDebugTraceEnabled = TRUE;
static BOOL used = FALSE;

void debug_trace ( const char * fmt, ... )
{
  FILE * debugLog;

  if (!bDebugTraceEnabled)
    return;

  if (!used)    // if first call to debug_trace => create empty file
  {
    used = TRUE;
    debugLog = fopen( szDebugLogName, "wt" );
  }
  else   // else just append to end of file
    debugLog = fopen( szDebugLogName, "at" );

  if (debugLog)
  {
    va_list ap;
    va_start( ap, fmt );

    vfprintf( debugLog, fmt, ap );
    fclose( debugLog );

    va_end( ap );
  }
  else
  {
    printf("Fatal: log file (%s) error - %s\n", szDebugLogName, sys_errlist[errno]);
    abort();
  };
};

/*
-------------------------------------------
Small diagnostic pseudo terminal functions.
-------------------------------------------
*/

static char *screen[256];
static int term_x;
static int term_y;
static int last_ln;
static char spaces[512];
#ifdef _DEBUG
static BOOLEAN bInit = FALSE;
#define TERM_INIT()  bInit
#else
#define TERM_INIT()  (1)
#endif

/* ************************************************************************
   Function: CloseSmallTerminal
   Description:
     Closes the small terminal.
*/
void CloseSmallTerminal(void)
{
  int i;

  ASSERT(TERM_INIT());

  for (i = 0; i < 256; ++i)
    if (screen[i] != NULL)
      free(screen[i]);
  #ifdef _DEBUG
  bInit = FALSE;
  #endif
}

/* ************************************************************************
   Function: OpenSmallTerminal
   Description:
     Opens a small pseudo terminal.
     Q: Why pseudo?
     A: Because all the scroll and cursor positioning functions
     are done by hand.
     Q: Why not using the natural printf() functionality?
     A: Because when working in UNIX terminals, returning natural
     terminal modes disables our editor keyboard handling? Furthermore
     doing this by hand, gives an opportunity to provide some
     output beauty stuff.
*/
void OpenSmallTerminal(void)
{
  int i;

  ASSERT(ScreenHeight < 256);
  ASSERT(!TERM_INIT());

  #if _DEBUG
  bInit = TRUE;
  #endif

  memset(screen, 0, sizeof(screen));  /* Set all to null */

  for (i = 0; i < ScreenHeight; ++i)
  {
    screen[i] = malloc(ScreenWidth * 2);
    if (screen[i] == NULL)
    {
      CloseSmallTerminal();
      return;
    }
    screen[i][0] = '\0';
  }

  for (i = 0; i < ScreenHeight; ++i)
    FillXY(' ', GetColor(coEnterLnPrompt), 0, i, ScreenWidth);

  term_x = 0;
  term_y = 0;
  last_ln = 0;
  memset(spaces, ' ', 510);
  spaces[511] = '\0';
}

/* ************************************************************************
   Function: CalcFlexLen
   Description:
     Calculates Flex write string length. That means not to count
     the '~' characters.
*/
static int CalcFlexLen(const char *s)
{
  const char *p;
  int c;

  if (s == 0)  /* If the address of the string is NULL */
    return 0;

  c = 0;
  for (p = s; *p; ++p)
    if (*p != '~')
      c++;

  return c;
}

/* ************************************************************************
   Function: PrintString
   Description:
     Displays a string on the small terminal.
*/
void PrintString(const char *fmt, ...)
{
  va_list ap;
  char buf[1024];
  char *p;
  char *last;
  int i;
  int nlastline;
  int len;

  ASSERT(TERM_INIT());

  va_start(ap, fmt);
  vsprintf(buf, fmt, ap);
  va_end( ap );

  p = buf;
  last = p;
  nlastline = term_y;
  while (*p != '\0')
  {
    if (*p == '\n')
    {
      if (term_y == ScreenHeight - 1)
      {
        /* do a scroll: copy all the lines to an upper position */
        for (i = 1; i <= term_y; ++i)
          strcpy(screen[i - 1], screen[i]);
        screen[term_y][0] = '\0';  /* plus clear last line on the screen */
        FillXY(' ', GetColor(coEnterLnPrompt), 0, ScreenHeight - 1, ScreenWidth);
        --term_y;  /* The work line is positionized a line up */
        nlastline = 0;  /* redisplay whole the screen */
      }

      *p = '\0';  /* from last up to *p is a line to be displayed */
      strcat(screen[term_y], last);
      last = p + 1;
      term_x = 0;
      len = CalcFlexLen(screen[term_y]);
      if (len < ScreenWidth)  /* Right pad the line with spaces */
        strcat(screen[term_y], &spaces[512 - (ScreenWidth - len) - 1]);
      screen[term_y][ScreenWidth - 1] = '\0';
      term_y += 1;
    }
    ++p;
  }

  if (last == buf)
  {
    /* This line has no newline marker '\n' */
    strcat(screen[term_y], last);
    term_x = CalcFlexLen(screen[term_y]);
    FlexWriteXY(screen[term_y], 0, term_y, GetColor(coEnterLnBraces), GetColor(coEnterLn));
  }

  for (i = nlastline; i < term_y; ++i)
    FlexWriteXY(screen[i], 0, i, GetColor(coTerm1), GetColor(coTerm2));
  GotoXY(term_x, term_y);
}

/* ************************************************************************
   Function: DiagKeyNames
   Description:
*/
void DiagKeyNames(void)
{
  DWORD Key;
  char sKeyName[35];
  #ifdef UNIX
  extern BOOLEAN bTraceKbd;
  #endif

  PrintString("Press <ESC> to cancel\n");
  #ifdef UNIX
  bTraceKbd = TRUE;
  #endif
  while ((Key = ReadKey()) != KEY(0, kbEsc))
  {
    GetKeyName(Key, sKeyName);
    if (Key == 0xffff)
      continue;  /* This is a time event indicator */
    PrintString("%s %#lx state:%#x\n", sKeyName, NO_SH_STATE(Key), SH_STATE(Key));
  }
  #ifdef UNIX
  bTraceKbd = FALSE;
  #endif
}

void main(void)
{
  if (!OpenConsole())
  {
    printf("OpenConsole() failed\n");
    return;
  }

  if (!MapPallete(CPallete, MAX_PALLETE))
  {
    printf("MapPallete() failed\n");
    CloseConsole();
    return;
  }

  GetScreenMetrix();

  OpenSmallTerminal();

  DiagKeyNames();

  CloseSmallTerminal();

  CloseConsole();
}

