(* Copyright (C) 2000, Luca Padovani <luca.padovani@cs.unibo.it>.
 *
 * This file is part of lablgtkmathview, the Ocaml binding
 * for the GtkMathView widget.
 * 
 * lablgtkmathview is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * lablgtkmathview is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with lablgtkmathview; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * For details, send a mail to the author.
 *)

(******************************************************************************)
(*                Claudio Sacerdoti Coen <sacerdot@cs.unibo.it>               *)
(*                                 25/09/2000                                 *)
(*                                                                            *)
(*     This is a simple test for the OCaml (LablGtk indeed) binding of the    *)
(*                             MathView widget                                *)
(******************************************************************************)

(* Callbacks *)
let selection_changed mathview (element : Gdome.element option) =
 print_endline ("selection_changed: " ^
  (match element with
      None -> "selection_changed on nothing"
    | Some element -> element#get_tagName#to_string
  )
 ) ;
 mathview#set_selection element;
 flush stdout
;;

let element_changed mathview (element : Gdome.element option) =
 print_endline ("element_changed: " ^
  (match element with
      None -> "element_changed on nothing"
    | Some element -> element#get_tagName#to_string
  )
 ) ;
 flush stdout
;;

let rec jump (element : Gdome.element) =
 let module G = Gdome in
  let attr = (element#getAttribute ~name:(G.domString "href"))#to_string in
   if attr = "" then
    match element#get_parentNode with
       Some p ->
        begin
         try
          jump (new Gdome.element_of_node p)
         with
          GdomeInit.DOMCastException _ ->
           print_string "jump: NO HREF FOR THIS NODE\n" ;
           flush stdout ;
           false
        end
     | None -> assert false (* every element has a parent *)
   else
    begin
     print_endline ("jump: " ^ attr) ;
     flush stdout ;
     true
    end
;;

let clicked (mathview : GMathView.math_view) (element : Gdome.element) =
 let module G = Gdome in
  if not (jump element) then
   match mathview#get_action with
      Some n ->
       mathview#action_toggle ;
       print_string ("current action selection: " ^
        string_of_int mathview#action_get_selected ^ "\n") ;
       flush stdout ;
    | None ->
       print_string ("mouse is on: " ^
        (match (mathview#get_element : Gdome.element option) with
            Some e -> e#get_tagName#to_string
          | None -> "NO ELEMENT!\n"
         ) ^ "\n") ;
       print_endline ("clicked: " ^ element#get_tagName#to_string) ;
       flush stdout
;;


let activate_t1 (mathview : GMathView.math_view) () =
 mathview#set_font_manager_type `font_manager_t1;
 print_string "WIDGET SET WITH T1 FONTS\n" ;
 flush stdout
;;

let activate_gtk (mathview : GMathView.math_view) () =
 mathview#set_font_manager_type `font_manager_gtk ;
 print_string "WIDGET SET WITH GTK FONTS\n" ;
 flush stdout
;;

let get_font_manager_type mathview () =
 print_string "CURRENT FONT MANAGER TYPE: ";
 begin
  match mathview#get_font_manager_type with
  | `font_manager_t1 -> print_string "T1"
  | `font_manager_gtk -> print_string "GTK"
 end;
 print_newline();
 flush stdout
;;

let get_transparency mathview () =
 print_string ("CURRENT TRANSPARENCY: " ^
  (if mathview#get_transparency then "ON" else "OFF") ^ "\n") ;
 flush stdout
;;

let set_transparency mathview () =
 mathview#set_transparency (not mathview#get_transparency) ;
 print_string "TRANSPARENCY CHANGED\n" ;
 flush stdout
;;


let load (mathview : GMathView.math_view) () =
 mathview#load "test.xml" ;
 print_string "load: SEEMS TO WORK\n" ;
 flush stdout
;;

let load_dom (mathview : GMathView.math_view) () =
 mathview#load_tree ((Gdome.domImplementation ())#createDocumentFromURI ~uri:"test.xml" ()) ;
 print_string "load from DOM: SEEMS TO WORK\n" ;
 flush stdout
;;

let get_selection (mathview : GMathView.math_view) () =
 let selection =
   match mathview#get_selection with
      Some element -> element#get_tagName#to_string
    | None -> "no selection!"
 in
  print_string ("get_selection: " ^ selection ^ "\n") ;
  flush stdout
;;

let set_selection (mathview : GMathView.math_view) () =
 begin
   match mathview#get_selection with
      Some element -> 
       begin
        match element#get_parentNode with
           Some p ->
            begin
             try
              mathview#set_selection (Some (new Gdome.element_of_node p));
              print_string "set selection: SEEMS TO WORK\n"
             with
              GdomeInit.DOMCastException _ ->
               print_string "EXCEPTION: no parent\n"
            end
         | None -> assert false (* every element has a parent *)
       end
    | None ->
       mathview#set_selection None;
       print_string "no selection\n"
 end ;
 flush stdout
;;

let unload mathview () =
 mathview#unload ;
 print_string "unload: SEEMS TO WORK\n" ;
 flush stdout
;;

let get_width mathview () =
 print_string ("get_width: " ^ string_of_int (mathview#get_width) ^ "\n") ;
 flush stdout
;;

let get_height mathview () =
 print_string ("get_height: " ^ string_of_int (mathview#get_height) ^ "\n") ;
 flush stdout
;;

let get_top mathview () =
 let (x,y) = mathview#get_top in
  print_string ("get_top: ("^ string_of_int x ^ "," ^ string_of_int y ^ ")\n") ;
  flush stdout
;;

let set_top mathview () =
 mathview#set_top 0 0;
 print_string "set_top: SEEM TO WORK\n" ;
 flush stdout
;;

let set_adjustments mathview () =
 let adj1 = GData.adjustment () in
 let adj2 = GData.adjustment () in
  mathview#set_adjustments adj1 adj2 ;
  adj1#set_value ((adj1#lower +. adj1#upper) /. 2.0) ;
  adj2#set_value ((adj2#lower +. adj2#upper) /. 2.0) ;
  print_string "set_adjustments: SEEM TO WORK\n" ;
  flush stdout
;;

let get_hadjustment mathview () =
 let adj = mathview#get_hadjustment in
  adj#set_value ((adj#lower +. adj#upper) /. 2.0) ;
  print_string "get_hadjustment: SEEM TO WORK\n" ;
  flush stdout
;;

let get_vadjustment mathview () =
 let adj = mathview#get_vadjustment in
  adj#set_value ((adj#lower +. adj#upper) /. 2.0) ;
  print_string "get_vadjustment: SEEM TO WORK\n" ;
  flush stdout
;;

let get_buffer mathview () =
 let buffer = mathview#get_buffer in
  Gdk.Draw.rectangle buffer (Gdk.GC.create buffer) ~x:0 ~y:0
   ~width:50 ~height:50 ~filled:true () ;
  print_string "get_buffer: SEEMS TO WORK (hint: force the widget redrawing)\n";
  flush stdout
;;

let get_frame mathview () =
 let frame = mathview#get_frame in
  frame#set_shadow_type `NONE ;
  print_string "get_frame: SEEMS TO WORK\n" ;
  flush stdout
;;

let set_font_size mathview () =
 mathview#set_font_size 24 ;
 print_string "set_font_size: FONT IS NOW 24\n" ;
 flush stdout
;;
 
let get_font_size mathview () =
 print_string ("get_font_size: " ^ string_of_int (mathview#get_font_size) ^ "\n") ;
 flush stdout
;;
 
let set_anti_aliasing mathview () =
 mathview#set_anti_aliasing true ;
 print_string "set_anti_aliasing: ON\n" ;
 flush stdout
;;
 
let get_anti_aliasing mathview () =
 print_string ("get_anti_aliasing: " ^
  (match mathview#get_anti_aliasing with true -> "ON" | false -> "OFF") ^
  "\n") ;
 flush stdout
;;
 
let set_kerning mathview () =
 mathview#set_kerning true ;
 print_string "set_kerning: ON\n" ;
 flush stdout
;;
 
let get_kerning mathview () =
 print_string ("get_kerning: " ^
  (match mathview#get_kerning with true -> "ON" | false -> "OFF") ^
  "\n") ;
 flush stdout
;;

let set_log_verbosity mathview () =
 mathview#set_log_verbosity 3 ;
 print_string "set_log_verbosity: NOW IS 3\n" ;
 flush stdout
;;
 
let get_log_verbosity mathview () =
 print_string ("get_log_verbosity: " ^
  string_of_int mathview#get_log_verbosity ^
  "\n") ;
 flush stdout
;;

let export_to_postscript (mathview : GMathView.math_view) () =
 mathview#export_to_postscript ~filename:"test.ps" ();
 print_string "expor_to_postscript: SEEMS TO WORK (hint: look at test.ps)\n";
 flush stdout
;;
 
(* Widget creation *)
let main_window = GWindow.window ~title:"GtkMathView test" () in
let vbox = GPack.vbox ~packing:main_window#add () in
let sw = GBin.scrolled_window ~width:50 ~height:50 ~packing:vbox#pack () in
let mathview= GMathView.math_view ~packing:sw#add ~width:50 ~height:50 () in
let table = GPack.table ~rows:6 ~columns:5 ~packing:vbox#pack () in
let button_gtk=GButton.button ~label:"activate Gtk fonts" ~packing:(table#attach ~left:0 ~top:0) () in
let button_load = GButton.button ~label:"load" ~packing:(table#attach ~left:1 ~top:0) () in
let button_unload = GButton.button ~label:"unload" ~packing:(table#attach ~left:2 ~top:0) () in
let button_get_selection = GButton.button ~label:"get_selection" ~packing:(table#attach ~left:3 ~top:0) () in
let button_set_selection = GButton.button ~label:"set_selection" ~packing:(table#attach ~left:4 ~top:0) () in
let button_get_width = GButton.button ~label:"get_width" ~packing:(table#attach ~left:0 ~top:1) () in
let button_get_height = GButton.button ~label:"get_height" ~packing:(table#attach ~left:1 ~top:1) () in
let button_get_top = GButton.button ~label:"get_top" ~packing:(table#attach ~left:2 ~top:1) () in
let button_set_top = GButton.button ~label:"set_top" ~packing:(table#attach ~left:3 ~top:1) () in
let button_set_adjustments = GButton.button ~label:"set_adjustments" ~packing:(table#attach ~left:4 ~top:1) () in
let button_get_hadjustment = GButton.button ~label:"get_hadjustment" ~packing:(table#attach ~left:0 ~top:2) () in
let button_get_vadjustment = GButton.button ~label:"get_vadjustment" ~packing:(table#attach ~left:1 ~top:2) () in
let button_get_buffer = GButton.button ~label:"get_buffer" ~packing:(table#attach ~left:2 ~top:2) () in
let button_get_frame = GButton.button ~label:"get_frame" ~packing:(table#attach ~left:3 ~top:2) () in
let button_set_font_size = GButton.button ~label:"set_font_size" ~packing:(table#attach ~left:4 ~top:2) () in
let button_get_font_size = GButton.button ~label:"get_font_size" ~packing:(table#attach ~left:0 ~top:3) () in
let button_set_anti_aliasing = GButton.button ~label:"set_anti_aliasing" ~packing:(table#attach ~left:1 ~top:3) () in
let button_get_anti_aliasing = GButton.button ~label:"get_anti_aliasing" ~packing:(table#attach ~left:2 ~top:3) () in
let button_set_kerning = GButton.button ~label:"set_kerning" ~packing:(table#attach ~left:3 ~top:3) () in
let button_get_kerning = GButton.button ~label:"get_kerning" ~packing:(table#attach ~left:4 ~top:3) () in
let button_set_log_verbosity = GButton.button ~label:"set_log_verbosity" ~packing:(table#attach ~left:0 ~top:4) () in
let button_get_log_verbosity = GButton.button ~label:"get_log_verbosity" ~packing:(table#attach ~left:1 ~top:4) () in
let button_export_to_postscript = GButton.button ~label:"export_to_postscript" ~packing:(table#attach ~left:2 ~top:4) () in
let button_t1 = GButton.button ~label:"activate T1 fonts" ~packing:(table#attach ~left:3 ~top:4) () in
let button_get_font_manager_type = GButton.button ~label:"get_font_manager" ~packing:(table#attach ~left:4 ~top:4) () in
let button_get_transparency = GButton.button ~label:"get_transparency" ~packing:(table#attach ~left:0 ~top:5) () in
let button_set_transparency = GButton.button ~label:"set_transparency" ~packing:(table#attach ~left:1 ~top:5) () in
let button_load_dom = GButton.button ~label:"load from DOM" ~packing:(table#attach ~left:2 ~top:5) () in
(* Signals connection *)
ignore(button_gtk#connect#clicked (activate_gtk mathview)) ;
ignore(button_load#connect#clicked (load mathview)) ;
ignore(button_unload#connect#clicked (unload mathview)) ;
ignore(button_get_selection#connect#clicked (get_selection mathview)) ;
ignore(button_set_selection#connect#clicked (set_selection mathview)) ;
ignore(button_get_width#connect#clicked (get_width mathview)) ;
ignore(button_get_height#connect#clicked (get_height mathview)) ;
ignore(button_get_top#connect#clicked (get_top mathview)) ;
ignore(button_set_top#connect#clicked (set_top mathview)) ;
ignore(button_set_adjustments#connect#clicked (set_adjustments mathview)) ;
ignore(button_get_hadjustment#connect#clicked (get_hadjustment mathview)) ;
ignore(button_get_vadjustment#connect#clicked (get_vadjustment mathview)) ;
ignore(button_get_buffer#connect#clicked (get_buffer mathview)) ;
ignore(button_get_frame#connect#clicked (get_frame mathview)) ;
ignore(button_set_font_size#connect#clicked (set_font_size mathview)) ;
ignore(button_get_font_size#connect#clicked (get_font_size mathview)) ;
ignore(button_set_anti_aliasing#connect#clicked (set_anti_aliasing mathview)) ;
ignore(button_get_anti_aliasing#connect#clicked (get_anti_aliasing mathview)) ;
ignore(button_set_kerning#connect#clicked (set_kerning mathview)) ;
ignore(button_get_kerning#connect#clicked (get_kerning mathview)) ;
ignore(button_set_log_verbosity#connect#clicked (set_log_verbosity mathview)) ;
ignore(button_get_log_verbosity#connect#clicked (get_log_verbosity mathview)) ;
ignore(button_export_to_postscript#connect#clicked (export_to_postscript mathview)) ;
ignore(button_t1#connect#clicked (activate_t1 mathview)) ;
ignore(button_get_font_manager_type#connect#clicked (get_font_manager_type mathview)) ;
ignore(button_get_transparency#connect#clicked (get_transparency mathview)) ;
ignore(button_set_transparency#connect#clicked (set_transparency mathview)) ;
ignore(mathview#connect#clicked (clicked mathview)) ;
ignore(mathview#connect#selection_changed (selection_changed mathview)) ;
ignore(mathview#connect#element_changed (element_changed mathview)) ;
ignore(button_load_dom#connect#clicked (load_dom mathview)) ;
(* Main Loop *)
main_window#show () ;
GMain.Main.main ()
;;
