/***************************************************************************
                              kiojobwrapper.cpp
                             -------------------
    copyright            : (C) 2008+ by Csaba Karai
    email                : krusader@users.sourceforge.net
    web site             : http://krusader.sourceforge.net
 ---------------------------------------------------------------------------
  Description
 ***************************************************************************

  A

     db   dD d8888b. db    db .d8888.  .d8b.  d8888b. d88888b d8888b.
     88 ,8P' 88  `8D 88    88 88'  YP d8' `8b 88  `8D 88'     88  `8D
     88,8P   88oobY' 88    88 `8bo.   88ooo88 88   88 88ooooo 88oobY'
     88`8b   88`8b   88    88   `Y8b. 88~~~88 88   88 88~~~~~ 88`8b
     88 `88. 88 `88. 88b  d88 db   8D 88   88 88  .8D 88.     88 `88.
     YP   YD 88   YD ~Y8888P' `8888Y' YP   YP Y8888D' Y88888P 88   YD

                                                     S o u r c e    F i l e

 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kiojobwrapper.h"

#include <cstdio>
#include <iostream>

// QtCore
#include <QEvent>
// QtGui
#include <QTextDocument>
// QtWidgets
#include <QApplication>

#include <KI18n/KLocalizedString>
#include <KIO/Global>
#include <KIO/DirectorySizeJob>
#include <KIO/JobUiDelegate>
#include <KIO/Job>

#include "virtualcopyjob.h"
#include "packjob.h"

class JobStartEvent : public QEvent
{
public:
    JobStartEvent(KIOJobWrapper * wrapperIn) : QEvent(QEvent::User),
            m_wrapper(wrapperIn) {}
    virtual ~JobStartEvent() {}

    KIOJobWrapper * wrapper() {
        return m_wrapper;
    }
private:
    QPointer<KIOJobWrapper> m_wrapper;
};

KrJobStarter * KrJobStarter::m_self = 0;

bool KrJobStarter::event(QEvent * e)
{
    if (e->type() == QEvent::User) {
        JobStartEvent *je = (JobStartEvent *)e;
        je->wrapper()->createJob();
        return true;
    }
    return QObject::event(e);
}

KIOJobWrapper::KIOJobWrapper(KIOJobWrapperType type, const QUrl &url) : QObject(0),
        m_autoErrorHandling(false), m_started(false),
        m_suspended(false)
{
    moveToThread(QApplication::instance()->thread());
    m_type = type;
    m_url = url;
}

KIOJobWrapper::KIOJobWrapper(KIOJobWrapperType type, const QUrl &url, void * userData) : QObject(0),
        m_autoErrorHandling(false), m_started(false),
        m_suspended(false)
{
    moveToThread(QApplication::instance()->thread());
    m_type = type;
    m_url = url;
    m_userData = userData;
}

KIOJobWrapper::KIOJobWrapper(KIOJobWrapperType type, const QUrl &url, const QList<QUrl> &list,
                             int pmode, bool showp) : QObject(0),
        m_autoErrorHandling(false), m_started(false),
        m_suspended(false)
{
    moveToThread(QApplication::instance()->thread());
    m_type = type;
    m_url = url;
    m_urlList = list;
    m_pmode = pmode;
    m_showProgress = showp;
}

KIOJobWrapper::KIOJobWrapper(KIOJobWrapperType type, const QUrl &url, const QUrl &dest, const QStringList &names,
                             bool showp, const QString &atype, const QMap<QString, QString> &packProps) : QObject(0),
        m_urlList(), m_autoErrorHandling(false), m_started(false),
        m_suspended(false)
{
    m_type = type;
    m_url = dest;
    m_archiveSourceBase = url;
    foreach(const QString &name , names) {
        QUrl srcUrl = url;
        srcUrl = srcUrl.adjusted(QUrl::StripTrailingSlash);
        srcUrl.setPath(srcUrl.path() + '/' + (name));
        m_urlList << srcUrl;
    }
    m_archiveFileNames = names;
    m_showProgress = showp;
    m_archiveType = atype;
    m_archiveProperties = packProps;
}

KIOJobWrapper::~KIOJobWrapper()
{
}

void KIOJobWrapper::createJob()
{
    KIO::Job * job = 0;
    switch (m_type) {
    case Stat:
        job = KIO::stat(m_url);
        break;
    case DirectorySize:
        job = KIO::directorySize(m_url);
        break;
    case VirtualMove:
    case VirtualCopy: {
        VirtualCopyJob * vcj = (VirtualCopyJob *)m_userData;
        vcj->slotStart();
        job = vcj;
    }
    break;
    case Copy:
        job = PreservingCopyJob::createCopyJob((PreserveMode)m_pmode, m_urlList, m_url, KIO::CopyJob::Copy, false, m_showProgress);
        break;
    case Move:
        job = PreservingCopyJob::createCopyJob((PreserveMode)m_pmode, m_urlList, m_url, KIO::CopyJob::Move, false, m_showProgress);
        break;
    case Pack:
        job = PackJob::createPacker(m_archiveSourceBase, m_url, m_archiveFileNames, m_archiveType, m_archiveProperties);
        break;
    case Unpack:
        job = UnpackJob::createUnpacker(m_archiveSourceBase, m_url, m_archiveFileNames);
        break;
    default:
        fprintf(stderr, "Internal error: invalid job!\n");
        break;
    }
    if (job) {
        m_job = job;
        connect(job, SIGNAL(destroyed()), this, SLOT(deleteLater()));
        for (int i = 0; i != m_signals.count(); i++)
            if (!m_receivers[ i ].isNull())
                connect(job, m_signals[ i ], m_receivers[ i ], m_methods[ i ]);

        if (m_autoErrorHandling && job->ui())
            job->ui()->setAutoErrorHandlingEnabled(true);
        if (m_suspended)
            job->suspend();
    } else
        deleteLater();
}

KIOJobWrapper * KIOJobWrapper::stat(QUrl &url)
{
    return new KIOJobWrapper(Stat, url);
}

KIOJobWrapper * KIOJobWrapper::directorySize(QUrl &url)
{
    return new KIOJobWrapper(DirectorySize, url);
}

KIOJobWrapper * KIOJobWrapper::copy(int pmode, QList<QUrl> &list, QUrl &url, bool showProgress)
{
    return new KIOJobWrapper(Copy, url, list, pmode, showProgress);
}

KIOJobWrapper * KIOJobWrapper::move(int pmode, QList<QUrl> &list, QUrl &url, bool showProgress)
{
    return new KIOJobWrapper(Move, url, list, pmode, showProgress);
}

KIOJobWrapper * KIOJobWrapper::virtualCopy(const QStringList *names, vfs * vfs, QUrl& dest,
        const QUrl &baseURL, int pmode, bool showProgressInfo)
{
    return new KIOJobWrapper(VirtualCopy, dest,
                             new VirtualCopyJob(names, vfs, dest, baseURL, (PreserveMode)pmode, KIO::CopyJob::Copy, showProgressInfo, false));
}

KIOJobWrapper * KIOJobWrapper::virtualMove(const QStringList *names, vfs * vfs, QUrl& dest,
        const QUrl &baseURL, int pmode, bool showProgressInfo)
{
    return new KIOJobWrapper(VirtualMove, dest,
                             new VirtualCopyJob(names, vfs, dest, baseURL, (PreserveMode)pmode, KIO::CopyJob::Move, showProgressInfo, false));
}

KIOJobWrapper * KIOJobWrapper::pack(const QUrl &srcUrl, const QUrl &destUrl, const QStringList & fileNames,
                                    const QString &type, const QMap<QString, QString> &packProps, bool showProgressInfo)
{
    return new KIOJobWrapper(Pack, srcUrl, destUrl, fileNames, showProgressInfo, type, packProps);
}

KIOJobWrapper * KIOJobWrapper::unpack(const QUrl &srcUrl, const QUrl &destUrl, const QStringList & fileNames,
                                      bool showProgressInfo)
{
    return new KIOJobWrapper(Unpack, srcUrl, destUrl, fileNames, showProgressInfo, QString(), QMap<QString, QString> ());
}

void KIOJobWrapper::start()
{
    m_started = true;
    KrJobStarter *self = KrJobStarter::self();
    QApplication::postEvent(self, new JobStartEvent(this));
}

void KIOJobWrapper::connectTo(const char * signal, const QObject * receiver, const char * method)
{
    m_signals.append(signal);
    m_receivers.append((QObject *)receiver);
    m_methods.append(method);
}

QString KIOJobWrapper::typeStr()
{
    switch (m_type) {
    case Stat:
        return i18nc("Job type", "Status");
    case DirectorySize:
        return i18nc("Job type", "Folder Size");
    case Copy:
    case VirtualCopy:
        return i18nc("Job type", "Copy");
    case Move:
    case VirtualMove:
        return i18nc("Job type", "Move");
    case Pack:
        return i18nc("Job type", "Pack");
    case Unpack:
        return i18nc("Job type", "Unpack");
    default:
        return i18nc("Job type", "Unknown");
    }
}

void KIOJobWrapper::suspend()
{
    m_suspended = true;
    if (m_job)
        m_job->suspend();
}

void KIOJobWrapper::resume()
{
    m_suspended = false;
    if (m_job)
        m_job->resume();
}

void KIOJobWrapper::abort()
{
    if (m_job)
        m_job->kill();
}

QString KIOJobWrapper::toolTip()
{
    QString tip = "<qt><div align=\"center\">";
    tip += "<h3>" + typeStr().toHtmlEscaped() + "</h3>";
    tip += "<table frame=\"box\" border=\"1\"><tbody>";
    tip += "<tr><td>" + i18n("Target").toHtmlEscaped() + "</td><td>" + url().toDisplayString().toHtmlEscaped() + "</td></tr>";
    tip += "<tr><td>" + i18n("Source").toHtmlEscaped() + "</td><td>";
    foreach(const QUrl &urlIn, urlList()) {
        tip += "<li>" + urlIn.toDisplayString().toHtmlEscaped() + "</li>";
    }
    tip += "</td></tr>";
    tip += "</tbody></table>";
    tip += "</div></qt>";
    return tip;
}
