;;; -*- syntax: common-lisp; package: cmn; base: 10; mode: lisp -*-
;;;
;;; music symbols
;;;
;;; created from MusicTex font starting point as translated to PostScript by Matti Koskinen
;;; then edited extensively to try to approach the Sonata font using glfed.c.  All symbols
;;; are expressed here nominally as 0..1.0 with scaling by font-size handled elsewhere.
;;; (that is, from cmn's point of view, a staff is 1.0 high and these glyphs fit that).
;;;
;;; Multi-glyph symbols are mostly drawn as needed from scratch, not saved as separate functions.
;;; Order of curves matters in several cases -- we want these to draw as filled polygons in X
;;; as well as PostScript, but X's filling algorithm is dumb.
;;;
;;; this set of glyphs could be turned back into the MusicTex style of font -- the overall
;;; scaler (going back) is 15.873.  See also make-font.cl.

;;; The Sonata font needs work -- placements are mostly wrong

(in-package :cmn)

;;; -------------------------------- CLEFS --------------------------------

(defun draw-treble-clef (score &optional style)
  (if (music-font score)
      (g-mustext score #o46 0 -.25)
    (let ((x-output (eq (output-type score) :X)))
      ;; in the X case, we need to split this up (fill unhappy)
      (declare (ignore x-output))
      (comment score "treble clef")
      (moveto score 0.490 -0.258)
      (curveto score 0.516 -0.430 0.546 -0.672 0.298 -0.590)
      (curveto score 0.404 -0.580 0.432 -0.436 0.320 -0.398)
      (curveto score 0.210 -0.398 0.180 -0.518 0.256 -0.600)
      (curveto score 0.290 -0.622 0.310 -0.630 0.338 -0.638)
      (curveto score 0.576 -0.668 0.554 -0.402 0.522 -0.252)
      (curveto score 0.892 -0.126 0.746 0.314 0.442 0.236)
      (curveto score 0.436 0.286 0.410 0.388 0.420 0.440)
      (curveto score 0.430 0.490 0.484 0.558 0.510 0.606)
      (curveto score 0.624 0.814 0.616 1.000 0.496 1.108)
      (curveto score 0.410 1.118 0.348 0.888 0.348 0.744)
      (curveto score 0.348 0.696 0.364 0.618 0.348 0.576)
      (curveto score 0.332 0.530 0.290 0.482 0.264 0.440)
      (curveto score 0.152 0.260 0.054 0.082 0.182 -0.120)
      (curveto score 0.256 -0.238 0.358 -0.258 0.490 -0.258)
      (moveto score 0.394 0.622)
      (curveto score 0.374 0.696 0.370 0.978 0.512 0.996)
      (curveto score 0.666 0.948 0.454 0.668 0.394 0.622)
      (moveto score 0.382 0.398)
      (lineto score 0.410 0.224)
      (curveto score 0.252 0.126 0.252 -0.062 0.358 -0.106)
      (lineto score 0.372 -0.088)
      (curveto score 0.284 0.004 0.346 0.132 0.424 0.136)
      (lineto score 0.482 -0.222)
      (curveto score 0.382 -0.220 0.306 -0.222 0.236 -0.134)
      (curveto score 0.096 0.038 0.278 0.242 0.384 0.406)
      (moveto score 0.516 -0.220)
      (lineto score 0.458 0.146)
      (curveto score 0.678 0.176 0.744 -0.146 0.516 -0.220)
      (if (not style) (fill-in score) (draw score style)))))

(defvar treble-clef-bounds '(0.000 -0.650 0.740 1.108))


(defun draw-percussion-clef (score &optional style)
  (comment score "percussion clef")
  (moveto score 0 0)
  (lineto score 0 .5)
  (rlineto score .045 0)
  (rlineto score 0 -.5)
  (rlineto score -.045 0)
  (if (not style) (fill-in score) (draw score style))
  (moveto score .085 0)
  (rlineto score 0 .5)
  (rlineto score .045 0)
  (rlineto score 0 -.5)
  (if (not style) (fill-in score) (draw score style))) ;closepath?

(defvar percussion-clef-bounds '(0.000 0.000 .13 .5))


(defun draw-c-clef (score &optional style)
  (if (music-font score)
      (g-mustext score #o102 0 -0.5)
    (progn
      (comment score "c clef")
      (moveto score 0.465 0.442)
      (lineto score 0.465 0.475)
      (curveto score 0.765 0.503 0.643 0.012 0.515 0.080)
      (curveto score 0.453 0.132 0.450 0.197 0.423 0.257)
      (curveto score 0.382 0.115 0.352 0.042 0.305 0.005)
      (lineto score 0.305 0.500)
      (lineto score 0.282 0.500)
      (lineto score 0.282 -0.500)
      (lineto score 0.305 -0.500)
      (lineto score 0.305 -0.005)
      (curveto score 0.353 -0.043 0.380 -0.112 0.423 -0.257)
      (curveto score 0.452 -0.192 0.455 -0.132 0.517 -0.080)
      (curveto score 0.643 -0.012 0.765 -0.503 0.467 -0.475)
      (lineto score 0.467 -0.438)
      (curveto score 0.533 -0.385 0.503 -0.293 0.417 -0.305)
      (curveto score 0.263 -0.383 0.483 -0.645 0.692 -0.437)
      (curveto score 0.825 -0.283 0.693 0.032 0.440 -0.075)
      (lineto score 0.410 0.000)
      (lineto score 0.440 0.075)
      (curveto score 0.693 -0.032 0.825 0.283 0.692 0.437)
      (curveto score 0.493 0.647 0.263 0.377 0.430 0.303)
      (curveto score 0.502 0.297 0.547 0.383 0.468 0.442)
      (if (not style) (fill-in score) (draw score style))  
      (moveto score 0.120 0.500)
      (lineto score 0.120 -0.500)
      (lineto score 0.227 -0.500)
      (lineto score 0.227 0.500)
      (lineto score 0.120 0.500)
      (if (not style) (fill-in score) (draw score style)))))

(defvar c-clef-bounds '(0.000 -0.512 0.744 0.514))


(defun draw-bass-clef (score &optional style)
  (if (music-font score)
      (g-mustext score #o77 0 -0.75)
    (progn
      (comment score "bass clef")
      (moveto score 0.058 0.075)
      (curveto score 0.115 0.053 0.108 0.145 0.223 0.098)
      (curveto score 0.292 0.052 0.278 -0.057 0.173 -0.072)
      (curveto score 0.085 -0.080 0.003 -0.018 0.007 0.098)
      (curveto score 0.045 0.262 0.238 0.288 0.343 0.250)
      (curveto score 0.492 0.192 0.532 0.067 0.517 -0.047)
      (curveto score 0.497 -0.292 0.072 -0.522 0.017 -0.535)
      (lineto score 0.007 -0.512)
      (curveto score 0.127 -0.452 0.302 -0.342 0.367 -0.152)
      (curveto score 0.437 0.045 0.377 0.187 0.275 0.227)
      (curveto score 0.128 0.267 0.012 0.165 0.062 0.075)
      (if (not style) (fill-in score) (draw score style))
      (circle score .6 .1 .0525 0 360 (not style))
      (circle score .6 -.1 .0525 0 360 (not style)))))

(defvar bass-clef-bounds '(0.000 -0.525 0.715 0.252))



;;; -------------------------------- ORNAMENTS and ACCENTS --------------------------------

(defun draw-turn (score &optional style)
  (if (music-font score)
      (g-mustext score #o124 0 0.0)
    (progn
      (comment score "turn")
      (moveto score -0.096 0.062)
      (curveto score -0.130 0.104 -0.124 0.208 -0.048 0.214)
      (curveto score 0.060 0.204 0.154 0.040 0.258 0.010)
      (curveto score 0.384 -0.028 0.524 0.116 0.418 0.220)
      (curveto score 0.372 0.266 0.264 0.262 0.278 0.180)
      (curveto score 0.310 0.122 0.370 0.154 0.394 0.196)
      (curveto score 0.432 0.144 0.428 0.054 0.342 0.044)
      (curveto score 0.242 0.054 0.148 0.216 0.046 0.248)
      (curveto score -0.080 0.296 -0.220 0.142 -0.114 0.036)
      (curveto score -0.070 -0.008 0.040 -0.004 0.028 0.080)
      (curveto score -0.002 0.142 -0.066 0.102 -0.096 0.062)
      (if (not style) (fill-in score) (draw score style)))))

(defvar turn-bounds '(-0.150 0.000 0.454 0.256))


(defun draw-mordent (score &optional style)
  (if (music-font score)
      (g-mustext score #o155 0 0.0)
    (progn
      (comment score "mordent")
      (moveto score 0.310 0.103)
      (curveto score 0.335 0.129 0.334 0.129 0.359 0.104)
      (lineto score 0.425 0.031)
      (curveto score 0.459 -0.001 0.444 -0.002 0.478 0.028)
      (lineto score 0.606 0.156)
      (lineto score 0.606 0.193)
      (lineto score 0.563 0.146)
      (curveto score 0.536 0.121 0.535 0.121 0.505 0.151)
      (lineto score 0.436 0.231)
      (curveto score 0.407 0.262 0.416 0.258 0.388 0.232)
      (lineto score 0.307 0.156)
      (curveto score 0.282 0.132 0.283 0.130 0.262 0.153)
      (lineto score 0.190 0.232)
      (curveto score 0.161 0.265 0.167 0.265 0.136 0.237)
      (lineto score 0.000 0.111)
      (lineto score 0.000 0.075)
      (lineto score 0.051 0.122)
      (curveto score 0.083 0.153 0.086 0.152 0.109 0.120)
      (lineto score 0.189 0.030)
      (curveto score 0.214 0.003 0.206 0.003 0.236 0.031)
      (lineto score 0.310 0.103)
      (if (not style) (fill-in score) (draw score style)))))

(defvar mordent-bounds '(0.000 0.000 0.606 0.257))


(defun draw-double-mordent (score &optional style)
  (if (music-font score)
      (g-mustext score #o265 0 0.0)
    (progn
      (comment score "double mordent")
      (moveto score 0.560 0.106)
      (curveto score 0.586 0.132 0.583 0.132 0.620 0.092)
      (lineto score 0.681 0.031)
      (curveto score 0.708 0.000 0.707 0.004 0.736 0.027)
      (lineto score 0.870 0.146)
      (lineto score 0.870 0.187)
      (lineto score 0.821 0.138)
      (curveto score 0.793 0.115 0.796 0.111 0.761 0.145)
      (lineto score 0.691 0.220)
      (curveto score 0.662 0.257 0.655 0.257 0.629 0.224)
      (lineto score 0.558 0.145)
      (curveto score 0.533 0.121 0.537 0.118 0.505 0.151)
      (lineto score 0.439 0.224)
      (curveto score 0.416 0.251 0.418 0.252 0.393 0.227)
      (lineto score 0.320 0.159)
      (curveto score 0.286 0.121 0.286 0.125 0.256 0.156)
      (lineto score 0.189 0.226)
      (curveto score 0.154 0.266 0.160 0.266 0.131 0.238)
      (lineto score 0.000 0.115)
      (lineto score 0.000 0.075)
      (lineto score 0.050 0.121)
      (curveto score 0.081 0.151 0.077 0.146 0.108 0.117)
      (lineto score 0.188 0.029)
      (curveto score 0.210 0.000 0.210 0.006 0.236 0.030)
      (lineto score 0.310 0.104)
      (curveto score 0.333 0.130 0.331 0.128 0.358 0.100)
      (lineto score 0.425 0.027)
      (curveto score 0.448 0.004 0.444 0.007 0.476 0.031)
      (lineto score 0.561 0.105)
      (if (not style) (fill-in score) (draw score style)))))

(defvar double-mordent-bounds '(0.000 0.000 0.870 0.257))


(defun draw-trill-section (score &optional style)
  (comment score "trill")
  (moveto score -0.045 0.053)
  (lineto score -0.045 0.075)
  (curveto score -0.028 0.099 0.058 0.171 0.113 0.158)
  (curveto score 0.179 0.142 0.245 0.076 0.287 0.079)
  (curveto score 0.339 0.076 0.354 0.095 0.383 0.118)
  (lineto score 0.383 0.097)
  (curveto score 0.366 0.076 0.279 0.001 0.223 0.008)
  (curveto score 0.150 0.014 0.095 0.091 0.050 0.094)
  (curveto score 0.002 0.097 -0.017 0.079 -0.045 0.053)
  (if (not style) (fill-in score) (draw score style)))

(defvar trill-section-bounds '(-0.045 0.000 0.383 0.159))

(defun draw-trill-sections (score count &optional style)
  (comment score (format nil "~D trill sections" count))
  (loop for i from 0 below count and x0 from 0.0 by 0.385 do
    (moveto score (+ x0 -0.045) 0.053)
    (lineto score (+ x0 -0.045) 0.075)
    (curveto score (+ x0 -0.028) 0.099 (+ x0 0.058) 0.171 (+ x0 0.113) 0.158)
    (curveto score (+ x0 0.179) 0.142 (+ x0 0.245) 0.076 (+ x0 0.287) 0.079)
    (curveto score (+ x0 0.339) 0.076 (+ x0 0.354) 0.095 (+ x0 0.383) 0.118)
    (lineto score (+ x0 0.383) 0.097)
    (curveto score (+ x0 0.366) 0.076 (+ x0 0.279) 0.001 (+ x0 0.223) 0.008)
    (curveto score (+ x0 0.150) 0.014 (+ x0 0.095) 0.091 (+ x0 0.050) 0.094)
    (curveto score (+ x0 0.002) 0.097 (+ x0 -0.017) 0.079 (+ x0 -0.045) 0.053)
    (if (not style) (fill-in score) (draw score style))))


(defun draw-arpeggio (score &optional style)
  (comment score "arpeggio")
  (moveto score 0.005 0.147)
  (curveto score -0.004 0.115 0.042 0.046 0.047 0.039)
  (curveto score 0.049 0.034 0.068 0.005 0.071 0.005)
  (curveto score 0.097 0.009 0.074 0.027 0.070 0.041)
  (curveto score 0.054 0.071 0.055 0.083 0.056 0.098)
  (curveto score 0.067 0.137 0.112 0.153 0.132 0.203)
  (curveto score 0.138 0.261 0.042 0.318 0.058 0.347)
  (curveto score 0.061 0.389 0.125 0.414 0.141 0.455)
  (curveto score 0.142 0.497 0.105 0.527 0.097 0.539)
  (curveto score 0.094 0.541 0.074 0.573 0.070 0.574)
  (curveto score 0.047 0.568 0.065 0.547 0.075 0.534)
  (curveto score 0.085 0.511 0.099 0.513 0.085 0.485)
  (curveto score 0.061 0.451 0.010 0.432 0.004 0.389)
  (curveto score 0.015 0.326 0.084 0.284 0.081 0.245)
  (curveto score 0.083 0.206 0.018 0.193 0.004 0.146)
  (if (not style) (fill-in score) (draw score style)))

(defvar arpeggio-bounds '(0.000 0.000 0.141 0.574))

(defun draw-arpeggios (score count &optional style)
  (comment score (format nil "~D arpeggios" count))
  (loop for i from 0 below count and y0 from 0.0 by 0.52 do
    (moveto score 0.005 (+ y0 0.147))
    (curveto score -0.004 (+ y0 0.115) 0.042 (+ y0 0.046) 0.047 (+ y0 0.039))
    (curveto score 0.049 (+ y0 0.034) 0.068 (+ y0 0.005) 0.071 (+ y0 0.005))
    (curveto score 0.097 (+ y0 0.009) 0.074 (+ y0 0.027) 0.070 (+ y0 0.041))
    (curveto score 0.054 (+ y0 0.071) 0.055 (+ y0 0.083) 0.056 (+ y0 0.098))
    (curveto score 0.067 (+ y0 0.137) 0.112 (+ y0 0.153) 0.132 (+ y0 0.203))
    (curveto score 0.138 (+ y0 0.261) 0.042 (+ y0 0.318) 0.058 (+ y0 0.347))
    (curveto score 0.061 (+ y0 0.389) 0.125 (+ y0 0.414) 0.141 (+ y0 0.455))
    (curveto score 0.142 (+ y0 0.497) 0.105 (+ y0 0.527) 0.097 (+ y0 0.539))
    (curveto score 0.094 (+ y0 0.541) 0.074 (+ y0 0.573) 0.070 (+ y0 0.574))
    (curveto score 0.047 (+ y0 0.568) 0.065 (+ y0 0.547) 0.075 (+ y0 0.534))
    (curveto score 0.085 (+ y0 0.511) 0.099 (+ y0 0.513) 0.085 (+ y0 0.485))
    (curveto score 0.061 (+ y0 0.451) 0.010 (+ y0 0.432) 0.004 (+ y0 0.389))
    (curveto score 0.015 (+ y0 0.326) 0.084 (+ y0 0.284) 0.081 (+ y0 0.245))
    (curveto score 0.083 (+ y0 0.206) 0.018 (+ y0 0.193) 0.004 (+ y0 0.146))
    (if (not style) (fill-in score) (draw score style))))


(defun draw-tr (score &optional style)
  (if (music-font score)
      (g-mustext score #o331 0 0.0)
    (progn
      (comment score "tr")
      (moveto score 0.162 0.252)
      (lineto score 0.198 0.380)
      (lineto score 0.183 0.380)
      (lineto score 0.121 0.341)
      (lineto score 0.093 0.248)
      (lineto score -0.005 0.244)
      (lineto score -0.020 0.210)
      (lineto score 0.083 0.216)
      (lineto score 0.032 0.050)
      (curveto score 0.002 -0.038 0.044 -0.034 0.205 0.021)
      (lineto score 0.198 -0.005)
      (lineto score 0.262 -0.005)
      (lineto score 0.324 0.205)
      (curveto score 0.325 0.207 0.339 0.239 0.359 0.248)
      (curveto score 0.384 0.240 0.368 0.222 0.369 0.209)
      (curveto score 0.380 0.164 0.429 0.169 0.434 0.215)
      (curveto score 0.435 0.281 0.333 0.294 0.305 0.222)
      (curveto score 0.300 0.276 0.243 0.260 0.209 0.256)
      (lineto score 0.162 0.252)
      (moveto score 0.146 0.211)
      (curveto score 0.228 0.220 0.269 0.237 0.266 0.216)
      (lineto score 0.222 0.076)
      (curveto score 0.224 0.058 0.075 0.005 0.099 0.041)
      (lineto score 0.146 0.211)
      (if (not style) (fill-in score) (draw score style)))))

(defvar tr-bounds '(-0.020 -0.026 0.434 0.380))


(defun draw-accent (score &optional style)
  (comment score "accent")
  (moveto score 0 0)
  (lineto score .4 .124)
  (lineto score 0 .248)
  (lineto score 0 .216)
  (lineto score .3 .124)
  (lineto score .0 .032)
  (lineto score 0 0)
  (if (not style) (fill-in score) (draw score style)))

(defvar accent-bounds '(0 0 .4 .248))


(defun draw-tnecca (score &optional style) ;accent backwards
  (comment score "accent reversed")
  (moveto score 0 .124)
  (lineto score .4 .248)
  (lineto score .4 .216)
  (lineto score .1 .124)
  (lineto score .4 .032)
  (lineto score .4 0)
  (lineto score 0 .124)
  (if (not style) (fill-in score) (draw score style)))

(defvar tnecca-bounds '(0 0 .4 .248))




;;; -------------------------------- PAUSES --------------------------------

(defun draw-breath-mark (score &optional style)
  (comment score "breath mark")
  (moveto score 0.027 -0.005)
  (curveto score 0.047 0.093 0.192 0.085 0.186 -0.055)
  (curveto score 0.183 -0.144 0.104 -0.198 0.049 -0.224)
  (lineto score 0.045 -0.215)
  (curveto score 0.138 -0.135 0.161 -0.087 0.127 -0.060)
  (curveto score 0.093 -0.094 0.025 -0.075 0.027 -0.005)
  (if (not style) (fill-in score) (draw score style)))

(defvar breath-mark-bounds '(0.000 -0.217 0.185 0.057))


(defun draw-caesura (score &optional style)
  (comment score "caesura")
  (moveto score 0 0)
  (lineto score .05 0)
  (lineto score .33 .5)
  (lineto score .28 .5)
  (lineto score 0 0)
  (if (not style) (fill-in score) (draw score style))
  (moveto score .17 0)
  (lineto score .22 0)
  (lineto score .5 .5)
  (lineto score .45 .5)
  (lineto score .17 0)
  (if (not style) (fill-in score) (draw score style)))

(defvar caesura-bounds '(0.000 0.000 0.500 0.500))


(defun draw-fermata (score &optional style)
  (if (music-font score)
      (g-mustext score #o125 0 0.0)
    (progn
      (comment score "fermata")
      (moveto score 0.0 0.0)
      (curveto score -0.023 0.197 0.14 0.38 0.338 0.38)
      (curveto score 0.535 0.38 0.698 0.197 0.675 0.0)
      (rlineto score -.01 0.0)
      (curveto score 0.652 0.173 0.513 0.296 0.338 0.296)
      (curveto score 0.163 0.296 0.023 0.173 0.01 0.0)
      (lineto score 0 0)
      (if (not style) (fill-in score) (draw score style))
      (circle score 0.338 0.0925 .05 0 360 (not style)))))

(defvar fermata-bounds '(-0.002 0.000 0.676 0.380))


(defun draw-upside-down-fermata (score &optional style)
  (if (music-font score)
      (g-mustext score #o165 0 0.0)
    (progn
      (comment score "fermata reversed")
      (moveto score 0.0000 -0.003)
      (curveto score -0.023 -0.199 0.14 -0.383 0.338 -0.383)
      (curveto score 0.535 -0.383 0.698 -0.199 0.675 -0.003)
      (rlineto score -.01 0)
      (curveto score 0.652 -0.176 0.513 -0.298 0.338 -0.298)
      (curveto score 0.163 -0.298 0.023 -0.176 0.01 -0.003)
      (lineto score 0 -0.003)
      (if (not style) (fill-in score) (draw score style))
      (circle score .338 -0.0925 .05 0 360 (not style)))))

(defvar upside-down-fermata-bounds '(-0.002 -0.382 0.676 0.000))



;;; -------------------------------- MISCELLANEOUS --------------------------------

(defun draw-repeat-sign (score &optional style)
  (comment score "repeat")
  (moveto score 0 0)
  (lineto score .425 .5)
  (rlineto score .173 0)
  (rlineto score -.425 -.5)
  (if (not style) (fill-in score) (draw score style))
  (circle score .121 .375 .06 0 360 (not style))
  (circle score .490 .121 .06 0 360 (not style)))

(defvar repeat-sign-bounds '(0 0 .6 .5))


(defun draw-upper-bracket (score &optional style)
  (comment score "upper bracket")
  (moveto score 0.100 0.365)
  (rlineto score 0.000 -0.145)
  (rlineto score 0.075 0.000)
  (rlineto score 0.000 0.110)
  (curveto score 0.230 0.350 0.270 0.370 0.310 0.430)
  (rlineto score -0.012 0.003)
  (curveto score 0.270 0.400 0.230 0.370 0.100 0.365)
  (if (not style) (fill-in score) (draw score style)))

(defvar upper-bracket-bounds '(0.000 0.000 0.310 0.433))


(defun draw-lower-bracket (score &optional style)
  (comment score "lower bracket")
  (moveto score 0.100 -0.365)
  (rlineto score 0.000 0.145)
  (rlineto score 0.075 0.000)
  (rlineto score 0.000 -0.110)
  (curveto score 0.230 -0.350 0.270 -0.370 0.310 -0.430)
  (rlineto score -0.012 -0.003)
  (curveto score 0.270 -0.400 0.230 -0.370 0.100 -0.365)
  (if (not style) (fill-in score) (draw score style)))

(defvar lower-bracket-bounds '(0.000 -0.433 0.310 0.000))


(defun draw-segno (score &optional style)
  (if (music-font score)
      (g-mustext score #o45 0 0.0)
    (progn
      (comment score "segno")
      (moveto score 0.533 0.688)
      (lineto score 0.479 0.688)
      (lineto score 0.278 0.396)
      (curveto score 0.144 0.515 0.086 0.550 0.093 0.611)
      (curveto score 0.097 0.705 0.264 0.704 0.236 0.626)
      (curveto score 0.237 0.601 0.210 0.596 0.204 0.567)
      (curveto score 0.211 0.476 0.334 0.494 0.315 0.587)
      (curveto score 0.271 0.850 -0.209 0.656 0.220 0.312)
      (lineto score 0.016 0.007)
      (lineto score 0.068 0.007)
      (lineto score 0.262 0.287)
      (curveto score 0.340 0.236 0.486 0.141 0.450 0.059)
      (curveto score 0.431 -0.023 0.305 -0.005 0.306 0.060)
      (curveto score 0.303 0.094 0.344 0.087 0.338 0.130)
      (curveto score 0.331 0.209 0.231 0.211 0.224 0.121)
      (curveto score 0.261 -0.190 0.776 0.047 0.319 0.374)
      (lineto score 0.533 0.688)
      (if (not style) (fill-in score) (draw score style))
      (circle score 0.09 .28 .05  0 360 (not style)) 
      (circle score .45 .40 .05 9 360 (not style)))))

(defvar segno-bounds '(0.000 0.000 0.532 0.696))


(defun draw-coda (score &optional style)
  (if (music-font score)
      (g-mustext score #o336 0 0.0)
    (progn
      (comment score "coda")
      (moveto score 0.241 0.379)
      (lineto score 0.241 0.429)
      (lineto score 0.216 0.429)
      (lineto score 0.216 0.379)
      (curveto score 0.130 0.375 0.071 0.280 0.075 0.196)
      (lineto score 0.020 0.196)
      (lineto score 0.020 0.171)
      (lineto score 0.077 0.171)
      (curveto score 0.071 0.090 0.132 -0.002 0.216 -0.006)
      (lineto score 0.216 -0.056)
      (lineto score 0.241 -0.056)
      (lineto score 0.241 -0.006)
      (curveto score 0.321 0.002 0.381 0.090 0.376 0.171)
      (lineto score 0.431 0.171)
      (lineto score 0.431 0.196)
      (lineto score 0.376 0.196)
      (curveto score 0.382 0.279 0.324 0.379 0.241 0.379)
      (moveto score 0.308 0.169)
      (curveto score 0.308 0.097 0.294 0.037 0.241 0.020)
      (lineto score 0.241 0.171)
      (lineto score 0.307 0.171)
      (moveto score 0.216 0.020)
      (curveto score 0.163 0.040 0.144 0.109 0.141 0.171)
      (lineto score 0.216 0.171)
      (lineto score 0.216 0.020)
      (moveto score 0.141 0.196)
      (curveto score 0.145 0.265 0.166 0.346 0.216 0.353)
      (lineto score 0.216 0.196)
      (lineto score 0.137 0.196)
      (moveto score 0.241 0.353)
      (curveto score 0.286 0.345 0.308 0.265 0.312 0.196)
      (lineto score 0.241 0.196)
      (lineto score 0.241 0.353)
      (if (not style) (fill-in score) (draw score style)))))

(defvar coda-bounds '(0.000 -0.056 0.431 0.429))


(defun draw-pedal-off (score &optional style)
  (if (music-font score)
      (g-mustext score #o52 0 0.0)
    (progn
      (comment score "pedal off")
      (moveto score 0.219 0.198)
      (curveto score 0.231 0.172 0.195 0.138 0.162 0.173)
      (curveto score 0.149 0.219 0.206 0.231 0.219 0.198)
      (moveto score 0.144 0.242)
      (curveto score 0.166 0.223 0.193 0.230 0.181 0.267)
      (curveto score 0.178 0.306 0.144 0.302 0.151 0.335)
      (curveto score 0.160 0.381 0.225 0.377 0.224 0.330)
      (curveto score 0.228 0.302 0.198 0.306 0.197 0.267)
      (curveto score 0.194 0.237 0.213 0.222 0.237 0.247)
      (curveto score 0.263 0.276 0.234 0.297 0.268 0.322)
      (curveto score 0.314 0.347 0.354 0.297 0.316 0.259)
      (curveto score 0.296 0.237 0.273 0.266 0.246 0.237)
      (curveto score 0.223 0.217 0.232 0.194 0.266 0.197)
      (curveto score 0.303 0.202 0.302 0.232 0.332 0.228)
      (curveto score 0.381 0.232 0.388 0.156 0.332 0.152)
      (curveto score 0.302 0.148 0.302 0.185 0.266 0.183)
      (curveto score 0.231 0.186 0.228 0.169 0.245 0.143)
      (curveto score 0.273 0.116 0.297 0.141 0.316 0.117)
      (curveto score 0.350 0.075 0.303 0.029 0.258 0.062)
      (curveto score 0.237 0.082 0.261 0.102 0.233 0.133)
      (curveto score 0.212 0.151 0.194 0.147 0.197 0.113)
      (curveto score 0.203 0.075 0.232 0.075 0.230 0.043)
      (curveto score 0.223 -0.004 0.159 -0.002 0.152 0.042)
      (curveto score 0.148 0.075 0.185 0.076 0.183 0.113)
      (curveto score 0.183 0.147 0.163 0.150 0.141 0.133)
      (curveto score 0.113 0.104 0.140 0.079 0.113 0.059)
      (curveto score 0.069 0.037 0.033 0.077 0.063 0.117)
      (curveto score 0.082 0.141 0.104 0.117 0.132 0.142)
      (curveto score 0.153 0.163 0.144 0.188 0.113 0.182)
      (curveto score 0.073 0.182 0.075 0.147 0.046 0.152)
      (curveto score -0.003 0.152 -0.003 0.227 0.048 0.227)
      (curveto score 0.075 0.231 0.075 0.198 0.113 0.196)
      (curveto score 0.141 0.197 0.147 0.207 0.133 0.237)
      (curveto score 0.102 0.264 0.082 0.237 0.062 0.261)
      (curveto score 0.028 0.302 0.077 0.347 0.118 0.318)
      (curveto score 0.138 0.297 0.116 0.275 0.144 0.242)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar pedal-off-bounds '(0.000 0.000 0.370 0.370))


(defun draw-ped (score &optional style)
  (if (music-font score)
      (g-mustext score #o241 0 0.0)
    (progn
      (comment score "ped")
      (moveto score 0.368 0.074)
      (curveto score 0.341 0.121 0.335 0.147 0.371 0.203)
      (curveto score 0.435 0.289 0.531 0.243 0.488 0.155)
      (curveto score 0.472 0.117 0.434 0.096 0.414 0.080)
      (curveto score 0.429 0.038 0.494 -0.006 0.541 0.075)
      (curveto score 0.559 0.123 0.558 0.224 0.663 0.252)
      (curveto score 0.603 0.354 0.449 0.393 0.461 0.405)
      (curveto score 0.902 0.262 0.705 -0.124 0.555 0.046)
      (curveto score 0.488 -0.032 0.417 0.021 0.389 0.055)
      (curveto score 0.303 -0.018 0.303 -0.020 0.248 0.040)
      (curveto score 0.218 0.108 0.191 0.062 0.164 0.047)
      (curveto score 0.010 -0.056 0.032 0.019 0.124 0.062)
      (curveto score 0.229 0.117 0.200 0.091 0.228 0.195)
      (curveto score 0.240 0.241 0.149 0.250 0.166 0.311)
      (lineto score 0.207 0.493)
      (curveto score -0.041 0.441 0.049 0.261 0.126 0.387)
      (lineto score 0.138 0.381)
      (curveto score -0.020 0.119 -0.100 0.472 0.220 0.507)
      (curveto score 0.548 0.486 0.399 0.171 0.254 0.374)
      (lineto score 0.264 0.384)
      (curveto score 0.338 0.259 0.521 0.449 0.228 0.488)
      (lineto score 0.198 0.356)
      (curveto score 0.181 0.304 0.273 0.294 0.262 0.241)
      (lineto score 0.229 0.101)
      (curveto score 0.273 0.070 0.282 -0.038 0.368 0.074)
      (moveto score 0.391 0.094)
      (curveto score 0.456 0.130 0.476 0.171 0.468 0.213)
      (curveto score 0.452 0.276 0.333 0.171 0.391 0.094)
      (moveto score 0.627 0.019)
      (curveto score 0.533 0.041 0.586 0.228 0.678 0.229)
      (curveto score 0.729 0.170 0.712 0.025 0.627 0.019)
      (if (not style) (fill-in score :even-odd t) (draw score style))
      (circle score 0.8 0.04 .04 0 360 (not style)))))

(defvar ped-bounds '(-0.006 -0.003 0.731 0.507))


(defun draw-left-paren (score &optional style)
  (comment score "left paren")
  (moveto score 0.157 0.580)
  (curveto score 0.090 0.540 -0.015 0.442 -0.012 0.287)
  (curveto score -0.007 0.145 0.082 0.040 0.147 -0.005)
  (lineto score 0.153 0.003)
  (curveto score 0.035 0.122 0.047 0.193 0.047 0.297)
  (curveto score 0.042 0.417 0.067 0.490 0.162 0.570)
  (lineto score 0.157 0.580)
  (if (not style) (fill-in score) (draw score style)))

(defvar left-paren-bounds '(-0.012 -0.005 0.162 0.580))


(defun draw-right-paren (score &optional style)
  (comment score "right paren")
  (moveto score 0.005 0.580)
  (curveto score 0.072 0.540 0.177 0.442 0.174 0.287)
  (curveto score 0.169 0.145 0.080 0.040 0.015 -0.005)
  (lineto score 0.009 0.003)
  (curveto score 0.127 0.122 0.115 0.193 0.115 0.297)
  (curveto score 0.120 0.417 0.095 0.490 0.0 0.570)
  (lineto score 0.005 0.580)
  (if (not style) (fill-in score) (draw score style)))

(defvar right-paren-bounds '(0.000 -0.005 0.173 0.580))


(defun draw-wedge (score &optional style)
  (comment score "wedge")  
  (moveto score 0 0)
  (lineto score -.075 .25)
  (lineto score .075 .25)
  (lineto score 0 0)
  (if (not style) (fill-in score) (draw score style)))

(defvar wedge-bounds '(-.075 0 .075 .25))


(defun draw-down-bow (score &optional style)
  (comment score "down bow")
  (moveto score 0 0)
  (lineto score 0 .15)
  (lineto score .3 .15)
  (lineto score .3 0)
  (lineto score .29 0)
  (lineto score .29 .075)
  (lineto score .01 .075)
  (lineto score .01 0)
  (lineto score 0 0)
  (if (not style) (fill-in score) (draw score style)))

(defvar down-bow-bounds '(0.000 0.000 0.300 0.150))


(defun draw-up-bow (score &optional style)
  (comment score "up bow")
  (moveto score 0.075 0.000)
  (lineto score 0.000 0.250)
  (lineto score 0.010 0.250)
  (lineto score 0.075 0.055)
  (lineto score 0.140 0.250)
  (lineto score 0.150 0.250)
  (lineto score 0.075 0.000)
  (if (not style) (fill-in score) (draw score style)))

(defvar up-bow-bounds '(0.000 0.000 0.150 0.250))


;;; -------------------------------- NUMBERS and METERS --------------------------------

(defun draw-zero (score &optional style)
  (if (music-font score)
      (g-mustext score #o60 0 0.0)
    (progn
      (comment score "zero")
      (moveto score 0.159 0.233)
      (curveto score 0.272 0.233 0.333 0.117 0.333 0.000)
      (curveto score 0.333 -0.130 0.270 -0.235 0.159 -0.233)
      (curveto score 0.041 -0.233 -0.013 -0.122 -0.019 -0.006)
      (curveto score -0.015 0.113 0.043 0.233 0.159 0.233)
      (moveto score 0.159 0.207)
      (curveto score 0.218 0.207 0.228 0.083 0.229 -0.002)
      (curveto score 0.228 -0.085 0.216 -0.207 0.159 -0.207)
      (curveto score 0.100 -0.207 0.089 -0.085 0.085 0.000)
      (curveto score 0.085 0.085 0.096 0.207 0.159 0.207)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar zero-bounds '(-0.018 -0.233 0.333 0.233))


(defun draw-one (score &optional style)
  (if (music-font score)
      (g-mustext score #o61 0 0.0)
    (progn
      (comment score "one")
      (moveto score 0.070 -0.182)
      (curveto score 0.068 -0.199 0.042 -0.203 0.026 -0.203)
      (lineto score 0.026 -0.233)
      (lineto score 0.209 -0.233)
      (lineto score 0.209 -0.203)
      (curveto score 0.199 -0.202 0.171 -0.203 0.167 -0.187)
      (lineto score 0.167 0.233)
      (lineto score 0.070 0.233)
      (lineto score -0.011 0.034)
      (lineto score 0.010 0.024)
      (lineto score 0.070 0.130)
      (lineto score 0.070 -0.182)
      (if (not style) (fill-in score) (draw score style)))))

(defvar one-bounds '(0.000 -0.233 0.212 0.233))


(defun draw-two (score &optional style)
  (if (music-font score)
      (g-mustext score #o62 0 0.0)
    (progn
      (comment score "two")
      (moveto score 0.068 0.170)
      (curveto score 0.093 0.168 0.130 0.167 0.130 0.113)
      (curveto score 0.123 0.042 0.007 0.044 0.006 0.131)
      (curveto score 0.007 0.179 0.074 0.233 0.146 0.233)
      (curveto score 0.224 0.233 0.318 0.202 0.329 0.107)
      (curveto score 0.320 0.010 0.238 -0.013 0.152 -0.052)
      (curveto score 0.105 -0.071 0.063 -0.114 0.067 -0.137)
      (curveto score 0.113 -0.107 0.157 -0.100 0.205 -0.122)
      (curveto score 0.239 -0.138 0.252 -0.170 0.274 -0.170)
      (curveto score 0.329 -0.170 0.316 -0.111 0.316 -0.107)
      (lineto score 0.337 -0.107)
      (curveto score 0.335 -0.158 0.326 -0.231 0.244 -0.237)
      (curveto score 0.165 -0.235 0.146 -0.172 0.083 -0.172)
      (curveto score 0.015 -0.167 0.024 -0.217 0.019 -0.226)
      (lineto score 0.000 -0.226)
      (lineto score 0.000 -0.161)
      (curveto score 0.011 -0.072 0.231 0.014 0.233 0.104)
      (curveto score 0.233 0.167 0.194 0.205 0.111 0.200)
      (curveto score 0.068 0.198 0.050 0.168 0.068 0.170)
      (if (not style) (fill-in score) (draw score style)))))

(defvar two-bounds '(0.000 -0.234 0.336 0.233))


(defun draw-three (score &optional style)
  (if (music-font score)
      (g-mustext score #o63 0 0.0)
    (progn
      (comment score "three")
      (moveto score 0.094 0.007)
      (curveto score 0.163 0.028 0.204 0.039 0.203 0.102)
      (curveto score 0.212 0.180 0.159 0.209 0.117 0.207)
      (curveto score 0.072 0.210 0.026 0.165 0.052 0.155)
      (curveto score 0.113 0.148 0.113 0.074 0.050 0.068)
      (curveto score 0.020 0.070 -0.010 0.092 -0.005 0.147)
      (curveto score 0.034 0.273 0.291 0.257 0.292 0.103)
      (curveto score 0.292 0.039 0.265 0.019 0.220 0.000)
      (curveto score 0.259 -0.017 0.291 -0.043 0.289 -0.114)
      (curveto score 0.272 -0.281 -0.001 -0.262 -0.011 -0.134)
      (curveto score -0.013 -0.091 0.020 -0.063 0.050 -0.067)
      (curveto score 0.098 -0.070 0.113 -0.144 0.052 -0.157)
      (curveto score 0.033 -0.172 0.065 -0.207 0.117 -0.207)
      (curveto score 0.170 -0.201 0.208 -0.177 0.204 -0.099)
      (curveto score 0.200 -0.038 0.163 -0.028 0.094 -0.007)
      (lineto score 0.094 0.007)
      (if (not style) (fill-in score) (draw score style)))))

(defvar three-bounds '(-0.007 -0.234 0.292 0.234))


(defun draw-four (score &optional style)
  (if (music-font score)
      (g-mustext score #o64 0 0.0)
    (progn
      (comment score "four")
      (moveto score 0.252 0.233)
      (lineto score 0.113 0.233)
      (curveto score 0.107 0.128 0.108 0.003 -0.002 -0.097)
      (lineto score -0.002 -0.122)
      (lineto score 0.159 -0.122)
      (lineto score 0.159 -0.188)
      (curveto score 0.157 -0.198 0.130 -0.207 0.115 -0.206)
      (lineto score 0.115 -0.233)
      (lineto score 0.309 -0.233)
      (lineto score 0.309 -0.207)
      (curveto score 0.299 -0.207 0.274 -0.203 0.268 -0.187)
      (lineto score 0.268 -0.122)
      (lineto score 0.309 -0.122)
      (lineto score 0.309 -0.096)
      (lineto score 0.265 -0.096)
      (lineto score 0.265 0.146)
      (lineto score 0.159 0.017)
      (lineto score 0.159 -0.096)
      (lineto score 0.043 -0.096)
      (curveto score 0.139 0.021 0.217 0.149 0.252 0.233)
      (if (not style) (fill-in score) (draw score style)))))

(defvar four-bounds '(-0.004 -0.233 0.308 0.233))


(defun draw-five (score &optional style)
  (if (music-font score)
      (g-mustext score #o65 0 0.0)
    (progn
      (comment score "five")
      (moveto score 0.022 0.233)
      (lineto score 0.022 -0.002)
      (lineto score 0.050 -0.002)
      (curveto score 0.067 0.026 0.093 0.041 0.124 0.041)
      (curveto score 0.178 0.041 0.207 -0.015 0.207 -0.070)
      (curveto score 0.209 -0.165 0.169 -0.211 0.120 -0.212)
      (curveto score 0.098 -0.215 0.076 -0.211 0.056 -0.187)
      (curveto score 0.039 -0.149 0.114 -0.196 0.118 -0.126)
      (curveto score 0.104 -0.067 0.015 -0.068 0.008 -0.145)
      (curveto score 0.005 -0.275 0.306 -0.297 0.296 -0.063)
      (curveto score 0.283 0.028 0.224 0.068 0.137 0.068)
      (curveto score 0.109 0.068 0.076 0.056 0.050 0.030)
      (lineto score 0.050 0.159)
      (curveto score 0.057 0.152 0.152 0.135 0.199 0.155)
      (curveto score 0.248 0.176 0.264 0.203 0.269 0.236)
      (curveto score 0.181 0.212 0.087 0.211 0.022 0.233)
      (if (not style) (fill-in score) (draw score style)))))

(defvar five-bounds '(0.000 -0.233 0.296 0.233))


(defun draw-six (score &optional style)
  (if (music-font score)
      (g-mustext score #o66 0 0.0)
    (progn
      (comment score "six")
      (moveto score 0.168 -0.238)
      (curveto score 0.244 -0.235 0.303 -0.182 0.305 -0.096)
      (curveto score 0.305 -0.043 0.274 0.030 0.205 0.030)
      (curveto score 0.167 0.030 0.137 0.015 0.115 -0.015)
      (curveto score 0.093 -0.006 0.089 0.176 0.150 0.207)
      (curveto score 0.194 0.235 0.242 0.198 0.231 0.181)
      (curveto score 0.207 0.176 0.188 0.159 0.190 0.119)
      (curveto score 0.208 0.062 0.291 0.069 0.300 0.128)
      (curveto score 0.299 0.175 0.276 0.236 0.179 0.240)
      (curveto score 0.074 0.235 0.007 0.117 0.007 0.004)
      (curveto score 0.011 -0.108 0.057 -0.231 0.168 -0.238)
      (moveto score 0.174 -0.215)
      (curveto score 0.220 -0.212 0.228 -0.155 0.228 -0.108)
      (curveto score 0.229 -0.061 0.222 0.006 0.174 0.006)
      (curveto score 0.124 0.006 0.115 -0.058 0.117 -0.108)
      (curveto score 0.118 -0.157 0.126 -0.215 0.174 -0.215)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar six-bounds '(0.000 -0.238 0.305 0.242))


(defun draw-seven (score &optional style)
  (if (music-font score)
      (g-mustext score #o67 0 0.0)
    (progn
      (comment score "seven")
      (moveto score 0.068 -0.233)
      (lineto score 0.202 -0.233)
      (curveto score 0.202 -0.228 0.185 -0.117 0.229 -0.048)
      (curveto score 0.263 0.007 0.338 0.076 0.330 0.234)
      (lineto score 0.306 0.234)
      (curveto score 0.266 0.147 0.235 0.205 0.169 0.225)
      (curveto score 0.080 0.235 0.061 0.169 0.028 0.220)
      (lineto score 0.004 0.220)
      (lineto score 0.004 0.120)
      (lineto score 0.028 0.120)
      (curveto score 0.037 0.164 0.106 0.177 0.151 0.139)
      (curveto score 0.214 0.099 0.265 0.112 0.294 0.142)
      (curveto score 0.286 0.060 0.218 0.004 0.159 -0.046)
      (curveto score 0.077 -0.112 0.067 -0.226 0.068 -0.233)
      (if (not style) (fill-in score) (draw score style)))))

(defvar seven-bounds '(0.000 -0.233 0.307 0.233))


(defun draw-eight (score &optional style)
  (if (music-font score)
      (g-mustext score #o70 0 0.0)
    (progn
      (comment score "eight")
      (moveto score 0.146 0.235)
      (curveto score 0.068 0.233 0.015 0.198 -0.004 0.125)
      (curveto score -0.011 0.048 0.055 0.024 0.068 0.009)
      (curveto score 0.042 -0.006 -0.006 -0.051 -0.007 -0.117)
      (curveto score 0.002 -0.189 0.054 -0.234 0.154 -0.238)
      (curveto score 0.228 -0.237 0.295 -0.195 0.296 -0.109)
      (curveto score 0.301 -0.073 0.279 -0.025 0.220 0.015)
      (curveto score 0.265 0.045 0.290 0.082 0.287 0.131)
      (curveto score 0.279 0.193 0.213 0.233 0.146 0.235)
      (moveto score 0.146 0.209)
      (curveto score 0.200 0.208 0.233 0.170 0.235 0.114)
      (curveto score 0.237 0.083 0.208 0.052 0.185 0.038)
      (curveto score 0.149 0.070 0.085 0.083 0.069 0.140)
      (curveto score 0.069 0.172 0.101 0.208 0.146 0.209)
      (moveto score 0.099 -0.014)
      (curveto score 0.132 -0.040 0.191 -0.055 0.224 -0.114)
      (curveto score 0.240 -0.166 0.207 -0.207 0.146 -0.207)
      (curveto score 0.080 -0.202 0.051 -0.150 0.047 -0.098)
      (curveto score 0.043 -0.059 0.073 -0.024 0.098 -0.013)
      (if (not style) (fill-in score) (draw score style)))))

(defvar eight-bounds '(-0.002 -0.238 0.290 0.233))


(defun draw-nine (score &optional style)
  (if (music-font score)
      (g-mustext score #o71 0 0.0)
    (progn
      (comment score "nine")
      (moveto score 0.129 0.235)
      (curveto score -0.113 0.212 -0.049 -0.123 0.189 -0.003)
      (curveto score 0.211 0.006 0.216 -0.176 0.157 -0.198)
      (curveto score 0.106 -0.232 0.072 -0.203 0.079 -0.197)
      (curveto score 0.125 -0.179 0.135 -0.142 0.114 -0.105)
      (curveto score 0.096 -0.065 0.019 -0.072 0.003 -0.133)
      (curveto score 0.000 -0.193 0.053 -0.244 0.132 -0.235)
      (curveto score 0.220 -0.231 0.307 -0.117 0.307 -0.003)
      (curveto score 0.309 0.111 0.264 0.240 0.131 0.235)
      (moveto score 0.131 0.215)
      (curveto score 0.085 0.213 0.075 0.155 0.076 0.109)
      (curveto score 0.076 0.059 0.080 -0.006 0.131 -0.006)
      (curveto score 0.181 -0.007 0.191 0.059 0.187 0.109)
      (curveto score 0.187 0.157 0.178 0.215 0.131 0.215)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar nine-bounds '(-0.024 -0.234 0.307 0.234))

(defun draw-common-time (score &optional style)
  (if (music-font score)
      (g-mustext score #o143 0 0.0)
    (progn
      (comment score "common time")
      (moveto score 0.004 0.000)
      (moveto score 0.004 0.000)
      (curveto score 0.004 -0.128 0.096 -0.247 0.228 -0.247)
      (curveto score 0.328 -0.246 0.391 -0.171 0.410 -0.070)
      (lineto score 0.383 -0.070)
      (curveto score 0.366 -0.160 0.310 -0.215 0.230 -0.223)
      (curveto score 0.136 -0.220 0.110 -0.114 0.113 -0.002)
      (curveto score 0.113 0.102 0.128 0.200 0.224 0.220)
      (curveto score 0.266 0.230 0.312 0.220 0.344 0.186)
      (curveto score 0.358 0.172 0.356 0.162 0.334 0.156)
      (curveto score 0.300 0.159 0.263 0.134 0.263 0.084)
      (curveto score 0.270 -0.004 0.414 -0.001 0.418 0.096)
      (curveto score 0.418 0.210 0.310 0.258 0.218 0.252)
      (curveto score 0.092 0.246 0.004 0.128 0.004 0.000)
      (if (not style) (fill-in score) (draw score style)))))

(defvar common-time-bounds '(0.000 -0.247 0.417 0.252))


(defun draw-cut-time (score &optional style)
  (if (music-font score)
      (g-mustext score #o103 0 0.0)
    (progn
      (comment score "cut time")
      (draw-common-time score style)
      (moveto score 0.194 0.374)
      (lineto score 0.194 -0.414)
      (lineto score 0.222 -0.414)
      (lineto score 0.222 0.374)
      (if (not style) (fill-in score) (draw score style)))))

(defvar cut-time-bounds '(0.000 -0.414 0.418 0.374))


(defun draw-plus (score &optional style)
  (comment score "plus")
  (moveto score 0.000 -0.020)
  (lineto score 0.100 -0.020)
  (lineto score 0.100 -0.140)
  (lineto score 0.140 -0.140)
  (lineto score 0.140 -0.020)
  (lineto score 0.240 -0.020)
  (lineto score 0.240 0.020)
  (lineto score 0.140 0.020)
  (lineto score 0.140 0.140)
  (lineto score 0.100 0.140)
  (lineto score 0.100 0.020)
  (lineto score 0.000 0.020)
  (lineto score 0.000 -0.020)
  (if (not style) (fill-in score) (draw score style)))

(defvar plus-bounds '(0.000 -0.140 0.240 0.140))


;;; -------------------------------- ACCIDENTALS --------------------------------

(defun draw-sharp (score &optional style)
  (if (music-font score)
      (g-mustext score #o43 0 0.0)
    (progn
      (comment score "sharp")
      (moveto score 0.168 0.098)
      (lineto score 0.168 -0.050)
      (lineto score 0.210 -0.032)
      (lineto score 0.210 -0.136)
      (lineto score 0.168 -0.154)
      (lineto score 0.168 -0.338)
      (lineto score 0.140 -0.338)
      (lineto score 0.140 -0.166)
      (lineto score 0.072 -0.194)
      (lineto score 0.072 -0.380)
      (lineto score 0.044 -0.380)
      (lineto score 0.044 -0.206)
      (lineto score 0.000 -0.222)
      (lineto score 0.000 -0.116)
      (lineto score 0.044 -0.100)
      (lineto score 0.044 0.048)
      (lineto score 0.000 0.032)
      (lineto score 0.000 0.138)
      (lineto score 0.044 0.154)
      (lineto score 0.044 0.338)
      (lineto score 0.072 0.338)
      (lineto score 0.072 0.166)
      (lineto score 0.140 0.192)
      (lineto score 0.140 0.380)
      (lineto score 0.168 0.380)
      (lineto score 0.168 0.204)
      (lineto score 0.210 0.222)
      (lineto score 0.210 0.116)
      (lineto score 0.168 0.098)
      (moveto score 0.140 0.088)
      (lineto score 0.072 0.060)
      (lineto score 0.072 -0.088)
      (lineto score 0.140 -0.060)
      (lineto score 0.140 0.088)
      (if (not style) (fill-in score) (draw score style)))))

(defvar sharp-bounds '(0.000 -0.380 0.210 0.380))


(defun draw-flat (score &optional style)
  (if (music-font score)
      (g-mustext score #o142 0 0.0)
    (progn
      (comment score "flat")
      (moveto score 0.027 0.086)
      (lineto score 0.027 0.483)
      (lineto score 0.000 0.483)
      (lineto score 0.000 -0.193)
      (curveto score 0.012 -0.186 0.070 -0.139 0.097 -0.119)
      (curveto score 0.141 -0.086 0.244 -0.024 0.215 0.082)
      (curveto score 0.165 0.204 0.027 0.087 0.027 0.086)
      (moveto score 0.027 0.037)
      (lineto score 0.027 -0.137)
      (curveto score 0.032 -0.140 0.206 0.029 0.120 0.090)
      (curveto score 0.112 0.093 0.085 0.105 0.027 0.037)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar flat-bounds '(0.000 -0.193 0.222 0.483))


(defun draw-double-sharp (score &optional style)
  (if (music-font score)
      (g-mustext score #o334 0 0.0)
    (progn
      (comment score "double sharp")
      (moveto score 0.000 0.130)
      (lineto score 0.090 0.130)
      (curveto score 0.091 -0.002 0.174 -0.001 0.170 0.130)
      (lineto score 0.260 0.130)
      (lineto score 0.260 0.040)
      (curveto score 0.127 0.028 0.136 -0.040 0.260 -0.040)
      (lineto score 0.260 -0.130)
      (lineto score 0.170 -0.130)
      (curveto score 0.165 0.002 0.091 0.000 0.090 -0.130)
      (lineto score 0.000 -0.130)
      (lineto score 0.000 -0.043)
      (curveto score 0.127 -0.042 0.127 0.027 0.000 0.040)
      (lineto score 0.000 0.130)
      (if (not style) (fill-in score) (draw score style)))))

(defvar double-sharp-bounds '(0.000 -0.130 0.260 0.130))


(defun draw-natural (score &optional style)
  (if (music-font score)
      (g-mustext score #o156 0 0.0)
    (progn
      (comment score "natural")
      (moveto score 0.000 -0.180)
      (lineto score 0.144 -0.142)
      (lineto score 0.144 -0.348)
      (lineto score 0.170 -0.348)
      (lineto score 0.170 0.176)
      (lineto score 0.026 0.138)
      (lineto score 0.026 0.348)
      (lineto score 0.000 0.348)
      (lineto score 0.000 -0.180)
      (moveto score 0.026 -0.076)
      (lineto score 0.026 0.048)
      (lineto score 0.144 0.078)
      (lineto score 0.144 -0.046)
      (lineto score 0.026 -0.076)
      (if (not style) (fill-in score) (draw score style)))))

(defvar natural-bounds '(0.000 -0.348 0.170 0.348))


(defun draw-double-flat (score &optional style)
  (if (music-font score)
      (g-mustext score #o272 0 0.0)
    (progn
      (comment score "double flat")
      (draw-flat score style)
      (matrix-front score (list 1 0 0 1 (* .2 (scr-size score)) 0))
      (draw-flat score style)
      (matrix-back score))))

(defvar double-flat-bounds '(-0.002 -0.194 0.44 0.484))



;;; -------------------------------- DYNAMICS --------------------------------

(defun draw-f (score &optional style)
  (if (music-font score)
      (g-mustext score #o146 0 0.0)
    (progn
      (comment score "f")
      (moveto score 0.420 0.238)
      (lineto score 0.330 0.238)
      (curveto score 0.330 0.230 0.334 0.348 0.418 0.386)
      (curveto score 0.444 0.398 0.496 0.400 0.490 0.374)
      (curveto score 0.488 0.362 0.462 0.370 0.452 0.360)
      (curveto score 0.429 0.342 0.426 0.308 0.448 0.292)
      (curveto score 0.478 0.276 0.521 0.292 0.514 0.344)
      (curveto score 0.514 0.400 0.456 0.432 0.386 0.414)
      (curveto score 0.258 0.380 0.232 0.240 0.234 0.238)
      (lineto score 0.158 0.238)
      (lineto score 0.158 0.210)
      (lineto score 0.228 0.210)
      (curveto score 0.228 0.208 0.172 -0.046 0.122 -0.117)
      (curveto score 0.114 -0.133 0.096 -0.158 0.078 -0.158)
      (curveto score 0.056 -0.162 0.046 -0.142 0.060 -0.139)
      (curveto score 0.099 -0.138 0.113 -0.074 0.056 -0.059)
      (curveto score 0.022 -0.052 -0.018 -0.102 0.007 -0.146)
      (curveto score 0.026 -0.176 0.088 -0.188 0.134 -0.162)
      (curveto score 0.244 -0.100 0.287 0.046 0.324 0.210)
      (lineto score 0.420 0.210)
      (lineto score 0.420 0.238)
      (if (not style) (fill-in score) (draw score style)))))

(defvar f-bounds '(0.000 -0.176 0.514 0.418))


(defun draw-p (score &optional style)
  (if (music-font score)
      (g-mustext score #o160 0 0.0)
    (progn
      (comment score "p")
      (moveto score 0.184 0.047)
      (curveto score 0.203 0.016 0.218 0.006 0.248 0.004)
      (curveto score 0.379 0.002 0.464 0.230 0.336 0.290)
      (curveto score 0.280 0.310 0.228 0.262 0.211 0.246)
      (curveto score 0.210 0.333 0.126 0.312 0.068 0.153)
      (lineto score 0.089 0.147)
      (curveto score 0.155 0.310 0.205 0.279 0.185 0.224)
      (lineto score 0.040 -0.120)
      (lineto score 0.000 -0.120)
      (lineto score 0.000 -0.146)
      (lineto score 0.157 -0.146)
      (lineto score 0.157 -0.120)
      (lineto score 0.117 -0.120)
      (lineto score 0.184 0.047)
      (moveto score 0.222 0.047)
      (curveto score 0.160 0.086 0.245 0.261 0.321 0.268)
      (curveto score 0.377 0.218 0.303 0.029 0.222 0.047)
      (if (not style) (fill-in score) (draw score style)))))

(defvar p-bounds '(0.000 -0.148 0.404 0.292))

(defun draw-lig-p (score &optional style)
  ;; p without the pronounced upstroke (to connect to f in fp)
  (comment score "P")
  (moveto score 0.184 0.047)
  (curveto score 0.203 0.016 0.218 0.006 0.248 0.004)
  (curveto score 0.379 0.002 0.464 0.230 0.336 0.290)
  (curveto score 0.280 0.310 0.228 0.262 0.211 0.246)
  (curveto score 0.205 0.285 0.168 0.304 0.138 0.296)
  (lineto score 0.136 0.270)
  (curveto score 0.162 0.278 0.190 0.257 0.185 0.224)
  (lineto score 0.040 -0.120)
  (lineto score 0.000 -0.120)
  (lineto score 0.000 -0.145)
  (lineto score 0.157 -0.145)
  (lineto score 0.157 -0.120)
  (lineto score 0.117 -0.120)
  (lineto score 0.184 0.047)
  (moveto score 0.222 0.047)
  (curveto score 0.160 0.086 0.245 0.261 0.321 0.268)
  (curveto score 0.377 0.218 0.303 0.029 0.222 0.047)
  (if (not style) (fill-in score) (draw score style)))

(defvar lig-p-bounds '(0.000 -0.147 0.404 0.299))


(defun draw-m (score &optional style)
  (comment score "m")
  (moveto score 0.188 0.000)
  (lineto score 0.271 0.226)
  (curveto score 0.287 0.270 0.252 0.303 0.207 0.225)
  (lineto score 0.120 0.000)
  (lineto score 0.052 0.000)
  (lineto score 0.157 0.261)
  (curveto score 0.160 0.304 0.072 0.266 0.045 0.184)
  (lineto score 0.027 0.184)
  (curveto score 0.056 0.245 0.100 0.301 0.159 0.292)
  (curveto score 0.188 0.286 0.200 0.262 0.200 0.243)
  (curveto score 0.228 0.303 0.327 0.312 0.333 0.243)
  (curveto score 0.373 0.317 0.488 0.303 0.468 0.236)
  (lineto score 0.396 0.038)
  (curveto score 0.386 0.001 0.435 0.024 0.460 0.074)
  (lineto score 0.472 0.071)
  (curveto score 0.429 -0.026 0.310 -0.065 0.335 0.019)
  (lineto score 0.403 0.222)
  (curveto score 0.424 0.273 0.375 0.291 0.346 0.228)
  (lineto score 0.264 0.000)
  (lineto score 0.186 0.000)
  (if (not style) (fill-in score) (draw score style)))

(defvar m-bounds '(0.000 -0.012 0.472 0.292))


(defun draw-n (score &optional style)
  (comment score "n")
  (moveto score 0.210 0.019)
  (lineto score 0.280 0.226)
  (curveto score 0.287 0.270 0.253 0.302 0.207 0.225)
  (lineto score 0.120 0.000)
  (lineto score 0.052 0.000)
  (lineto score 0.158 0.261)
  (curveto score 0.160 0.304 0.072 0.266 0.045 0.184)
  (lineto score 0.028 0.184)
  (curveto score 0.056 0.245 0.100 0.301 0.159 0.292)
  (curveto score 0.188 0.286 0.200 0.263 0.200 0.242)
  (curveto score 0.228 0.302 0.338 0.312 0.341 0.243)
  (lineto score 0.265 0.024)
  (curveto score 0.264 0.001 0.331 0.024 0.340 0.074)
  (lineto score 0.352 0.071)
  (curveto score 0.33 -0.026 0.210 -0.065 0.210 0.019)
  (if (not style) (fill-in score) (draw score style))
  )

(defvar n-bounds '(0.000 -0.034 0.367 0.292))

(defun draw-niente (score &optional style)
  (if *use-circle-as-niente*
      (progn
        (comment score "n.")
        (setf (line-width score) 0.02)
        (circle score 0.1 0.125 .05 0 360 nil)
        (setf (line-width score) 0.0))
    (progn
      (draw-n score style)
      (circle score 0.425 0.025 0.03 0 360 t))))

(defvar niente-bounds '(0.00 0 0.45 0.175))


(defun draw-subito (score &optional style)
  (declare (ignore style))
  (comment score "I")
  (setf (line-width score) 0.02)
  (moveto score 0.3 -0.2)
  (rlineto score 0 0.7)
  (draw score t)
  (setf (line-width score) 0.0))

(defvar subito-bounds '(0.0 -0.2 0.6 0.7))


(defun draw-z (score &optional style)
  (comment score "z")
  (moveto score 0.082 0.287)
  (lineto score 0.052 0.222)
  (lineto score 0.069 0.222)
  (lineto score 0.089 0.240)
  (lineto score 0.238 0.240)
  (lineto score -0.003 0.006)
  (lineto score 0.034 0.006)
  (curveto score 0.072 0.043 0.156 0.028 0.191 0.003)
  (curveto score 0.274 -0.021 0.302 0.046 0.265 0.101)
  (curveto score 0.251 0.117 0.219 0.118 0.205 0.093)
  (curveto score 0.206 0.050 0.249 0.075 0.246 0.041)
  (curveto score 0.206 0.000 0.196 0.113 0.092 0.061)
  (lineto score 0.299 0.261)
  (lineto score 0.299 0.287)
  (lineto score 0.082 0.287)
  (if (not style) (fill-in score) (draw score style)))

(defvar z-bounds '(-0.003 -0.009 0.299 0.287))


(defun draw-s (score &optional style)
  (comment score "s")
  (moveto score 0.148 0.066)
  (curveto score 0.161 0.028 0.130 0.002 0.083 0.009)
  (curveto score 0.032 0.038 0.107 0.045 0.077 0.088)
  (curveto score 0.055 0.113 0.018 0.101 0.014 0.068)
  (curveto score 0.019 -0.043 0.216 -0.041 0.208 0.076)
  (curveto score 0.200 0.141 0.113 0.167 0.091 0.211)
  (curveto score 0.072 0.255 0.134 0.297 0.172 0.268)
  (curveto score 0.212 0.237 0.145 0.225 0.188 0.188)
  (curveto score 0.223 0.179 0.242 0.199 0.236 0.224)
  (curveto score 0.209 0.327 0.028 0.306 0.032 0.212)
  (curveto score 0.037 0.149 0.142 0.116 0.148 0.066)
  (if (not style) (fill-in score) (draw score style)))

(defvar s-bounds '(0.000 -0.009 0.236 0.293))


(defun draw-r (score &optional style)
  (comment score "r")
  (moveto score 0.210 0.010)
  (curveto score 0.243 0.184 0.262 0.219 0.295 0.246)
  (curveto score 0.350 0.261 0.311 0.213 0.349 0.198)
  (curveto score 0.398 0.181 0.431 0.252 0.368 0.291)
  (curveto score 0.327 0.308 0.278 0.268 0.256 0.235)
  (curveto score 0.303 0.336 0.172 0.345 0.072 0.198)
  (lineto score 0.085 0.186)
  (curveto score 0.123 0.248 0.206 0.312 0.185 0.239)
  (lineto score 0.129 0.010)
  (lineto score 0.210 0.010)
  (if (not style) (fill-in score) (draw score style)))

(defvar r-bounds '(0.000 0.000 0.396 0.297))



;;; -------------------------------- NOTE HEADS and FLAGS --------------------------------

(defun draw-double-whole-note (score &optional style)
  (if (music-font score)
      (g-mustext score #o127 0 0.0)
    (progn
      (comment score "double whole note")
      (moveto score .298 .127)
      (curveto score 0.393 0.127 0.501 0.087 0.505 0.000)
      (curveto score 0.508 -0.095 0.393 -0.128 0.298 -0.127)
      (curveto score 0.207 -0.125 0.100 -0.087 0.100 0.000)
      (curveto score 0.102 0.091 0.204 0.127 0.298 0.127)
      (moveto score 0.263 .108)
      (curveto score 0.325 0.124 0.370 0.065 0.381 0.001)
      (curveto score 0.389 -0.050 0.387 -0.091 0.342 -0.108)
      (curveto score 0.286 -0.125 0.247 -0.075 0.230 -0.023)
      (curveto score 0.214 0.039 0.212 0.089 0.263 0.108)
      (if (not style) (fill-in score :even-odd t) (draw score style))
      (moveto score 0.000 0.166)
      (lineto score 0.031 0.166)
      (lineto score 0.031 -0.166)
      (lineto score 0.000 -0.166)
      (lineto score 0.000 0.166)
      (if (not style) (fill-in score) (draw score style))
      (moveto score 0.069 0.166)
      (lineto score 0.100 0.166)
      (lineto score 0.100 -0.166)
      (lineto score 0.069 -0.166)
      (lineto score 0.069 0.166)
      (if (not style) (fill-in score) (draw score style))
      (moveto score 0.505 0.166)
      (lineto score 0.537 0.166)
      (lineto score 0.537 -0.166)
      (lineto score 0.505 -0.166)
      (lineto score 0.505 0.166)
      (if (not style) (fill-in score) (draw score style))
      (moveto score 0.576 0.166)
      (lineto score 0.607 0.166)
      (lineto score 0.607 -0.166)
      (lineto score 0.576 -0.166)
      (lineto score 0.576 0.166)
      (if (not style) (fill-in score) (draw score style)))))

(defvar double-whole-note-bounds '(0.000 -0.166 0.607 0.166))


(defun draw-whole-note (score &optional style)
  (if (music-font score)
      (g-mustext score #o167 0 0.0)
    (progn
      (comment score "whole note")
      (moveto score 0.198 0.127)
      (curveto score 0.293 0.127 0.402 0.087 0.405 0.000)
      (curveto score 0.408 -0.095 0.293 -0.128 0.198 -0.127)
      (curveto score 0.107 -0.125 0.000 -0.087 0.000 0.000)
      (curveto score 0.002 0.091 0.104 0.127 0.198 0.127)
      (moveto score 0.163 0.108)
      (curveto score 0.225 0.124 0.270 0.065 0.281 0.001)
      (curveto score 0.289 -0.050 0.287 -0.091 0.242 -0.108)
      (curveto score 0.186 -0.125 0.147 -0.075 0.130 -0.023)
      (curveto score 0.114 0.039 0.112 0.089 0.163 0.108)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar whole-note-bounds '(0.000 -0.127 0.405 0.127))


(defun draw-half-note (score &optional style)
  (if (music-font score)
      (g-mustext score #o372 0 0.0)
    (progn
      (comment score "half note")
      (moveto score 0.020 -0.101)
      (curveto score -0.063 0.037 0.185 0.197 0.268 0.107)
      (curveto score 0.379 -0.010 0.137 -0.193 0.021 -0.101)
      (moveto score 0.043 -0.082)
      (curveto score 0.015 -0.007 0.207 0.125 0.239 0.087)
      (curveto score 0.291 0.027 0.097 -0.139 0.043 -0.082)
      (if (not style) (fill-in score :even-odd t) (draw score style)))))

(defvar half-note-bounds '(0.000 -0.131 0.291 0.134))


(defun draw-quarter-note (score &optional style)
  (if (music-font score)
      (g-mustext score #o317 0 0.0)
    (progn
      (comment score "quarter note")
      (moveto score 0.014 -0.088)
      (curveto score -0.014 -0.030 0.026 0.056 0.090 0.096)
      (curveto score 0.144 0.128 0.230 0.142 0.270 0.092)
      (curveto score 0.316 0.024 0.258 -0.060 0.190 -0.100)
      (curveto score 0.130 -0.126 0.066 -0.136 0.014 -0.088)
      (if (not style) (fill-in score) (draw score style)))))

(defvar quarter-note-bounds '(0.000 -0.120 0.286 0.124))


(defun draw-diamond (score &optional style)
  (comment score "diamond")
  (setf (line-width score) .03)
  (moveto score 0 0)
  (rlineto score .14 .14)
  (rlineto score .14 -.14)
  (rlineto score -.14 -.14)
  (rlineto score -.14 .14)
  (if (not style) (fill-in score) (draw score style))
  (setf (line-width score) 0))

(defvar diamond-bounds '(0.0 -.14 .28 .14))


(defun draw-diamond-1 (score &optional style)
  (comment score "diamond-1") ;Anders Vinjar
  (setf (line-width score) .03)
  (moveto score 0 0)
  (curveto score 0.04 0.03 0.10 .07 .14 .14)
  (curveto score .12 .05 .08 .03 0.0 0.0)
  (moveto score .28 .0)
  (curveto score  .24 -.03 .18 -.07 .14 -.14)
  (curveto score .16 -.07 .20 -.03 .28 .0)
  (moveto score .0 .0)
  (lineto score .05 .03)
  (curveto score .1 .0 .13 -.02 .17 -.08)
  (lineto score .14 -.14)
  (curveto score .10 -.07 .04 -0.03 0 0)
  (moveto score .28 .0)
  (lineto score .23 -.03)
  (curveto score .20 -.02 .14 .01 .11 .09)
  (lineto score .14 .14)
  (curveto score .18 .07 .24 .03 .28 .0)
  (fill-in score)
  (if (not style) (fill-in score :even-odd t) (draw score style))
  (setf (line-width score) 0))
  
(defun draw-filled-diamond-1 (score &optional style)
  (comment score "filled diamond-1")
  (setf (line-width score) .03)
  (moveto score 0 0)
  (curveto score 0.04 0.03 0.10 .07 .14 .14)
  (curveto score .18 .07 .24 .03  .28 .0)
  (curveto score .24 -.03 .18 -.07 .14 -.14)
  (curveto score .10 -.07 .04 -0.03 0 0)
  (if (not style) (fill-in score :even-odd t) (draw score style))
  (setf (line-width score) 0))
                              
(defvar diamond-1-bounds '(0.0 -.14 .28 .14))



(defun draw-rhythmX (score &optional style)
  (comment score "X")
  (moveto score 0.128 0.000)
  (lineto score 0.002 0.102)
  (lineto score 0.020 0.124)
  (lineto score 0.150 0.020)
  (lineto score 0.278 0.124)
  (lineto score 0.296 0.102)
  (lineto score 0.170 0.000)
  (lineto score 0.298 -0.102)
  (lineto score 0.278 -0.124)
  (lineto score 0.150 -0.020)
  (lineto score 0.020 -0.124)
  (lineto score 0.004 -0.104)
  (lineto score 0.128 0.000)
  (if (not style) (fill-in score) (draw score style)))

(defvar rhythmX-bounds '(0.000 -0.124 0.298 0.124))


(defun draw-circled-x (score &optional style)
  (let ((off .06)
	(size .75))
    (comment score "circled-X")
    (moveto score (* size (+ off 0.020)) (* size 0.124))
    (lineto score (* size (+ off 0.150)) (* size 0.020))
    (lineto score (* size (+ off 0.278)) (* size 0.124))
    (curveto score (* size (+ off 0.224)) (* size 0.190) (* size (+ off 0.085)) (* size 0.188) (* size (+ off 0.020)) (* size 0.124))
    (moveto score (* size (+ off 0.296)) (* size 0.102))
    (lineto score (* size (+ off 0.170)) (* size 0.000))
    (lineto score (* size (+ off 0.297)) (* size -0.101))
    (curveto score (* size (+ off 0.341)) (* size -0.056) (* size (+ off 0.339)) (* size 0.050) (* size (+ off 0.296)) (* size 0.102))
    (moveto score (* size (+ off 0.278)) (* size -0.124))
    (lineto score (* size (+ off 0.150)) (* size -0.020))
    (lineto score (* size (+ off 0.020)) (* size -0.124))
    (curveto score (* size (+ off 0.080)) (* size -0.189) (* size (+ off 0.221)) (* size -0.194) (* size (+ off 0.278)) (* size -0.124))
    (moveto score (* size (+ off 0.004)) (* size -0.104))
    (lineto score (* size (+ off 0.128)) (* size 0.000))
    (lineto score (* size (+ off 0.003)) (* size 0.102))
    (curveto score (* size (+ off -0.048)) (* size 0.055) (* size (+ off -0.044)) (* size -0.061) (* size (+ off 0.004)) (* size -0.104))
    (moveto score (* size (+ off -0.065)) (* size 0.003))
    (curveto score (* size (+ off -0.064)) (* size 0.264) (* size (+ off 0.360)) (* size 0.280) (* size (+ off 0.361)) (* size 0.000))
    (curveto score (* size (+ off 0.361)) (* size -0.278) (* size (+ off -0.059)) (* size -0.279) (* size (+ off -0.064)) (* size 0.000))
    (if (not style) (fill-in score) (draw score style))))
  
;(defvar circled-x-bounds (list (* .75 (+ .06 -0.065)) (* .75 -0.207) (* .75 (+ .06 0.361)) (* .75 0.204)))
(defvar circled-x-bounds '(-0.00375 -.015525 0.31575 0.153))


(defun draw-slash (score &optional style)
  (comment score "slash")
  (setf (line-width score) .1) 
  (moveto score 0 -.15) 
  (rlineto score .275 .275) 
  (draw score style) 
  (setf (line-width score) 0))

(defvar slash-bounds '(0.0 -.15 .275 .3))


(defun draw-mslash (score &optional style)
  (comment score "slash")
  (setf (line-width score) .05) 
  (moveto score .05 -.225) 
  (rlineto score .2 .45) 
  (draw score style) 
  (setf (line-width score) 0))

(defvar mslash-bounds '(0.0 -.2 .225 .4))


;;; these versions of 'triangle and 'square makes sure that the start
;;; and end of the drawn line is on the middle of one of its sides,
;;; ensuring 'sharp corners

(defun draw-triangle (score &optional style)
  (comment score "triangle")
  (setf (line-width score) .03)
  (moveto score  .14 -.11)
  (rlineto score -.14 0)
  (rlineto score .14 .25)
  (rlineto score .14 -.25)
  (rlineto score -.14 .0)
  (if (not style) (fill-in score) (draw score style))
  (setf (line-width score) 0))

(defvar triangle-bounds '(0.0 -.11 .30 .14))


(defun draw-square (score &optional style)
  (comment score "square")
  (setf (line-width score) .04)
  (moveto score .13 -.13)
  (rlineto score -.13 0)
  (rlineto score 0 .26)
  (rlineto score .26 0)
  (rlineto score 0 -.26)
  (rlineto score -.13 0)
  (if (not style) (fill-in score) (draw score style))
  (setf (line-width score) 0))

(defvar square-bounds '(0 -.13 .28 .13))


(defun draw-8th-flag-up (score &optional style) ;stem up, flag hangs down
  (if (music-font score)
      (g-mustext score #o152 -.265 -0.45)
    (progn
      (comment score "8th flag up")
      (moveto score 0.000 0.296)
      (lineto score 0.019 0.296)
      (curveto score 0.022 0.239 0.031 0.146 0.101 0.092)
      (curveto score 0.246 -0.053 0.275 -0.121 0.268 -0.260)
      (curveto score 0.260 -0.411 0.213 -0.482 0.182 -0.540)
      (lineto score 0.175 -0.535)
      (curveto score 0.221 -0.453 0.247 -0.358 0.244 -0.257)
      (curveto score 0.243 -0.119 0.118 0.000 0.021 0.000)
      (lineto score 0.021 -0.024)
      (lineto score 0.000 -0.024)
      (lineto score 0.000 0.296)
      (if (not style) (fill-in score) (draw score style)))))

(defvar up8th-bounds '(0.000 -0.543 0.282 0.296))


(defun draw-extend-flag-up (score &optional style)
  (if (music-font score)
      (g-mustext score #o373 -.265 0.2)
    (progn
      (comment score "extend flag up")
      (moveto score 0.000 0.296)
      (lineto score 0.021 0.296)
      (curveto score 0.025 0.196 0.065 0.158 0.122 0.094)
      (curveto score 0.213 -0.011 0.400 -0.160 0.264 -0.512)
      (lineto score 0.262 -0.494)
      (curveto score 0.319 -0.325 0.290 -0.185 0.240 -0.118)
      (curveto score 0.194 -0.051 0.065 0.076 0.026 0.033)
      (lineto score 0.026 -0.025)
      (lineto score 0.000 -0.025)
      (lineto score 0.000 0.296)
      (if (not style) (fill-in score) (draw score style)))))

(defvar extend-flag-up-bounds '(0.000 -0.401 0.280 0.212))


(defun draw-8th-flag-down (score &optional style)  ;invert by hand...
  (if (music-font score)
      (g-mustext score #o112 0 0.5)
    (progn
      (comment score "8th flag down")
      (moveto score 0.000 -0.296)
      (lineto score 0.019 -0.296)
      (curveto score 0.022 -0.239 0.031 -0.146 0.101 -0.092)
      (curveto score 0.246 0.053 0.275 0.121 0.268 0.260)
      (curveto score 0.260 0.411 0.213 0.482 0.182 0.540)
      (lineto score 0.175 0.535)
      (curveto score 0.221 0.453 0.247 0.358 0.244 0.257)
      (curveto score 0.243 0.119 0.118 0.000 0.021 0.000)
      (lineto score 0.021 0.024)
      (lineto score 0.000 0.024)
      (lineto score 0.000 -0.296)
      (if (not style) (fill-in score) (draw score style)))))

(defvar down8th-bounds '(0.000 -0.296 0.280 0.542))


(defun draw-extend-flag-down (score &optional style)
  (if (music-font score)
      (g-mustext score #o360 0 -0.2)
    (progn
      (comment score "extend flag down")
      (moveto score 0.000 -0.296)
      (lineto score 0.021 -0.296)
      (curveto score 0.025 -0.196 0.065 -0.158 0.122 -0.094)
      (curveto score 0.213 0.011 0.400 0.160 0.264 0.512)
      (lineto score 0.262 0.494)
      (curveto score 0.319 0.325 0.290 0.185 0.240 0.118)
      (curveto score 0.194 0.051 0.065 -0.076 0.026 -0.033)
      (lineto score 0.026 0.025)
      (lineto score 0.000 0.025)
      (lineto score 0.000 -0.296)
      (if (not style) (fill-in score) (draw score style)))))

(defvar extend-flag-down-bounds '(0.000 -0.214 0.280 0.382))



;;; -------------------------------- RESTS --------------------------------

(defun draw-whole-rest (score &optional style)
  (comment score "whole rest")
  (moveto score 0.063 0.253)
  (lineto score 0.063 0.127)
  (lineto score 0.359 0.127)
  (lineto score 0.359 0.253)
  (lineto score 0.422 0.253)
  (lineto score 0.422 0.256)
  (lineto score 0.000 0.256)
  (lineto score 0.000 0.253)
  (lineto score 0.063 0.253)
  (if (not style) (fill-in score) (draw score style)))

(defvar whole-rest-bounds '(0.000 0.000 0.420 0.256))


(defun draw-half-rest (score &optional style)
  (comment score "half rest")
  (moveto score 0.063 0.127)
  (lineto score 0.063 0.000)
  (lineto score 0.000 0.000)
  (lineto score 0.000 -0.003)
  (lineto score 0.422 -0.003)
  (lineto score 0.422 0.000)
  (lineto score 0.359 0.000)
  (lineto score 0.359 0.127)
  (lineto score 0.063 0.127)
  (if (not style) (fill-in score) (draw score style)))

(defvar half-rest-bounds '(0.000 -0.003 0.420 0.126))


(defun draw-quarter-rest (score &optional style)
  (if (music-font score)
      (g-mustext score #o316 0 0.0)
    (progn
      (comment score "quarter rest")
      (moveto score 0.072 0.358)
      (curveto score 0.120 0.253 0.160 0.158 0.038 0.058)
      (lineto score 0.182 -0.103)
      (curveto score -0.028 -0.032 -0.058 -0.248 0.180 -0.343)
      (lineto score 0.187 -0.322)
      (curveto score 0.042 -0.222 0.150 -0.103 0.230 -0.142)
      (lineto score 0.243 -0.127)
      (curveto score 0.220 -0.082 0.170 -0.043 0.157 0.012)
      (curveto score 0.165 0.120 0.290 0.113 0.237 0.187)
      (lineto score 0.103 0.355)
      (lineto score 0.072 0.358)
      (if (not style) (fill-in score) (draw score style)))))

(defvar quarter-rest-bounds '(0.000 -0.343 0.248 0.358))


(defun draw-8th-rest (score &optional style)
  (if (music-font score)
      (g-mustext score #o344 0 0.125)
    (progn
      (comment score "8th rest")
      (moveto score 0.164 0.131)
      (curveto score 0.171 0.222 0.046 0.236 0.028 0.136)
      (curveto score 0.032 0.030 0.159 0.033 0.224 0.086)
      (lineto score 0.140 -0.256)
      (lineto score 0.174 -0.256)
      (lineto score 0.286 0.200)
      (lineto score 0.259 0.200)
      (curveto score 0.249 0.140 0.188 0.074 0.142 0.079)
      (curveto score 0.154 0.096 0.166 0.114 0.164 0.131)
      (if (not style) (fill-in score) (draw score style)))))

(defvar rest8-bounds '(0.000 -0.256 0.286 0.202))


(defun draw-16th-rest (score &optional style)
  (if (music-font score)
      (g-mustext score #o305 0 0.125)
    (progn
      (comment score "16th rest")
      (moveto score 0.102 -0.170)
      (curveto score 0.128 -0.152 0.141 -0.128 0.136 -0.113)
      (curveto score 0.144 -0.032 0.017 -0.020 0.000 -0.113)
      (curveto score 0.000 -0.224 0.146 -0.224 0.208 -0.163)
      (lineto score 0.118 -0.503)
      (lineto score 0.150 -0.503)
      (lineto score 0.334 0.200)
      (lineto score 0.302 0.200)
      (curveto score 0.292 0.141 0.221 0.065 0.168 0.077)
      (curveto score 0.167 0.079 0.162 0.081 0.164 0.082)
      (curveto score 0.186 0.096 0.196 0.119 0.194 0.136)
      (curveto score 0.197 0.221 0.070 0.234 0.056 0.136)
      (curveto score 0.058 0.029 0.205 0.018 0.276 0.100)
      (lineto score 0.232 -0.068)
      (curveto score 0.221 -0.126 0.165 -0.181 0.106 -0.171)
      (if (not style) (fill-in score) (draw score style)))))

(defvar rest16-bounds '(0.000 -0.508 0.334 0.202))


(defun draw-32nd-rest (score &optional style)
  (if (music-font score)
      (g-mustext score #o250 0 -0.125)
    (progn
      (comment score "32nd rest")
      (moveto score 0.164 0.077)
      (curveto score 0.182 0.099 0.194 0.114 0.199 0.136)
      (curveto score 0.201 0.228 0.070 0.241 0.058 0.136)
      (curveto score 0.064 0.034 0.211 0.028 0.270 0.083)
      (lineto score 0.228 -0.076)
      (curveto score 0.219 -0.126 0.165 -0.174 0.107 -0.168)
      (curveto score 0.102 -0.170 0.109 -0.168 0.104 -0.168)
      (curveto score 0.124 -0.151 0.136 -0.133 0.136 -0.115)
      (curveto score 0.142 -0.031 0.007 -0.019 0.000 -0.115)
      (curveto score 0.003 -0.216 0.146 -0.216 0.206 -0.170)
      (lineto score 0.128 -0.510)
      (lineto score 0.172 -0.510)
      (lineto score 0.400 0.453)
      (lineto score 0.364 0.453)
      (curveto score 0.354 0.393 0.295 0.329 0.229 0.334)
      (curveto score 0.260 0.372 0.253 0.354 0.260 0.388)
      (curveto score 0.268 0.470 0.129 0.486 0.117 0.388)
      (curveto score 0.125 0.285 0.261 0.281 0.330 0.333)
      (lineto score 0.294 0.196)
      (curveto score 0.282 0.133 0.235 0.077 0.160 0.077)
      (if (not style) (fill-in score) (draw score style)))))

(defvar rest32-bounds '(0.000 -0.510 0.400 0.456))


(defun draw-64th-rest (score &optional style)
  (if (music-font score)
      (g-mustext score #o364 0 -0.125)
    (progn
      (comment score "64th rest")
      (moveto score 0.161 0.081)
      (curveto score 0.188 0.096 0.196 0.117 0.194 0.136)
      (curveto score 0.205 0.210 0.076 0.231 0.058 0.136)
      (curveto score 0.054 0.029 0.214 0.019 0.270 0.087)
      (lineto score 0.228 -0.076)
      (curveto score 0.216 -0.124 0.163 -0.184 0.102 -0.170)
      (curveto score 0.129 -0.151 0.138 -0.134 0.136 -0.115)
      (curveto score 0.147 -0.039 0.018 -0.013 0.000 -0.115)
      (curveto score 0.000 -0.220 0.151 -0.228 0.206 -0.170)
      (lineto score 0.122 -0.510)
      (lineto score 0.164 -0.510)
      (lineto score 0.454 0.708)
      (lineto score 0.419 0.708)
      (curveto score 0.409 0.644 0.341 0.566 0.281 0.588)
      (curveto score 0.306 0.610 0.314 0.630 0.314 0.642)
      (curveto score 0.325 0.721 0.189 0.734 0.176 0.642)
      (curveto score 0.179 0.536 0.333 0.532 0.386 0.590)
      (lineto score 0.347 0.434)
      (curveto score 0.341 0.384 0.284 0.325 0.225 0.334)
      (curveto score 0.246 0.350 0.254 0.370 0.254 0.388)
      (curveto score 0.261 0.466 0.134 0.486 0.117 0.388)
      (curveto score 0.119 0.284 0.266 0.279 0.328 0.338)
      (lineto score 0.296 0.207)
      (curveto score 0.285 0.154 0.235 0.063 0.164 0.080)
      (if (not style) (fill-in score) (draw score style)))))

(defvar rest64-bounds '(0.000 -0.510 0.454 0.708))


(defun draw-128th-rest (score &optional style)
  (if (music-font score)
      (g-mustext score #o345 0 -0.125)
    (progn
      (comment score "128th rest")
      (moveto score 0.277 0.586)
      (curveto score 0.306 0.604 0.314 0.623 0.314 0.641)
      (curveto score 0.320 0.727 0.189 0.733 0.176 0.641)
      (curveto score 0.181 0.536 0.343 0.526 0.386 0.589)
      (lineto score 0.350 0.434)
      (curveto score 0.341 0.400 0.309 0.323 0.226 0.333)
      (curveto score 0.246 0.350 0.254 0.369 0.254 0.389)
      (curveto score 0.267 0.469 0.134 0.480 0.119 0.389)
      (curveto score 0.111 0.286 0.284 0.274 0.326 0.336)
      (lineto score 0.296 0.209)
      (curveto score 0.286 0.179 0.253 0.067 0.164 0.079)
      (curveto score 0.189 0.096 0.194 0.114 0.194 0.136)
      (curveto score 0.200 0.223 0.073 0.226 0.059 0.136)
      (curveto score 0.059 0.023 0.221 0.027 0.269 0.079)
      (lineto score 0.229 -0.076)
      (curveto score 0.214 -0.117 0.180 -0.183 0.106 -0.170)
      (curveto score 0.129 -0.153 0.136 -0.133 0.136 -0.116)
      (curveto score 0.147 -0.033 0.004 -0.020 0.000 -0.116)
      (curveto score 0.004 -0.229 0.160 -0.224 0.204 -0.176)
      (lineto score 0.121 -0.510)
      (lineto score 0.164 -0.510)
      (lineto score 0.514 0.954)
      (lineto score 0.481 0.954)
      (curveto score 0.467 0.903 0.436 0.820 0.340 0.836)
      (curveto score 0.367 0.849 0.377 0.874 0.371 0.890)
      (curveto score 0.383 0.967 0.250 0.993 0.236 0.890)
      (curveto score 0.240 0.787 0.406 0.776 0.446 0.830)
      (lineto score 0.416 0.704)
      (curveto score 0.406 0.684 0.370 0.576 0.279 0.583)
      (if (not style) (fill-in score) (draw score style)))))

(defvar rest128-bounds '(0.000 -0.510 0.514 0.956))


(defun draw-measure-rest (score &optional style)
  (comment score "measure rest")
  (moveto score 0 .25)
  (rlineto score 0 -.5)
  (moveto score .5 .25)
  (rlineto score 0 -.5)
  (draw score style)
  (setf (line-width score) .225)
  (moveto score 0 0)
  (lineto score .5 0)
  (draw score style)
  (setf (line-width score) 0))

(defvar measure-rest-bounds '(0 -.25 .5 .25))

(defun draw-double-whole-rest (score &optional style)
  (comment score "double whole rest")
  (moveto score 0 .5)
  (rlineto score 0 .25)
  (rlineto score .15 0)
  (rlineto score 0 -.25)
  (rlineto score -.15 0)
  (if (not style) (fill-in score) (draw score style)))

(defvar double-whole-rest-bounds '(0 0 .5 .75))
  
  
