# -*- coding: utf-8 -*-

# Copyright (c) 2008 - 2010 Lukas Hetzenecker <LuHe@gmx.at>

import os
import sys
import math
from PyQt4.QtCore import *
from devices.status_numbers import *
from lib.classes import *
import lib.helper

USE_PYBLUEZ = False
USE_LIGHTBLUE = False

try:
   # PyBluez module for Linux and Windows
   import bluetooth
   USE_PYBLUEZ = True
except ImportError:
   # Lightblue for Mac OS X
   import lightblue
   import socket
   USE_LIGHTBLUE = True

class BluetoothError(Exception): pass
class InvalidAddressError(BluetoothError): pass
class InvalidProtocolVersion(BluetoothError): pass
class NotConnectedError(BluetoothError): pass
class NotFoundError(BluetoothError): pass

class Connection(QObject):
    def __init__(self,  parent):
        """Plugin for connections to series60 mobile phones over bluetooth"""
        super(Connection,  self).__init__(parent)
        self.__helper = lib.helper.Helper(self,  None)

        self.PLUGIN_NAME = "series60"
        self.PLUGIN_VERSION = 0.1
        self.PLUGIN_DEVICES = "Series60"

    def useThisPlugin(self,  database,  saveAllMessages=False,  logger=None):
        """Use the plugin"""
        # Create logger
        if logger == None:
            import logging
            import sys
            handler = logging.StreamHandler(sys.stdout)
            format = logging.Formatter("%(levelname)-8s - %(message)s",
                            "%d.%m.%Y %H:%M:%S")
            handler.setFormatter(format)
            self.__log = logging.getLogger()
            self.__log.addHandler(handler)
            self.__log.setLevel(logging.ERROR)
        else:
            self.__log = logger

        self.__database = database

        self.__saveAllMessages = saveAllMessages

        self.__connected = False
        self.__scan = ScanDevices(self)
        self.__listenThread = None
        self.__connectionStates = 4
    
    def __str__(self):
        return "\"Connection with series60\""

    def scan(self):
        """Scan for devices"""        
        self.__scan.start()

    def scanStop(self):
        self.__scan.stop()

    def connected(self):
        """Return connection state as Boolean"""
        return self.__connected

    def isValidBluetoothAddress(self,  address):
        if USE_PYBLUEZ:
           return bluetooth.is_valid_address(address)
        elif USE_LIGHTBLUE:
           return lightblue._lightbluecommon._isbtaddr(address)

    def connectToDevice(self,  device):
        """Connect to device"""
        if not self.isValidBluetoothAddress(device.bluetoothAddress()):
            raise InvalidAddressError,  str(device.bluetoothAddress()) + " is no valid address"

        self.__listenThread = Listen(self)
        self.__connected = False
        self.__connectionEstablishment = True
        self.__device = device
        self.__contacts = dict()
        self.__contactHashes = dict()
        self.__addedContacts = list()
        self.__changedContacts = list()
        self.__deletedContacts = list()
        self.__setRead = list()
        self.__waitForContactEndReply = False
        self.__newMessagesNumber = 0
        self.__firstRequest = True
        self.__partialMessage = ""
        self.__messageInQueue = False
        self.__currentMessage = None
        self.__messagesSent = list()
        self.__messagesPending = list()
        self.__contactAddQueue = list()

        if USE_PYBLUEZ:
            self.connect(self.__listenThread,  SIGNAL("finished()"),  self._lostConnection)
        else:
            self.connect(self.__listenThread,  SIGNAL("lostConnection"),  self._lostConnection)
        self.connect(self.__listenThread,  SIGNAL("dataAvailable"),  self.__parseData)

        self.__listenThread.start()

    def __parseData(self,  header,  message):
        self.__log.debug(QString("Received data: Header: %1 - Message: %2").arg(header).arg(message.replace(NUM_SEPERATOR,  " - ")))

        if header != NUM_PARTIAL_MESSAGE and self.__partialMessage:
            message = self.__partialMessage + message
            self.__partialMessage = ""

        if header == NUM_PARTIAL_MESSAGE:
            self.__partialMessage += message

        elif header == NUM_CONNECTED:
            if not float(message) == PROTOCOL_VERSION:
                self.emit(SIGNAL("connectionVersionMismatchError"),  float(message),  PROTOCOL_VERSION)
                #raise InvalidProtocolVersion,  str(message) + " protocol version on device, but " + str(PROTOCOL_VERSION)  + " needed!"
                return

            self.emit(SIGNAL("connectionStateChanged"),  1)
            self.__send(NUM_HELLO_REQUEST)

        elif header == NUM_HELLO_REPLY:
            self.emit(SIGNAL("connectionStateChanged"),  2)

            if self.__connectionEstablishment:
                self.refreshSysinfo()

        elif header == NUM_SYSINFO_REPLY_START:
            self.__device.clear()

        elif header == NUM_SYSINFO_REPLY_LINE:
            type = message.split(NUM_SEPERATOR)[0]
            value = message.split(NUM_SEPERATOR)[1]

            self.__log.debug(QString("Systeminfo - Type: %1 - Value: %2").arg(type,  value))

            if type == "model":
                value = self.__helper.getModel(value)

            self.__device.addValue(type,  value)

        elif header == NUM_SYSINFO_REPLY_END:
            if self.__database.deviceCheckValues(self.__device):
                for type,  value in self.__database.devices(bluetoothAddress=self.__device.bluetoothAddress()).next().values():
                    self.__device.addValue(type,  value)
            else:
                self.__database.deviceAddDetails(self.__device)

            self.emit(SIGNAL("sysinfoCompleted"))
            self.emit(SIGNAL("connectionStateChanged"),  3)

            if self.__connectionEstablishment:
                if self.__database.contactCount() > 0:
                    self.__send(NUM_CONTACTS_REQUEST_HASH_ALL)
                    self.__waitForContactEndReply = False
                else:
                    self.__log.info("No contactes saved, requesting all...")
                    self.__send(NUM_CONTACTS_REQUEST_CONTACTS_ALL)
                    self.__waitForContactEndReply = True

        elif header == NUM_CONTACTS_REPLY_HASH_ALL:
            mobileHash = message
            pcHash = self.__database.contactHash()

            self.__log.info(QString("Contact hash from device is %1 and pc hash is %2").arg(mobileHash).arg(pcHash))
            if mobileHash != pcHash:
                self.__log.info("Checksum of contacts aren't equal, requesting hash of each contact...")
                self.__send(NUM_CONTACTS_REQUEST_HASH_SINGLE)
            else:
                self.__log.info("Checksum of contacts on device and pc are equal")
                self.emit(SIGNAL("contactsCompleted"))
                self.__connectionCompleted()

        elif header == NUM_CONTACTS_REPLY_HASH_SINGLE_START:
            self.__contactHashes = dict()

        elif header == NUM_CONTACTS_REPLY_HASH_SINGLE_LINE:
            id = int(message.split(NUM_SEPERATOR)[0])
            hash = message.split(NUM_SEPERATOR)[1]
            self.__contactHashes[id] = hash

        elif header == NUM_CONTACTS_REPLY_HASH_SINGLE_END:
            self.__checkHashes()
            self.__contacts = dict()

        elif header == NUM_CONTACTS_REPLY_CONTACT_START:
            id = int(message.split(NUM_SEPERATOR)[0])
            name = message.split(NUM_SEPERATOR)[1]
            self.__contact = Contact(id=0,  idOnPhone=id,  name=name)

        elif header == NUM_CONTACTS_REPLY_CONTACT_LINE:
            id = int(message.split(NUM_SEPERATOR)[0])
            type = message.split(NUM_SEPERATOR)[1]
            location = message.split(NUM_SEPERATOR)[2]
            value = message.split(NUM_SEPERATOR)[3]
            self.__contact.addValue(ContactField(type,  location),  value)

        elif header == NUM_CONTACTS_REPLY_CONTACT_END:
            id = self.__contact.idOnPhone()
            assert id == int(message.split(NUM_SEPERATOR)[0]),  "ID mismatch"
            if id in self.__addedContacts:
                self.__addedContacts.remove(id)
                self.__database.contactAdd(self.__contact)
            elif self.__waitForContactEndReply:
                self.__database.contactAdd(self.__contact)
            elif id in self.__changedContacts:
                self.__changedContacts.remove(id)
                self.__database.contactChange(self.__contact)

            if not (self.__addedContacts  or self.__changedContacts or self.__waitForContactEndReply):
                self.emit(SIGNAL("contactsCompleted"))
                self.__connectionCompleted()

        elif header == NUM_CONTACTS_REPLY_CONTACTS_ALL_END:
            self.emit(SIGNAL("contactsCompleted"))
            self.__connectionCompleted()

        elif header == NUM_MESSAGE_REPLY_LINE:
            msg = Message()

            type = message.split(NUM_SEPERATOR)[0]
            if type == "inbox":
                msg.setType(MessageType.Incoming)
            elif type == "sent":
                msg.setType(MessageType.Outgoing)

            msg.setDevice(self.__device)
            msg.setIdOnPhone(int(message.split(NUM_SEPERATOR)[1]))
            msg.setDateTime(float(message.split(NUM_SEPERATOR)[2]))
            msg.setContact(Contact(name=message.split(NUM_SEPERATOR)[3]))
            msg.setMessage(message.split(NUM_SEPERATOR)[4])
            msg.setId(self.__database.messageUpdate(msg))

            self.emit(SIGNAL("messageLine"),  msg)
            self.__newMessagesNumber += 1

            if msg.id() in self.__setRead:
                self.__log.info(QString("We got message with id %1 - set it read.").arg(msg.id()))
                self.setRead(msg)
                self.__setRead.remove(msg.id())

        elif header == NUM_MESSAGE_REPLY_END:
            self.emit(SIGNAL("messagesRequestComplete"),  self.__newMessagesNumber)
            if self.__firstRequest:
                self.__send(NUM_MESSAGE_REQUEST_UNREAD)
                self.__firstRequest = False

        elif header == NUM_MESSAGE_REPLY_UNREAD:
            if message:
                for id in message.split(NUM_SEPERATOR):
                    if id:
                        id = int(id)
                        msg = self.__database.messages(messageIdOnPhone=id).next()
                        self.emit(SIGNAL("messageUnread"),  msg)

        elif header == NUM_MESSAGE_SEND_REPLY_OK:
            self.__currentMessage.addState(MessageState.SendOk,  message)
            self.__prepareNextMessage()
            self.__newMessages()

        elif header == NUM_MESSAGE_SEND_REPLY_STATUS:
            self.__currentMessage.addState(MessageState.Unknown,  message)

        elif header == NUM_MESSAGE_SEND_REPLY_FAILURE:
            self.__currentMessage.addState(MessageState.SendFailed,  message)
            self.__prepareNextMessage()

        elif header == NUM_MESSAGE_SEND_REPLY_RETRY:
            if "wait" not in self.__currentMessage.contact().internalValues():
                self.__currentMessage.contact().addInternalValue("wait",  True)
            self.__currentMessage.addState(MessageState.SendFailed,  "retry sending (%s) " % message)
            self.__prepareNextMessage(retry=True)

        elif header == NUM_MESSAGE_NEW:
            msg = Message()
            msg.setType(MessageType.Incoming)
            msg.setDevice(self.__device)
            msg.setContact(self.__database.contactByName(message.split(NUM_SEPERATOR)[2]))
            msg.setDateTime(float(message.split(NUM_SEPERATOR)[1]))
            msg.setMessage(message.split(NUM_SEPERATOR)[3])

            # The message id shouldn't be saved in the database, because there could be other messages with a higher id
            msg.setIdOnPhone(0)
            msg.setId(self.__database.messageAdd(msg))

            msg.setIdOnPhone(int(message.split(NUM_SEPERATOR)[0]))
            self.__newMessages()

            self.emit(SIGNAL("messageNew"),  msg)

        elif header == NUM_CONTACTS_ADD_REPLY_ID:
            id = int(message)
            contact = self.__contactAddQueue.pop()
            contact.setIdOnPhone(id)
            contact.setId(self.__database.contactAdd(contact))
            self.contactChange(contact,  list(),  contact.values())

            self.emit(SIGNAL("contactsCompleted"))

        elif header == NUM_DEBUG:
            self.__log.error(QString("Debug data: %1").arg(message))

        else:
            self.__log.error(QString("Unknown header: %1 - Data: %2").arg(header).arg(message))

    def __connectionCompleted(self):
        if self.__device and self.__device.name():
            self.__connected = True
            self.emit(SIGNAL("connectionStateChanged"),  4)
            self.__connectionEstablishment = False
            self.emit(SIGNAL("connectionCompleted"))
            self.__newMessages()

    def __checkHashes(self):
        pcHashes = self.__database.contactHashSingle()
        self.__addedContacts = list()
        self.__changedContacts = list()
        self.__deletedContacts = list()
        for id,  hash in self.__contactHashes.iteritems():
            if not id in pcHashes:
                self.__addedContacts.append(id)
            elif not hash == pcHashes[id]:
                self.__changedContacts.append(id)

        for id,  hash in pcHashes.iteritems():
            if not id in self.__contactHashes:
                self.__deletedContacts.append(id)

        self.__log.info(QString("Requesting added contacts with id %1").arg(str(self.__addedContacts)))
        self.__requestContacts(self.__addedContacts)

        self.__log.info(QString("Requesting changed contacts with id %1").arg(str(self.__changedContacts)))
        self.__requestContacts(self.__changedContacts)

        self.__log.info(QString("Removing deleted contacts with id %1").arg(str(self.__deletedContacts)))
        self.__database.contactsRemove(self.__deletedContacts)

        if (self.__deletedContacts and not (self.__addedContacts or self.__changedContacts)):
            self.emit(SIGNAL("contactsCompleted"))
            self.__connectionCompleted()

    def __send(self,  header,  *message):
        new_message = ""

        if len(message) == 1:
            new_message = unicode(message[0])
        else:
            for part in message:
                new_message += unicode(part) + str(NUM_SEPERATOR)

        length = 600
        if len(new_message) > length:
            parts = int(math.ceil(len(new_message) / float(length)))
            sentParts = 0
            for i in range(parts):
                part = new_message[sentParts*length:sentParts*length+length]
                if sentParts == parts-1:
                    self.__send(header,  part)
                else:
                    self.__send(NUM_PARTIAL_MESSAGE,  part)
                sentParts += 1
            return

        self.__log.debug(QString("Sent data: Header: %1 - Message: %2").arg(header).arg(new_message))
        
        self.emit(SIGNAL("requestSend(QString)"),  QString(str(header) + str(NUM_END_HEADER) + unicode(new_message).encode("utf8") + '\n'))

    def __sendMessage(self,  message):
        self.__messageInQueue = True
        self.__currentMessage = message

        name = message.contact().name()
        phone = message.contact().internalValue("phone")
        msg = message.messageForPhone()
        assert name and phone and message,  "invalid message"

        self.__send(NUM_MESSAGE_SEND_REQUEST,  name,  phone,  msg)

        message.addState(MessageState.Sending,  "Message transfered to device.")
        #self.emit(SIGNAL("messageStateChanged"))

    def __prepareNextMessage(self,  retry=False):
        if self.__messageInQueue:
            self.__messagesPending.remove(self.__currentMessage)

            if retry:
                self.__messagesPending.append(self.__currentMessage)
            else:
                self.__messagesSent.append(self.__currentMessage)

            self.emit(SIGNAL("messageSent"),  self.__currentMessage)
        self.__messageInQueue = False

        if self.__messagesPending:
            message = self.__messagesPending[0]
            if "wait" in message.contact().internalValues():
                sleep = QTimer(self)
                sleep.setInterval(10000)
                sleep.setSingleShot(True)
                sleep.start()
                self.connect(sleep,  SIGNAL("timeout()"),  lambda: self.__sendMessage(message))
            else:
                self.__sendMessage(message)

    def _lostConnection(self):
        if self.__device and self.__device.name():
            if self.__connected:
                self.emit(SIGNAL("connectionClosed"),  self.__device.name())
            else:
                self.emit(SIGNAL("connectionAborted"),  self.__device.name())
        
        self.__connected = False
        self.__newMessagesCompleted = False
        self.__contacts = dict()
        self.__device = Device()
    
    def __newMessages(self):
        if self.__saveAllMessages:
            self.__newMessagesNumber = 0
            last = self.__database.messageLastIdOnPhone(self.__device)
            self.emit(SIGNAL("messagesRequest"))
            self.__send(NUM_MESSAGE_REQUEST,  last)

    def __requestContacts(self,  list):
        for id in list:
            self.__send(NUM_CONTACTS_REQUEST_CONTACT,  id)

    def sendMessage(self,  message):
        """Send a text message to the contact message.contact()"""
        message.setId(self.__database.messageAdd(message))
        message.addState(MessageState.Created, "Message created on computer.")
        self.__messagesPending.append(message)
        self.emit(SIGNAL("messageQueued"),  message)

        if not self.__messageInQueue:
            self.__prepareNextMessage()

    def setRead(self,  message,  state=True,  send=True):
        """Set a message as read or unread"""
        if send:
            if message.idOnPhone() == 0 and message.id() > 0:
                self.__log.info(QString("We don't know the message id on the phone, but we'll wait until we get it. PC id is %1").arg(message.id()))
                if message.id() not in self.__setRead:
                    self.__setRead.append(message.id())
            else:
                self.__send(NUM_SET_READ,  message.idOnPhone(),  state)
        if state:
            self.emit(SIGNAL("messageRead"),  message)

    def contactAdd(self,  contact):
        self.__contactAddQueue.append(contact)
        self.__send(NUM_CONTACTS_ADD)

    def contactRemove(self,  contact):
        self.__send(NUM_CONTACTS_DELETE,  contact.idOnPhone())

    def contactChange(self,  contact,  remove,  add):
        assert contact.idOnPhone() != 0
        for field,  value in remove:
            value = value.replace(u'\n', u'\u2029') # LINE FEED (\u000a) replaced by PARAGRAPH SEPARATOR (\u2029)
            self.__send(NUM_CONTACTS_CHANGE_REMOVEFIELD,  contact.idOnPhone(),  field.type(),  field.location(),  value)
        for field,  value in add:
            value = value.replace(u'\n', u'\u2029') # LINE FEED (\u000a) replaced by PARAGRAPH SEPARATOR (\u2029)
            self.__send(NUM_CONTACTS_CHANGE_ADDFIELD,  contact.idOnPhone(),  field.type(),  field.location(),  value)

    def refreshSysinfo(self):
        """Refresh system informations"""
        self.__send(NUM_SYSINFO_REQUEST,  int(not self.__database.deviceCheckValues(self.__device)))

    def device(self):
        """Return connected device"""
        return self.__device

    def pendingMessages(self):
        """Return a list with all pending messages"""
        if not self.__connected:
            raise NotConnectedError,  "No active connection!"

        return self.__messagesPending

    def sentMessages(self):
        """Return a list with all sent messages"""
        if not self.__connected:
            raise NotConnectedError,  "No active connection!"

        return self.__messagesSent

    def closeConnection(self,  sendQuit=True):
        """Close connection"""
        self.__log.info(QString("Trying to close the connection"))

        if sendQuit:
            self.__send(NUM_QUIT)
        
        self._lostConnection()

if USE_PYBLUEZ:
    if sys.platform == "win32":
        # Asynchronous mode isn't implemented
        class ScanDevices(QThread):
            def __init__(self,  parent):
                super(ScanDevices,  self).__init__(parent)
                self.__parent = parent

            def stop(self):
                # TODO: Kill this thread?
                pass

            def run(self):
                try:
                    self.__parent.emit(SIGNAL("scanStarted"))
                    
                    devices = dict(bluetooth.discover_devices( lookup_names=True))
                    for address,  name in devices.iteritems():
                        self.__parent.emit(SIGNAL("foundDevice"),  address,  name)
                    self.__parent.emit(SIGNAL("scanCompleted"))
                except IOError,  msg:
                    self.__parent.emit(SIGNAL("scanFailed"), msg[0])
        
    else:
        class ScanDevices(QThread,  bluetooth.DeviceDiscoverer):
            def __init__(self,  parent):
                QThread.__init__(self,  parent)
                bluetooth.DeviceDiscoverer.__init__(self)
                self.__parent = parent
                
                self.__running = False

            def run(self):
                self.__running = True
                
                try:
                   self.find_devices(lookup_names = True, flush_cache = True)
                   self.process_inquiry()
                except bluetooth.btcommon.BluetoothError, msg:
                   self.__parent.emit(SIGNAL("scanFailed"), msg[0])
                   self.__running = False

            def stop(self):
                if self.__running:
                    self.cancel_inquiry()
                    self.__parent.emit(SIGNAL("scanCompleted"))

            def pre_inquiry(self):
                self.__parent.emit(SIGNAL("scanStarted"))

            def device_discovered(self, address, device_class, name) :
                self.__parent.emit(SIGNAL("foundDevice"),  address,  name)

            def inquiry_complete(self):
                self.__parent.emit(SIGNAL("scanCompleted"))
                self.__running = False

    class Listen(QThread):
        def __init__(self,  parent):
            super(Listen,  self).__init__(parent)
            self.parent = parent
        
        def run(self):
            self.connect(self.parent,  SIGNAL("requestSend(QString)"),  self.send)
            
            try:
                self.sock = bluetooth.BluetoothSocket( bluetooth.RFCOMM )
                self.sock.connect((self.parent.device().bluetoothAddress(),  self.parent.device().port()))
            except Exception, msg:
                if os.name == "nt":
                    errno = msg.errno
                    errmsg = msg.message
                else:
                    errno, errmsg = eval(msg[0]) # msg.message has been deprecated as of Python 2.6
                errmsg = unicode(errmsg,  "utf8")
                self.parent.emit(SIGNAL("connectionFailed"),  errno,  errmsg)
                return

            data = u""
            while True:
                try:
                    recv = unicode(self.sock.recv(1000),  "utf8")
                    if recv:
                        data += recv
                        
                        # Last part is either empty or the beginning of the next data segment
                        data = data.split(NUM_END_TEXT)
                        for part in data[:-1]:
                            # Extract header and message 
                            header = int(part.split(NUM_END_HEADER)[0])
                            message = unicode(part.split(NUM_END_HEADER)[1])

                            # Quit the thread -> finished() signal emitted
                            if header  == NUM_QUIT:
                                return
                            self.emit(SIGNAL("dataAvailable"),  header,  message)
                            
                        data = data[-1]
                    else:
                        # Got an empty string -> connection is closed
                        self.disconnect(self.parent,  SIGNAL("requestSend(QString)"),  self.send)
                        return
                except Exception:
                    self.disconnect(self.parent,  SIGNAL("requestSend(QString)"),  self.send)
                    return
        
        def send(self,  data):
            data = str(data.toAscii())
            try:
                self.sock.send(data)
            except Exception:
                self.disconnect(self.parent,  SIGNAL("requestSend(QString)"),  self.send)
                self.emit(SIGNAL("lostConnection"))
                self.exit(1)

elif USE_LIGHTBLUE:
    class ScanDevices(object):
        def __init__(self,  parent):
            self.__parent = parent
            self.__scan = lightblue._lightblue._AsyncDeviceInquiry.alloc().init()
            self.__scan._setupdatenames(True)
            self.__scan.cb_started= self.scanStarted
            self.__scan.cb_completed = self.scanCompleted
            self.__scan.cb_founddevice = self.scanFoundDevice
        
        def start(self):
            ret = self.__scan.start()
            if ret != lightblue._macutil.kIOReturnSuccess:
                self.__parent.emit(SIGNAL("scanFailed"), str(ret))
 
        def stop(self):
            self.__scan.stop()

        def scanStarted(self):
            self.__parent.emit(SIGNAL("scanStarted"))

        def scanFoundDevice(self,  device):
            name = device.getName()
            if not name:
                name = lightblue.finddevicename(device.getAddressString())
            addr = device.getAddressString().replace("-", ":").encode('ascii').upper()
            self.__parent.emit(SIGNAL("foundDevice"),  addr,  name)
        
        def scanCompleted(self,  err, aborted):
            if err:
                self.__parent.emit(SIGNAL("scanFailed"), str(err))
            else:
                self.__parent.emit(SIGNAL("scanCompleted"))

    class Listen(QObject):
        def __init__(self,  parent):
            super(Listen,  self).__init__(parent)
            self.parent = parent
            
            self.connect(self.parent,  SIGNAL("requestSend(QString)"),  self.send)
            
        def start(self):
            self.__data = u""
            
            self.sock = lightblue.socket( lightblue.RFCOMM )
            self.sock.setblocking(False)
            try:
                self.sock.connect((self.parent.device().bluetoothAddress(),  self.parent.device().port()))
                self.startTimer(60)
            except socket.error,  msg:
                self.parent.emit(SIGNAL("connectionFailed"),  msg[0],  msg[1])
        
        def timerEvent(self,  event):
            try:
                recv = unicode(self.sock.recv(10000),  "utf8")
            except socket.error,  msg:
                if msg[0] != 35: # 35 = Resource temporarily unavailable (due to non-blocking mode)
                    self.parent._lostConnection()
                return
            except Exception:
                self.parent._lostConnection()
                return
            
            self.__data += recv
            # Last part is either empty or the beginning of the next data segment
            self.__data = self.__data.split(NUM_END_TEXT)
            for part in self.__data[:-1]:
                # Extract header and message 
                header = int(part.split(NUM_END_HEADER)[0])
                message = unicode(part.split(NUM_END_HEADER)[1])

                # Quit the thread -> finished() signal emitted
                if header  == NUM_QUIT:
                    self.parent._lostConnection()
                    return
                self.emit(SIGNAL("dataAvailable"),  header,  message)
                
            self.__data = self.__data[-1]

        def send(self,  data):
            data = str(data.toAscii())
            try:
                self.sock.send(data)
            except:
                self.emit(SIGNAL("lostConnection"))
