// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/interface/jit_intf.h,v 1.13 2001/11/22 01:29:32 gwu2 Exp $
//



#ifndef _JIT_INTF_H_
#define _JIT_INTF_H_


#include "orp_types.h"



//
// The implementation of those types is private to the ORP.
// The JIT uses them as handles.
//

#ifndef ORP_POSIX
enum Loader_Exception;
#endif

class ORP;

typedef void *Compile_Handle;
typedef void *Class_Loader_Handle;
typedef void *Class_Handle;
typedef void *Field_Handle;
typedef void *Method_Handle;
typedef void *JIT_Handle;
typedef void *Method_Signature_Handle;
typedef const void *Arg_List_Iterator;       // Java only
typedef const void *Argument_List_Iterator;  // Java and CLI
typedef const void *Arg_List_Iter;
typedef void *Method_Iterator;




enum JIT_Result {
    JIT_SUCCESS,
    JIT_RESOLVE_ERROR,
    JIT_FAILURE
}; //JIT_Result


enum Method_Side_Effects {
    MSE_Unknown,
    MSE_True,
    MSE_False
}; //Method_Side_Effects



//
// This is currently not used.
//
// Max 32 bits, so that it fits in one word.
struct JIT_Flags {

    // The JIT should generate write barriers.
    Boolean insert_write_barriers : 1;

    ///////////////////////////////////////////////////////////////////////////
    // Below are just examples of what we could have.

    // The JIT should generate code with debug info.
    Boolean generate_debug_info : 1;

    // The JIT should generate code to poll for GC on back edges and at call sites.
    Boolean poll_gc : 1;

    // Write barrier type:
    // 1 - tracing
    // 0 - replicating
    // For replicating barriers the JIT must insert write barriers for all
    // fields.  For tracing barriers only writes to fields which hold references
    // must be recorded.
    Boolean tracing_write_barriers : 1;

}; //JIT_Flags





///////////////////////////////////////////////////////////////////////////////
// Access flags for Class, Class_Member, Field and Method.
///////////////////////////////////////////////////////////////////////////////
									//		used by
									//
#define ACC_PUBLIC			0x0001	// Class	Field	Method
#define ACC_PRIVATE			0x0002	//			Field	Method
#define ACC_PROTECTED		0x0004	//			Field	Method
#define ACC_STATIC			0x0008	//			Field	Method
#define ACC_FINAL			0x0010	// Class	Field	Method
#define ACC_SUPER			0x0020	// Class
#define	ACC_SYNCHRONIZED	0x0020	//					Method
#define	ACC_VOLATILE		0x0040	//			Field
#define	ACC_TRANSIENT		0x0080	//			Field
#define ACC_NATIVE			0x0100	//					Method
#define	ACC_INTERFACE		0x0200	// Class
#define	ACC_ABSTRACT		0x0400	// Class			Method



//////////////// begin C interface


//
// calls from Compiler to ORP
//
//-----------------------------------------------------------------------------
// Constant pool resolution
//-----------------------------------------------------------------------------
//
// The following byte codes reference constant pool entries:
//
//  field
//      getstatic           static field
//      putstatic           static field
//      getfield            non-static field
//      putfield            non-static field
//
//  method
//      invokevirtual       virtual method
//      invokespecial       special method
//      invokestatic        static method
//      invokeinterface     interface method
//
//  class
//      new                 class
//      anewarray           class
//      checkcast           class
//      instanceof          class
//      multianewarray      class
//

//
// For the method invocation byte codes, certain linkage exceptions are thrown 
// at run-time:
//
//  (1) invocation of a native methods throws the UnsatisfiedLinkError if the 
//      code that implements the method cannot be loaded or linked.
//  (2) invocation of an interface method throws 
//          - IncompatibleClassChangeError if the object does not implement 
//            the called method, or the method is implemented as static
//          - IllegalAccessError if the implemented method is not public
//          - AbstractMethodError if the implemented method is abstract
//



//
// resolve constant pool reference to a non-static field
// used for getfield and putfield
//
ORPExport Field_Handle 
resolve_nonstatic_field(Compile_Handle h,
                        Class_Handle c,
                        unsigned index,
                        Loader_Exception *exc);



//
// resolve constant pool reference to a static field
// used for getstatic and putstatic
//
ORPExport Field_Handle
resolve_static_field(Compile_Handle h,
                     Class_Handle c,
                     unsigned index,
                     Loader_Exception *exc);



ORPExport Field_Handle
resolve_field(Compile_Handle h,
              Class_Handle c,
              unsigned index,
              Loader_Exception *exc);



//
// resolve constant pool reference to a virtual method
// used for invokevirtual and invoke special.
//
// DEPRECATED
//
ORPExport Method_Handle 
resolve_nonstatic_method(Compile_Handle h,
                         Class_Handle c,
                         unsigned index,
                         Loader_Exception *exc);



//
// resolve constant pool reference to a virtual method
// used for invokevirtual
//
ORPExport Method_Handle 
resolve_virtual_method(Compile_Handle h,
                         Class_Handle c,
                         unsigned index,
                         Loader_Exception *exc);



//
// resolve constant pool reference to a virtual method
// used for invokespecial
//
ORPExport Method_Handle 
resolve_special_method(Compile_Handle h,
                       Class_Handle c,
                       unsigned index,
                       Loader_Exception *exc);



//
// resolve constant pool reference to a static method
// used for invokestatic
//
ORPExport Method_Handle 
resolve_static_method(Compile_Handle h,
                      Class_Handle c,
                      unsigned index,
                      Loader_Exception *exc);



//
// resolve constant pool reference to a special method
// used for invokeinterface
//
ORPExport Method_Handle 
resolve_interface_method(Compile_Handle h,
                         Class_Handle c,
                         unsigned index,
                         Loader_Exception *exc);



// New API for both Java and CLI
ORPExport Method_Handle 
resolve_method(Compile_Handle h,
               Class_Handle c,
               unsigned index,
               Loader_Exception *exc);


// Standalone signatures are for CLI only, but it may make sense
// to use them in Java for consistency
Method_Signature_Handle 
resolve_method_signature(Compile_Handle h,
                         Class_Handle c,
                         unsigned index,
                         Loader_Exception *exc);


//
// resolve constant pool reference to a class
// used for
//      (1) new 
//              - InstantiationError exception if resolved class is abstract
//      (2) anewarray
//      (3) multianewarray
//
// resolve_class_new is used for resolving references to class entries by the
// the new byte code.
//
ORPExport Class_Handle 
resolve_class_new(Compile_Handle h,
                  Class_Handle c,
                  unsigned index,
                  Loader_Exception *exc);

ORPExport Class_Handle 
orp_resolve_class_new(Compile_Handle h,
                      Class_Handle c,
                      unsigned index,
                      Loader_Exception *exc);


//
// resolve_class is used by all the other byte codes that reference classes,
// as well as exception handlers.
//
ORPExport Class_Handle 
resolve_class(Compile_Handle h,
              Class_Handle c,
              unsigned index,
              Loader_Exception *exc);

ORPExport Class_Handle 
orp_resolve_class(Compile_Handle h,
                  Class_Handle c,
                  unsigned index,
                  Loader_Exception *exc);



ORPExport Class_Handle 
resolve_class_array_of_class(Compile_Handle h,
                             Class_Handle c,
                             Loader_Exception *exc);





//
// Field
//
ORPExport Boolean      field_is_static(Field_Handle f);
ORPExport Boolean      field_is_final(Field_Handle f);
ORPExport Boolean      field_is_private(Field_Handle f);
ORPExport unsigned     field_get_flags(Field_Handle f);
ORPExport unsigned     field_get_offset(Field_Handle f);
ORPExport void        *field_get_addr(Field_Handle f);
ORPExport const char  *field_get_name(Field_Handle f);
ORPExport Java_Type    field_get_type(Field_Handle f);
ORPExport const char  *field_get_descriptor(Field_Handle f);
ORPExport Class_Handle field_get_class(Field_Handle m);




//
// Method
//
ORPExport Boolean      method_is_static(Method_Handle m);
ORPExport Boolean      method_is_final(Method_Handle m);
ORPExport Boolean      method_is_synchronized(Method_Handle m);
ORPExport Boolean      method_is_private(Method_Handle m);
ORPExport Boolean      method_is_native(Method_Handle m);
ORPExport Byte        *method_allocate_code_block(Method_Handle m, JIT_Handle j,
                                                  size_t size);
ORPExport Byte        *method_allocate_data_block(Method_Handle m, JIT_Handle j,
                                                  size_t size);
ORPExport Byte        *method_allocate_info_block(Method_Handle m, JIT_Handle j,
                                                  size_t size);
ORPExport Byte        *method_allocate_jit_data_block(Method_Handle m, JIT_Handle j,
                                                      size_t size);
ORPExport Byte        *method_get_info_block(Method_Handle m);
ORPExport Byte        *method_get_info_block(Method_Handle m, JIT_Handle j);
ORPExport unsigned     method_get_info_block_size(Method_Handle m);
ORPExport unsigned     method_get_info_block_size(Method_Handle m, JIT_Handle j);
ORPExport Byte        *method_get_code_block_addr(Method_Handle m);
ORPExport Byte        *method_get_code_block_addr(Method_Handle m, JIT_Handle j);
ORPExport unsigned     method_get_code_block_size(Method_Handle m);
ORPExport unsigned     method_get_code_block_size(Method_Handle m, JIT_Handle j);
ORPExport const Byte  *method_get_byte_code_addr(Method_Handle m);
ORPExport size_t       method_get_byte_code_size(Method_Handle m);
ORPExport unsigned     method_get_max_locals(Method_Handle m);
ORPExport unsigned     method_get_max_stack(Method_Handle m);
ORPExport unsigned     method_get_flags(Method_Handle m);
ORPExport unsigned     method_get_offset(Method_Handle m);
ORPExport void        *method_get_indirect_address(Method_Handle m);
ORPExport const char  *method_get_name(Method_Handle m);
ORPExport const char  *method_get_descriptor(Method_Handle m);
ORPExport Class_Handle method_get_class(Method_Handle m);
ORPExport Boolean      method_is_overridden(Method_Handle m);
ORPExport Boolean      method_uses_fastcall(Method_Handle m);

ORPExport unsigned method_get_line_number_table_size(Method_Handle m);
ORPExport void     method_get_line_number(Method_Handle m,
                                          unsigned index,
                                          unsigned *start_pc,
                                          unsigned *line_number);

#ifdef STACK_TRACE_ADDITIONAL_INFO
ORPExport void     method_set_nativecode_offset_for_line(Method_Handle m,
                                          unsigned index,
										  unsigned npc_offset);
#endif

ORPExport unsigned method_get_local_vars_table_size(Method_Handle m);

ORPExport JIT_Handle   method_get_JIT_id(Compile_Handle h);

// Functions to iterate over all methods compiled by a given JIT.
#define JAVA_METHOD_END 0
ORPExport Method_Iterator method_get_first_method_jit(JIT_Handle j);
ORPExport Method_Iterator method_get_next_method_jit(Method_Iterator m);
ORPExport Method_Handle   method_get_method_jit(Method_Iterator m);

ORPExport unsigned     method_get_num_handlers(Method_Handle m);
ORPExport void         method_get_handler_info(Method_Handle m,
                                               unsigned handler_id,
                                               unsigned *begin_offset,
                                               unsigned *end_offset,
                                               unsigned *handler_offset,
                                               unsigned *handler_class_index);

ORPExport void         method_set_num_target_handlers(Method_Handle m,
                                                      JIT_Handle j,
                                                      unsigned num_handlers);
ORPExport void         method_set_target_handler_info(Method_Handle m,
                                                      JIT_Handle j,
                                                      unsigned      eh_number,
                                                      void         *start_ip,
                                                      void         *end_ip,
                                                      void         *handler_ip,
                                                      Class_Handle  catch_cl,
                                                      bool exc_obj_is_dead = false
                                                      );
#ifdef CLI_TESTING
ORPExport Boolean method_handler_has_filter(Method_Handle m,
                                            unsigned handler_id);
ORPExport Boolean method_handler_has_finally(Method_Handle m,
                                             unsigned handler_id);
ORPExport Boolean method_handler_has_fault(Method_Handle m,
                                           unsigned handler_id);
ORPExport void         method_get_handler_info_full(Method_Handle m,
                                                    unsigned handler_id,
                                                    unsigned *begin_offset,
                                                    unsigned *end_offset,
                                                    unsigned *handler_offset,
                                                    unsigned *handler_len,
                                                    unsigned *filter_offset,
                                                    unsigned *handler_class_index);
#endif

ORPExport Method_Side_Effects method_get_side_effects(Method_Handle m);
ORPExport void method_set_side_effects(Method_Handle m, Method_Side_Effects mse);


ORPExport Java_Type    method_get_return_type(Method_Handle m);
ORPExport Class_Handle method_get_return_type_class(Method_Handle m,
                                                    Loader_Exception *exc);

ORPExport Arg_List_Iterator  initialize_arg_list_iterator(const char *descr);
ORPExport Arg_List_Iterator  method_get_argument_list(Method_Handle m);
ORPExport Java_Type          curr_arg(Arg_List_Iterator iter);
ORPExport Class_Handle       get_curr_arg_class(Arg_List_Iterator iter,
                                                Method_Handle m,
                                                Loader_Exception *exc);
ORPExport Arg_List_Iterator  advance_arg_iterator(Arg_List_Iterator iter);

#ifdef CLI_TESTING

ORPExport Argument_List_Iterator  method_get_arg_iter(Method_Handle m);

ORPExport bool is_cli_native(Method_Handle m);

ORPExport ORP_Data_Type           arg_iter_get_current(Method_Handle m,
                                                       Argument_List_Iterator iter);
ORPExport Class_Handle            arg_iter_get_current_class(Method_Handle m,
                                                             Argument_List_Iterator iter,
                                                             Loader_Exception *exc);
ORPExport Argument_List_Iterator  arg_iter_advance(Method_Handle m,
                                                   Argument_List_Iterator iter);


ORPExport Arg_List_Iter method_sig_get_arg_iter(Method_Signature_Handle s);
ORPExport ORP_Data_Type method_sig_arg_iter_get_current(Method_Signature_Handle s,
                                                        Arg_List_Iter iter);
ORPExport Class_Handle  method_sig_arg_iter_get_current_class(Method_Signature_Handle s,
                                                              Arg_List_Iter iter,
                                                              Loader_Exception *exc);
ORPExport Arg_List_Iter method_sig_arg_iter_advance(Method_Signature_Handle s,
                                                    Arg_List_Iter iter);
ORPExport ORP_Data_Type  method_sig_get_ret_type(Method_Signature_Handle s);
ORPExport Class_Handle   method_sig_get_ret_type_class(Method_Signature_Handle s,
                                                       Loader_Exception *exc);


ORPExport ORP_Data_Type  method_get_ret_type(Method_Handle m);
ORPExport Class_Handle   method_get_ret_type_class(Method_Handle m,
                                                   Loader_Exception *exc);


ORPExport int  method_get_num_local_vars(Method_Handle m);
ORPExport ORP_Data_Type  method_get_local_var_type(Method_Handle m,
                                                   int idx);
ORPExport Class_Handle   method_get_local_var_type_class(Method_Handle m,
                                                         int idx,
                                                         Loader_Exception *exc);

ORPExport Method_Signature_Handle method_get_signature(Method_Handle m);

ORPExport Boolean method_sig_has_this(Method_Signature_Handle s);
ORPExport const char *method_sig_get_descriptor(Method_Signature_Handle s);


ORPExport ORP_Data_Type    field_get_type1(Field_Handle f);
ORPExport Class_Handle     field_get_class1(Field_Handle m);

ORPExport ORP_Data_Type    class_get_primitive_type(Class_Handle clss);


// Returns one of:
// - System.RuntimeTypeHandle,
// - System.RuntimeFieldHandle, or
// - System.RuntimeMethodHandle
// depending on the kind of the token.
ORPExport Class_Handle token_get_type(Class_Handle cl, uint32 token);


// Returns a natural int-sized value of the internal ORP
// representation of the token.  The token must be one of:
// TypeRef, TypeDef, FieldDef or MethodDef.
ORPExport POINTER_SIZE_INT token_get_value(Class_Handle cl, uint32 token);


#endif

ORPExport void method_set_inline_assumption(Compile_Handle h,
                                            Method_Handle caller,
                                            Method_Handle callee);


//
// Class
//        
ORPExport const char  *class_get_name(Class_Handle cl);
ORPExport unsigned     class_get_flags(Class_Handle cl);
ORPExport Boolean      class_property_is_interface(Class_Handle cl);
ORPExport int          class_get_super_offset();
//::
//#ifdef FAST_INSTOF
ORPExport int          vtable_get_super_array_offset();
ORPExport int          class_get_depth(Class_Handle cl);
//#endif
//::
ORPExport Boolean      class_is_initialized(Class_Handle cl);
ORPExport Boolean      class_has_non_default_finalizer(Class_Handle cl);
ORPExport Class_Handle class_get_super_class(Class_Handle cl);
ORPExport Boolean      class_is_instanceof(Class_Handle s, Class_Handle t);
ORPExport Class_Handle class_get_array_element_class(Class_Handle cl);
ORPExport Class_Handle class_get_array_of_class(Class_Handle cl);
ORPExport void        *class_get_vtable(Class_Handle cl);
ORPExport const char  *class_get_source_file_name(Class_Handle cl);

#ifdef CLI_TESTING
ORPExport Boolean      class_is_valuetype(Class_Handle cl);
ORPExport Boolean      class_is_enum(Class_Handle cl);
ORPExport ORP_Data_Type class_get_enum_int_type(Class_Handle cl);
ORPExport Boolean      class_has_gc_fields(Class_Handle cl);
ORPExport unsigned     class_get_unboxed_data_offset(Class_Handle cl);
ORPExport unsigned     class_get_unboxed_data_size(Class_Handle cl);
ORPExport unsigned     field_get_offset_unboxed(Field_Handle f);
#endif

// class_get_const_string is generally only for JIT internal use,
// e.g. printing a string pool constant in a bytecode disassembler.
// The resulting const char* should of course not be inserted into
// the jitted code.
ORPExport const char  *class_get_const_string(Class_Handle cl, unsigned index);

// Return the address where the interned version of the string
// is stored.  Calling class_get_const_string_intern_addr has
// a side-effect of interning the string, so that the JIT can
// load a reference to the interned string without checking if
// it is null.
ORPExport void *class_get_const_string_intern_addr(Class_Handle cl, unsigned index);

// For const Strings, the return value is JAVA_TYPE_STRING
ORPExport Java_Type    class_get_const_type(Class_Handle c, unsigned index);

// This function shouldn't be called for const Strings.  
// Instead, get the java.lang.String object at runtime by calling ORP_RT_LDC_STRING.
ORPExport const void  *class_get_const_addr(Class_Handle c, unsigned index);

ORPExport Class_Loader_Handle class_get_class_loader(Class_Handle c);

ORPExport Class_Handle
class_load_class_by_name(const char *name,
                         Class_Handle c,
                         Loader_Exception *exc);

ORPExport Class_Handle
class_load_class_by_name_using_system_class_loader(const char *name,
                                                   Loader_Exception *exc);

ORPExport Class_Handle
class_load_class_by_descriptor(const char *descr,
                               Class_Handle c,
                               Loader_Exception *exc);

ORPExport Method_Handle
class_lookup_method_recursively(Class_Handle clss,
                                const char *name,
                                const char *descr);



//
// The following do not cause constant pools to be resolve, if they are not
// resolved already
//
ORPExport Class_Handle resolve_class_from_constant_pool(Class_Handle c_handle,
                                                        unsigned index);
ORPExport const char  *const_pool_get_field_name(Class_Handle cl,
                                                 unsigned index);
ORPExport const char  *const_pool_get_field_class_name(Class_Handle cl,
                                                       unsigned index);
ORPExport const char  *const_pool_get_field_descriptor(Class_Handle cl,
                                                       unsigned index);
ORPExport const char  *const_pool_get_method_name(Class_Handle cl,
                                                  unsigned index);
ORPExport const char  *const_pool_get_method_class_name(Class_Handle cl,
                                                        unsigned index);
ORPExport const char  *const_pool_get_method_descriptor(Class_Handle cl,
                                                        unsigned index);
ORPExport const char  *const_pool_get_class_name(Class_Handle cl,
                                                 unsigned index);
ORPExport const char  *const_pool_get_interface_method_name(Class_Handle cl,
                                                            unsigned index);
ORPExport const char  *const_pool_get_interface_method_class_name(Class_Handle cl,
                                                                  unsigned index);
ORPExport const char  *const_pool_get_interface_method_descriptor(Class_Handle cl,
                                                                  unsigned index);

ORPExport Java_Type get_java_type(const char *descriptor);

ORPExport Compile_Handle jit_get_comp_handle(JIT_Handle j);
ORPExport Method_Handle  frame_get_method(Frame_Context *frame);

//
// Java objects
//        
ORPExport int object_get_vtable_offset();
ORPExport int vtable_get_class_offset();



//
// Functions for getting offsets to the length and the
// first element of an array.
// The JIT should use those values to generate code to
// access arrays.
//
ORPExport int array_length_offset();
ORPExport int array_first_element_offset(ORP_Data_Type element_type);


ORPExport int get_array_offset(Java_Type element_type);


//////////////// end C interface



#endif
