// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/include/properties.h,v 1.11 2002/01/09 03:29:08 gwu2 Exp $
//


#ifndef _ORP_PROPERTIES_H
#define _ORP_PROPERTIES_H

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifndef ORP_NT
extern "C"{
    char *strdup(const char *s);
}
#endif
 
/***************************************************************************************/
class Prop_Value
{
public:
	Prop_Value()
	{
		value = NULL;
	}

	Prop_Value(void* v)						/*Do not clone v*/
	{
		value = v;
	}
	virtual Prop_Value* clone() = 0;		/*Clone this Prop_Value*/
	virtual void print() = 0;
public:
	void *value;
};

class Prop_String: public Prop_Value
{
public:
	Prop_String()
	{
		value = NULL;
	}

	Prop_String(char* v)					/*Do not clone v*/
	{
		value = v;
	}
	~Prop_String()
	{
		if(value)
			delete value;
	}
	Prop_Value* clone()
	{
		return (new Prop_String(strdup(value)));	/*use strdup() to clone the value*/
	}
	void print()
	{
		printf(value);
	}
public:
	char* value;
};

/***************************************************************************************/

class Prop_uLong: public Prop_Value
{
public:
	Prop_uLong()
	{
		value = NULL;
	}
	Prop_uLong(unsigned long* v)
	{
		value = v;
	}
	~Prop_uLong()
	{
		if(value)
			delete value;
	}
	Prop_Value* clone()
	{
		unsigned long* u = new unsigned long[1];
		*u = *value;
		return (new Prop_uLong(u));
	}
	void print()
	{
		printf("%Iu",*value);
	}
public:
	unsigned long* value;
};


/***************************************************************************************/
struct Prop_entry
{
	char *key;
	Prop_Value *value;					/*value can be Prop_Sting,Prop_Properties,etc.*/
	struct Prop_entry *next;
	int compareTo(const char *akey)
	{
		return strcmp(akey, key);
	}
	void replace(Prop_entry *e)
	{
		if(value)delete value;
			value = e->value->clone();	/*clone e's value*/
	}
public:
	Prop_entry()
	{
		key = NULL;
		value = NULL;
		next = NULL;
	}
	~Prop_entry()
	{
		if(key)delete key;
		if(value)delete value;
		if(next)delete next;
	}
};


/**************** The following is Properties *********************/

class Properties
{
	//enum { BUCKET_SIZE = 10	};

public:
	class Iterator
	{
	private:
		Properties *prop;
		Prop_entry *current;
		int idx;
		Iterator(Properties *prop_)			/*Couldn't new instances except Properties*/
		{
			prop = prop_;
		}
	public:
		void reset()
		{
			current = NULL;
			idx = 0;
		}
		const Prop_entry* next()
		{
			if(current && current->next)
				return current = current->next;
			int i = current? idx+1 : idx;
			for(; i < prop->BUCKET_SIZE; i++)
				if(prop->bucket[i])
					break;
			if(i == prop->BUCKET_SIZE)
				return NULL;
			current = prop->bucket[i];
			idx = i;
			return current;
		}
		friend Properties;
	};

	friend Iterator;
public:
	Properties()
	{
		BUCKET_SIZE = 10;
		init(BUCKET_SIZE);
	}
	Properties(long size_)
	{
		BUCKET_SIZE = size_;
		init(BUCKET_SIZE);
	}
	~Properties(){
		delete iterator;
		clear();
		free(bucket);
	}
	Iterator *getIterator(){
		iterator->reset();
		return iterator;
	}

	void add(const char* k,Prop_Value* v);		/*Caller delete k and v*/
	void add(Prop_entry* e);				/*Caller can not delete e*/
	void add(const char *line);
	void clear();
	Properties* clone();
	Prop_Value* get(const char* key);
	void print();
	bool contains_key(const char* key);

private:
	//Prop_entry* bucket[BUCKET_SIZE];
	Prop_entry** bucket;
	int index_of(const char*key);
	Iterator *iterator;
	long BUCKET_SIZE;
	void init(long size_){
		bucket = (Prop_entry**)calloc(sizeof(Prop_entry*), size_);
		iterator = new Iterator(this);
	}
};

inline int Properties::index_of(const char *key)
{
    unsigned int hash = 0;
    for (; *key; key++)
      hash = hash * 31 + *key;
	return hash % BUCKET_SIZE;
}

inline void Properties::add(Prop_entry* e)	/*Caller can NOT delete e*/
{
	int idx = index_of(e->key);
	if(bucket[idx]){
		Prop_entry *itr = bucket[idx];
		Prop_entry *prev = NULL;
		while(itr)
		{
			int cmp = itr->compareTo(e->key);
			if(cmp > 0)
			{
				e->next = itr;
				if(!prev)
					bucket[idx] = e;
				else
					prev->next = e;
				break;
			}
			if(cmp == 0)
			{
				itr->replace(e);
				delete e;					/*Because this, e will be invalid*/
				break;
			}
			prev = itr;
			itr = itr->next;
		}
		if(!itr)
			prev->next = e;
	}else
		bucket[idx] = e;
}

inline void Properties::add(const char* k,Prop_Value* v)
{
	Prop_entry* e = new Prop_entry();
	e->key = strdup(k);						/*caller can delete k*/
	e->value = v->clone();					/*caller can delete v*/
	add(e);									/*e will be invalid*/
}

inline char* unquote(char *str)
{
	const char *tokens = " \t\n\r\'\"";
	int i = strspn(str, tokens);
	str += i;
	char *p = str + strlen(str) - 1;
	while(strchr(tokens, *p) && p >= str)
		*(p--) = '\0';
	return str;
}

inline void Properties::add(const char *line)
{
	char *src = strdup(line);
	char *tok = strchr(src, '=');
	if(tok)
	{
		*tok = '\0';
		Prop_entry *e = new Prop_entry();
		e->key = strdup(unquote(src));
		e->value = new Prop_String(strdup(unquote(tok + 1)));
		if((e->key[0] == '\0') || (((Prop_String*)e->value)->value[0] == '\0'))
			delete e;
		else
			add(e);
	}
	delete src;
}

inline void Properties::print()
{
	int i = 0;
	for(i = 0; i < BUCKET_SIZE; i++)
		if(bucket[i]){
			Prop_entry *e = bucket[i];
			while(e)
			{
				printf("%s=", e->key);
				e->value->print();
				printf("\n");
				e = e->next;
			}
		}
}

inline void Properties::clear()
{
	for(int i = 0; i < BUCKET_SIZE; i++)
		if(bucket[i])
		{
			delete bucket[i];
			bucket[i]=NULL;
		}
}

inline Properties* Properties::clone()
{
	Properties* cloned_p = new Properties();
	for(int i = 0; i < BUCKET_SIZE; i++)
		if(bucket[i])
		{
			Prop_entry *itr = bucket[i];
			while(itr)
			{
				Prop_entry *newe = new Prop_entry();
				newe->key = strdup(itr->key);
				newe->value = itr->value->clone();
				cloned_p->add(newe);				/*Can NOT delete newe*/
				itr = itr->next;
			}
		}
	return cloned_p;

}

inline Prop_Value* Properties::get(const char* key)
{
	int idx = index_of(key);
	if(bucket[idx] == NULL)
		return NULL;

	Prop_entry *itr = bucket[idx];
	while(itr)
	{
		int cmp = itr->compareTo(key);
		if(cmp > 0)
			return NULL;
		if(cmp == 0)
			return itr->value;
		itr = itr->next;
	}
	return NULL;
}

inline bool Properties::contains_key(const char* key)
{
	int idx = index_of(key);
	if(bucket[idx] == NULL)
		return false;

	Prop_entry *itr = bucket[idx];
	while(itr)
	{
		int cmp = itr->compareTo(key);
		if(cmp > 0)
			return false;
		if(cmp == 0)
			return true;
		itr = itr->next;
	}
	return false;
}


/***************************************************************************************/

class Prop_Properties: public Prop_Value
{
public:
	Prop_Properties()
	{
		value = NULL;
	}
	Prop_Properties(Properties* v)
	{
		value = v;
	}
	~Prop_Properties()
	{
		if(value)
			delete value;
	}
	Prop_Value* clone()
	{
		return (new Prop_Properties(value->clone()));
	}
	void print()
	{
		value->print();
	}
public:
	Properties* value;
};

void initialize_properties(Properties &prop);
void add_to_properties(Properties& prop, const char* line);

#endif

