// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/include/jit_intf_cpp.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//




#ifndef _JIT_INTF_CPP_H_
#define _JIT_INTF_CPP_H_


#include "orp_types.h"
#include "jit_intf.h"



//
// A JIT is invoked through virtual functions declared in this class.
//
class JIT {

public:

    Compile_Handle     global_compile_handle;
	JIT_Flags  jit_flags; // KEN fix later: if jit keeps jit_flags, there is no need to pass flags

    void init();

    //
    // Gracefully terminate the JIT.
    //
    virtual ~JIT() {};


    virtual void 
    next_command_line_argument(const char *, const char *) {};


    //
    // The ORP call into JIT to compile a method to generate method info
    //
    virtual JIT_Result 
    gen_method_info(Compile_Handle      compilation,              // in
                    Method_Handle       method,                   // in
                    JIT_Flags           flags                     // in
                    ) = 0;
    //
    // The ORP call into JIT to compile a method.
    //
    // Arguments:
    // - compilation -- A compilation handle must be passed as an argument for
    //                  some ORP functions called by the JIT.
    // - method      -- A handle for the method which corresponds to the frame
    //                  that we want to unwind.
    // - flags       -- Currently unused.
    // 
    virtual JIT_Result 
    compile_method(Compile_Handle     compilation,   // in
                   Method_Handle      method,        // in
                   JIT_Flags          flags          // in
                   ) = 0;




    virtual void
    method_was_overridden(Method_Handle         caller,
                          Method_Handle         callee
                          ) {};



    //
    // Unwind a stack frame for a method give the context.
    //
    // Arguments:
    // - method   -- A handle for the method which corresponds to the frame
    //               that we want to unwind.
    // - context  -- The register context.
    // - is_first -- Set to true if the current frame is the top frame on the stack.
    // 
    virtual void
    unwind_stack_frame(Method_Handle       method,                // in
                       Frame_Context      *context,               // in out
                       Boolean             is_first               // in
                       ) = 0;



    //
    // Call from the ORP into the JIT.
    // The JIT:
    // 1. Enumerates all references for the current stack.
    //    The enumeration is precise, i.e., non-references are not
    //    enumerated.
    //    Enumeration can be done with either orp_enumerate_root_reference.
    //    The enum_handle argument must be used in those callbacks.
    // 2. Unwinds the stack and modifies the context in-place.
    //
    // Arguments:
    // - method      -- A handle for the method which corresponds to the frame
    //                  of the exception handler.
    // - enum_handle -- This value should used as an argument to
    //                  orp_enumerate_root_reference
    // - context     -- The register context.
    //                  Note that the eip corresponds to the location which throws
    //                  the exception not to the beggining address of the handler.
    //
    virtual void 
    get_root_set_from_stack_frame(Method_Handle           method,         // in
                                  GC_Enumeration_Handle   enum_handle,    // in
                                  Frame_Context          *context,        // in out
                                  Boolean                 is_first        // in
                                  ) = 0;



    virtual Boolean
    can_enumerate(Method_Handle method,
                  uint32        eip
                  ) = 0;



    virtual unsigned
    num_breakpoints(Method_Handle method,
                    uint32        eip
                    ) = 0;



    // The JIT needs to write the breakpoint addresses into bp[],
    // and to make context->p_eip point to the return eip for the method.
    virtual void
    get_breakpoints(Method_Handle method,      // in
                    uint32        *bp,         // out
                    Frame_Context *context     // in out
                    ) = 0;

    //
    // Called by the ORP before control is transferred to an exception handler.
    // The JIT can restore the registers to the values expected by the handler.
    //
    // Arguments:
    // - method   -- A handle for the method which corresponds to the frame
    //               of the exception handler.
    // - context  -- The register context.
    //               Note that the eip corresponds to the location which throws
    //               the exception not to the beggining address of the handler.
    // - is_first -- Set to true if the current frame is the top frame on the stack.
    //
    virtual void
    fix_handler_context(Method_Handle      method,              // in
                        Frame_Context     *context,             // in out
                        Boolean            is_first             // in
                        ) = 0;



    //
    // Returns the address of the this pointer of the method.
    // This function is called when an exception is propagated out of
    // a stack frame of a synchronized method.  The ORP uses this address
    // to call monitor_exit for the object associated with the method.
    //
    // Arguments:
    // - method   -- A handle for the method which corresponds to the current frame.
    // - context  -- The register context.
    // - is_first -- Set to true if the current frame is the top frame on the stack.
    //
    virtual void *
    get_address_of_this(Method_Handle         method,              // in
                        const Frame_Context  *context,             // in
                        Boolean               is_first             // in
                        ) = 0;




    //
    // This function is called for a call site (ie with and eip pointing to
    // the instruction following a call).
    // Iff the function at this call site (be it a Java method or a runtime
    // support function) returns a reference, the result is TRUE.
    // This function is used during GC to decide if eax contains a reference
    // and if it should be enumerated as part of the root set.
    //
    virtual Boolean
    call_returns_a_reference(Method_Handle         method,              // in
                             const Frame_Context  *context              // in
                             ) = 0;

    //
    // return the native code offset of the bytecode location (for jvmdi support)
    //
    virtual int
    get_break_point_offset(Compile_Handle compilation,
                           Method_Handle  meth,
                           JIT_Flags      flags,
                           unsigned       bc_location) = 0;

    //
    // return the address of a var (for jvmdi support)
    //
    virtual void *
    get_address_of_var(Frame_Context *context,
                       Boolean       is_first,
                       unsigned      var_no) = 0;

    //
    // thread for triggering recompilation
    //
    virtual void 
    thread_recompile_methods() = 0;

#ifdef _DEBUG
    virtual void
        gc_start() {};

    virtual void
        gc_end() {};

    virtual void
        gc_object_died(void *java_ref) {};
#endif


}; //JIT






#endif
