// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/los_container.cpp,v 1.9 2002/01/11 15:47:38 weldon Exp $
//

 
#include "gc_for_orp.h"
 
#include "los.h"
#include "los_container.h"
extern bool verbose_gc;

#include "root_set_enum.h"
#include "orp_threads.h"

#ifdef ORP_POSIX
#include "orp_synch.h"
#include "platform2.h"
#endif

// The backup stores are set to zero for the non-concurrent GC code.
#ifdef GC_SAPPHIRE
int initial_backup_object_stores;;
int backup_object_stores;
#else
int initial_backup_object_stores = 0;
int backup_object_stores = 0;
#endif // GC_SAPPHIRE

// Blocks that hold only a single object are linked here.

block_info *single_object_blocks = NULL;


LOS_Container::LOS_Container(Gc_Interface  *p_gc_interface,
                             Gc_Fast_Hooks *p_gc_hooks,
                             Gc_Plan       *p_gc_plan,
		                     Generation    *p_container,
		                     Card_Table    *p_card_table,
			                 Block_Store   *p_bs)

             : Gc_Component(p_gc_hooks, 
                            p_gc_plan,
                            p_bs)
{
	//
	// Create only one LOS for starters. Others will be
	// created upon demand.
	//
    _p_card_table = p_card_table;
    _p_container = p_container;
	_number_of_large_object_stores = p_gc_plan->number_of_large_object_stores ();;
    _los_block_size_bytes = p_gc_plan->los_block_size_bytes ();
    _p_gc_interface = p_gc_interface;

void init_large_object_stores();
    
    init_large_object_stores();
    

#ifdef GC_SAPPHIRE
    // Reserve unused space so GC can happen while we have space to allocate in.
    initial_backup_object_stores = _number_of_large_object_stores/2;
    backup_object_stores = initial_backup_object_stores;
#endif // GC_SAPPHIRE

}

//
// We failed to allocate an object even after triggering a
// collection. This is the time to try to extend the LOS.
//

void 
LOS_Container::extend_los(unsigned int segment_size_bytes)
{

    if (stats_gc) {
        orp_cout << " ** out of LOS, extending for " << segment_size_bytes << endl;
    }

    // First see if the segment_size_bytes that we need will fit into 
    // the LOS increment size.
    unsigned int size_to_extend_bytes = _los_block_size_bytes;

    if (segment_size_bytes > size_to_extend_bytes) {
        // Double it so that we are sure this block will fit.
        size_to_extend_bytes = segment_size_bytes * 2;
    }

    // Normalize the size to extend to something that falls on a block boundary.

    size_to_extend_bytes = p_global_bs->size_up_to_los_block_size(size_to_extend_bytes);

    _p_gc_plan->set_los_block_size_bytes (size_to_extend_bytes);

    p_global_bs->_extend_heap (size_to_extend_bytes);

    return;
}
//
// Run through the mark table for every block in this bucket making sure it is clear.
bool all_marks_are_clear (block_info *bucket)
{
    block_info *this_block = bucket;
    while (this_block) {
        int i;
        for (i = 0; i < GC_LOS_MAX_MARK_INDEX; i++) {
            assert (!this_block->mark_table[i]); // Make sure the bucket isn't marked.
        }
        this_block = this_block->next;
    }
    return true;
}


void sweep_los_bucket (block_info *bucket)
{
    bool whole_block_free = true; // If the entire block has no live objects release the entire block.
    block_info *this_block = bucket;
    block_info *last_block = NULL;
    this_block->free = NULL;
    unsigned int size = this_block->los_object_size;
    while (this_block) {
        Object_Gc_Header *first_obj_start = (Object_Gc_Header *)GC_BLOCK_ALLOC_START(this_block);
        // There may be some space at the end that is unused, this is the last possible place to start a valid object.
        Object_Gc_Header *last_possible_obj_start = (Object_Gc_Header *)((GC_BLOCK_SIZE_BYTES + (POINTER_SIZE_INT)this_block) - size);
        Object_Gc_Header *an_obj_start = first_obj_start;
        while ((POINTER_SIZE_INT)an_obj_start <= (POINTER_SIZE_INT)last_possible_obj_start) {
            if (!GC_LOS_MARK(an_obj_start)) {
                memset (an_obj_start, 0x0, size); // For debugging clear this object to 0. This could be avoided with a little work.
                ((los_free_link *)an_obj_start)->next = (los_free_link *)(this_block->free);
                this_block->free = (void *)an_obj_start;
            } else {
                whole_block_free = false;
                set_object_unmarked(an_obj_start); // Unmark the object in preparation for the next collection.
            }
            an_obj_start = (Object_Gc_Header *)((POINTER_SIZE_INT)an_obj_start + (POINTER_SIZE_INT)size);
        }        
        if ((whole_block_free) && (bucket != this_block)) {
            assert (last_block); // If there isn't a last block then bucket == this_block
            last_block->next = this_block->next; // decouple this block and return.
            // Now return the block to the free block list
            assert (this_block->number_of_blocks == 1);
            p_global_bs->link_free_blocks (this_block, this_block->number_of_blocks);
            this_block = last_block;
        }
        last_block = this_block;
        this_block = this_block->next;
        if (this_block) {
            assert (this_block->los_object_size == size);
            whole_block_free = true;
            this_block->free = NULL;
        }
    }
    
#ifdef _DEBUG
    all_marks_are_clear(bucket);
#endif
}

void sweep_single_object_blocks ()
{
    block_info *this_block = single_object_blocks;
    block_info *block = this_block;
    block_info *orig_single_object_blocks = single_object_blocks;
    block_info *new_single_object_blocks = NULL;
    while (block != NULL) {
        this_block = block;
        block = block->next;   // Get the next block before we destroy this block.
        assert (this_block->in_los_p);
        assert (this_block->c_area_p);
        assert (this_block->train_birthday == 0);
        
        Object_Gc_Header *obj_header = (Object_Gc_Header *)GC_BLOCK_ALLOC_START(this_block);
        if (!GC_LOS_MARK(obj_header)) {
            // It is free, relink the blocks onto the free list.
            assert (this_block->number_of_blocks);
            p_global_bs->link_free_blocks (this_block, this_block->number_of_blocks);
        } else {
            set_object_unmarked(obj_header); // Unmark the object in preparation for the next collection.
            this_block->next = new_single_object_blocks;
            new_single_object_blocks = this_block;
        }
    }
    single_object_blocks = new_single_object_blocks;
#ifdef _DEBUG
    all_marks_are_clear(single_object_blocks);
#endif 
}


void
LOS_Container::cleanup()
{

    int i;
    for (i = 0; i < GC_LOS_BUCKETS; i++)  {
        sweep_los_bucket (los_buckets[i]);
        los_buckets_free_hint[i] = los_buckets[i]; // Start looking for a free block at the start of the buckets.
    }

    sweep_single_object_blocks();

    return;
}

//
// No more LOS, even after a collection.
//
void
LOS_Container::_notify_large_object_store_exhausted_error(unsigned int size)
{
#if (GC_DEBUG>2)
	assert(_p_container);
#endif
	cout << "Error: Large Object Space Exhausted when allocating ";
	cout << size << " bytes" << endl;
	orp_exit(1);
}

// External from root_set_enum.h.
Boolean orp_is_gc_disabled(void *thread_id);

#ifdef _DEBUG
//
// Allocate an object into large object space.
//
int _calls_to_pinned_malloc = 0;
int _pinned_malloc_size_allocated = 0;
#endif

Java_java_lang_Object  *
LOS_Container::gc_pinned_malloc(unsigned size, 
		                        VTable *p_vtable,
				                bool return_null_on_fail,
                                bool double_align
                                )
{
    Java_java_lang_Object *result = NULL;
    assert (orp_is_gc_disabled(p_TLS_orpthread));
#ifdef _DEBUG
    _calls_to_pinned_malloc++;
    _pinned_malloc_size_allocated += size;
#endif

    // If the size is greater that the half a block size then allocate 1 or more blocks for it.
    assert (MAX_BUCKET_OBJECT_SIZE < 32768); // Assumes block is 65536 in size after we subtract block overhead this needs to be less than half a block.
    if (size > MAX_BUCKET_OBJECT_SIZE) {
        // The gc is disabled so this block will never be seen in a incoherent state.
        // Sapphire will need to grab a lock to make this work...
        block_info *block = p_global_bs->p_get_multi_block (size, false, false, false /*not in a gc */ );   // Do not extend the heap yet.
        unsigned int i;
        for (i = block->block_status_table_index; i < block->block_status_table_index + block->number_of_blocks; i++) {
            assert (gc_block_status_table[i] == block_in_free);
            gc_block_status_table[i] = block_in_los;
        }
        block->in_los_p = true;    
        block->in_free_p = false;
        block_info *orig_single_object_blocks = single_object_blocks;
        block->next = orig_single_object_blocks;
        while ((block_info *)InterlockedCompareExchange((PVOID *)&single_object_blocks, (PVOID)block, (PVOID)orig_single_object_blocks) != orig_single_object_blocks) {
            orig_single_object_blocks = single_object_blocks;
            block->next = orig_single_object_blocks;
        }
        block->c_area_p = true;
        block->los_object_size = size;
        if (size < GC_BLOCK_ALLOC_SIZE) {
            block->free = ((char *)GC_BLOCK_ALLOC_START(block) + size);
            block->ceiling = ((char *)GC_BLOCK_ALLOC_START(block) + size);
        }
        Object_Gc_Header *p_obj_start = (Object_Gc_Header *)GC_BLOCK_ALLOC_START(block);
        memset (p_obj_start, 0, size); // Clear the new object.
        result = P_OBJ_FROM_START(p_obj_start);
        result->vt = (VTable *)p_vtable;
        assert (result);
        gc_trace (result, "Created in pinned single object in block(s) area.");
        
        return result;
    }

    // returns a cleared object    
    result = gc_los_malloc_noclass(size);
    result->vt = (VTable *)p_vtable;
    
    gc_trace (result, "Created in pinned many objects in block area.");
#ifdef GC_SAPPHIRE
                            // If we are in Sapphire doing a concurrent MS GC 
                            // and the allocation of a new object is in the C space
                            // Which it is when we are just doing a simple MS then the
                            // object is allocated black.
                            if (p_TLS_orpthread->gc_sapphire_status != sapphire_normal) {
                                // We don't have to scan it since it only contains 
                                // empty slots.
                                set_object_marked (result);
                            }                       
#endif

        
    return result;
}

//
// A special version during bootstrapping for objects
// that still don't have classes.
//
Java_java_lang_Object  *
LOS_Container::gc_pinned_malloc_noclass(unsigned size)
{
	//
	// FOR THE TIME BEING, HARD WIRE IT TO THE ONLY LOS:
	// 

	return _p_los_array[0]->gc_pinned_malloc_noclass(size);
}

#ifdef GC_STATS
void
LOS_Container::prepare_for_collection()
{

	for (int idx = 0; 
	         idx < _number_of_large_object_stores;
			 idx++) {

		_p_los_array[idx]->prepare_for_collection();
	}
}
#else

void
LOS_Container::prepare_for_collection()
{
}
#endif

void 
LOS_Container::set_generation(Generation *p_gen) 
{
	//
	// FOR THE TIME BEING, HARDWIRE THIS. NOTE THAT
	// WE WILL NEED TO DYNAMICALLY SET THIS FOR ALL
	// FUTURE LOSs THAT ARE CREATED.
	//
	for (int idx = 0; 
	         idx < _number_of_large_object_stores;
			 idx++) {

		_p_los_array[idx]->set_generation(p_gen);
	}
}

//
// Is there a block of this size available, if not caller will
// extend heap.
//
bool
LOS_Container::is_block_available(unsigned int size)
{
	for (int idx = 0; idx < _number_of_large_object_stores; idx++) {
        if (_p_los_array[idx]->is_block_available(size)) {
            return true;
        }
	}
    return false;
}
// end file los_container.cpp

