// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/los.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#ifndef _los_H_
#define _los_H_

//
// The Large Object Store contains objects that are either too
// big to move efficiently, or are not position-independent.
// A Los_Container contains several of these large object stores.
//


#include "Block_Store.h"
#include "card_table.h"
#include "object_layout.h"
#include "gc_space.h"
#include "gc_consts.h"
#include "gc_interface.h"
#include "orp_threads.h"

class LOS_Container;
class Gc_Fast_Hooks;
class Gc_Plan;

class Large_Object_Store : public Gc_Space {
public:

    Large_Object_Store(Gc_Interface  *p_gc_interface,
                       Gc_Fast_Hooks *p_gc_hooks,
                       Gc_Plan       *p_gc_plan,
		               Generation    *p_container,
		               Card_Table    *p_card_table,
					   Block_Store   *p_bs,
                       LOS_Container *p_los_container);

    virtual ~Large_Object_Store() {
		// Unnecessary for the time being.
	}

	bool static is_address_in_los (void *address);

    //
    // Debugging routine for printing out LOS stats.
    //
    void dump_stats();

	//
	// Allocate an object into large object space.
	//
 
    Java_java_lang_Object *gc_pinned_malloc(unsigned size, 
		                   Partial_Reveal_VTable *p_vtable,
						   bool return_null_on_fail,
                           bool double_align                     
                           );

 
    //
	// A special version during bootstrapping for objects
	// that still don't have classes.
	//
	Java_java_lang_Object *gc_pinned_malloc_noclass(unsigned size);

	//
	// Do any required cleanup at the end of the collection.
	// This may include running finalizers on dead objects,
	// etc.. 
	//
	virtual void cleanup();

    virtual Java_java_lang_Object *p_evict_object(Java_java_lang_Object **pp_obj, bool doing_mos_collection);

	//
	// routines for introspection
	//

	virtual bool is_large_object_space() {
		return true;
	}

    //
    // The LOS is being give a chance to set up the instrumentation
    // to do things like space accounting, just before a GC hits.
    //
    void prepare_for_collection();

    //
    // BUG: NEED TO ALSO SCAN LIVE REFERENCES FOR INCOMING
    //
	inline bool no_incoming_pointers() {
		if (_p_container_write_barrier->is_empty())
			return true;

		return false;
	}

	Java_java_lang_Object *p_scavenge_object(Java_java_lang_Object **pp_obj) {
		orp_exit(1); // illegal call
		return NULL;
	}

#if (GC_DEBUG>3)
	//
	// Debug routine to inspect details.
	//
	virtual void inspect(unsigned int level);
#endif // _DEBUG

	//
	// The generation may not be available when the first LOS
	// is created. Therefore it may need to be patched later.
	// Earlier we created LOSs first because they were large,
	// monolithic beasts. Now that we have non-contiguous,
	// variable sized loses that grow on demand, we don't need
	// to create them first. Need to refine this.
	//
	void set_generation(Generation *p_gen);

	//
	// Merge the entries in the incoming remembered set with entry.
    // This is only valid for GC Spaces with their own remembered sets.
    // (Such as cars, and unlike nurseries and steps.)
	//
	virtual void update_remembered_set(Java_java_lang_Object **pp_obj) {
		cout << "Error: attempt to add entry to a LOS rs (I don't have one!)" << endl;
        orp_exit(1);
	}

#if (GC_DEBUG>3)
	//
	// Each space provides a routine to verify that it is consistent.
	// i.e. contains only well-formed objects.
	//
	virtual bool verify_space();
#endif // _DEBUG

    bool _walk_los(bool (*func)(Object_Gc_Header *));

	virtual int my_block_size_bytes() {
		return _los_block_size_bytes;
	}

    // #ifdef GC_MARK_FLAGS implements non-resident mark bits

    // Each mark is for so many MARK_AREA_BYTES, the 1 << MARK_SHIFT is equal to
    // MARK_AREA_BYTES. 

#ifdef POINTER64
#define MARK_SHIFT 3
#define MARK_AREA_BYTES 8
#else
// mark every word, only the first one counts for the entire object.
#define MARK_SHIFT 2
#define MARK_AREA_BYTES 4
#endif // POINTER64


    char *_mark_flags; // An array of mark flags, each taking up a byte.
	
	void Large_Object_Store::clear_mark_flags ();
    // #endif // GC_MARK_BITS

    //
    // The base of this LOS
    //
    void *_p_base;

    // Similarly, there is an equivalent ceiling pointer.
    //
    void *_p_ceiling;


    // Each LOS has a light weight lock that is grabbed by thread trying to
    // allocate an object. If an LOS's lock is held the thread just moves
    // on to another LOS and tries again. If all of the LOS's are unavailable
    // and the number of threads is greater than the number of LOS's then
    // a new LOS is added. It is possible that starvation can happen if one
    // thread races in front of the other grabbing locks but this is unlikely since
    // once a thread has a lock is spends a lot of time allocating objects and so forth
    // and this should allow the trailing thread to move ahead and find a free LOS.
    //

    ORP_thread *this_los_lock; // NULL is free, volatile forces .rel semantics.

    ORP_thread *get_lock_owner();

    bool try_lock();

    void release_lock();

    bool is_block_available(unsigned int size);

private:

	int _los_block_size_bytes;

    //
    // This routine takes two free blocks and creates one larger
    // block that is pointed to by _p_last_free.
    //
    void _coalesce_free_blocks(unsigned int obj_size,
        void *previous_free_block,
        void *this_free_block);

	void * _best_fit(unsigned int real_size_bytes);

	//
	// We failed to allocate an object even after triggering a
	// collection. This is the time to try to extend the LOS.
	//
//	void _extend_los(unsigned int real_size_bytes);

	Java_java_lang_Object *_first_fit(unsigned int real_size_bytes,
                     bool double_align
        );

    //
    // Start a first fit search with this free block.
    //
    Java_java_lang_Object *_first_fit_with_start(unsigned int real_size_bytes, 
        void *start_here,
        bool double_align
                                );
	//
	// Pointer to free list of available space.
	//
	void *_p_free_list;

    //
    // Pointer of where to start searching the free list.
    //
    void *_p_start_search_here;

    //
    // Support pointer when scanning an coalesing free list.
    //
    void *_p_last_free;

    //
    // State variable to help with the scan process.
    //
    bool _last_object;

    // 
    // The container holding this large object store.
    //
    LOS_Container *_p_los_container;

    //
	// A pointer to the enclosing GC Interface.
	//
	Gc_Interface *_p_gc_interface;

	unsigned int _largest_object_size;

	//
	// This is a pointer to the live object which has
	// the highest address in the LOS. It is used when
	// doing linear scans, so that we don't fall of the
	// high end (into unallocated space).
	//
	Java_java_lang_Object *_p_last_object;
	//
	// When doing a linear scan to reclaim dead objects,
	// we keep track of the last live object we scanned,
	// so that when we get to the end, if we find that the
	// last object is no longer live, we know what the 
	// new last object is.
	//
	Java_java_lang_Object *_p_last_live_object;

	void _linear_sweep_heap_and_reclaim();

	//
	// This flag is set when the LOS can't do a first-fit of the
	// required block size. This flag is checked again after the
	// subsequent collection to see if this allocation still fails.
	// This is a signal that the heap needs extending. This boolean
	// is reset after any successful allocation.
	//
	bool _los_space_may_be_critically_low;

	//
	// Out of space - need a collection.
	//
	void _notify_large_object_store_exhausted(unsigned int real_size_bytes);
	//
	// Really out of space, despite a collection - error.
	//
	void _notify_large_object_store_exhausted_error(unsigned int real_size_bytes);

	//
	// Number of objects scanned when linearly sweeping the heap.
	// (This includes live and dead objects.)
	//
	unsigned long _number_scanned_objects;

	void _reclaim_object(Object_Gc_Header *p_gc_hdr);

	void *_skip_free_block_and_update_free_list(void *p_thing);

    void *_skip_free_block(void *p_thing);

	void *_skip_java_object(void *p_thing);

#ifdef GC_STATS 
    //
    // The following variables are used for space accounting.
    //

    //
    // Total size of objects encountered during the mark phase.
    //
    int _stat_marked_space;
    //
    // Total size of free blocks encountered during the sweep phase.
    // This plus _stat_swept_reclaimed gives the total memory available.
    //
    int _stat_swept_free;
    //
    // Total size of live objects encountered during the sweep phase.
    // This should equal _stat_marked_space when the sweep is done.
    //
    int _stat_swept_live;
    //
    // Total size of dead objects reclaimed during the sweep phase.
    // This plus _stat_swept_free gives the total memory available.
    //
    int _stat_swept_reclaimed;
    // If an object is too small to hold a free header block it is 
    // is not collected unless it can be coalesced with the preceeding
    // block. This tracks how much space we are loosing to such objects.
    // Dividing the size by 8 will give the total number of such objects.
    // If this becomes a problem we can try to coalesce with the following
    // block (or better yet) change the class loader to make all objects at
    // large enough to hold a free header.
    int _stat_free_space_not_recovered;
#endif // GC_STATS

	unsigned int _smallest_object_size;

	void *_sweep_java_object(void *p_thing);

	void *_sweep_thing(void *p_thing);
 
	void _update_allocation_stats(Partial_Reveal_Class *p_class, 
				                  unsigned int real_size_bytes);
 
    //
    // This routine is used when scanning the LOS and linking
    // up free blocks.
    //
    void _update_free_list(void *p_obj, unsigned int obj_size);

	bool _verify_object_is_in_large_object_store(Java_java_lang_Object *p_obj);

    //
    // Support routine used by verify_space.
    //
    void *_p_verify_and_skip_over_free_block(void *p_scan);
    //
    // Support routine used by verify_space.
    //
    void *_p_verify_and_skip_over_object(void *p_scan);

};

#endif // _los_H_

