// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/card_table.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#ifndef _CARD_TABLE_H_
#define _CARD_TABLE_H_

//
// A card is a 2-power-k-aligned region of memory of size 2-power-k
// for some compile time selected "k" (typically 128 bytes to 10K).
// It is necessary for a card to be smaller than a train car.
// 
// The JIT generates code to mark cards, and a concurrent thread
// converts the marked cards into remembered sets.
//

#include "platform.h"
#include <fstream.h>
#include <iostream.h>
#include "gc_globals.h"
#include "gc_header.h"
#include "gc_consts.h"

#ifdef __cplusplus 
extern "C" {
#endif

#ifdef __cplusplus 
}
#endif 
 
class Block_Store;
class Gc_Fast_Hooks;
class Gc_Plan;
class Generation;
class Remembered_Set;

extern Card_Table *p_global_card_table_manager;

class Card_Table {

public:

    Card_Table(Gc_Fast_Hooks  *p_gc_hooks,
               Gc_Plan        *p_gc_plan,
               unsigned long  final_heap_size,
               unsigned long  initial_heap_size,
               unsigned long card_size,
               Block_Store    *p_block_store,
			   POINTER_SIZE_INT  heap_base);

    void extend_card_table (unsigned long heap_extension_size_bytes);
    //
    // When adding a block into a block-list (representing a
    // car or a step), the card table manager is being forced
    // to clear out the "last object" entries corresponding
    // to that card. This is a requisite initialization that
    // was missing earlier.
    //
    void clear_last_object_addresses(void *p_block_address,
		                             int super_block_size_bytes);

    void clear_cards_in_super_block(void *p_block_start_start,
		                            unsigned long super_block_size_bytes);

    void mark_card(POINTER_SIZE_INT card);

    void mark_card(void *address);

    void set_last_object_address(Object_Gc_Header *address);


    //
    // We are handed two generations and a remembered set.
    // We scan the dirty cards looking for pointers from 
    // mature to young space, as well as from younger to older
    // cars in mature space. These interesting references are
    // inserted into the remembered set that is also handed to us.
    //
    void scan_cards(Generation *p_mature_generation,
		Generation *p_young_generation);


    bool object_in_legal_card(void *address);

#if (GC_DEBUG>2)
    //
    // Verify that this card has a valid last_object_table entry associated
    // with it.

    void verify_last_object_table (POINTER_SIZE_INT card);


	//
	// Verify that the card index in question is that of a
	// card that has some objects resident. (i.e. it is not
	// a card representing unallocated/unused memory).
    //
    // Since the card last object table is only valid for cards in
    // mature space OK all cards not in mature space.
    // 
	//

	void verify_is_populated_card(POINTER_SIZE_INT card);
 
#endif // GC_DEBUG>2

private:

    // Holds the address of the last address allocated in this card. Not valid for object in YOS.
    Object_Gc_Header **_p_card_last_object_table;

    Card_Table();

    unsigned long _card_size_bytes;    

	// needed to calculate card.
	POINTER_SIZE_INT _heap_base;

    POINTER_SIZE_INT _number_of_cards;

    POINTER_SIZE_INT _max_number_of_cards;

    inline POINTER_SIZE_INT _address_to_card(void *address);

    inline void *_p_card_to_start_address(POINTER_SIZE_INT card);

    inline void _clear_card(POINTER_SIZE_INT card);

    inline void _clear_card(void *address);

    inline bool _is_card_marked(POINTER_SIZE_INT card);

    inline bool _is_card_marked(void *address);

    bool _is_card_in_use (POINTER_SIZE_INT card_index);
    //
    // Is this card a part of mature space?
    //

    // This could be inlined if it is  performance issue.
    bool _is_focus_car_card(POINTER_SIZE_INT card);

#if (GC_DEBUG>0)
    bool _is_mature_card(POINTER_SIZE_INT card);
#else

    // Double check using a different much slower path.

    inline bool _is_mature_card(POINTER_SIZE_INT card_index) {

        void *p_card_start_address = _p_card_to_start_address(card_index);

        int the_gen_number = p_global_bs->get_address_generation (p_card_start_address);

        if ((the_gen_number == YOS_GEN_NUMBER)|| (the_gen_number == POINTER_SIZE_MINUS_ONE)) {
            return false;
        } else {
            if (p_global_bs->p_get_address_container(p_card_start_address)->is_in_allocated_area(p_card_start_address)) {
                return true;
            } else {
                return false;
            }
        }
    }
#endif

    bool _is_first_card_in_super_block (POINTER_SIZE_INT card);
#ifdef GC_PT_WB
    
    // Space made available to hold the addresses of pages that have been dirtied.
    PVOID *_p_page_base_buffer;

    void _scan_page (PVOID page_base,
                    Generation *p_mature_gen,
                    Generation *p_young_gen);
#endif // GC_PT_WB
    void _scan_card(POINTER_SIZE_INT card,
                    Generation *p_mature_gen,
                    Generation *p_young_gen);

    void _scan_object(Object_Gc_Header *p_gc_hdr,
                      Generation *p_older_gen,
                      Generation *p_younger_gen);

    //#ifdef GC_OFFSET_LOGIC
        void _scan_array_object(Object_Gc_Header  *p_gc_hdr,
                                Generation *p_older_gen,
                                Generation *p_younger_gen);
    //#endif // GC_OFFSET_LOGIC

    Block_Store *_p_block_store;
};
#ifdef GC_PT_WB
UINT WINAPI dll_GetWriteWatch (IN DWORD dwFlags,          // write-tracking state
							  IN PVOID lpBaseAddress,    // base address of region
							  IN SIZE_T dwRegionSize,    // size of region
							  IN OUT PVOID  *lpAddresses,    // array of page addresses
							  IN OUT PULONG lpdwCount,   //  IN OUT PULONG_PTR lpdwCount,   // number of addresses returned
							  OUT PULONG lpdwGranularity  // page size
							  );
UINT WINAPI dll_ResetWriteWatch (PVOID lpBaseAddress,
                          SIZE_T dwRegionSize
                          );
#endif // GC_PT_WB
#endif // _CARD_TABLE_H


