
// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/Block_Store.h,v 1.2 2001/08/02 01:24:54 Rlhudson Exp $
//

#ifndef _block_store_H_
#define _block_store_H_

// The garbage collected heap is divided into blocks of size 2-power-k,
// aligned on 2-power-k bounderies. The specific 'k' will be tuned,
// but initially we may pick something like 16. 
// All the garbage collected regions (cars, steps, LOS) are composed
// from blocks that are managed by the block manager.
//

#include "platform.h"
#include "object_layout.h"

#include "gc_space.h"
#include "gc_consts.h"

#define NURSERY_TYPE 0
#define INVALID_TYPE 255

class Car;
class Gc_Fast_Hooks;
class Gc_Space;
class Gc_Plan;
//class Generation;
class Large_Object_Store;
class Nursery;
class Obsolete_Space;
class Remembered_Set;
class Step;
class Train;
class Train_Generation;
class Step_Plus_Nursery_Generation;
class LOS_Container;

extern POINTER_SIZE_INT global_heap_base; // in gc_globals.

	// The number of bits to shift an address right by,
	// in order to get the offset into the block table.
	//
#ifdef POINTER64
    // Pages are 8K 2**13 on IA64
#define GC_BLOCK_SHIFT_COUNT 13
    // This is the same as GC_CARD_SHIFT_COUNT
#else
    // Pages are 4K 2**12 on IA32
#define GC_BLOCK_SHIFT_COUNT 12
#endif

class Block_Store {
public:

    Block_Store(Gc_Fast_Hooks *p_gc_hooks,
                Gc_Plan *p_gc_plan,
                unsigned long initial_heap_size_bytes, 
		        unsigned long final_heap_size_bytes,
		        unsigned int block_size_bytes);

    virtual ~Block_Store();

	//
	// Set the generation of the sub block.
	//
	// There is a table _p_block_generation_table which contains an
	// integer per sub-block. It is either -1 if the sub block is
	// unallocated. Otherwise it is 0 if the sub block is part of
	// the young generation, or an encoding of the train/car if it
	// is in mature space.
	//

	inline int get_address_generation(void *p_address)
	{
		int block_id = (int) (((Byte *)p_address - (Byte *)_p_effective_heap_base) >> GC_BLOCK_SHIFT_COUNT); // sub_block_id
		int super_block_id = _get_super_block_id_from_sub_block_id(block_id);
		return _p_block_generation_table[block_id];
	}

	inline void set_address_generation(void *p_address, int generation)
	{
		int block_id = (int) (((Byte *)p_address - (Byte *)_p_effective_heap_base) >> GC_BLOCK_SHIFT_COUNT); // sub_block_id
		_p_block_generation_table[block_id] = generation;
	}


    inline Car *
    Block_Store::p_get_address_car(void *p_addr) 
    {
        Gc_Space *p_container = p_get_address_container(p_addr);

    #if (GC_DEBUG>2)
        assert(p_container->is_car());
    #endif // _DEBUG

        return (Car *)p_container;
    }

    //
    // Return the LOS that this address belongs in.
    //
    Large_Object_Store *p_get_address_los(void *p_addr);
    //
    // Return the Nursery that this address is contained in.
    //
    Nursery *p_get_address_nursery(void *p_addr);
    //
    // Return the Train this address is contained in.
    //
    Train *p_get_address_train(void *p_addr);
    //
    // Return the Step this address is contained in.
    //
    Step *p_get_address_step(void *p_addr);

    inline Car *p_get_object_car(Java_java_lang_Object *p_obj) {
        return p_get_address_car((void *)p_obj);
    }

    inline Generation *
    Block_Store::p_get_address_generation(void *p_addr) 
    {
        Gc_Space *p_container = p_get_address_container(p_addr);
        //Generation *p_gen = p_container->p_my_generation();//old hotspot 5/28
        Generation *p_gen = p_container->p_container;
    #if (GC_DEBUG>2)
        //
        // Catch nulls here.
        //
        assert(p_gen->is_generation());
    #endif // _DEBUG
        return p_gen;
    }

    inline Generation *p_get_object_generation(Java_java_lang_Object *p_obj) {
        return p_get_address_generation((void *)p_obj);
    }

    inline Generation *p_get_reference_generation(Java_java_lang_Object **pp_obj_ref) {
        return p_get_address_generation((void *)pp_obj_ref);
    }

    inline Train *p_get_object_train(Java_java_lang_Object *p_obj) {
        return p_get_address_train((void *)p_obj);
    }

    inline Car *p_get_reference_car(Java_java_lang_Object **pp_obj_ref) {
        return p_get_address_car((void *)pp_obj_ref);
    }

    inline Train *p_get_reference_train(Java_java_lang_Object **pp_obj_ref) {
        return p_get_address_train((void *)pp_obj_ref);
    }

    inline bool 
    Block_Store::is_address_in_heap(void *p_addr) 
    {
    #ifdef GC_NONE_V0
        return true; // We can't tell so everything can be in heap.
    #endif
	    return (((Byte *)p_addr >= (Byte *)_p_effective_heap_base) &&
		    (((Byte *)p_addr) < 
                ((Byte *)_p_effective_heap_base + _current_heap_size_bytes)));
    }

    inline bool is_object_in_heap(Java_java_lang_Object *p_obj) {
		return is_address_in_heap((void *)p_obj);
	}

	inline bool is_object_outside_heap(Java_java_lang_Object *p_obj) {
		return is_address_outside_heap((void *)p_obj);
	}

    inline bool is_reference_in_heap(Java_java_lang_Object **pp_obj_ref) {
        return is_address_in_heap((void *)pp_obj_ref);
    }

	inline bool is_reference_outside_heap(Java_java_lang_Object **pp_obj_ref) {
		if (is_reference_in_heap(pp_obj_ref)) {
			return false;
		}
		return true;
	}

	inline bool is_address_outside_heap(void *p_addr) {
		if (is_address_in_heap(p_addr)) {
			return false;
		}
		return true;
	}

	bool is_object_in_moving_area(Java_java_lang_Object *p_obj);

	bool is_reference_in_moving_area(Java_java_lang_Object **pp_obj_ref);

	bool is_address_in_moving_area(void *p_address);

	bool is_object_in_fixed_area(Java_java_lang_Object *p_obj);

	bool is_reference_in_fixed_area(Java_java_lang_Object **pp_obj_ref);

	bool is_address_in_fixed_area(void *p_address);


	bool is_address_in_young_generation(void *p_addr);

    inline bool 
    Block_Store::is_address_in_mature_generation(void *p_addr)
    {
	    if (is_address_outside_heap(p_addr)) {
		    return false;
	    }
        bool result = get_address_generation (p_addr) != YOS_GEN_NUMBER;
        return result;
    }

    inline bool
    Block_Store::is_address_in_large_object_space(void *p_addr)
    {
	    Gc_Space *p_container = p_get_address_container(p_addr);
	    if (p_container->is_large_object_space()) {
		    return true;
	    }

	    return false;
    }

    inline bool is_object_in_large_object_space(Java_java_lang_Object *p_obj) {
		return is_address_in_large_object_space((void *)p_obj);
	}

	bool is_address_in_car(void *p_addr);

	inline bool is_object_in_car(Java_java_lang_Object *p_obj) {
		return is_address_in_car((void *)p_obj);
	}

	bool is_address_in_nursery(void *p_addr);

	inline bool is_object_in_nursery(Java_java_lang_Object *p_obj) {
		return is_address_in_nursery((void *)p_obj);
	}

	bool is_address_in_step(void *p_addr);

	inline bool is_object_in_step(Java_java_lang_Object *p_obj) {
		return is_address_in_step((void *)p_obj);
	}
	
	bool is_obsolete_object(Java_java_lang_Object *p_obj);

#if (GC_DEBUG>3)
    void set_address_obsolete_container(void *p_addr);
#endif

	inline void set_mature_generation(Train_Generation *p_gen) {
		_p_mature_generation = p_gen;
	}
	inline void set_young_generation(Step_Plus_Nursery_Generation *p_gen) {
		_p_young_generation = p_gen;
	}
	
	void set_los_container(LOS_Container *p_los_container) {
		_p_los_container = p_los_container;
	}

    inline Train_Generation *get_mature_generation() {
		return _p_mature_generation;
	}

	inline Step_Plus_Nursery_Generation *get_young_generation() {
		return _p_young_generation;
	}

	inline LOS_Container *get_los_container() {
		return _p_los_container;
	}


    //
    // This routine takes an address and returns the start address of the
    // super block containing that address. If the address is not in the
    // heap, or if the containing sub block is unallocated, the result is
    // undefined for GC_DEBUG levels less than 4, but an error is flagged
    // for GC_DEBUG levels of 4 or greater.
    //
    inline void *
    Block_Store::p_get_address_super_block_start_address(void *p_address) 
    {
    #if (GC_DEBUG>3)
	    assert(p_global_bs->is_address_in_heap(p_address));
    #endif

	    //
	    // First get the ID of the sub block.
	    //
	    int sub_block_id = (int) (((Byte *)p_address - (Byte *)_p_effective_heap_base) >> GC_BLOCK_SHIFT_COUNT); // sub_block_id

    #if (GC_DEBUG>3)
	    assert((sub_block_id>=0) && (_is_sub_block_allocated(sub_block_id)));
    #endif
	    //
	    // Next get the block ID of the first sub-block in
	    // the super-block. If the debug level is > 3, it will ensure
	    // that we didn't index into an unused block.
	    //
	    int super_block_id = 
                _get_super_block_id_from_sub_block_id(sub_block_id);

    #if (GC_DEBUG>3)
	    assert((super_block_id>=0) && (_is_sub_block_allocated(super_block_id)));
    #endif
	    //
	    // Return the start address of this first sub-block in the
	    // super-block.
	    //
	    return _p_get_sub_block_start_address(super_block_id);
    }

	inline Gc_Space *p_get_address_container(void *p_address) {
		//
		// This will return the block_id of the first sub-block
		// in the super-block. If the GC_DEBUG level is greater
		// than 3, it checks to ensure that we have indexed into
		// a block that is currently used.
		//
        int super_block_id = _get_super_block_id_of_address(p_address);

        Gc_Space *p_container = 
            p_get_block_container(super_block_id);

        return p_container;
    }

	//
	// This returns the block ID of the sub block containing
	// the specified address.
	//
	//	int get_sub_block_id_of_address(void *p_addr);

    inline int get_sub_block_id_of_address(void *p_addr) {
#if (GC_DEBUG>3)
    	if (is_address_outside_heap(p_addr)) {
	    	orp_cout << "Error: Block Store asked for container of object outside GC heap." << endl;
            assert (0);
//            orp_exit(1);
	    }
#endif // _DEBUG

        int sub_block_id = 
	    	((unsigned int)((Byte *)p_addr - (Byte *)_p_effective_heap_base)) >> GC_BLOCK_SHIFT_COUNT; // sub_block_id


#ifdef GC_WB_FOR_WILLIE

#include gc_wb_for_willie.h

#endif // 

#if (GC_DEBUG>3)
	    assert((sub_block_id >= 0) && (sub_block_id < _maximum_table_index));
#endif

    	return (int)sub_block_id;
    }

    inline Gc_Space *p_get_object_container(Java_java_lang_Object *p_obj) {
	    return p_get_address_container((void *)p_obj);
    }

	inline Gc_Space *p_get_reference_container(Java_java_lang_Object **pp_obj_ref) {
		return p_get_address_container((void *)pp_obj_ref);
	}

 
	//
	// This routine exists to support setting of obsolete containers to
	// super blocks while they still remain allocated from the block store's
	// point of view. This facility is for debugging only, since entities
	// outside the block store shouldn't be able to set address containers
	// for normal operation.
	//
	void set_address_container(void *p_addr,
							   Gc_Space *p_container);


    //
    // Return a new block from the Block Store and update the
    // corresponding entry in the Block Table to point to the
    // container requesting this block. NOTE that the third 
	// argument, the generation, is an integer encoding of the
	// generation that the block store records in a "generation
	// table" for fast lookup.
    //
    void *p_get_new_super_block(Gc_Space *p_container,
		                        int block_size_bytes,
								int generation,
                                bool return_null_on_fail);

    inline unsigned int _get_sub_block_size_bytes(void ) {
		return _sub_block_size_bytes;
	}


    //
    // This routine takes an address into the heap and returns the
    // last legal address in the super block that contains this
    // address.
    //
    inline void *
    Block_Store::p_get_address_super_block_end_address(void *p_address)
    {
        void *fast_result;

        // This needs to be tested before we put in SS.
        // The fast code...
        int sub_block_id = (int) (((Byte *)p_address - (Byte *)_p_effective_heap_base) >> GC_BLOCK_SHIFT_COUNT); // sub_block_id
 
        fast_result = (void *)_p_block_end_table[sub_block_id]; 
        // This is just returned if fast code.

        // This improvement helps javac but it needs to be inlined.
        // return fast_result;
#if (GC_DEBUG>1)

        // The slow check code that does not use the _p_block_end_table code.

	    //
	    // Get the start of the super block corresponding to this address.
	    //
	    void *p_start = p_get_address_super_block_start_address(p_address);
	    //
	    // Get the super block ID.
	    //
	    int super_block_id = _get_super_block_id_of_address(p_start);
	    //
	    // Get the size of that super block in blocks.
	    //
	    int super_block_size_blocks = _p_block_size_blocks_table[super_block_id];
	    //
	    // Get the corresponding size in bytes.
	    //
	    int super_block_size_bytes = super_block_size_blocks * _sub_block_size_bytes;
	    //
	    // Calculate the end address.
	    //
        void *result = (void *)((char *)p_start + super_block_size_bytes - 1);

        assert (fast_result == result);

        // The end of the block is the byte before the start of the next
        // block so I added a -1.
#endif // GC_DEBUG>0

	    return fast_result;
    }
	
    inline void *get_object_super_block_start_address(Java_java_lang_Object *p_obj) {
	    return p_get_address_super_block_start_address((void *)p_obj);
    }
	
    inline Gc_Space *
    Block_Store::p_get_block_container(unsigned long block_id)
    {
    #if (GC_DEBUG>0)
        assert(block_id < _table_size_entries);
    #endif
	    // Why isn't this locked by p_gc_lock????? This is during initialization so this
        // should not be needed but to be safe we use it.
        //lock();
        Gc_Space *p_space = _p_block_container_table[block_id];
        //unlock();

	    return(p_space);
    }

	//
    // Card table logic needs to know where heap starts to calculate
	// card.
	//
	Byte * get_heap_base();

	//
	// Return the total number of bytes in the heap (allocated
	// and free).
	//
    inline unsigned long get_heap_size_bytes() {
		return _current_heap_size_bytes;
	}

	//
	// Return the number of bytes free in the block store.
	//
	inline unsigned long get_free_size_bytes() {
		return _free_sub_blocks * _sub_block_size_bytes;
	}

    inline bool
    Block_Store::is_address_in_super_block(void *p_block_addr, void *p_addr)
    {
	    void *p_real_start_addr = p_get_address_super_block_start_address(p_block_addr);
	    void *p_end_addr = p_get_address_super_block_end_address(p_block_addr);

    #if (GC_DEBUG>2)
	    assert(p_block_addr == p_real_start_addr);
    #endif // _DEBUG

	    return ((p_addr <= p_end_addr) && (p_addr >= p_real_start_addr));
    }

	//
	// A GC container will use this to return a super block
	// that it doesn't need any more. This may happen when
	// cars or FROM spaces are discarded, or when LOS objects
	// are freed, etc.
	//
	void release_super_block(void *p_super_block_start_address);

    inline int get_total_current_sub_blocks () {
        return _total_current_sub_blocks;
    }

    inline unsigned long get_free_sub_blocks () {
        return _free_sub_blocks;
    }
    
    //
    // Round up to the next block size.
    //
    unsigned int size_up_to_block_size (unsigned int number_of_bytes);

    unsigned int size_up_to_los_block_size (unsigned int number_of_bytes);

    inline unsigned long get_final_heap_size_blocks () {
       return (unsigned long) _final_heap_size_bytes / _sub_block_size_bytes;   
    } 

	void dump_block_store_info();

private:

    int _number_of_tables;

	Train_Generation *_p_mature_generation;

	Step_Plus_Nursery_Generation *_p_young_generation;

	LOS_Container *_p_los_container;

	//
	// This routine is used to get the number of specified sub
	// blocks from the block store using an address order first
	// fit policy. The super block corresponding to the contiguous
	// sub blocks is marked as in use (corresponding to the 
	// specified container.)
	//
	int _address_order_first_fit(Gc_Space *p_container,
		                         int generation_code,
		                         int number_of_sub_blocks);

	//
	// When creating the block store, create one large super
	// block out of all the blocks.
	//
	void _initialize_block_tables(void *p_start,
		                          void *p_last_block);
	//
	// This routine is called to extend the block tables when
	// the heap is extended.
	//
	int _extend_block_tables(void *p_start,
		                     void *p_last_block);

	//
	// The Block Store is being asked to extend the heap
	// by the specified number of bytes. 
	//
	bool _extend_heap();

	//
	// Private copy of the plan object.
	//
	Gc_Plan *_p_gc_plan;

	//
	// This returns the block ID of the first sub-block in the 
	// enclosing super-block. If the block is currently
	// un-used, the return value is undefined 
	// if GC_DEBUG is less than 4. Otherwise it asserts if
	// the GC_DEBUG is 4 or greater.
	//
    //

    inline int
    Block_Store::_get_super_block_id_of_address(void *p_addr)
    {
    //    if ((void *)p_addr == (void *)0x29dd16c) {
    //        p_addr = p_addr;
    //    }
    #if (GC_DEBUG>0)
	    if (is_address_outside_heap(p_addr)) {
		    orp_cout << "Error: Block Store asked for container of object outside GC heap." << endl;
		    assert(0);
		    orp_exit(1);
	    }
    #endif // _DEBUG

	    //
	    // First get the block ID of the enclosing sub block.
	    //
        int sub_block_id = (int) (((Byte *)p_addr - (Byte *)_p_effective_heap_base) >> GC_BLOCK_SHIFT_COUNT); // sub_block_id

	    //
	    // Next discover the start of the super-block.
	    //
	    int real_block_id = (int)_p_block_start_id_table[sub_block_id];

    #if (GC_DEBUG>0)
	    assert(real_block_id >= 0);
    #endif
	    return real_block_id;
    }

    inline void *
    Block_Store::_p_get_sub_block_start_address(int block_id) 
    {	
	    block_id <<= GC_BLOCK_SHIFT_COUNT;
	    return (void *)(block_id + (char *)_p_effective_heap_base);
    }

    //
	// For bounds checking, we maintain the upper bound of the 
	// meta information tables in the block store.
	// NOTE: This reflects the maximum heap size.
	//
	int _maximum_table_index;

	//
	// This routine checks if the specified sub block is allocated
	// out to some GC container. 
	//
	bool _is_sub_block_allocated(int sub_block_id);

	//
	// The following routine checks to make sure that the specified
	// index is indeed into a super block. A super-block may be
	// allocated or unallocated
	//
	bool _is_super_block(int super_block_id);

	//
	// This routine takes a block ID of a sub block and returns the block
	// ID of the containing super block. It does not check to see if the
	// sub block is unallocated.
	//
    inline int 
    Block_Store::_get_super_block_id_from_sub_block_id(int sub_block_id)
    {
	    //
	    // The following table is indexed by sub block ID. It contains -1 if
	    // the sub block is unallocated. Otherwise it contains the block ID of
	    // the super block containing that sub block. The block ID of the first
	    // sub block in the super block has the same ID as that super block.
	    //
	    return _p_block_start_id_table[sub_block_id];
    }

    //
    // This is a global that we can use for performance, instead
    // of creating a new one each time on demand.
    //
    Obsolete_Space *_p_global_obsolete_space;

	//
	// Allocate the block tables that maps blocks to containers.
	//
	void _allocate_block_tables();

	//
	// Given a super block size in bytes, return the corresponding
	// number of blocks needed.
	//
	unsigned long _get_sub_block_count_from_bytes(unsigned int number_of_bytes);

	//
	// Get relevant information about the underlying machine/os
	// (ex: page size, number of active processors, etc..)
	//
	void _get_system_info();
    //
    // Protecting the block store from concurrent access.
    //
    CRITICAL_SECTION _lock;

	//
	// Round up the address to the specified page multiple boundary
	//
	void *_p_page_align_up(void *p_addr, unsigned int number_of_pages);
	//
	// Round down the address to the specified page multiple boundary.
	//
	void *_p_page_align_down(void *p_addr, unsigned int number_of_pages);

	//
	// This table has an entry per block, and maintains
	// a pointer to the container that owns that block.
	//
    Gc_Space **_p_block_container_table;

	//
	// This table has an entry per block, and maintains
	// the size of the block if it is allocated, and the
	// first block of the super-block. Otherwise it is 
	// -1 if unallocated, and 0 if not the first block.
	//
    int *_p_block_size_blocks_table;

	//
	// For all blocks in a super-block that are not the first,
	// this gives the offset of the first block in each of the
	// three tables. The field is -1 if unallocated, and 0 if
	// not the first block in the super-block.
	//
	int *_p_block_start_id_table;

   	//
	// For all blocks in a super-block that are allocated, the
    // end of the block.
	//
	POINTER_SIZE_INT *_p_block_end_table;
	//
	// Holds the generation of the sub blocks.
	// Is -1 if unallocated.
	//
	int *_p_block_generation_table;

	//
	// A pointer to the blocks that are in the garbage_collected
	// portion of the heap.
	//
	void * _gc_free_list;

    //
    // This is the size of blocks doled out to steps and cars.
    //
	unsigned int _sub_block_size_bytes;

	//
	// This is the base of the allocated region, including the
	// meta data region.
	//
	void *_p_real_heap_base;
	//
	// THis is the base of the actual garbage collected portion of
	// the allocated heap.
	//
	void *_p_effective_heap_base;
	//
	// This is the current size of the heap, including the garbage 
	// collected area and the associated meta information (tables, etc).
	//
	unsigned long _current_heap_size_bytes;
	//
	// This is just the garbage collected heap size, not including
	// the ancilliary meta information.
	//
	unsigned long _effective_heap_size_bytes;
	//
	// The initial number of actual pages committed is this.
	//
    unsigned long _initial_heap_size_bytes;
	//
	// The final heap size is this (the virtual space is allocated 
	// at startup time.)
	//
	unsigned long _final_heap_size_bytes;
	//
	// Pointer to the last block in committed space.
	//
	void *_p_last_block;

	//
	// The machine (OS) page size.
	//
	unsigned int _machine_page_size_bytes;
	//
	// The ENABLED processors on this machine.
	//
	unsigned int _number_of_active_processors;

	//
	// The number of entries in the table.
	// NOTE: this reflects the final heap size.	
	//
    unsigned long _table_size_entries;
	//
	// The corresponding table size in bytes
	// NOTE: this reflects the final heap size.
	//
	unsigned long _table_size_bytes;
	//
	// This is a count of the total number of blocks in the 
	// block store.
	// NOTE: this reflects the CURRENT heap size.	
	//
	int _total_current_sub_blocks;
	
	//
	// This is a count of the free blocks in the block store.
	//
	unsigned long _free_sub_blocks;

};

#endif // _Block_Store_H_
