// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/block_list.cpp,v 1.1.1.1 2001/07/23 07:25:38 xli18 Exp $
//

#include "platform.h"

#include "gc_header.h"
#include "remembered_set.h"
#include "Block_Store.h"
#include "generation.h"
#include "descendents.h"
#include "gc_hooks.h"
#include "block_list.h"
#include "gc_asserts.h"
#include "card_table.h"
#include "step.h"

#include "nursery_step_gen.h"

extern Step_Plus_Nursery_Generation *p_young_gen;

#ifdef GC_SAPPHIRE
//
//  This takes blocks in arg1's block list and moves them to arg2's block list.
//  The blocks in arg1 list are removed and interesting param are updated.
//
void Block_List::scavenge_block(void *the_block, 
                                 int the_size_bytes,
                                 void *block_end)
{
    assert ((char *)block_end <= ((char *)the_block+the_size_bytes));
    _p_block[_number_of_blocks] = the_block;
    _p_block_end[_number_of_blocks] = (Object_Gc_Header *)block_end;
    _p_free = block_end; // Since this can be for a step _p_free should not be changed.
    _p_scan = block_end; // One can only scavenge scanned blocks.
    _p_scan_start = block_end;
    _p_base = the_block;
    // Since we are scavenging blocks then we won't be doing any scanning 
    // so _current_block_id is not used.
    _current_block_id = _number_of_blocks; // point to this the last block.
    _number_of_blocks++;
    p_global_bs->set_address_container (the_block, this);
    assert (((Byte *)the_block + (unsigned long)the_size_bytes) > (Byte *)block_end);
    assert ((Byte *)the_block <= (Byte *)block_end);
    assert (_current_block_id == (_number_of_blocks - 1));
}
#else

void Block_List::scavenge_block(void *the_block, 
                                 int the_size_bytes,
                                 void *block_end)
{
    _p_block[_number_of_blocks] = the_block;
    _p_block_end[_number_of_blocks] = (Object_Gc_Header *)block_end;
    _p_free = block_end; // Since this can be for a step _p_free should not be changed.
    _p_scan = block_end; // One can only scavenge scanned blocks.
    _current_block_id = _number_of_blocks;
    _number_of_blocks++;
    p_global_bs->set_address_container (the_block, this);

    assert (((Byte *)the_block + the_size_bytes) > (Byte *)block_end);
    assert ((Byte *)the_block <= (Byte *)block_end);
    assert (_current_block_id == (_number_of_blocks - 1));
}
#endif // GC_SAPPHIRE

//
// Add another block into the current step or car.
//
void 
Block_List::_add_block() 
{
    int super_block_size_bytes = my_block_size_bytes();

    void *p_save_free = _p_free;

    //
    // The container (step or car) provides an encoding of the
    // generation to be used in marking this block that is to
    // be allocated.
    //
    int generation = _get_generation_encoding();

    _p_free =
        _p_block[_number_of_blocks] = 
            _p_block_store->
                 p_get_new_super_block(this,
                                       super_block_size_bytes,
                                       generation,
                                       false); 

    _p_block_end[_number_of_blocks] = (Object_Gc_Header *)_p_free;
    
    _p_base = _p_free;

    _p_ceiling = (void *)((char *)_p_base + super_block_size_bytes);


#ifdef OBJECT_SPLITTING
	_p_cold_free = (void *) ((POINTER_SIZE_INT) _p_base + (super_block_size_bytes / 2));
#endif // OBJECT_SPLITTING


#if (GC_DEBUG>3)
    memset (_p_free, 0xE5, super_block_size_bytes);  // fill the block before adding it.
                                                // use E5 since CPM used E5
                                                // Also setting the low bit will
                                                // break lock code looking at a 
                                                // stray pointer ;)_                                               
#endif

    //
    // If this is the first TO block in a car or step, initialize _p_base and _p_free.
    //
    if (_number_of_blocks == 0) {
        assert (_current_block_id == -1);
        _current_block_id = 0;
        _p_base = _p_free;
        assert (p_save_free == NULL);  // No blocks - no previous free pointer.
        assert (_p_scan == NULL);   // We could have set this scan pointer if there
                                    // are no blocks in this list yet.
    } else {
        //
        // This is not the first TO block. Therefore, we need to
        // check if the scan pointer needs to be updated to reflect
        // the addition of this new block.
        //
        if (p_save_free == _p_scan) {
            //
            // We had already scanned the last object, since
            // we observe that _p_free == _p_scan. Since we
            // changed _p_free we know nothing will ever be added to this
            // previous block.
            // So let's update the scan pointer.
            //
            _p_scan = (Object_Gc_Header *)_p_free;
            // And set the current_block_id to the new block.
            _current_block_id++;
        }

    }
    _number_of_blocks += 1;
 
    assert(_number_of_blocks < MAXIMUM_BLOCKS_IN_LIST); 

#ifdef GC_GEN_V3
    //
    // Finally we tell the card manager to purge any "last object"
    // entries corresponding to the cards that this new block 
    // represents.
    //
    _p_card_table->clear_last_object_addresses(_p_free,
                                               super_block_size_bytes);

    _p_card_table->clear_cards_in_super_block(_p_free, 
                                              super_block_size_bytes);
#endif // GC_GEN_V3
 
    assert(_p_free);
//    assert (_p_scan <= _p_free); - not true, if the scan pointer is into another block
                                    // That happens to be in a block that is at a higher 
                                    // location in memory. 
    return;
}

// 
// Ancilliary function of _add_into_block that checks if the object will cross
// a sub block. It does it very fast. And without a shift or memory reference
// for those of us who care.
//

#if (GC_DEBUG>0)
// Do not inline for debug.
#else
// inline for release code..
inline
#endif //(GC_DEBUG>0)

bool
Block_List::_will_fit_in_sub_block (void *p_free_ptr, int size) {

#ifdef OBJECT_SPLITTING
    bool result = ( ((POINTER_SIZE_INT)p_free_ptr ^ ( (POINTER_SIZE_INT)p_free_ptr + (size / 2))) 
                    < (p_global_bs->_get_sub_block_size_bytes() / 2)
                  );
#else
    bool result = ( ((POINTER_SIZE_INT)p_free_ptr ^ ( (POINTER_SIZE_INT)p_free_ptr + size)) 
                    < p_global_bs->_get_sub_block_size_bytes()
                  );
#endif // OBJECT_SPLITTING

#if (GC_DEBUG>1) 
    if (_will_not_fit_in_block (size) & result) {  // It will fit into the 
                                        // sub block but not the superblock 
        assert (0);
    }
#endif //(GC_DEBUG>1)

    return result;
}


//
// fast_reserve_space_in_block (Java_java_lang_Object **pp_obj_ref)
// Ancilliary function of _add_into_block that reserves the space for the object.
// It is inlined in the hopes of keeping _add_into_block fast. 
//
#if (GC_DEBUG>0)
// Do not inline for debug.
#else
// inline for release code..
inline
#endif //(GC_DEBUG>0)
Object_Gc_Header *
Block_List::p_fast_reserve_space (int real_object_size_bytes)
{
     
    assert(_number_of_blocks > 0);
#if (GC_DEBUG>3)
    _debug_check_block_integrity();
#endif 

    //
    // Now check to see if there is enough space in this sub block to host object.
    // We check the sub block since it can be done very fast and if the sub block is
    // 16K and average object size is 80 (a guess) bytes then we get 2000 quick checks
    // for each expensive check.
    // 
    if (!(_will_fit_in_sub_block (_p_free, real_object_size_bytes))) {              //Fast
        // The quick check failed now do the expensive check.
        if (_will_not_fit_in_block(real_object_size_bytes)) {
            //
            // No. Not enough space - add another block to this step.
            //
            _add_block();
 
            //
            // We assume here that a movable object is smaller than a
            // block. Therefore we don't need another check here.
            //  
            assert(real_object_size_bytes < my_block_size_bytes()); 

#if (GC_DEBUG>3)
            _debug_check_block_integrity();
#endif

        } // end (_will_not_fit_in_block(real_object_size_bytes))
    } // end(!(_will_fit_in_sub_block (_p_free, real_object_size_bytes)))

#if (GC_DEBUG>2)
    //
    // Then get the end of this block.
    // 
    void *p_block_end = 
            _p_block_store->p_get_address_super_block_end_address(_p_free);    
#endif // GC_DEBUG>2

    //
    // Locate the target address in this step to copy the object into.
    //
    Object_Gc_Header *p_new_hdr =
                    (Object_Gc_Header *)_p_free;

  
#if (GC_DEBUG>0) // help with debugging
    Gc_Space *p_container = p_global_bs->p_get_address_container(p_new_hdr);
    assert (p_container);

#endif

#if (GC_DEBUG>2) 
    _debug_check_block_integrity(); // DEBUG
#endif // _DEBUG
    //
    // OK, there is space. Copy is done in caller to support Sapphire.
    //
//    fast_copy_forward_and_update_locks(pp_obj_ref,                                  // Fast 
//                                     p_old_hdr, 
//                                     p_new_hdr, 
//                                     real_object_size_bytes);

#if (GC_DEBUG>2)
    _debug_check_block_integrity(); // DEBUG
#endif

    //
    // Record this object as the last object in that
    // corresponding card.
    //

    // Only do it if the object is in MOS since no one else needs it...

    if (p_global_bs->is_address_in_mature_generation ((void *)p_new_hdr)) {        // Fast
#ifdef GC_COPY_V2
        assert (0);
#endif
#ifdef GC_FIXED_V1
        assert(0);
#endif
#ifdef GC_GEN_V3
        // do nothing
#endif
#ifdef GC_GEN_V3
        _p_card_table->set_last_object_address(p_new_hdr);                        // Fast?
#endif // GC_GEN_V3
#ifdef GC_TRAIN_V5
        // done by GC_GEN_V3 _p_card_table->set_last_object_address(p_new_hdr);
#endif // GC_TRAIN_V5
    }

    //
    // Record the last object in this block to facilitate
    // subsequent scanning of this step or car...
    //
    _p_block_end[_number_of_blocks - 1] = p_new_hdr;                              // Fast

#if (GC_DEBUG>1)
    _debug_check_block_integrity(); // DEBUG
#endif

    //
	// Bump up the free pointer past the newly allocated object.
	//
#ifdef OBJECT_SPLITTING
	_p_free = (Object_Gc_Header *)((POINTER_SIZE_INT)_p_free + (real_object_size_bytes / 2));
#else
	_p_free = (Object_Gc_Header *)((POINTER_SIZE_INT)_p_free + real_object_size_bytes);
#endif // OBJECT_SPLITTING
    
    
#if (GC_DEBUG>1)
    _debug_check_block_integrity(); // DEBUG
#endif

    //
    // Set up the scan pointer if this is the first object in the step.
    //
    if (_p_scan == NULL) {
        _p_scan = p_new_hdr;
    }
#if (GC_DEBUG>1)
    //
    // Update statistics.
    //
    _resident_occupied_bytes += real_object_size_bytes;
    _resident_object_count   += 1;
#endif // (GC_DEBUG>1)
#if (GC_DEBUG>2)
    _debug_check_block_integrity();
#endif

    return p_new_hdr;
}


// A wrapper for fast_reserve_space_in_block for those who only need to reserve the space. 
Java_java_lang_Object *
Block_List::p_reserve_space (Java_java_lang_Object **pp_obj_ref)
{   
    //
    // Get the real start of the object.
    //
    Object_Gc_Header *p_old_hdr = get_object_gc_header(*pp_obj_ref);                // Fast

    //
    // Get the object's real size so we know the size of the copy.
    //
    int real_object_size_bytes = 
        get_real_object_size_bytes(*pp_obj_ref);                                    // Fast
    Object_Gc_Header *p_new_hdr = 
        p_fast_reserve_space (real_object_size_bytes);
 
    gc_trace ((void *)*pp_obj_ref, "fast_reserve_space_in_block in to space for this from object"); 

#ifdef GC_SAPPHIRE
    memset (p_new_hdr, 0, real_object_size_bytes);
#endif // GC_SAPPHIRE
    return get_object_from_gc_header(p_new_hdr);
}

// A wrapper for fast_reserve_space_in_block for those who only need to reserve the space. 
Java_java_lang_Object *
Block_List::p_reserve_bytes (unsigned size)
{
    Object_Gc_Header *p_new_hdr = 
        p_fast_reserve_space (size);
    return get_object_from_gc_header(p_new_hdr);
}

//
// Ancilliary function for p_scavenge_object.
//
Java_java_lang_Object *
Block_List::_add_into_block(Java_java_lang_Object **pp_obj_ref)
{   
    //
    // Get the real start of the object.
    //
    Object_Gc_Header *p_old_hdr = get_object_gc_header(*pp_obj_ref);                // Fast

    //
    // Get the object's real size so we know the size of the copy.
    //
    int real_object_size_bytes = 
        get_real_object_size_bytes(*pp_obj_ref);                                    // Fast
 
    assert (real_object_size_bytes <= my_block_size_bytes()); 

    //
    // Locate the target address in this step to copy the object into.
    //
    Object_Gc_Header *p_new_hdr =
                    p_fast_reserve_space(real_object_size_bytes);
 
    gc_trace ((void *)*pp_obj_ref, "evacuated by _add_into_block"); 

    //
    // OK, there is space. Do the copy.
    //
    fast_copy_forward_and_update_locks(pp_obj_ref,                                  // Fast 
                                       p_old_hdr, 
                                       p_new_hdr, 
                                       real_object_size_bytes);

    assert (*pp_obj_ref == get_object_from_gc_header(p_new_hdr));
 
    gc_trace ((void *)*pp_obj_ref, "Slot updated to point to this object."); 

    return get_object_from_gc_header(p_new_hdr);
}
/**
bool 
Block_List::more_to_scan()
{
    
    if (_p_free == _p_scan) {
        //
        // Done with scan.
        //
        assert (_p_free != NULL); // ?? is this true or can we scan an empty block???
        assert(_current_block_id == (_number_of_blocks - 1));
        assert(_number_of_blocks > 0);
        return false;
    }
    assert (_p_scan <= _p_block_end[_current_block_id]);
    assert (_p_free <= _p_block_end[_number_of_blocks - 1]);
    assert (_current_block_id < _number_of_blocks);
    return true;

}
**/


// Is there a pending cheney scan pending here.
bool
Block_List::cheney_scan_pending()
{
    if (_number_of_blocks == 0) {
        assert (_p_scan == NULL);
        assert (_p_free == NULL);
        assert (_p_base == NULL);
        assert(_current_block_id == (_number_of_blocks - 1));
        return false;
    }
    if (_p_scan == _p_free) {
        assert(_current_block_id == (_number_of_blocks - 1));
        return false;
    }
    if (_p_free == _p_base) {
        assert (_p_scan == NULL);
        return false;
    }
    return true;
}

// Scan this block list. Return T if this results in an object being
// scanned otherwise return false. This result is used to determine
// if we are done with cheney scanning.


// RLH - for some reason 
//  get_real_object_size_bytes,
//  _advance_scan_pointer and
//  cheney_scan_pending are not inlined...
//  I wonder if this is on the VTune hit parade

bool
Block_List::cheney_scan_block_list(bool doing_mos_collection)
{
#if (GC_DEBUG>0)
    _debug_check_block_integrity();
#endif
    if (!(cheney_scan_pending())) {
        //
        // Looks like the party was over before I got here.
        // Let the caller know that not objects were processed.
        //
        return false;
    }
    // The scan pointer can be into one block but the free pointer can be in another.
    // So this assert is not correct.
    //    assert (_p_scan < _p_free);

    //
    // Get the first object in the cheney scan queue.
    //
    Object_Gc_Header *p_gc_hdr_for_gen = (Object_Gc_Header *)_p_scan;

#if (GC_DEBUG>2)
    verify_is_object(p_gc_hdr_for_gen);
#endif 

    //
    // Get the real object pointer from the GC header.
    //
    Java_java_lang_Object *p_obj_for_gen = get_object_from_gc_header(p_gc_hdr_for_gen);

    // Get the generation so we can use it to decide policy on the scavenge below.
    // This should be the same generation as the generation associated with "this"
    Generation *p_gen = 
        p_global_bs->p_get_object_generation(p_obj_for_gen);

    assert (p_container == p_gen);
    //orp_cout << " block list scanning generation " << p_gen->get_generation_number() << endl;

    //
    // Do a breadth-first cheney scan:
    //

    while (cheney_scan_pending()) { // This could be T since I return when we are done.
        //
        // Get the first object in the scan queue.
        //
        Object_Gc_Header *p_gc_hdr = (Object_Gc_Header *)_p_scan;

#if (GC_DEBUG>0)
       
        verify_is_object(p_gc_hdr);
#endif 
        //
        // Get the real object pointer from the GC header.
        //
        Java_java_lang_Object *p_obj = get_object_from_gc_header(p_gc_hdr);
        gc_trace (p_obj, "cheney scanning this object."); 
        p_gen->cheney_scan_execute_policy(p_obj, doing_mos_collection);

        //
        // Bump the scan pointer forward.
        //
        //
        unsigned int real_object_size_bytes =
            get_real_object_size_bytes(p_gc_hdr);

        if (_advance_scan_pointer(real_object_size_bytes)) {
            //
            // Function returns true if there are objects remaining
            // to be scanned.
            //
            continue;
        } else {
            //
            // No. Nothing left.
            //
            assert ( !( cheney_scan_pending() ) );
#if (GC_DEBUG>1)
            _debug_check_block_integrity();
#endif
            return true;
        }
    }
    assert (0); // The return in the _advance_scan_pointer logic above should 
                // have done the return.
    // Let the caller know that objects were processed
    return true;
}

//
// After a reclaim, the Mrl_Gc_Vxxx calls each generation 
// to give it an opportunity to clean up. Step_Generation calls us.
//
void
Block_List::cleanup()
{
    orp_cout << "Error: Block_List::cleanup stub" << endl;
    orp_exit(1); // who calls me?
}

    
#if (GC_DEBUG>3)
//
// At debug time, when a car has been reclaimed, or when
// the FROM blocks are being reclaimed, we may choose to
// avoid re-using the blocks, instead marking them as
// in an obsolete container.
//
void 
Block_List::debug_set_blocks_obsolete()
{
    for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
        p_global_bs->set_address_obsolete_container(_p_block[idx]);
    }
}
#endif // GC_DEBUG>3

#if (GC_DEBUG>3)
void
Block_List::debug_write_protect_blocks()
{
    for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
        unsigned long old_prot;
        int success = VirtualProtect(_p_block[idx],
                                     my_block_size_bytes(),
                                     PAGE_NOACCESS,
                                     &old_prot);
        if (success == 0) {
            printf("Virtual protect error %d (0x%x)\n", GetLastError(), GetLastError());
            orp_exit(GetLastError());
        }
    }
}
#endif // GC_DEBUG>3

//
// A subordinate container is asking us to scavenge an object
// from that container into ours.
//
Java_java_lang_Object *
Block_List::p_scavenge_object(Java_java_lang_Object **pp_obj_ref) 
{
    Java_java_lang_Object *p_old_object = *pp_obj_ref;

    Java_java_lang_Object *p_return_object = NULL;

//    assert(not_in_to_block(p_old_object)); // We should never 
                                            // scavenge objects in to space.
    //
    // By default, a TO step should have at least one block.
    //
    assert(_p_free);
#if (GC_DEBUG>3)
    _debug_check_block_integrity();
#endif 
    //
    // Import the object into our step or car.
    //
    p_return_object = _add_into_block(pp_obj_ref); 

    gc_trace (p_old_object, 
        "p_scavenge_object scavenges, ln 593 in block_list.cpp");
    gc_trace (p_return_object, 
        "p_scavenge_object returns, ln 595 in block_list.cpp"); 
 
    assert(_p_scan != _p_free);
    assert(_p_scan != NULL); 

    //
    // Inform container (generation) that we have a cheney scan needed.
    //
    // The scan will update any needed write barriers.
    //
    // Perhaps also need to update write barriers to reflect this object
    // being moved. Always send the request to the generation of the older
    // object. That way, scans related to younger collections go to the
    // younger generation (for either steps or cars), and vice versa.
    //
    // ***************** SHOULDN'T THIS INFORM ITSELF, WHO CARES ABOUT WHERE IT
    //    CAME FROM. IN any case since we only have one gen in GC_COPY_V2
    //    this shouldn't be a problem. WHY ARE CHENEY SCANS REGISTERED WITH 
    //    GENERATIONS???
    //

//    orp_cout << "p_gen registering cheney_scan is " << p_gen << endl;

#if (GC_DEBUG>0)
    //
    // Provide opportunity for debug routines to use this.
    //
    scavenge_hook(pp_obj_ref, p_return_object);

#endif // GC_DEBUG>0
#if (GC_DEBUG>3)
    _debug_check_block_integrity();
#endif

    return p_return_object;
}
//    
// Centralize integrity tests to one routine.
//
void 
Block_List::_debug_check_block_integrity() 
{
    if (_number_of_blocks == 0) { 
        // We have an empty block list
        assert (_p_free == 0);
        assert (_p_scan == 0);
        // The _current_block_id should -1 since there are no blocks.
        assert (_current_block_id == -1);
        return;
    }

    void *p_block_start =
        _p_block_store->p_get_address_super_block_start_address(_p_free);
    void *p_block_end =
        _p_block_store->p_get_address_super_block_end_address(_p_free);
    assert((char *)_p_free <= (char *)p_block_end);
    assert(_p_block[_number_of_blocks - 1] == p_block_start);
    assert(_p_free >= _p_block_end[_number_of_blocks - 1]);
    assert(_p_block_end[_number_of_blocks - 1] <= p_block_end);
    assert(_p_block_end[_number_of_blocks - 1] >= p_block_start);
    assert(_p_block_end[_number_of_blocks - 1] <= _p_free);
    assert(_current_block_id < _number_of_blocks); // _current_block_id is zero based.
}

#if (GC_DEBUG>3)
//
// Verify all objects in specified block are consistent.
//
bool 
Block_List::_verify_block(void *p_block_start, void *p_block_end) 
{
    if (_resident_object_count == 0) {
        return true;
    }   
    //
    // Make sure contents are properly aligned valid objects.
    //
    _walk_block(p_block_start, p_block_end, assert_is_object);
    //
    // Make sure none of the objects have their busy bit set.
    //
    _walk_block(p_block_start, p_block_end, assert_busy_bit_not_set);
    //
    // Make sure none of the objects have their forward bit set.
    //
    _walk_block(p_block_start, p_block_end, assert_object_not_forwarded);
    //
    // Make sure that live objects don't point to dead objects.
    //
    _walk_block(p_block_start, p_block_end, assert_reachable_objects_not_forwarded);

    return true;
}
#endif // _DEBUG

#if (GC_DEBUG>3)
//
// Each space provides a routine to verify that it is consistent.
// i.e. contains only well-formed objects.
//
bool Block_List::verify_space() 
{
    for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
        if (_verify_block(_p_block[idx],
                          _p_block_end[idx])) {
            continue;
        } else {
            return false;
        }
    }
    return true;
}
#endif // _DEBUG

//
// This routine takes a function argument that accepts a GC header
// and walks the entire block applying that function to each
// object. The walk terminates if the function returns false.
//
void Block_List::_walk_block(void *p_block_start, 
                             void *p_block_end,
                             bool (*func)(Object_Gc_Header *,
                                          Remembered_Set *))
{
    Object_Gc_Header *p_scan_gc_hdr = 
        (Object_Gc_Header *)p_block_start;

    if (p_block_start == p_block_end) {
        //
        // The block is empty.
        //
        return;
    }

    Remembered_Set *p_rs = p_young_gen->p_old_to_young_remembered_set();

    while ((char *)p_scan_gc_hdr < p_block_end) {
        // was < (char *)_p_base + my_block_size_bytes()) but that would include
        // (non-existent) objects past the frontier.

        func(p_scan_gc_hdr, p_rs);

        if (p_scan_gc_hdr == p_block_end) {
            //
            // We have reached the last object in this block.
            //
            return; 

        } else {
            p_scan_gc_hdr = p_scan_forward_over_object(p_scan_gc_hdr);
        }
    }
    return;
}
//
// Walk the entire list and apply the function header.
//
void 
Block_List::walk_list(bool (*func)(Object_Gc_Header *,
                                   Remembered_Set *)) {
    for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
        _walk_block(_p_block[idx], _p_block_end[idx], func);
    }
    return;
}

bool
Block_List::_will_not_fit_in_block(Java_java_lang_Object **pp_obj_ref)
{
    unsigned long real_object_size_bytes =
        get_real_object_size_bytes(*pp_obj_ref);

    return _will_not_fit_in_block(real_object_size_bytes);
}

bool 
Block_List::_will_not_fit_in_block(unsigned long real_object_size_bytes)
{
#ifdef OBJECT_SPLITTING

	// In the case of Object Splitting, there is only one sub-block per super-block.
	// Which means that the last usable byte of the current block is also the last 
	// byte of the superblock. This means that we can simply compare with _p_cold_free

    if (((char *)_p_free + (real_object_size_bytes / 2)) < ((char *) _p_cold_free)) 
		return false;
	else
		return true;
#else
    //
    // Get the end of newest block.
    // 
    // This is pretty slow so it should only follow a
    // check that ensures _will_fit_in_sub_block is false.
    //

    void *p_block_end = 
        _p_block_store->p_get_address_super_block_end_address(_p_free);
    // Exactly fill up a block if we can. p_block_end is last byte in block
    // so it can be used so we inc by 1.
    if (((char *)_p_free + real_object_size_bytes) >= ((char *)p_block_end + 1)) {
        // What is no objects have been allocated in LOS?
        return true;  // Yes, will not fit.
    } else {
        return false; // No, will fit.
    }
#endif // OBJECT_SPLITTING
}

// end file gc\block_list.cpp
