// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/class_loader/Class_File_Loader.cpp,v 1.23 2002/01/04 21:08:27 ssubram5 Exp $
//


#ifdef ORP_POSIX
#include "Loader_Result.h"
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#endif

#include "platform.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>

#ifndef ORP_POSIX
#include <io.h>
//#include <orp_io.h>
#endif

#include <fcntl.h> 
#include <fstream.h>
#include <string.h>
#include <assert.h>

#include "Class.h"
#include "Class_Loader.h"
#include "Package.h"
#include "environment.h"
#include "nogc.h"
#include "orp_synch.h"
#include "orp_utils.h"
#include "root_set_enum.h"
#include "ini.h"
#include "exceptions.h"
#include "bytereader.h"

#include "Class_File_Loader.h"

/////////////////////////////////////////////////////////////////////////////////////////////////.
#ifdef ZIP_JAR_ARCHIVE
#include <string.h>
#include <time.h>
#include <errno.h>

#ifdef ORP_POSIX
# include <unistd.h>
# include <utime.h>
#else
# include <direct.h>
# include <io.h>
#endif

#include "jarfile_util.h"


#ifdef CLI_OCL
#include "../include/cli_misc.h"
#endif // CLI_OCL


Properties JarFile_Properties;
BOOL classpath_changed = FALSE;

extern ExpandableMemBlock class_path;
ExpandableMemBlock extra_path;

inline BOOL isPathExist(char *path){
	BOOL exist = FALSE;
	char r[_MAX_PATH];
	sprintf(r, "%c%s%c", PATH_SEPARATOR, path, PATH_SEPARATOR);
	if(strstr(ORP_Global_State::loader_env->classpath, r))
		exist = TRUE;
	else{
		if(strstr(ORP_Global_State::loader_env->classpath, r+1))
			exist = TRUE;
		else{
			*(r+strlen(r)-1) = '\0';
			if(strstr(ORP_Global_State::loader_env->classpath, r))
				exist = TRUE;
		}
	}
	return exist;
}

inline BOOL MergeClasspath(char *classpath){
	char sep[] = { PATH_SEPARATOR };
	char *p = strtok(classpath, sep);
	BOOL changed = FALSE;
	while(p){
		if(*p && !isPathExist(p)){
			changed = TRUE;
			class_path.AppendFormatBlock("%c%s", PATH_SEPARATOR, p);
			ORP_Global_State::loader_env->classpath = (char*)class_path.toString();
		}
		p = strtok(NULL, sep);
	}
	return changed;
}

// This routine copies java/lang/System.properties("java.class.path")
// to ORP_Global_State::loader_env->classpath;
inline int UpdateJavaClassPathToNative()
{
	ExpandableMemBlock mb; 
	void __stdcall PropGet(char *pProp, ExpandableMemBlock &mb);
	PropGet("java.class.path", mb);	
	char *newpath = (char*)mb.toString();
	if(*newpath && strcmp(newpath, ORP_Global_State::loader_env->classpath)){
		MergeClasspath(newpath);
		return 1;
	}
	//no change
	return 0;
}

// This routine copies ORP_Global_State::loader_env->classpath to
// java/lang/System.properties("java.class.path");
inline void UpdateNativeClassPathToJava()
{
	if(classpath_changed){
		ExpandableMemBlock mb; 
		void __stdcall PropGet(char *pProp, ExpandableMemBlock &mb);
		PropGet("java.class.path", mb);	
		char *newpath = (char*)mb.toString();
		if(*newpath && strcmp(newpath, ORP_Global_State::loader_env->classpath)){
			classpath_changed = MergeClasspath(newpath);
		}
		newpath = (char*)extra_path.toString();
		classpath_changed |= MergeClasspath(newpath);
		if(classpath_changed){
			void __stdcall PropPut(char *pProp, char *pVal);
			PropPut("java.class.path", ORP_Global_State::loader_env->classpath);
		}
		classpath_changed = FALSE;
	}
}

void UpdateClasspathForJarFile(JarFile *jarfl)
{
	const char *classpath = getQualifiedClassPathFromJARFile(jarfl, extra_path);
	if(classpath){
		class_path.AppendBlock((char*)classpath);
		ORP_Global_State::loader_env->classpath = (char*)class_path.toString();
		//?? classpath_changed = TRUE;
	}
}

JarFile *getZIPJARFileFromPool(const char *filename)
{
	JarFile *jarfl = NULL;
	Prop_JarFile* propj = (Prop_JarFile*)JarFile_Properties.get(filename);
	if(propj)
		jarfl = propj->value;

	/*This jar file has not been opened*/
	if(NULL == jarfl)
	{
		jarfl = new JarFile(filename);
		if(!jarfl->Is_Open_Jar_OK())
		{
			delete jarfl;
			return NULL;
		}
		/*
		Because there is not this key in the HashTable,So we add a Prop_entry,whose value
		is not be the value cloned. And DO NOT delete this Prop_entry
		*/
		Prop_entry* propen = new Prop_entry();
		propen->key = strdup(filename);
		propen->value = new Prop_JarFile(jarfl);
		JarFile_Properties.add(propen);
		
		UpdateClasspathForJarFile(jarfl); 
	}

	return jarfl;
}

#endif


//	detects loading errors and exceptions: (p 43 of the JVM spec)
//		- ClassCircularityError
//		- ClassFormatError
//			* occurs when a class does not specify
//			  a super class, and it is not java.lang.Object
//		- NoClassDefFoundError 
//			* all other parsing errors
//
static Class *locked_load_class_by_system_loader(Global_Env *env,
                                                 const String *classname,
                                                 Loader_Exception *exception);


//
// To do: 
//
//	(1) Check that ConstantValue attribute's type matches field's type
//	(2) Set the value field of Field, when ConstantValue is parsed
//	(3) verify correctness of Method access_flags (see pg 104-105 of the JVM spec)
//	(4) verify correctness of Field access_flags (pg 102-103 of the JVM spec)
//	(5) verify Class access_flags (pg 86-87 of the JVM spec)
//	(6) verify that for interfaces, super is always java.lang.Object (pg 87)
//	(7) interfaces cannot have super classes (assumed in preparation)
//
#ifdef _DEBUG
//#define TRACE
#endif

#ifdef TRACE
static ostream& trace = cerr;
#endif



String *cp_check_utf8(Const_Pool *cp,
					 unsigned cp_size,
					 unsigned utf8_index)
{
	if (utf8_index >= cp_size || !cp_is_utf8(cp,utf8_index)) {
#ifdef _DEBUG
		cerr << "cp_check_utf8: illegal const pool utf8 index" 
			 << utf8_index << endl;
#endif
		return NULL;
	}	
	return cp[utf8_index].string;
} //cp_check_utf8



String *cp_check_class(Const_Pool *cp,
					  unsigned cp_size,
					  unsigned class_index)
{
	if (class_index >= cp_size || !cp_is_class(cp,class_index)) {
#ifdef _DEBUG
		cerr << "cp_check_class: illegal const pool class index" 
			 << class_index << endl;
#endif
		return NULL;
	}	
	return cp[class_index].string;
} //cp_check_class



String *field_attr_strings[N_FIELD_ATTR+1];
Attributes field_attrs[N_FIELD_ATTR];

String *method_attr_strings[N_METHOD_ATTR+1];
Attributes method_attrs[N_METHOD_ATTR];

String *class_attr_strings[N_CLASS_ATTR+1];
Attributes class_attrs[N_CLASS_ATTR];

String *code_attr_strings[N_CODE_ATTR+1];
Attributes code_attrs[N_CODE_ATTR];

String *must_recognize_attr_strings[4];
Attributes must_recognize_attrs[3];


Attributes parse_attribute(ByteReader &cfs,
						   Const_Pool *cp,
						   unsigned cp_size,
						   String *attr_strings[],
						   Attributes attrs[],
                           uint32 *attr_len)
{
    uint16 attr_name_index;
    //Loader_Result result = cfs.parse_u2_be(&attr_name_index);
    bool result = cfs.parse_u2_be(&attr_name_index);
    if (!result)
        return ATTR_ERROR;

    result = cfs.parse_u4_be(attr_len);
    if (!result)
        return ATTR_ERROR;

    String *attr_name = cp_check_utf8(cp,cp_size,attr_name_index);
    if (attr_name == NULL) {
#ifdef _DEBUG
        cerr << "Field::parse: illegal const pool attr_name_index" << endl;
#endif
        return ATTR_ERROR;
    }
    unsigned i;
    for (i=0; attr_strings[i] != NULL; i++) {
        if (attr_strings[i] == attr_name)
            return attrs[i];
    }
    //
    // unrecognized attribute; skip
    //
    while ( (*attr_len)-- > 0) {
        uint8 val;
        bool result = cfs.parse_u1(&val);
        if(!result)
            return ATTR_ERROR;
    }
    // Code, ConstantValue, Exceptions must be recognized even if illegal in
    // a particular context
    for (i = 0; must_recognize_attr_strings[i] != NULL; i++) {
        if (must_recognize_attr_strings[i] == attr_name) {
            return must_recognize_attrs[i];
        }
    }

    return ATTR_UNDEF;
} //parse_attribute



Loader_Result Class_Member::parse(ByteReader &cfs,
                                  Const_Pool *cp,
                                  unsigned cp_size,
                                  Signature_Table& sig_table,
                                  Class *clss)
{
    bool result = cfs.parse_u2_be(&_access_flags);
	if (!result)
		return LD_ParseError;

	_class = clss;
	uint16 name_index;
	result = cfs.parse_u2_be(&name_index);
	if (!result)
		return LD_ParseError;

	uint16 descriptor_index;
	result = cfs.parse_u2_be(&descriptor_index);
	if (!result)
		return LD_ParseError;

	//
	// look up the name_index and descriptor_index 
	// utf8 string const pool entries
	//
	String *name = cp_check_utf8(cp,cp_size,name_index);
	String *descriptor = cp_check_utf8(cp,cp_size,descriptor_index);
	if (name == NULL || descriptor == NULL) {
#ifdef _DEBUG
		cerr << "Class_Member::parse: illegal const pool index" << endl;
#endif
		return LD_ParseError;
	} 
	_signature = sig_table.lookup(name,descriptor);
#ifdef TRACE
	trace << "Class_Member::parse: " << name << "  " << descriptor << endl;
#endif
	return LD_OK;
} //Class_Member::parse



Loader_Result Field::parse(ByteReader &cfs,
                           Const_Pool *cp,
                           unsigned cp_size,
                           Signature_Table& sig_table,
                           Class *clss)
{
	Loader_Result result = Class_Member::parse(cfs, cp, cp_size, sig_table, clss);
	if (result != LD_OK)
		return result;

	uint16 attr_count;
	bool cfs_result = cfs.parse_u2_be(&attr_count);
	if (!cfs_result)
		return LD_ParseError;

	unsigned n_constval_attr = 0;

    uint32 attr_len = 0;

	for (unsigned j=0; j<attr_count; j++) {
		switch (parse_attribute(cfs, cp, cp_size, field_attr_strings, field_attrs, &attr_len)){
		case ATTR_ConstantValue:
		{	// constant value attribute
			// a field can have at most 1 ConstantValue attribute
			if (++n_constval_attr > 1) {
#ifdef _DEBUG
				cerr << "Field::parse: more than 1 ConstantValue attribute "; 
				//dump(cerr);
				cerr << endl;
#endif
				return LD_ParseError;
			}
            // attribute length must be two (orp spec reference 4.7.3)
            if (attr_len != 2) {
                return LD_ParseError;
            }

#if 0  // somewhere in orp spec it says a NON final constant value is OK
			// field must be final
			if (!is_final()) {
#ifdef _DEBUG
				cerr << "Field::parse: non-final field with ConstantValue attribute ";
				dump(cerr);
				cerr << endl;
#endif
				return LD_ParseError;
			}
#endif // 0

            cfs_result = cfs.parse_u2_be(&_const_value_index);
            if(!cfs_result) {
				return LD_ParseError;
            }

            if(_const_value_index == 0) {
                return LD_ParseError;
            }
			// type of constant must match field's type
            Const_Pool_Tags tag = (Const_Pool_Tags)cp_tag(cp, _const_value_index);

            Java_Type java_type = get_java_type();

            switch(tag) {
            case CONSTANT_Long:
                {
                if (java_type != JAVA_TYPE_LONG)
                    return LD_ParseError;
#ifdef POINTER64
                const_value.l.lo_bytes = cp[_const_value_index].cp64x.low_bytes;
                const_value.l.hi_bytes = cp[_const_value_index].cp64x.high_bytes;
#else
                const_value.l.lo_bytes = cp[_const_value_index].low_bytes;
                const_value.l.hi_bytes = cp[_const_value_index + 1].high_bytes;
#endif

                break;
                }
            case CONSTANT_Float:
                {
                if (java_type != JAVA_TYPE_FLOAT)
                    return LD_ParseError;
                
                const_value.f = cp[_const_value_index].float_value;

                break;
                }
            case CONSTANT_Double:
                {
                if (java_type != JAVA_TYPE_DOUBLE)
                    return LD_ParseError;
#ifdef POINTER64
                const_value.l.lo_bytes = cp[_const_value_index].cp64x.low_bytes;
                const_value.l.hi_bytes = cp[_const_value_index].cp64x.high_bytes;
#else
                const_value.l.lo_bytes = cp[_const_value_index].low_bytes;
                const_value.l.hi_bytes = cp[_const_value_index + 1].high_bytes;
#endif

                break;
                }
            case CONSTANT_Integer:
                {
                if ( !(java_type == JAVA_TYPE_INT         || 
                       java_type == JAVA_TYPE_SHORT       ||
                       java_type == JAVA_TYPE_BOOLEAN     || 
                       java_type == JAVA_TYPE_BYTE        ||
                       java_type == JAVA_TYPE_CHAR ) )

                    return LD_ParseError;
                
                const_value.i = cp[_const_value_index].int_value;
 
                break;
                }
            case CONSTANT_String:
                {
                if (java_type != JAVA_TYPE_CLASS)
                    return LD_ParseError;

                const_value.string = cp[_const_value_index].string;

                break;
                }
            default:
                {
#ifdef _DEBUG
                printf("Invalid tag: %d\n", tag);
#endif
                return LD_ParseError;

                break;
                }
            }
			break;
		}
		case ATTR_UNDEF:
			// unrecognized attribute; skipped
			break;
		default:
			// error occured
			return LD_ParseError;
		} // switch
	} // for
	return LD_OK;
} //Field::parse



Loader_Result Handler::parse(ByteReader &cfs,
                             unsigned code_length,
							 Const_Pool *cp,
                             unsigned cp_size)
{
	//Loader_Result result;
	bool result = cfs.parse_u2_be((uint16 *)&_start_pc);
	if(!result)
		return LD_ParseError;

	if (_start_pc >= code_length) {
#ifdef _DEBUG
		cerr << "Handler::parse: start_pc out of code length range" << endl;
#endif
		return LD_ParseError;
	}

	result = cfs.parse_u2_be((uint16 *)&_end_pc);
	if (!result)
		return LD_ParseError;

	if (_end_pc > code_length) {
#ifdef _DEBUG
		cerr << "Handler::parse: end_pc out of code length range" << endl;
#endif
		return LD_ParseError;
	}

    if (_start_pc >= _end_pc) {
#ifdef _DEBUG
		cerr << "Handler::parse: start_pc >= end_pc" << endl;
#endif
        return LD_ParseError;
    }

	result = cfs.parse_u2_be((uint16 *)&_handler_pc);
	if (!result)
		return LD_ParseError;

	if (_handler_pc >= code_length) {
#ifdef _DEBUG
		cerr << "Handler::parse: handler_pc out of code length range" << endl;
#endif
		return LD_ParseError;
	}

	uint16 catch_index;
	result = cfs.parse_u2_be(&catch_index);
	if (!result)
		return LD_ParseError;

    _catch_type_index = catch_index;

	if (catch_index == 0) {
		_catch_type = NULL;
	} else {
		_catch_type = cp_check_class(cp,cp_size,catch_index);
		if (_catch_type == NULL) {
#ifdef _DEBUG
			cerr << "Handler::parse:"
				 << " catch_type not a valid CONSTANT_ClassInfo index" << endl;
#endif
			return LD_ParseError;
		}
	}
	return LD_OK;
} //Handler::parse



Loader_Result Method::_parse_exceptions(ByteReader &cfs,
										Const_Pool *cp,
										unsigned cp_size,
                                        unsigned attr_len)
{
	//Loader_Result result;
	bool result = cfs.parse_u2_be(&_n_exceptions);
	if(!result)
		return LD_ParseError;

	_exceptions = new String*[_n_exceptions];
	for (unsigned i=0; i<_n_exceptions; i++) {
		uint16 index;
		result = cfs.parse_u2_be(&index);
		if(!result)
			return LD_ParseError;

		_exceptions[i] = cp_check_class(cp,cp_size,index);
		if (_exceptions[i] == NULL) {
#ifdef _DEBUG
			cerr << "Handler::_parse_exception:"
				 << " exception not a valid CONSTANT_ClassInfo index" << endl;
#endif
			return LD_ParseError;
		}
	}
    // attr_len must = 2 + (number_of_exceptions * 2),, (orp spec ref 4.7.5)
    if (attr_len != _n_exceptions * sizeof(uint16) + sizeof(_n_exceptions) ) {
        return LD_ParseError;
    }
	return LD_OK;
} //Method::_parse_exceptions


Loader_Result Method::_parse_code(ByteReader &cfs,
								  Const_Pool *cp,
								  unsigned cp_size,
								  Signature_Table& sig_table)
{
	//Loader_Result result;
	bool result = cfs.parse_u2_be(&_max_stack);
	if(!result)
		return LD_ParseError;

	result = cfs.parse_u2_be(&_max_locals);
	if(!result)
		return LD_ParseError;

    if(_max_locals < (get_num_arg_bytes() / 4) ) {  // Michal, please fix this for IA64
        return LD_ParseError;
    }

	result = cfs.parse_u4_be(&_byte_code_length);
	if(!result)
		return LD_ParseError;

    // code length must not = 0
    if(_byte_code_length == 0) {
        return LD_ParseError;
    }
	//
	// allocate & parse code array
	//
	_byte_codes = new Byte[_byte_code_length];

    unsigned i;
	for (i=0; i<_byte_code_length; i++) {
		result = cfs.parse_u1((uint8 *)&_byte_codes[i]);
		if(!result)
			return LD_ParseError;
	}

	result = cfs.parse_u2_be(&_n_handlers);
	if(!result)
		return LD_ParseError;

	// 
	// allocate & parse exception handler table
	//
	_handlers = new Handler[_n_handlers];

	for (i=0; i<_n_handlers; i++) {
		Loader_Result result = _handlers[i].parse(cfs, _byte_code_length, cp, cp_size);
		if (result != LD_OK)
			return result;
	}
	//
	// attributes of the Code attribute
	//
	uint16 n_attrs;
	result = cfs.parse_u2_be(&n_attrs);
	if(!result)
		return LD_ParseError;

    uint32 attr_len = 0;

	for (i=0; i<n_attrs; i++) {
		switch (parse_attribute(cfs, cp, cp_size, code_attr_strings, code_attrs, &attr_len)){
		case ATTR_LineNumberTable:
			{
				//
				// LineNumberTable attribute
				//
				uint16 n_line_numbers;
				result = cfs.parse_u2_be(&n_line_numbers);
				if(!result)
					return LD_ParseError;

                _line_number_table =
                    (Line_Number_Table *)malloc(sizeof(Line_Number_Table) +
                                                sizeof(Line_Number_Table_Entry) * (n_line_numbers - 1));
                _line_number_table->length = n_line_numbers;

				for (unsigned j=0; j<n_line_numbers; j++) {
					uint16 start_pc;
					result = cfs.parse_u2_be(&start_pc);
					if(!result)
						return LD_ParseError;

                    if(start_pc >= _byte_code_length) {
#ifdef _DEBUG
                        cerr << "Method::_parse_code: start_pc >= _byte_code_length" << endl;
#endif
                        return LD_ParseError;
                    }
					uint16 line_number;
					result = cfs.parse_u2_be(&line_number);
					if(!result)
						return LD_ParseError;
					_line_number_entries = 
						new Line_Number_Entry(start_pc,line_number,_line_number_entries);
                    _line_number_table->table[j].start_pc = start_pc;
                    _line_number_table->table[j].line_number = line_number;
				}
				break;
			}
		case ATTR_LocalVariableTable:
			{
				//
				// LocalVariableTable attribute
				//
				if (_max_locals > 0 && _local_vars_table == NULL) {
					//
					// create a local vars table for this method
					//
					_local_vars_table = new Local_Var_Entry*[_max_locals];
					for (unsigned i=0; i<_max_locals; i++) {
						_local_vars_table[i] = NULL;
					}
				}
				uint16 n_local_vars;
				result = cfs.parse_u2_be(&n_local_vars);
				if(!result)
					return LD_ParseError;

				for (unsigned j=0; j<n_local_vars; j++) {
					uint16 start_pc;
					result = cfs.parse_u2_be(&start_pc);
					if(!result)
						return LD_ParseError;

					uint16 length;
					result = cfs.parse_u2_be(&length);
					if(!result)
						return LD_ParseError;

                    if( (start_pc >= _byte_code_length)  ||
                        (start_pc + (unsigned)length) > _byte_code_length ) {
#ifdef _DEBUG
                        cerr << "Method::_parse_code: (start_pc >= _byte_code_length) || (start_pc + (unsigned)length) > _byte_code_length";
#endif  
                        return LD_ParseError;
                    }
					uint16 name_index;
					result = cfs.parse_u2_be(&name_index);
					if(!result)
						return LD_ParseError;

					uint16 descriptor_index;
					result = cfs.parse_u2_be(&descriptor_index);
					if(!result)
						return LD_ParseError;


					String *name = cp_check_utf8(cp,cp_size,name_index);
					String *descriptor = 
						cp_check_utf8(cp,cp_size,descriptor_index);
					if (name == NULL ||	descriptor == NULL) {

#ifdef _DEBUG
						cerr << "Method::parse:LocalVariableTable: not utf8 index"
							 << endl;
#endif
						return LD_ParseError;
					}

					uint16 index;
					result = cfs.parse_u2_be(&index);
					if(!result)
						return LD_ParseError;


					if (index >= _max_locals) {

#ifdef _DEBUG
						cerr << "Method::parse:LocalVariableTable: "
							 << "invalid local index " << index
							 << endl;
#endif
						return LD_ParseError;
					}

					Signature *sig = sig_table.lookup(name,descriptor);
					_local_vars_table[index] = 
						new Local_Var_Entry(sig,start_pc,length,
											_local_vars_table[index]);

                }
				break;
			}
		case ATTR_UNDEF:
			// unrecognized attribute; skipped
			break;
		default:
			// error occured
			return LD_ParseError;
		} // switch
	} // for

	return LD_OK;
} //Method::_parse_code



Loader_Result Method::parse(ByteReader &cfs,
                            Const_Pool *cp,
                            unsigned cp_size,
                            Global_Env& env,
                            Class *clss)
{
	Loader_Result result = Class_Member::parse(cfs, cp, cp_size, env.sig_table, clss);
	if (result != LD_OK)
		return result;
	uint16 attr_count;
	bool cfs_result = cfs.parse_u2_be(&attr_count);
	if (!cfs_result)
		return LD_ParseError;
	//
	// set the is_clinit and is_init flags
	//
    if (_signature == env.Finalize_Signature) {
		_flags.is_finalize = 1;
        if(strcmp("java/lang/Object", clss->name->bytes)) {
            clss->has_finalizer = 1;
        }
    }
	else if (_signature->name == env.Init_String)
		_flags.is_init = 1;
	else if (_signature->name == env.CLInit_String)
		_flags.is_clinit = 1;

	unsigned n_code_attr = 0;
	unsigned n_exceptions_attr = 0;
    uint32 attr_len = 0;

	for (unsigned j=0; j<attr_count; j++) {
		//
		// only code and exception attributes are defined for Method
		//
		switch (parse_attribute(cfs, cp, cp_size, method_attr_strings, method_attrs, &attr_len)){
		case ATTR_Code:
			n_code_attr++;
			result = _parse_code(cfs, cp, cp_size, env.sig_table);
			if (result != LD_OK)
				return result;
			break;
		case ATTR_Exceptions:
			n_exceptions_attr++;
			result = _parse_exceptions(cfs, cp, cp_size, attr_len);
			if (result != LD_OK)
				return result;
			break;
		case ATTR_UNDEF:
			// unrecognized attribute; skipped
			break;
		default:
			// error occured
			return LD_ParseError;
		} // switch
	} // for
	//
	// there must be no more than 1 code attribute and no more than 1 exceptions
	// attribute per method
	//
	if (n_code_attr > 1 || n_exceptions_attr > 1) {
#ifdef _DEBUG
		cerr << "Method::parse: too many code/exceptions attributes" << endl;
#endif
		return LD_ParseError;
	}

    if ( (is_abstract() || is_native()) && n_code_attr > 0 ) {
#ifdef _DEBUG
        cerr << "Method::_parse: code attribute not allowed";
#endif 
       return LD_ParseError;
    }

    if ( !(is_abstract() || is_native()) && n_code_attr == 0 ) {
#ifdef _DEBUG
        cerr << "Method::_parse: code attribute missing";
#endif 
        return LD_ParseError;
    }
	return LD_OK;
} //Method::parse



// The following is for new classpath, we add specific fields to some classes for orp usage.
struct FieldDescr{
    char *clssname;
    char *fieldname;
    char *descriptor;
    uint16 accessflag;
};

FieldDescr orp_extra_fields[] = {
    { "java/lang/Thread",    "data",      "J",                  ACC_PRIVATE},
    { "java/lang/Throwable", "traceinfo", "Ljava/lang/String;", ACC_PRIVATE|ACC_TRANSIENT}
};

void field_handler(Field *f, uint16 accflag, Class *clss, Signature *sig)
{ 
    f->set(accflag, sig, clss);
}

static Loader_Result class_parse_fields(Class *clss,
                                        ByteReader &cfs,
                                        Signature_Table& sig_table)
{
	bool result = cfs.parse_u2_be(&clss->n_fields);
	if(!result)
		return LD_ParseError;

    int num_fields_in_class_file = clss->n_fields;
    int i;
#ifndef OLD_VERSION_CLASSPATH
    for(i = 0; i < sizeof(orp_extra_fields)/sizeof(FieldDescr); i++)
        if(!strcmp(clss->name->bytes, orp_extra_fields[i].clssname))
            clss->n_fields++;
#endif

	clss->fields = new Field[clss->n_fields];
    for (i=0; i < num_fields_in_class_file; i++) {
	//for (unsigned i=0; i<clss->n_fields; i++) {
		Loader_Result result =
            clss->fields[i].parse(cfs, clss->const_pool, clss->cp_size, sig_table, clss);
		if (result != LD_OK)
			return result;
	}

#ifndef OLD_VERSION_CLASSPATH
    for(i = 0; i < sizeof(orp_extra_fields)/sizeof(FieldDescr); i++)
        if(!strcmp(clss->name->bytes, orp_extra_fields[i].clssname)){
            String *name = ORP_Global_State::loader_env->string_pool.lookup(orp_extra_fields[i].fieldname);
            String *descriptor = ORP_Global_State::loader_env->string_pool.lookup(orp_extra_fields[i].descriptor);
            field_handler(&(clss->fields[num_fields_in_class_file++]),
                orp_extra_fields[i].accessflag,
                clss,
                sig_table.lookup(name, descriptor)
                );
        }
#endif

	return LD_OK; // success
} //class_parse_fields



static Loader_Result class_parse_methods(Class *clss,
                                         ByteReader &cfs,
                                         Global_Env& env)
{
	bool result = cfs.parse_u2_be(&clss->n_methods);
	if(!result)
		return LD_ParseError;

	clss->methods = new Method[clss->n_methods];
	for (unsigned i=0; i<clss->n_methods; i++) {
		Loader_Result result =
            clss->methods[i].parse(cfs, clss->const_pool, clss->cp_size, env, clss);
		if (result != LD_OK)
			return result;
        if(clss->methods[i].is_clinit()) {
            // There can be at most one clinit per class.
            assert(!clss->static_initializer);
			clss->static_initializer = &(clss->methods[i]);
        }
	}
	return LD_OK; // success
} //class_parse_methods



static Loader_Result class_parse_const_pool(Class *clss,
                                            ByteReader &cfs,
                                            String_Pool& string_pool)
{
	bool result = cfs.parse_u2_be(&clss->cp_size);
	if(!result)
		return LD_ParseError;

	unsigned char *cp_tags = new unsigned char[clss->cp_size];
	clss->const_pool = new Const_Pool[clss->cp_size];
	//
	// 0'th constant pool entry is a pointer to the tags array
	//
	clss->const_pool[0].tags = cp_tags;
	cp_tags[0] = CONSTANT_Tags;
#ifdef TRACE
	trace << "parse_const_pool: parsing constant pool; size=" 
		  << clss->cp_size << endl;
#endif
	for (unsigned i=1; i<clss->cp_size; i++) {
		// parse tag into tag array
		bool result;
		uint8 tag;
		result = cfs.parse_u1(&tag);
		if(!result)
			return LD_ParseError;

		switch (cp_tags[i] = tag) {
			case CONSTANT_Class:
				result = cfs.parse_u2_be(&clss->const_pool[i].name_index);
				if(!result)
					return LD_ParseError;

				break;
			case CONSTANT_Fieldref:
			case CONSTANT_Methodref:
			case CONSTANT_InterfaceMethodref:
				result = cfs.parse_u2_be(&clss->const_pool[i].cp_tag.class_index);
				if(!result)
					return LD_ParseError;

				result = cfs.parse_u2_be(&clss->const_pool[i].cp_tag.name_and_type_index);
				if(!result)
					return LD_ParseError;

				break;
			case CONSTANT_String:
				result = cfs.parse_u2_be(&clss->const_pool[i].string_index);
				if(!result)
					return LD_ParseError;

				break;
			case CONSTANT_Integer:
			case CONSTANT_Float:
				result = cfs.parse_u4_be(&clss->const_pool[i].int_value );
				if(!result)
					return LD_ParseError;

				break;
			case CONSTANT_Long:
			case CONSTANT_Double:
				// longs and doubles take up two entries
				// on IA32, both allocated entries are used
				// on IA64, low element is used, high element unused
#ifdef  POINTER64
				result = cfs.parse_u4_be(&clss->const_pool[i+0].cp64x.high_bytes);
#else
				result = cfs.parse_u4_be(&clss->const_pool[i+1].high_bytes);
#endif
				if(!result)
					return LD_ParseError;

				cp_tags[i+1] = cp_tags[i];
				i++;
#ifdef POINTER64
				result = cfs.parse_u4_be(&clss->const_pool[i-1].cp64x.low_bytes);
#else
				result = cfs.parse_u4_be(&clss->const_pool[i-1].low_bytes);
#endif
				if(!result)
					return LD_ParseError;

				break;
			case CONSTANT_NameAndType:
				result = cfs.parse_u2_be(&clss->const_pool[i].ni_di.name_index);
				if(!result)
					return LD_ParseError;

				result = cfs.parse_u2_be(&clss->const_pool[i].ni_di.descriptor_index);
				if(!result)
					return LD_ParseError;

				break;
			case CONSTANT_Utf8:
				{
					uint16 len;
					result = cfs.parse_u2_be(&len);
					if(!result)
						return LD_ParseError;

					char *str = new char[len+1];
					unsigned j;
					for (j=0; j<len; j++) {
						uint8 c;
						result = cfs.parse_u1(&c);
						if(!result)
							return LD_ParseError;

                        if(c == 0)
                            return LD_ParseError;

                        str[j] = c;
					}
					str[j] = '\0';
#ifdef TRACE
					trace << "parse_const_pool: Utf8 [" << i << "] : "
						  << str << endl;
#endif
					// insert string into string table
					clss->const_pool[i].string = string_pool.lookup(str);
					delete[] str;
				}
				break;
			default:
#ifdef _DEBUG
				cerr << "parse_const_pool: unknown constant pool tag " 
					 << cp_tags[i] << endl;
#endif
				return LD_ParseError;	// signal error
		}
	}
	return LD_OK;
} //class_parse_const_pool



//
// check consistency of constant pool
//
// make sure all indices to other constant pool entries are in range
// make sure contents of constant pool entries are of the right type
//
// Set CONSTANT_Class entries to point directly to String representing
// internal form of fully qualified name of Class.
//
// Set CONSTANT_String entries to point directly to String representation
// of String.
//
// Peresolve CONSTANT_NameAndType entries to signature
//
Loader_Result check_const_pool(Const_Pool *cp,
								unsigned cp_size,
								Signature_Table& sig_table)
{
	unsigned char *cp_tags = cp[0].tags;
	for (unsigned i=1; i<cp_size; i++) {
		switch (cp_tags[i]) {
			case CONSTANT_Class:
			{
				unsigned name_index = cp[i].name_index;
				if (name_index >= cp_size ||
					cp_tag(cp,name_index) != CONSTANT_Utf8)
					// illegal name index
					return LD_ConstPoolError;
				cp[i].string = cp[name_index].string;
				break;
			}
			case CONSTANT_Fieldref:
			case CONSTANT_Methodref:
			case CONSTANT_InterfaceMethodref:
			{
				unsigned class_index = cp[i].cp_tag.class_index;
				unsigned name_type_index = cp[i].cp_tag.name_and_type_index;
				if (class_index >= cp_size ||
					cp_tag(cp,class_index) != CONSTANT_Class)
					// illegal class index
					return LD_ConstPoolError;
				if (name_type_index >= cp_size ||
					cp_tag(cp,name_type_index) != CONSTANT_NameAndType)
					// illegal name and type index
					return LD_ConstPoolError;
				break;
			}
			case CONSTANT_String:
			{
				unsigned string_index = cp[i].string_index;
				if (string_index >= cp_size ||
					cp_tag(cp,string_index) != CONSTANT_Utf8)
					// illegal string index
					return LD_ConstPoolError;
				// set entry to the actual string
				cp[i].string = cp[string_index].string;
				break;
			}
			case CONSTANT_Integer:
			case CONSTANT_Float:
			case CONSTANT_Long:
			case CONSTANT_Double:
				// not much to do here
				break;
			case CONSTANT_NameAndType:
			{
				unsigned name_index = cp[i].ni_di.name_index;
				unsigned descriptor_index = cp[i].ni_di.descriptor_index;
				if (name_index >= cp_size ||
					cp_tag(cp,name_index) != CONSTANT_Utf8)
					// illegal name index
					return LD_ConstPoolError;
				if (descriptor_index >= cp_size ||
					cp_tag(cp,descriptor_index) != CONSTANT_Utf8)
					// illegal descriptor index
					return LD_ConstPoolError;
				String *name = cp[name_index].string;
				String *descriptor = cp[descriptor_index].string;
				// look up in signature table
				cp[i].signature = sig_table.lookup(name,descriptor);
				cp_set_resolved(cp,i);
				break;
			}
			case CONSTANT_Utf8:
				// nothing to do here
				break;
		}
	}
	return LD_OK;
} //check_const_pool



static Loader_Result class_parse_interfaces(Class *clss, ByteReader &cfs)
{
	bool result = cfs.parse_u2_be(&clss->n_superinterfaces);
	if(!result)
		return LD_ParseError;

	//clss->superinterfaces = new _Intfc[clss->n_superinterfaces];
	clss->superinterfaces = (Class_Superinterface *)gc_malloc_fixed_data_C(
                sizeof(Class_Superinterface)*clss->n_superinterfaces);
	for (unsigned i=0; i<clss->n_superinterfaces; i++) {
		uint16 interface_index;
		bool result = cfs.parse_u2_be(&interface_index);
		if(!result)
			return LD_ParseError;
		//
		// verify that entry in constant pool is of type CONSTANT_Class
		//
		clss->superinterfaces[i].name = cp_check_class(clss->const_pool,clss->cp_size,interface_index);
		if (clss->superinterfaces[i].name == NULL) {
#ifdef _DEBUG
			cerr << "parse_interfaces: cp index " << i 
				 << " is not CONSTANT_Class_Info" << endl;
#endif
			return LD_ParseError;
		}
	}
	return LD_OK;
} //class_parse_interfaces



Loader_Result class_parse(Class *clss,
                          ByteReader &cfs,
                          Global_Env& env,
                          unsigned *super_class_cp_index,
                          Loader_Exception *exception)
{
	//Loader_Result result;

    *exception = LD_ClassFormatError;

	uint32 magic;
	bool result = cfs.parse_u4_be(&magic);
	if(!result)
		return LD_ParseError;

	if(magic != CLASSFILE_MAGIC)
		return LD_ParseError;

	uint16 minor_version;
	result = cfs.parse_u2_be(&minor_version);
	if(!result)
		return LD_ParseError;

   	//JDK 1.02: major:45 minor:0-3
	//JDK 1.1.X major:45 minor:0-65535
	//JDK 1.2>  major:45-46
	/*
	if(minor_version != CLASSFILE_MINOR)
		return LD_ParseError;
	*/

	uint16 major_version;
	result = cfs.parse_u2_be(&major_version);
	if(!result)
		return LD_ParseError;

    /*
	if(major_version != CLASSFILE_MAJOR)
		return LD_ParseError;
    */
    
    if( major_version != CLASSFILE_MAJOR &&
		major_version != CLASSFILE_MAJOR+1 )
		return LD_ParseError;
		
	//
	//  allocate and parse constant pool
	//
	Loader_Result parse_result = class_parse_const_pool(clss, cfs, env.string_pool);
	if(parse_result != LD_OK)
		return parse_result;

	//
	// check and preprocess the constant pool
	//
	parse_result = check_const_pool(clss->const_pool,clss->cp_size,env.sig_table);
	if (parse_result != LD_OK)
		return parse_result;

	result = cfs.parse_u2_be(&clss->access_flags);
	if(!result)
		return LD_ParseError;

	//
	// parse this_class & super_class & verify their constant pool entries
	//
	uint16 this_class;
	result = cfs.parse_u2_be(&this_class);
	if(!result)
		return LD_ParseError;

	String *name = cp_check_class(clss->const_pool, clss->cp_size, this_class);
	if (name == NULL) {
#ifdef _DEBUG
		cerr << "Class::parse: this_class constant pool entry " << this_class
			 << " is an illegal CONSTANT_Class entry" << endl;
#endif
		return LD_ParseError;
	}
	
	// When defineClass from byte stream, there are cases that clss->name is null,
	// so we should add a check here
    if (clss->name && name != clss->name) { 
    //if (name != clss->name) {
        *exception = LD_NoClassDefFoundError;
        return LD_ParseError;
    }

    if(!clss->name) {
        clss->name = name;
    }

    // Mark the current class as resolved.
    cp_resolve_to_class(clss->const_pool, this_class, clss);

    //
    // parse the super class name
    //
    uint16 super_class;
    result = cfs.parse_u2_be(&super_class);
    if(!result)
        return LD_ParseError;

    *super_class_cp_index = super_class;

    if (super_class == 0) {
        //
        // this class must represent java.lang.Object
        //
        clss->super_name = NULL;
    } else {
        clss->super_name = cp_check_class(clss->const_pool, clss->cp_size, super_class);
        if(clss->super_name == NULL) {
#ifdef _DEBUG
			cerr << "Class::parse: super_class entry " << super_class
				 << " is not CONSTANT_Class" << endl;
#endif
			// ClassFormatError exception in this case
            // already set in the beginning by default
			return LD_ParseError;
		}
	}
	//
	// allocate and parse class' interfaces
	//
	parse_result = class_parse_interfaces(clss, cfs);
	if (parse_result != LD_OK) {
		return parse_result;
	}
	// 
	// allocate and parse class' fields
	//
	parse_result = class_parse_fields(clss, cfs, env.sig_table);
	if (parse_result != LD_OK)
		return parse_result;

	// 
	// allocate and parse class' methods
	//
	parse_result = class_parse_methods(clss, cfs, env);
	if (parse_result != LD_OK) {
		return parse_result;
	}
	//
	// only the FileName attribute is defined for Class
	//
	uint16 n_attrs;
	result = cfs.parse_u2_be(&n_attrs);;
	if(!result)
		return LD_ParseError;

    unsigned n_source_file_attr = 0;
    uint32 attr_len = 0;

	for (unsigned i=0; i<n_attrs; i++) {
		switch(parse_attribute(cfs, clss->const_pool, clss->cp_size, class_attr_strings, class_attrs, &attr_len)){
		case ATTR_SourceFile:
		{
            // a class file can have at most one source file attribute
            if (++n_source_file_attr > 1) 
                return LD_ParseError;

            // attribute length must be two (orp spec 4.7.2)
            if (attr_len != 2)
                return LD_ParseError;

			// constant value attribute
			uint16 filename_index;
			result = cfs.parse_u2_be(&filename_index);
			if(!result)
				return LD_ParseError;

			clss->src_file_name = cp_check_utf8(clss->const_pool,clss->cp_size,filename_index);
			if (clss->src_file_name == NULL)
				return LD_ParseError;
			break;
		}
		case ATTR_UNDEF:
			// unrecognized attribute; skipped
			break;
			
	    case ATTR_InnerClasses:
	    {
			BOOL isinner = FALSE;
			// found_myself == 2: myself is not inner class or has passed myself when iterating inner class attribute arrays
			// found_myself == 1: myself is inner class, current index of inner class attribute arrays is just myself
			// found_myself == 0: myself is inner class, hasn't met myself in inner class attribute arrays
			int found_myself = 2;
			if(strchr(clss->name->bytes, '$')){
				isinner = TRUE;
				found_myself = 0;
			}
			//Only handle inner class
            uint16 num_of_classes;
            result = cfs.parse_u2_be(&num_of_classes);
			if(!result)
				return LD_ParseError;

			assert(num_of_classes >= 0);
			clss->declaringclass_index = 0; //????: would it be a valid index
			clss->innerclass_indexes = NULL;
			if(isinner)
				clss->n_innerclasses = num_of_classes - 1; //exclude itself
			else
				clss->n_innerclasses = num_of_classes;
			if(num_of_classes)
				clss->innerclass_indexes = (uint16*)gc_malloc_fixed_data_C(
					2*sizeof(uint16)*clss->n_innerclasses);
			int index = 0;
			for(int i = 0; i < num_of_classes; i++){
                uint16 inner_clss_info_idx;
                result = cfs.parse_u2_be(&inner_clss_info_idx);
			    if(!result)
				    return LD_ParseError;

				if(!found_myself){
					String *clssname;
					if(cp_is_resolved(clss->const_pool, inner_clss_info_idx))
						clssname = (String*)clss->const_pool[inner_clss_info_idx].clss->name;
					else
						clssname = cp_check_class(clss->const_pool, clss->cp_size, inner_clss_info_idx);
					// Only handle this class
					if(clss->name == clssname)
						found_myself = 1;
				}
				if(found_myself != 1)
					clss->innerclass_indexes[index++] = inner_clss_info_idx;

                uint16 outer_clss_info_idx;
                result = cfs.parse_u2_be(&outer_clss_info_idx);
			    if(!result)
				    return LD_ParseError;
				if(found_myself == 1 && outer_clss_info_idx){
					clss->declaringclass_index = outer_clss_info_idx;
				}

				uint16 inner_name_idx;
                result = cfs.parse_u2_be(&inner_name_idx);
			    if(!result)
				    return LD_ParseError;
#if 0				
                if(inner_name_idx){
                    String *name = cp_check_utf8(clss->const_pool, clss->cp_size, inner_name_idx);
                    if (name == NULL) {
#ifdef _DEBUG
                        cerr << "class_parse::parse: illegal const pool inner_name_index" << endl;
#endif
                        return LD_ParseError;
                    }
                }
#endif				
                uint16 inner_clss_access_flag;
                result = cfs.parse_u2_be(&inner_clss_access_flag);
			    if(!result)
				    return LD_ParseError;
				if(found_myself == 1)
					found_myself = 2;
				else
					clss->innerclass_indexes[index++] = inner_clss_access_flag;
            } // for num_of_classes
		}break; //case ATTR_InnerClasses

		default:
			// error occured
			return LD_ParseError;
		} // switch
	} // for

    // can't be both and final and interface, or both final and abstract
    if (class_is_final(clss) && 
       (class_is_interface(clss)  || class_is_abstract(clss) ) ) {
        return LD_ParseError;
    }

    *exception = LD_NoException;

	return LD_OK;
} //class_parse



void add_class_to_package(Class *clss, Global_Env& env)
{
	const String *classname = clss->name;
	//
	// add this class to its package
	//
	unsigned name_len = classname->len;
	char *buf = new char[name_len+1];
	strcpy(buf,classname->bytes);
	char *package_name = buf;
	unsigned j = 0;
	char c;
	Package *package = env.default_package;
	for (c = buf[0]; c != '\0'; c = buf[++j]) {
		if (c == '/') {
			//
			// delimiter for package name
			//
			buf[j] = '\0';
			String *package_string = env.string_pool.lookup(package_name);
#ifdef _DEBUG
//			cerr << "add_class_to_package: package_name is "
//				 << package_string->bytes << endl;
#endif
			Package *p = env.package_table.lookup(package_string);
			if (p == NULL) {
				//
				// create a new package
				//
				p = new Package(package_string,package);
				env.package_table.insert(p);
			}
			package = p;
			//
			// put back delimiter
			//
			buf[j] = '/';
		}
	}
	package->add_class(clss);
} //add_class_to_package



Class *create_array_class(const String *classname,
						  Class_Loader *class_loader,
                          Global_Env& env,
                          Loader_Exception *exception)
{
	//
	// Note (a BIG Note): env.JavaLangObject_Class better be loaded before this is called
	//
	Class *clss = NULL;
	const char *classname_string = classname->bytes;
	unsigned n_dimensions = 1;
	while (classname_string[n_dimensions] == '[') {
		n_dimensions++;
	}
	Class *array_base_class = NULL;
	unsigned is_array_of_primitives = 0;
	const char *basetype_string = &classname_string[n_dimensions];
	switch (*basetype_string) {
	case 'Z':	// boolean
		array_base_class = env.Boolean_Class;
		is_array_of_primitives = 1;
		break;
	case 'C':	// char
		array_base_class = env.Char_Class;
		is_array_of_primitives = 1;
		break;
	case 'F':	// float
		array_base_class = env.Float_Class;
		is_array_of_primitives = 1;
		break;
	case 'D':	// double
		array_base_class = env.Double_Class;
		is_array_of_primitives = 1;
		break;
	case 'B':	// byte
		array_base_class = env.Byte_Class;
		is_array_of_primitives = 1;
		break;
	case 'S':	// short
		array_base_class = env.Short_Class;
		is_array_of_primitives = 1;
		break;
	case 'I':	// int
		array_base_class = env.Int_Class;
		is_array_of_primitives = 1;
		break;
	case 'J':	// long
		array_base_class = env.Long_Class;
		is_array_of_primitives = 1;
		break;
#ifdef CLI_TESTING
	case 'b':	// ubyte
		array_base_class = env.UByte_Class;
		is_array_of_primitives = 1;
		break;
	case 's':	// ushort
		array_base_class = env.UShort_Class;
		is_array_of_primitives = 1;
		break;
	case 'i':	// uint
		array_base_class = env.UInt_Class;
		is_array_of_primitives = 1;
		break;
	case 'j':	// ulong
		array_base_class = env.ULong_Class;
		is_array_of_primitives = 1;
		break;
	case 'N':	// native int
		array_base_class = env.IntPtr_Class;
		is_array_of_primitives = 1;
		break;
	case 'n':	// native uint
		array_base_class = env.UIntPtr_Class;
		is_array_of_primitives = 1;
		break;
#endif
	case 'L':	// object
		{
			unsigned name_len;
			basetype_string++;	// skip over L
			for (name_len = 0; basetype_string[name_len] != ';'; name_len++) {
			}
			char *name = new char[name_len + 1];
			strncpy(name, basetype_string, name_len);
			name[name_len] = '\0';
			/*
			array_base_class = load_class_by_system_loader(&env,
                                                           env.string_pool.lookup(name),
										                   exception);
			*/
			array_base_class = load_class_by_loader(&env,
                                                    env.string_pool.lookup(name),
													class_loader,
										            exception);
			//assert(array_base_class);
			if(array_base_class)
				class_prepare(array_base_class);
			delete name;
			if (array_base_class == NULL) {
				// exception occured during loading of base class
				return NULL;
			}
			is_array_of_primitives = 0;
		}
		break;

    default:  // malformed input
        return NULL;
	}

    Class *array_element_class = 0;
    if(n_dimensions == 1) {
    	array_element_class = array_base_class;
    } else {
    	const char *elemtype_string = &classname_string[1];
        char *name;
        if(elemtype_string[0] == '[') {
            name = (char *)elemtype_string;
        } else {
            assert(elemtype_string[0] == 'L');
			unsigned name_len;
			elemtype_string++;	// skip over L
			for (name_len = 0; elemtype_string[name_len] != ';'; name_len++) {
			}
			name = new char[name_len+1];
			strncpy(name, elemtype_string, name_len);
			name[name_len] = '\0';
        }
    	array_element_class =
    	    /*
            load_class_by_system_loader(&env, env.string_pool.lookup(name), exception);
            */
            load_class_by_loader(&env, 
								 env.string_pool.lookup(name), 
								 class_loader,
								 exception);
								 
        if(elemtype_string[0] != '[') {
			delete name;
        }
    }

	clss = new_class(classname);
	clss->is_array = 1;
	clss->n_dimensions = n_dimensions;
	clss->array_base_class = array_base_class;
	clss->is_array_of_primitives = is_array_of_primitives;
    clss->array_element_class = array_element_class;

	//
	// set all the other field of struct Class
	//
	clss->n_fields = 1;		// give it a length field
	clss->fields = new Field[1];
	clss->fields[0].set(ACC_PUBLIC|ACC_FINAL,env.ArrayLength_Signature,clss);
#ifdef CLI_OCL
#ifndef CLI_TESTING
	assert(0);		// ERRRRRRRRR
#endif
	extern bool currently_creating_java_arrays;
	extern Global_CLI_Env *global_cli_env;

	// Set up the correct superclass information.
	if (currently_creating_java_arrays) { 
		// orp_init() in progress.
		assert(global_cli_env == NULL);
		clss->super_name = env.JavaLangObject_String;
	} else { 
		if (global_cli_env != NULL) { 
			// User-defined array being initialized/loaded.
			// I want to be flagged.
//			assert(0);
		}
		clss->super_name = env.SystemArray_String;
	}
#else	
	clss->super_name = env.JavaLangObject_String;
#endif // CLI_OCL

    // ACC_ABSTRACT was added to fix the following problem.
    // Class ba_class = some_byte_array.getClass(); 
    // byte [] bogus = ba_class.newInstance();  <<< THIS MUST FAIL
    // One way to make the above line fail is to add ACC_ABSTRACT the 
    // primitive array's Class clss->access_flags 

    clss->access_flags = (ACC_FINAL | ACC_ABSTRACT);
    if (is_array_of_primitives) {
        clss->access_flags |= ACC_PUBLIC;
    }
    else {
        clss->access_flags |= (array_base_class->access_flags & ACC_PUBLIC);
		clss->class_loader = array_base_class->class_loader; 
    }
    clss->package = clss->array_base_class->package;

    return clss;
} //create_array_class


#ifdef CLI_TESTING

Class *resolve_class_from_cli_type(Global_Env *env,

                                   CLI_Type *cli_type,

                                   Loader_Exception *exc);

#endif







Loader_Exception finish_class_loading(Class *clss,
                                      Global_Env *env,
                                      unsigned *super_class_cp_index)
{
    Loader_Exception exception;
	clss->state = ST_LoadingAncestors;

    if(clss->class_loader) {
        clss->class_loader->insert_class(clss);
        //?? env->class_table.insert(clss);
    } else {
//#ifdef CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH
		extern bool bootstrapped;
		if (!bootstrapped  && (strcmp(clss->name->bytes, "java/lang/Class") == 0)) { 
			// DO NOT insert and save the Class * for java/lang/Class since
			// bootstrap is in progress and we do not want to cache this Class *
		} else { 
			// do as before in all other cases.
			env->class_table.insert(clss);
		}
#else
		env->class_table.insert(clss);
#endif
    }
	String *super_name = clss->super_name;
	const String *class_name = clss->name;

	if (super_name == NULL) {

//#ifdef CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH
		extern bool bootstrapped;
		if ((strcmp(clss->name->bytes, "java/lang/Class") == 0) &&
			(bootstrapped == false)) { 
			// expected inconsistency at bootstrap stage
		} else { 
#endif
			//
			// this class better be java.lang.Object
			//
#ifdef CLI_OCL
			// System.Object in the case of OCL/CLI
            if ( !strcmp(clss->name->bytes, "System/Object") ) {
	            clss->state = ST_Loaded;
                return LD_NoException;
            }
#else
			if (clss->name != env->JavaLangObject_String) {
				// ClassFormatError
	#ifdef _DEBUG
				cerr << "ClassFormatError loading class "
					 << class_name->bytes << endl;
	#endif
				env->class_table.remove(clss);
				return LD_ClassFormatError;
			}
#endif //CLI_OCL
		
			//#ifdef CLASS_LOADER_BOOTSTRAP			
#ifndef OLD_VERSION_CLASSPATH
		}
#endif

	} else {
		//
		// load in super class
		//
		Class *super_class;
		super_class = load_class_by_loader(env,
                                           super_name,
                                           clss->class_loader,
                                           &exception);
#ifdef CLI_TESTING

        if(super_class == NULL && clss->super_type) {

            //printf("Loading CLI super class...\n");

            super_class = resolve_class_from_cli_type(env, clss->super_type, &exception);

        }

#endif

		if(super_class == NULL) {
			// NoClassDefFoundError, ClassFormatError, or ClassCircularityError
#ifdef _DEBUG
			cerr << "ClassFormatError loading superclass of class "
				 << class_name->bytes << " superclass is ";
			cerr << endl;
#endif
			env->class_table.remove(clss);
			return exception;
		}
        if (class_is_interface(super_class) || class_is_final(super_class) ) {
            env->class_table.remove(clss);
            return LD_IncompatibleClassChangeError;
        }

        Loader_Result result = class_prepare(super_class);

        if (result != LD_OK) {
            return LD_IncompatibleClassChangeError;
        }
		//
		// super class successfully loaded
		//
		clss->super_class = super_class;

        if(super_class_cp_index && *super_class_cp_index) {
            cp_resolve_to_class(clss->const_pool, *super_class_cp_index, super_class);
        }
        // if it's an interface, its superclass must be java/lang/Object
        if (class_is_interface(clss) 
            && env->JavaLangObject_Class
            && super_class != env->JavaLangObject_Class ) {
            env->class_table.remove(clss);
            return LD_IncompatibleClassChangeError;
        }
	}

    //
    // load in super interfaces
    //
    for (unsigned i = 0; i < clss->n_superinterfaces; i++) {
        // const String *intfc_name = clss->superinterfaces[i].name;
        String *intfc_name = clss->superinterfaces[i].name;	// meq
        Class *intfc = load_class_by_loader(env,
                                            intfc_name,
                                            clss->class_loader,
                                            &exception);
		if (intfc == NULL) {
			// NoClassDefFoundError, ClassFormatError, or ClassCircularityError
#ifdef _DEBUG
			cerr << "error: one of NoClassDefFoundError, ClassFormatError, or ClassCircularityError "
				 << class_name->bytes << " superinterface is " << intfc_name->bytes
				 << endl;
#endif
			env->class_table.remove(clss);
			return exception;
		}
        Loader_Result result = class_prepare(intfc);
    
        if (result != LD_OK) {
            return LD_IncompatibleClassChangeError;
        }

		//
		// super interface successfully loaded
		//
		clss->superinterfaces[i].clss = intfc;
	}
	// class, superclass, & superinterfaces successfully loaded
	clss->state = ST_Loaded;
    return LD_NoException;
} //finish_class_loading

#ifdef ZIP_JAR_ARCHIVE

//***********************************************************************************************
Class* load_class_from_ZIPJAR_file(const char* zipflname,const char* entryflname,Global_Env& env,
								   const String *classname, Loader_Exception *exception)
{
	JarFile* jarfl = getZIPJARFileFromPool(zipflname);
	if(!jarfl)return NULL;

	Jar_Entry jaren(jarfl, entryflname);

	uint8* buf = (uint8*)jaren.Read();
	if(buf == NULL)
		return NULL;
	int buf_len = jaren.Get_Entry_File_Length();
	
	Class *clss = new_class(classname);
	clss->class_file_name = env.string_pool.lookup(zipflname);;
    // Create a Class File Stream object
    ByteReader cfs(buf, 0, buf_len);
    unsigned super_class_cp_index;
	Loader_Result result = class_parse(clss, cfs, env, &super_class_cp_index, exception);

//#ifdef CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH
	extern bool bootstrapped;
	if ((bootstrapped == false) &&
		(strcmp(clss->name->bytes, "java/lang/Class") == 0)) {
		// make sure java/lang/class doesn't turn around and load java/lang/Object.
		// so zero out the superclass information just loaded.
		clss->super_name = 0;
	}
#endif

    if(result != LD_OK) {
		// NoClassDefFoundError
        assert(*exception);
        return NULL;
    }

    add_class_to_package(clss,env);
    return clss;
}

#endif		//ZIP_JAR_ARCHIVE.

Class* load_class_from_class_file(const char* filedirectory, const char* basename, Global_Env& env,
							      const String *classname, Loader_Exception *exception)
{
	int file_handle = -1;
    String *class_file_name = 0;
	
	unsigned baselen = strlen(basename);
	unsigned dirlen = strlen(filedirectory);
	char* filename = new char[baselen + dirlen + 2];
	strncpy(filename, filedirectory, dirlen);
	if (dirlen != 0) 
	{
		*(filename + dirlen) = DIR_SEPARATOR;
		dirlen++;
	}
	strcpy(filename + dirlen, basename);
#ifdef _DEBUG
	//		cerr << "load_class_from_file: trying filename " << filename << endl;
#endif
	struct _stat buf;

	/* Get data associated with "stat.c": */
	if (_stat(filename, &buf) != 0) 
	{
		delete[] filename;
		return NULL;
	}

	// file exists try to open it.
	class_file_name = env.string_pool.lookup(filename);
#ifdef ORP_NT
	file_handle = _open(filename, _O_RDONLY|O_BINARY|O_SEQUENTIAL);
#else
	file_handle = open(filename, O_RDONLY);
#endif

	delete[] filename;
	if(file_handle == -1)
		return NULL;

    //
    // I (RLH) think we have a race condition here. This routine can be called with the 
    // class loader lock locked and another thread can try to get that lock
    // while gc is disabled. This will cause the gc to hang.
    //

	Class *clss = new_class(classname);
    clss->class_file_name = class_file_name;
    // Create a Class File Stream object.
    ByteReader cfs(file_handle, 0, buf.st_size);
    unsigned super_class_cp_index;
	Loader_Result result = class_parse(clss, cfs, env, &super_class_cp_index, exception);

//#ifdef CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH
	extern bool bootstrapped;
	if ((bootstrapped == false) &&
		(strcmp(clss->name->bytes, "java/lang/Class") == 0)) {
		// make sure java/lang/class doesn't turn around and load java/lang/Object.
		// so zero out the superclass information just loaded.
		clss->super_name = 0;
	}
#endif

    if(result != LD_OK) {
		// NoClassDefFoundError
        assert(*exception);
        return NULL;
    }

    // close the file.
#ifdef ORP_NT
    int stat = _close(file_handle);
#else
    int stat = close(file_handle);
#endif

    add_class_to_package(clss,env);
    return clss;
}

Class *load_class_from_file(const String *classname, Global_Env& env, Loader_Exception *exception)
{
    // the symptom of circularity (illegal) is that a null classname is passed,
    // so detect this immediately

    if (!classname->bytes || *(classname->bytes) == 0) {
        *exception = LD_ClassCircularityError;
        return NULL;
    }

	//
	// first change classname from the internal fully qualified name
	// to an external file system name
	//
    unsigned baselen = classname->len + 7;  // includes the extra '\0'
    String *class_file_name = 0;
    int file_handle = -1;

    char *basename = new char[baselen];
	strcpy(basename, classname->bytes);
	// change fully qualified form to path name by replacing / with '\\' or vice versa.
	for (char *p = basename; *p != '\0'; p++)
		if ((*p == '/') || (*p == '\\'))
		    *p = DIR_SEPARATOR;
	// add the .class filename suffix
	strcat(basename, ".class");

#ifdef ZIP_JAR_ARCHIVE
    char *basename_in_zip = new char[baselen];
	strcpy(basename_in_zip, classname->bytes);
	strcat(basename_in_zip, ".class");
#endif

/*??
retry:
*/
	//
	// now find the .class file by searching through the classpath
	//
	char *dir = env.classpath;
	if (dir == NULL)
		dir = "";

	do {
		// find next directory delimiter and replace with end of string marker
		// problem: what if classpath ends with ;
		char *eos = strchr(dir,PATH_SEPARATOR);
		if (eos != NULL)
			*eos = '\0';

        Class *clss = NULL;
#ifdef ZIP_JAR_ARCHIVE
        if(isZIPJARFile(dir)){
			clss = load_class_from_ZIPJAR_file(dir, basename_in_zip, env, classname, exception);
        }
		else
#endif
			clss = load_class_from_class_file(dir, basename, env, classname, exception);

		if(clss != NULL){
			if (eos != NULL){
				*eos = PATH_SEPARATOR;
				//?? UpdateNativeClassPathToJava();
			}
			delete basename;
#ifdef ZIP_JAR_ARCHIVE
			delete basename_in_zip;
#endif
			return clss;
		}

		// replace '\0' with PATH_SEPARATOR for next use of class path
		if (eos != NULL) {
			*eos = PATH_SEPARATOR;
			//?? UpdateNativeClassPathToJava();
			dir = eos+1;	// starting point of next directory name
		} else {
#ifdef _DEBUG
//		cerr << "load_class_from_file: couldn't find class file for " 
//	 << basename << endl;
#endif
			/*??
			if(UpdateJavaClassPathToNative())
			    goto retry;
			*/

            *exception = LD_NoClassDefFoundError;
			delete basename;
#ifdef ZIP_JAR_ARCHIVE
			delete basename_in_zip;
#endif
			return NULL;	// we're done; no file found! 
		}
	} while (1);

	delete basename;
#ifdef ZIP_JAR_ARCHIVE
	delete basename_in_zip;
#endif

    return NULL;
} //load_class_from_file



Class *load_class_from_memory(uint8 *bytecode,
                              int offset,
                              int length,
                              const char *class_name,
                              Class_Loader *cl,
                              Global_Env *env,
                              Loader_Result *result,
                              Loader_Exception *ld_exc)
{
    String *classname;
    if(class_name) {
        classname = ORP_Global_State::loader_env->string_pool.lookup(class_name);
    } else {
        classname = 0;
    }
    Class *clss = new_class(classname);
    clss->class_loader = cl;
    // Create a Class File Stream object
    ByteReader cfs(bytecode, offset, length);
    unsigned super_class_cp_index;
	*result = class_parse(clss, cfs, *env, &super_class_cp_index, ld_exc);
	if(*result != LD_OK) {
        assert(*ld_exc);
		return NULL;
	}
	add_class_to_package(clss, *env);
    *ld_exc = finish_class_loading(clss, env, &super_class_cp_index);
    if(*ld_exc != LD_NoException) {
        *result = LD_ParseError;
        return NULL;
    }
	return clss;
} //load_class_from_memory



Class *load_class(Global_Env *env,
				  const String *classname,
				  Loader_Exception *exception)
{
	// According to Michal's analysis and my running Jigsaw, 
	// assert here is unreasonable especially in dynamic class loading
    //??assert(!orp_is_gc_enabled(p_TLS_orpthread));

    p_load_class_lock->_lock();
    //p_load_class_lock->_lock_enum();

    Class *p_class = locked_load_class_by_system_loader(env, classname, exception);

    p_load_class_lock->_unlock();
    //p_load_class_lock->_unlock_enum();
    
    return p_class;
} //load_class



Class *load_class_by_loader(Global_Env *env,
				            String *classname,
                            Class_Loader *cl,
				            Loader_Exception *exception)
{
    Class *clss = 0;
	if(cl) {
		/*?? if classname is just cl itself, ...
		char *clname = (char*)cl->get_loader()->vt->clss->name->bytes;
		if(!strcmp(clname, classname->bytes)){
			clss = cl->get_loader()->vt->clss;
			cl->insert_class(clss);
			env->class_table.insert(clss);
			return clss;
		}
		*/
		// In a circular reference scenario:
		// loadClass(A, true) -> resolveClass(A) -> loadClass(B) -> resolveClass(B)  
		// -> loadClass(A, true) ...
		// leads to infinite loop, so we should lookup in cache first
		if(clss = cl->lookup_class(classname)){
			if (clss->state == ST_LoadingAncestors) {
				// ClassCircularityError
				*exception = LD_ClassCircularityError;
				return NULL;
			}
			return clss;
		}
		// If classname is java.*, use system loader
		if( strstr(classname->bytes, "java/") == classname->bytes &&
			(clss = env->class_table.lookup(classname)) )
			return clss;
    
		// If the class is an array class
		if (classname->bytes[0] == '[') {
			//
			// array class
			//
			clss = create_array_class(classname, cl, *env, exception);
			if (clss == NULL) {
				// exception during loading of array base class
				return NULL;
			}
			Loader_Exception ex = finish_class_loading(clss, env, 0);
			if(ex != LD_NoException) {
				return NULL;
			}
		} 

        // Replace '/' with '.'
        unsigned class_name_len = classname->len + 1;
        char *class_name_buf = new char[class_name_len];
        for(unsigned i = 0; i < class_name_len; i++) {
            char c = classname->bytes[i];
            if(c == '/') {
                class_name_buf[i] = '.';
            } else {
                class_name_buf[i] = c;
            }
        }
        String *class_name_with_dots  = env->string_pool.lookup(class_name_buf);
        delete []class_name_buf;

        assert(env);
        String *name  = env->string_pool.lookup("loadClass");

        // call the version that takes the resolve flag
        // some subclasses of ClassLoader do NOT overload the (Ljava/lang/String;) version of the method
        // they all define (Ljava/lang/String;Z) version because it is abstract
		// wgs's comment here:
		//  * (Ljava/lang/String;Z) is not abstract in current JDK version
		//  * Generally (Ljava/lang/String;) are overloaded
        /*
        String *descr = env->string_pool.lookup("(Ljava/lang/String;Z)Ljava/lang/Class;");
        Signature *sig = env->sig_table.lookup(name, descr);
        assert(sig);
        */
		String *descr1 = env->string_pool.lookup("(Ljava/lang/String;)Ljava/lang/Class;");
        Signature *sig1 = env->sig_table.lookup(name, descr1);
        assert(sig1);

        //J_Value args[2]; 
		J_Value args[3];

        // Set the arg #1 first because calling orp_instantiate_cp_string_resolved
        // can cause GC and we would like to get away with code that doesn't
        // protect references from GC.
        if(class_name_with_dots->intern) {
            args[1].r = (void *)class_name_with_dots->intern;
        } else {
            args[1].r = (void *)orp_instantiate_cp_string_resolved(class_name_with_dots);
        }
        args[0].r = (void *)cl->get_loader();
#if 0
		printf("ClassLoader %s for class %s\n", cl->get_loader()->vt->clss->name->bytes, classname->bytes);
#endif
		Method *method = NULL;
		/*
		clss = cl->get_loader()->vt->clss;
		method = class_lookup_method(clss, sig1);
		if(!method){
			args[2].z = FALSE;
			method = class_lookup_method(clss, sig);
		}
		if(!method) 
			method = class_lookup_method_recursive(clss->super_class, sig);
		*/
		method = class_lookup_method_recursive(cl->get_loader()->vt->clss, sig1);
        assert(method);
		bool gc_was_enabled = orp_disable_gc();

        orp_execute_java_method_array(method, &clss, args);
        if(get_current_thread_exception()) {
            // Panic.
            //assert(0);
			throw_java_exception_from_native(get_current_thread_exception());
			return NULL;
        }

        if(gc_was_enabled) {
            orp_enable_gc();
        }

    } else {
        //
        // I (RLH) think we have a race condition here. If gc is disabled here and I have
        // seen this, and say the JIT is loading a class and the loading of the class
        // causes a garbage collection while allocating then the GC will hang.....
        //
        p_load_class_lock->_lock();
        clss = locked_load_class_by_system_loader(env, classname, exception);
        p_load_class_lock->_unlock();
    }
    return clss;

} //load_class_by_loader




static Class *locked_load_class_by_system_loader(Global_Env *env,
                                                 const String *classname,
                                                 Loader_Exception *exception)
{
#if 0
	// Left name of JAVA array to be "[B" and CHANGED CLI array to be "[BC"
	// SO code below is not needed.
#if CLI_OCL
	extern bool currently_creating_java_arrays;
	extern Global_CLI_Env *global_cli_env;
	Class *clss = NULL;
	if ((currently_creating_java_arrays == false) && ((classname->bytes)[0] == '[') &&
					(global_cli_env == NULL)){ 
		// orp_init() is running and CLI arrays are being preloaded.
		// THERE IS A NAME CLASH -- "[B" is there as both 
		// global_cli_env->ArrayOfByte_Class and  Java env->ArrayOfByte_Class 
		// NEED TO FIX...Until then bypass the cache and generate
		// CLI arrays from scratch.
		// THATS THE NAMING CONVENTION
		assert((classname->bytes)[2] == 'C');
		clss = NULL; // ?????????????????
	} else { 
		clss = env->class_table.lookup(classname);
	}
#else
	Class *clss = env->class_table.lookup(classname);
#endif // CLI_OCL
#endif

	Class *clss = env->class_table.lookup(classname);
	if (clss != NULL) {
		// class has already been loaded
		// check for cicularities in class hierarchy
		if (clss->state == ST_LoadingAncestors) {
			// ClassCircularityError
			*exception = LD_ClassCircularityError;
			return NULL;
		}
		return clss;
	}
	//
	// array classes require special handling
	//
	if (classname->bytes[0] == '[') {
		//
		// array class
		//
		clss = create_array_class(classname, NULL, *env, exception);
		if (clss == NULL) {
			// exception during loading of array base class
			return NULL;
		}
	} else {
		//
		// load class from file
		//
		clss = load_class_from_file(classname,*env, exception);
		if (clss == NULL) {
			// an error occured while loading the class
            assert(*exception);
            return NULL;
		}
	}
//wgs: I move the following from class_load_verify_prepare() to here, 
//      because java/lang/Class implements java/io/Serializable,
//      which will be loaded in finish_class_loading(), so we must fill in necessary
//      info into Class data structure before finish_class_loading().
#ifndef OLD_VERSION_CLASSPATH
	extern bool bootstrapped;
	if (bootstrapped == false) { 
		// Class class is the first class to be loaded and prepared.
		assert(strcmp(classname->bytes, "java/lang/Class") == 0);
		extern void assign_offsets_to_class_fields(Class *);
	    clss->unpadded_instance_data_size = 
	        ( (sizeof(Class) + (GC_OBJECT_ALIGNMENT - 1)) / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT;
		assign_offsets_to_class_fields(clss);
		clss->allocated_size = 
			(((clss->unpadded_instance_data_size + (GC_OBJECT_ALIGNMENT - 1)) 
			/ GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT) + OBJECT_HEADER_SIZE;
		clss->instance_data_size = clss->allocated_size;
		extern unsigned sizeof_class_class;
		assert(sizeof_class_class == 0);
		// Once sizeof_class_class is calculated, bootstrap is essentially done.
		sizeof_class_class = clss->instance_data_size ;
		return 0;
	}
#endif		

    *exception = finish_class_loading(clss, env, 0);
    if(*exception != LD_NoException) {
        return NULL;
    }
	return clss;
} //locked_load_class_by_system_loader



Class *load_class_by_system_loader(Global_Env *env,
                                   const String *classname,
                                   Loader_Exception *exception)
{
    p_load_class_lock->_lock();
    //p_load_class_lock->_lock_enum();

    Class *p_class = locked_load_class_by_system_loader(env, classname, exception);

    p_load_class_lock->_unlock();
    //p_load_class_lock->_unlock_enum();
    
    return p_class;
} //load_class_by_system_loader


Class *class_load_verify_prepare_by_loader(Global_Env *env,
                                           const String *classname,
                                           Class_Loader *cl,
                                           Loader_Exception *exception)
{
	// Class *clss = load_class(env, classname, exception);
	Class *clss = load_class_by_loader(env, (String *)classname, cl, exception);
    if (clss != NULL)  {
        Loader_Result result = class_prepare(clss);
        if (result != LD_OK) {
            *exception = LD_IncompatibleClassChangeError;
            return NULL;
        }
    }
	return clss;
} //class_load_verify_prepare_by_loader



#ifdef TRANSITIVE_CLASS_LOADING


Class *class_load_verify_prepare(Global_Env *env,
                                 const String *classname,
                                 Loader_Exception *exception)
{
    p_load_class_lock->_lock();

    Class *clss = load_class(env, classname, exception);
    if(clss != NULL) {

        Loader_Result result = class_prepare(clss);
        if (result != LD_OK) {
            *exception = LD_IncompatibleClassChangeError;
            return NULL;
        }

        const char *name = clss->name->bytes;
        if(strcmp(name, "java/lang/Object") &&
           strcmp(name, "java/lang/Class") &&
           strcmp(name, "java/io/Serializable")) {
            unsigned cp_size = clss->cp_size;
            Const_Pool *cp = clss->const_pool;
            for(unsigned i = 0; i < cp_size; i++) {
                if(cp_is_class(cp, i) && !cp_is_resolved(cp, i)) {
                    String *s = cp[i].string;
                    Loader_Exception ld_exc;
                    Class *c = load_class(env, s, &ld_exc);
                    if(c) {
                        cp_resolve_to_class(cp, i, c)
                        result = class_prepare(c);
                        if (result != LD_OK) {
                            *exception = LD_IncompatibleClassChangeError;
                            return NULL;
                        }
                    }
                }
            }
        }
    }

    p_load_class_lock->_unlock();
    return clss;
} //class_load_verify_prepare


#else


Class *class_load_verify_prepare(Global_Env *env,
                                 const String *classname,
                                 Loader_Exception *exception)
{
	Class *clss = load_class(env, classname, exception);
    if (clss != NULL) {

/*wgs: I moved the following to locked_load_class_by_system_loader()

//#ifdef CLASS_LOADER_BOOTSTRAP        
#ifndef OLD_VERSION_CLASSPATH
		extern bool bootstrapped;
		if (bootstrapped == false) { 
			// Class class is the first class to be loaded and prepared.
			assert(strcmp(classname->bytes, "java/lang/Class") == 0);
			extern void assign_offsets_to_class_fields(Class *);
	        clss->unpadded_instance_data_size = 
	            ( (sizeof(Class) + (GC_OBJECT_ALIGNMENT - 1)) / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT;
			assign_offsets_to_class_fields(clss);
			clss->allocated_size = 
				(((clss->unpadded_instance_data_size + (GC_OBJECT_ALIGNMENT - 1)) 
				/ GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT) + OBJECT_HEADER_SIZE;
			clss->instance_data_size = clss->allocated_size;
			extern unsigned sizeof_class_class;
			assert(sizeof_class_class == 0);
			// Once sizeof_class_class is calculated, bootstrap is essentially done.
			sizeof_class_class = clss->instance_data_size ;
			return 0;
		}
#endif
*/		
		Loader_Result result = class_prepare(clss);

        
		if (result != LD_OK) {
            *exception = LD_IncompatibleClassChangeError;
            return NULL;
        }
    }
	return clss;
} //class_load_verify_prepare

#endif


Class *class_load_verify_prepare_from_jni(Global_Env *env, const String *classname, Loader_Exception *exception)
{
    bool prev_state = orp_disable_gc();
    assert(prev_state);
    Class *clss = class_load_verify_prepare(env, classname, exception);
    orp_enable_gc();
    return clss;
}


//
// initialize string pool by preloading it with commonly used strings
//
Loader_Exception init_loader(Global_Env& env) {

	method_attr_strings[0] = env.string_pool.lookup("Code");
	method_attrs[0] = ATTR_Code;

	method_attr_strings[1] = env.string_pool.lookup("Exceptions");
	method_attrs[1] = ATTR_Exceptions;

	method_attr_strings[2] = NULL;

	field_attr_strings[0] = env.string_pool.lookup("ConstantValue");
	field_attrs[0] = ATTR_ConstantValue;

	field_attr_strings[1] = NULL;

	class_attr_strings[0] = env.string_pool.lookup("SourceFile");
	class_attrs[0] = ATTR_SourceFile;
	
	class_attr_strings[1] = env.string_pool.lookup("InnerClasses");
	class_attrs[1] = ATTR_InnerClasses;
    
	class_attr_strings[2] = NULL;

	class_attr_strings[1] = NULL;

	code_attr_strings[0] = env.string_pool.lookup("LineNumberTable");
	code_attrs[0] = ATTR_LineNumberTable;

	code_attr_strings[1] = env.string_pool.lookup("LocalVariableTable");
	code_attrs[1] = ATTR_LocalVariableTable;

	code_attr_strings[2] = NULL;

    must_recognize_attr_strings[0] = env.string_pool.lookup("Code");
    must_recognize_attrs[0] = ATTR_Code;

    must_recognize_attr_strings[1] = env.string_pool.lookup("ConstantValue");
    must_recognize_attrs[1] = ATTR_ConstantValue;

    must_recognize_attr_strings[2] = env.string_pool.lookup("Exceptions");
    must_recognize_attrs[2] = ATTR_Exceptions;

    must_recognize_attr_strings[3] = NULL;

    //
	// load in the preloaded classes
	//
	Loader_Exception exception = LD_NoException;

	return exception;
} //init_loader


