// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/jni_utils.cpp,v 1.11 2002/01/09 03:45:25 gwu2 Exp $
//



#include "platform.h"
#include "Class.h"
#include "Class_Loader.h"
#include "ini.h"

#include "native_utils.h"
#include "jni_utils.h"
#include "jni_direct.h"
#include "orp_utils.h"
#include "environment.h"
#include "exceptions.h"
#include "method_lookup.h"

#include "../../base_natives/gnu_classpath/include/gnu_classpath_jni_utils.h"

Class_Loader *class_loader_lookup(jobject loader)
{
    Object_Handle h = (Object_Handle)loader;

    orp_disable_gc();       //---------------------------------v

    Class_Loader *cl = Class_Loader::lookup(h->java_reference);

    orp_enable_gc();        //---------------------------------^

    return cl;
} //class_loader_lookup


Class_Loader *class_loader_find_if_exists(jobject loader)
{
    Object_Handle h = (Object_Handle)loader;

    orp_disable_gc();       //---------------------------------v

    Class_Loader *cl = Class_Loader::find_if_exists(h->java_reference);

    orp_enable_gc();        //---------------------------------^

    return cl;
} //class_loader_find_if_exists


ORPExport
jvalue *get_jvalue_arg_array(Method *method, va_list args)
{
    unsigned num_args = method->get_num_args();
    jvalue *jvalue_args = (jvalue *)orp_malloc_gc_safe(num_args * sizeof(jvalue));

    Arg_List_Iterator iter = method->get_argument_list();
    unsigned arg_number = 0;
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_CLASS:
        case JAVA_TYPE_ARRAY:
            jvalue_args[arg_number].l = va_arg(args, jobject);
            break;
        case JAVA_TYPE_INT:
            jvalue_args[arg_number].i = va_arg(args, int);
            break;
        case JAVA_TYPE_BYTE:
            jvalue_args[arg_number].b = va_arg(args, int);
            break;
        case JAVA_TYPE_BOOLEAN:
            jvalue_args[arg_number].z = va_arg(args, int);
            break;
        case JAVA_TYPE_CHAR:
            jvalue_args[arg_number].c = va_arg(args, int);
            break;
        case JAVA_TYPE_SHORT:
            jvalue_args[arg_number].s = va_arg(args, int);
            break;
        case JAVA_TYPE_LONG:
            jvalue_args[arg_number].j = va_arg(args, jlong);
            break;
        case JAVA_TYPE_FLOAT:
            //////jvalue_args[arg_number].f = va_arg(args, jfloat);
            // jfloat is passed as jdouble (8 bytes), not jfloat
            jvalue_args[arg_number].f = (float)va_arg(args, jdouble);
            break;
        case JAVA_TYPE_DOUBLE:
            jvalue_args[arg_number].d = va_arg(args, jdouble);
            break;
        default:
            assert(0);
            break;
        }
        iter = advance_arg_iterator(iter);
        arg_number++;
    }

    return jvalue_args;
} //get_jvalue_arg_array

/* This function is used to extract value from jobjectArray into J_Value arrays, use
 * Arg_List_Iterator as reference.
 * Parameter:
 *	  Arg_List_Iterator arglist(in): reference
 *    jobjectArray args(in): source
 *    J_Value *vargs(out): array which will store extracted values
 *    unsigned nvargs(in): size of array vargs, should be no less 
 *					     than array length of args
 * return: 
 *	  Exact number of unhandled values 
*/
int unhandle_jobjectArray(JNIEnv *jenv,
					   Arg_List_Iterator arglist,
                       jobjectArray args,
					   J_Value *vargs,
                       int nvargs)
{
	if(!args || !vargs)
		return 0;
	if(nvargs < GetArrayLength(jenv, args))
		return -1; //error
	nvargs = GetArrayLength(jenv, args);

	int i;
	for (i =0; i < nvargs; i++) {
		Java_Type type = curr_arg(arglist); 
		if(type == JAVA_TYPE_END){
			return -1;
		}
		jobject obj = GetObjectArrayElement(jenv, args, i);
		if(!obj){
			vargs[i].j = NULL; //set mem for the union to zero
		}else{
			jclass clzz = GetObjectClass(jenv, obj);
			assert(clzz);
			Class *clss = (Class*)(((Object_Handle)clzz)->java_reference); 
			if(type != JAVA_TYPE_ARRAY && type != JAVA_TYPE_CLASS){ //original is: if(clss->is_primitive){
				vargs[i] = UnwrapWrapperObjectJ(jenv, obj, type);
			}else
				vargs[i].r = (J_Reference)((Object_Handle)obj)->java_reference;
		}
		arglist = advance_arg_iterator(arglist);
	}
	return nvargs;
}
//
// Java references are represented in JNI using handles.  This function will
// replace all pointers to JNI handles with actual pointers to Java objects.
// Note that we must disable GC for this to work.  
// The size of the low-level union J_Value must be the same as the size of jvalue.
//
void unhandle_jni_args(JNIEnv *env,
                       Method *method,
                       J_Value *direct_args,
                       jvalue *args,
                       unsigned num_args)
{
    // First copy all arguments.
    // (This relies on the fact that the layouts of J_Value and jvalue
    // are indentical.)
    assert(sizeof(J_Value) == sizeof(jvalue));
    memcpy(direct_args, args, num_args * sizeof(jvalue));

    // Then unhandle those that are references.
    Arg_List_Iterator iter = method->get_argument_list();
    unsigned arg_number = 0;
    Object_Handle h;
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_CLASS:
        case JAVA_TYPE_ARRAY:
            h = (Object_Handle)(args[arg_number].l);
            if(h) {
				direct_args[arg_number].r = h->java_reference;
            }
            break;
        default:
            break;
        }
        iter = advance_arg_iterator(iter);
        arg_number++;
    }
} //unhandle_jni_args


Object_Handle put_class_in_handle(Class *clss)
{
    if(clss) {
        Object_Handle new_handle = orp_create_local_object_handle();
        new_handle->java_reference = (Java_java_lang_Object *)clss;
        return new_handle;
    } else {
        return 0;
    }
} //put_class_in_handle

//We'd better deprecate this routine.
jobject ConstructJavaConstructor (JNIEnv* env, Method* m)
{
	assert (m);

	const char* msig = m->get_descriptor ();

	jclass mclazz = FindClass (env, "java/lang/reflect/Constructor");

	// Construct a new java.lang.reflect.Constructor object:
	jobject jconst = AllocObject (env, mclazz);


	// Set "clazz" field to the declaring class of this field:
	Class* clss = m->get_class();
	//jclass clazz = FindClass (env, clss->name->bytes); 
	//Change FindClass to this
	jclass clazz = orp_create_local_object_handle();
	((Object_Handle)clazz)->java_reference = (Java_java_lang_Object *)clss;

	jfieldID clazz_id = GetFieldID (env, mclazz, "clazz", "Ljava/lang/Class;");
	SetObjectField (env, jconst, clazz_id, clazz);

	int nparams =0;	// nparams = number of parameters + return type

	// Identify individual parameter types using method signature:
	jclass*	ptypes = GetMethodParameterTypes (env, msig, &nparams);
	jclass cclazz = FindClass (env, "java/lang/Class");

	// Create array of java/lang/Class elements:
	jobjectArray parameterTypes = (jobjectArray) NewObjectArray (env, (nparams - 1), cclazz, 0);

    int ii;
	for (ii =0; ii < (nparams - 1); ii++) {
		SetObjectArrayElement (env, parameterTypes, ii, ptypes[ii]);
	}

	free(ptypes); //free the memory allocated in GetMethodParameterTypes()

	jfieldID parameterTypes_id = GetFieldID (env, mclazz, "parameterTypes", "[Ljava/lang/Class;");
	SetObjectField (env, jconst, parameterTypes_id, parameterTypes);

	// Create and fill exceptions array
	int n_exceptions = m->num_exceptions_method_can_throw ();

	// Create array of java/lang/Class elements:
	jobjectArray exceptionTypes = (jobjectArray) NewObjectArray (env, n_exceptions, cclazz, 0);

	Class_Loader *cl = m->get_class()->class_loader;
	for (ii =0; ii < n_exceptions; ii++) {
		//XXX: jclass exclass = FindClass (jenv, m->get_exception_name(ii)->bytes);
		jclass exclass = FindClassWithClassLoader(m->get_exception_name(ii)->bytes, cl);
		SetObjectArrayElement (env, exceptionTypes, ii, exclass);
	}

	jfieldID exceptionTypes_id = GetFieldID (env, mclazz, "exceptionTypes", "[Ljava/lang/Class;");
	SetObjectField (env, jconst, exceptionTypes_id, exceptionTypes);

	return jconst;
} // ConstructJavaConstructor

jobject ConstructJavaConstructor (JNIEnv* env, Method* m, Class_Loader *class_loader)
{
	assert (m);

	const char* msig = m->get_descriptor ();

	jclass mclazz = FindClass (env, "java/lang/reflect/Constructor");

	// Construct a new java.lang.reflect.Constructor object:
	jobject jconst = AllocObject (env, mclazz);


	// Set "clazz" field to the declaring class of this field:
	Class* clss = m->get_class();
	//jclass clazz = FindClass (env, clss->name->bytes); 
	//Change FindClass to this
	jclass clazz = orp_create_local_object_handle();
	((Object_Handle)clazz)->java_reference = (Java_java_lang_Object *)clss;

	jfieldID clazz_id = GetFieldID (env, mclazz, "clazz", "Ljava/lang/Class;");
	SetObjectField (env, jconst, clazz_id, clazz);

	int nparams =0;	// nparams = number of parameters + return type

	// Identify individual parameter types using method signature:
	jclass*	ptypes = GetMethodParameterTypes (env, msig, &nparams, class_loader);
	jclass cclazz = FindClass (env, "java/lang/Class");

	// Create array of java/lang/Class elements:
	jobjectArray parameterTypes = (jobjectArray) NewObjectArray (env, (nparams - 1), cclazz, 0);

    int ii;
	for (ii =0; ii < (nparams - 1); ii++) {
		SetObjectArrayElement (env, parameterTypes, ii, ptypes[ii]);
	}

	free(ptypes); //free the memory allocated in GetMethodParameterTypes()

	jfieldID parameterTypes_id = GetFieldID (env, mclazz, "parameterTypes", "[Ljava/lang/Class;");
	SetObjectField (env, jconst, parameterTypes_id, parameterTypes);

	// Create and fill exceptions array
	int n_exceptions = m->num_exceptions_method_can_throw ();

	// Create array of java/lang/Class elements:
	jobjectArray exceptionTypes = (jobjectArray) NewObjectArray (env, n_exceptions, cclazz, 0);

	for (ii =0; ii < n_exceptions; ii++) {
		jclass exclass = FindClassWithClassLoader(m->get_exception_name(ii)->bytes, class_loader);
		SetObjectArrayElement (env, exceptionTypes, ii, exclass);
	}

	jfieldID exceptionTypes_id = GetFieldID (env, mclazz, "exceptionTypes", "[Ljava/lang/Class;");
	SetObjectField (env, jconst, exceptionTypes_id, exceptionTypes);

	return jconst;
} // ConstructJavaConstructor

jobject ConstructJavaField (JNIEnv *env, Field* f)
{
	assert (f);

	const char* fname = f->get_name()->bytes;
	// toss int len = f->get_name()->len;
	const char* fsig  = f->get_descriptor ();
	
	jclass fclazz = FindClass (env, "java/lang/reflect/Field");

	// Construct a new java.lang.reflect.Field object:
	jobject jfield = AllocObject (env, fclazz);

	// Set "clazz" field to the declaring class of this field:
	Class* cls = f->get_class();
	//XXX: jclass clazz = FindClass (env, cls->name->bytes);
	jclass clazz = orp_create_local_object_handle();
	((Object_Handle)clazz)->java_reference = (Java_java_lang_Object*)cls;

	//Classpath's Field definition has no "clazz" field
	//Obsolete -> jfieldID clazz_id = GetFieldID (env, fclazz, "clazz", "Ljava/lang/Class;");
	jfieldID clazz_id = GetFieldID (env, fclazz, "declaringClass", "Ljava/lang/Class;");
	
	SetObjectField (env, jfield, clazz_id, clazz);

	// Set "name" field to the name of this field:
	// toss jstring name = CharArrayToJavaString (env, fname, len);
    jstring name = env->NewStringUTF(fname);

	jfieldID name_id = GetFieldID (env, fclazz, "name", "Ljava/lang/String;");
	SetObjectField (env, jfield, name_id, name);

//Classpath Field definition has no "type" field
/* Obsolete ->
	// Set "type" field to the type of this field:
	jclass type = SignatureToClass (env, fsig);

	jfieldID type_id = GetFieldID (env, fclazz, "type", "Ljava/lang/Class;");
	SetObjectField (env, jfield, type_id, type);
*/	
    return jfield;
} // ConstructJavaField


jobject ConstructJavaMethod (JNIEnv *env, Method* m)
{
	assert (m);

	const char* mname = m->get_name()->bytes;
	// toss int len = m->get_name()->len;
	const char* msig  = m->get_descriptor ();

	jclass mclazz = FindClass (env, "java/lang/reflect/Method");

    if ( !mclazz )
        return (jobject)NULL;

	// Construct a new java.lang.reflect.Method object:
	jobject jmethod = AllocObject (env, mclazz);

    if ( !jmethod )
        return (jobject)NULL;

	// Set "clazz" field to the declaring class of this field:
    Class* clss = m->get_class();
    //jclass clazz = FindClass (env, clss->name->bytes);
    //Change FindClass to this
    jclass clazz = orp_create_local_object_handle();
    ((Object_Handle)clazz)->java_reference = (Java_java_lang_Object *)clss;

    if ( !clazz )
        return (jobject)NULL;

	//Classpath Field definition has no "clazz" field
	//Obsolete -> jfieldID clazz_id = GetFieldID (env, mclazz, "clazz", "Ljava/lang/Class;");
	jfieldID clazz_id = GetFieldID (env, mclazz, "declaringClass", "Ljava/lang/Class;");

    if ( !clazz_id )
        return (jobject)NULL;

    SetObjectField (env, jmethod, clazz_id, clazz);

	// Set "name" field to the name of this field:
	// toss jstring name = CharArrayToJavaString (env, mname, len);

    jstring name = env->NewStringUTF(mname);

    if ( !name )
        return (jobject) NULL;

	jfieldID name_id = GetFieldID (env, mclazz, "name", "Ljava/lang/String;");

    if ( !name_id )
        return (jobject)NULL;

	SetObjectField (env, jmethod, name_id, name);

/*
	Classpath Method difinition has no "parameterTypes", "returnType" and "exceptionTypes" field,
	it's really a uneasy problem for VMers, I don't know exactly what "slot" field means,
	but here I use it as the method index. 
	I hope there's some better usage of this field to save method info.
*/
/*
	int nparams =0;	// nparams = number of parameters + return type
    // Identify individual parameter types using method signature:
	jclass*	ptypes = GetMethodParameterTypes (env, msig, &nparams);
	jclass cclazz = FindClass (env, "java/lang/Class");

    if ( !cclazz )
        return (jobject)NULL;

	// Create array of java/lang/Class elements:
	jobjectArray parameterTypes = NewObjectArray (env, (nparams - 1), cclazz, 0);

    if ( !parameterTypes )
        return (jobject)NULL;

    int ii;
	for (ii =0; ii < (nparams - 1); ii++) {
		SetObjectArrayElement (env, parameterTypes, ii, ptypes[ii]);
	}

	jfieldID parameterTypes_id = GetFieldID (env, mclazz, "parameterTypes", "[Ljava/lang/Class;");

    if ( !parameterTypes_id )
        return (jobject)NULL;

	SetObjectField (env, jmethod, parameterTypes_id, parameterTypes);

	jclass returnType = ptypes[nparams-1];	

	jfieldID returnType_id = GetFieldID (env, mclazz, "returnType", "Ljava/lang/Class;");

    if ( !returnType_id )
        return (jobject)NULL;
    
    SetObjectField (env, jmethod, returnType_id, returnType);

	// Create and fill exceptions array
	int n_exceptions = m->num_exceptions_method_can_throw ();

	// Create array of java/lang/Class elements:
	jobjectArray exceptionTypes = NewObjectArray (env, n_exceptions, cclazz, 0);
 
    if ( !exceptionTypes )
        return (jobject)NULL;   

	for (ii =0; ii < n_exceptions; ii++) {
		jclass exclass = FindClass (env, m->get_exception_name(ii)->bytes);

        if ( !exclass )
            return (jobject)NULL;   
        
        SetObjectArrayElement (env, exceptionTypes, ii, exclass);
	}

	jfieldID exceptionTypes_id = GetFieldID (env, mclazz, "exceptionTypes", "[Ljava/lang/Class;");

    if ( !exceptionTypes_id )
        return (jobject)NULL;   
    
    SetObjectField (env, jmethod, exceptionTypes_id, exceptionTypes);
*/
	jfieldID slot_id = GetFieldID (env, mclazz, "slot", "I");

    if ( !slot_id )
        return (jobject)NULL;   

	//use method signature to match, and set slot to the match times 
	
	int i, slot = 0;
	for (; clss; clss = clss->super_class) {        // for each superclass
		for (i = 0; i < clss->n_methods; i++, slot++) {   // for each method
			m = &clss->methods[i];
			const char* sig = m->get_descriptor();
			int len = strlen (msig);
			if (!strcmp(m->get_name()->bytes, mname) &&      // if names and signatures
				strncmp (msig, sig, len) == 0) {         // (excluding return types) match,   
				SetIntField (env, jmethod, slot_id, slot);
				return jmethod;
			}
		}
    }

	assert(0);

	return NULL;
} // ConstructJavaMethod

Method* GetConstructor (JNIEnv *env, jobject constructor) 
{
	jfieldID parameterTypes_id = GetFieldID_Quick
		(env, "java/lang/reflect/Constructor", "parameterTypes", "[Ljava/lang/Class;");
    jobjectArray parameterTypes = GetObjectField (env, constructor, parameterTypes_id);

	char* msig = ParameterTypesToMethodSignature (env, parameterTypes);

	// Access internal (Class) handle to declaring class for constructor:
    jclass dclazz = GetConstructorClass (env, constructor);
    Object_Handle_Struct *dclazz_h = (Object_Handle_Struct *)dclazz;
    Class *dcl = (Class *)dclazz_h->java_reference;

	// Find and return the constructor in the class definition. This method should
	// exist since the Constructor object is normally constructed by a call to
	// Class.getConstructor(...);

    Method *m = LookupMethod (dcl, "<init>", msig);

    //free memory allocated by ParameterTypesToMethodSignature
	free(msig);
	return m;
} // GetConstructor


Field* GetField (JNIEnv *env, jobject field) 
{
	// Access the field name as char array:
    jfieldID name_id = GetFieldID_Quick(env,
                                        "java/lang/reflect/Field",
                                        "name",
                                        "Ljava/lang/String;");
	if (!name_id) return (Field*)0;

    jstring name = GetObjectField(env, field, name_id);
	if (!name) return (Field*)0;

	// toss char* fname = JavaStringToCharArray (env, name, 0);
    const char* fname = env->GetStringUTFChars(name, NULL);

	// Access the field's internal class handle (Class):
    jclass clazz = GetFieldClass (env, field);
	if (!clazz) return (Field*)0;

    Object_Handle_Struct *clazz_h = (Object_Handle_Struct *)clazz;
    Class *fcl = (Class *)clazz_h->java_reference;

	// Find and return the field in the class definition. This field should exist
	// since the Field object is normally constructed by a call to Class.getField().

    Field *f = LookupField (fcl, fname);

    env->ReleaseStringUTFChars(name, fname);
    
	return f;
} // GetField


Method* GetMethod (JNIEnv *env, jobject method) 
{
	jclass mclazz = FindClass (env, "java/lang/reflect/Method");	

	// Access the method name as char array:
    jmethodID name_id = GetFieldID (env, mclazz, "name", "Ljava/lang/String;");
    jstring name = GetObjectField (env, method, name_id);

	// toss char* mname = JavaStringToCharArray (env, name, 0);
    const char* mname = env->GetStringUTFChars(name, NULL);

	// Access internal (Class) handle to class declaring method:
    jclass dclazz = GetMethodClass (env, method);
    Object_Handle_Struct *dclazz_h = (Object_Handle_Struct *)dclazz;
    Class *dcl = (Class *)dclazz_h->java_reference;

//GNU Class Method: no "parameterTypes" field
/*
	jfieldID parameterTypes_id = GetFieldID (env, mclazz, "parameterTypes",
		                                     "[Ljava/lang/Class;");
    jobjectArray parameterTypes = GetObjectField (env, method, parameterTypes_id);

	char* msig = ParameterTypesToMethodSignature (env, parameterTypes);
	// Find and return the method in the class definition. This method should exist
	// since the Method object is normally constructed by a call to Class.getMethod().

    Method *m = LookupMethod (dcl, mname, msig);
*/
	jfieldID slot_id = GetFieldID (env, mclazz, "slot", "I");

    if ( !slot_id )
        return NULL;   

    int slot = GetIntField(env, method, slot_id);

	for(; dcl; dcl = dcl->super_class)
		if(slot > dcl->n_methods)
			slot -= dcl->n_methods;
		else
			break;
	Method *m = &(dcl->methods[slot]);

    env->ReleaseStringUTFChars(name, mname);

    return m;
} // GetMethod


jclass GetConstructorClass (JNIEnv *env, jobject constructor)
{
	jclass cclazz = FindClass (env, "java/lang/reflect/Constructor");	
	// Get the fieldID of the *clazz* field of Constructor:
    jfieldID clazz_id = GetFieldID (env, cclazz, "clazz", "Ljava/lang/Class;");
    jclass clazz = GetObjectField (env, constructor, clazz_id);
	return clazz;
} // GetConstructorClass


jclass GetFieldClass (JNIEnv *env, jobject field)
{
	// Get the fieldID of the "declaringClass" ( not "clazz") field of Field:
    /*
	jfieldID clazz_id = GetFieldID_Quick (env,
                                          "java/lang/reflect/Field",
										  "clazz",
										  "Ljava/lang/Class;");
    */
	jfieldID clazz_id = GetFieldID_Quick (env,
                                          "java/lang/reflect/Field",
										  "declaringClass",
										  "Ljava/lang/Class;");

	jclass clazz = GetObjectField (env, field, clazz_id);
	return clazz;
} // GetFieldClass


jclass GetMethodClass (JNIEnv *env, jobject method)
{
	jclass mclazz = FindClass (env, "java/lang/reflect/Method");	

	// Get the fieldID of the "declaringClass" ( not "clazz") field of Method:
    //jfieldID clazz_id = GetFieldID (env, mclazz, "clazz", "Ljava/lang/Class;");
	jfieldID clazz_id = GetFieldID (env, mclazz, "declaringClass", "Ljava/lang/Class;");
    jclass clazz = GetObjectField (env, method, clazz_id);
	return clazz;
} // GetMethodClass

//We'd better deprecate this routine 
jclass* GetMethodParameterTypes (JNIEnv* env, const char* sig, int *nparams)
{
  int    _parsed =0;
  char*  param;
  const char** sparray = (const char**) malloc(256 * sizeof(char*));
  jclass* parray;
  const char*  s_ptr = sig;
  char*  p_ptr;

  assert (sig); assert (nparams);

  *nparams = 0;
  s_ptr++;

  while (*s_ptr != ')') {
    param = (char*) malloc(256);
    p_ptr = param;
    while (!_parsed) {
		switch (*s_ptr) {
		case 'L':
			while ((*p_ptr++ = *s_ptr++) != ';');
			*p_ptr = '\0';
			_parsed = 1;
			break;
		case '[':
			while (*s_ptr == '[')
				*p_ptr++ = *s_ptr++;
			break;
		case 'B': case 'C': case 'D':
		case 'F': case 'I': case 'J':
		case 'S': case 'V': case 'Z':
			*p_ptr++ = *s_ptr++;
			*p_ptr++ = '\0';
			_parsed = 1;
			break;
		}
    }

    (*nparams)++;
    sparray[*nparams-1] = param;
    _parsed = 0;
  }
  sparray[(*nparams)++] = ++s_ptr;

  parray = (jclass*) malloc (*nparams * sizeof(jclass));
  assert (parray);

  // Now convert parameter and return type signatures to class objects
  // and construct the parameterTypes array.
  int ii;
  for (ii =0; ii < *nparams; ii++) {
	  parray[ii] = SignatureToClass (env, sparray[ii]);
  }	
  for (ii =0; ii < *nparams-1; ii++) 
	  free((void*)sparray[ii]); 
  free(sparray);
  
  return parray;
} // GetMethodParameterTypes

jclass* GetMethodParameterTypes (JNIEnv* env, const char* sig, int *nparams, Class_Loader *class_loader)
{
  int    _parsed =0;
  char*  param;
  const char* sparray[256 * sizeof(char*)];
  jclass* parray;
  const char*  s_ptr = sig;
  char*  p_ptr = (char*)malloc(strlen(sig) + 257);
  char* o_ptr = p_ptr;

  assert (sig); assert (nparams);

  *nparams = 0;
  s_ptr++; //skip leading '('

  while (*s_ptr != ')') {
    param = p_ptr;
    while (!_parsed) {
		switch (*s_ptr) {
		case 'L':
			while ((*p_ptr++ = *s_ptr++) != ';');
			*p_ptr++ = '\0';
			_parsed = 1;
			break;
		case '[':
			while (*s_ptr == '[')
				*p_ptr++ = *s_ptr++;
			break;
		case 'B': case 'C': case 'D':
		case 'F': case 'I': case 'J':
		case 'S': case 'V': case 'Z':
			*p_ptr++ = *s_ptr++;
			*p_ptr++ = '\0';
            _parsed = 1;
			break;
		}
    }

    (*nparams)++;
    sparray[*nparams-1] = param;
    _parsed = 0;
  }
  sparray[(*nparams)++] = ++s_ptr;

  parray = (jclass*) malloc (*nparams * sizeof(jclass));
  assert (parray);

  // Now convert parameter and return type signatures to class objects
  // and construct the parameterTypes array.
  int ii = 0;
  for (ii =0; ii < *nparams; ii++) {
	  parray[ii] = SignatureToClass (env, sparray[ii], class_loader);
  }

  free(o_ptr);
  
  return parray;
} // GetMethodParameterTypes

char* ParameterTypesToMethodSignature (JNIEnv* env, jobjectArray parameterTypes)
{
	if (!parameterTypes) {
//			ThrowNew_Quick (env, "java/lang/NullPointerException", 0);
//		throw_java_exception ("java/lang/NullPointerException");
		return (char*)0;
	}

	jsize nelem = GetArrayLength (env, parameterTypes);

    char* sig = (char *) malloc(1024); //is 1024 enough?

    sig[0] = '(';
    sig[1] = '\0';

    char asig[512];
    for (jsize i =0; i < nelem; i++) {
		jclass pclazz = (jclass) GetObjectArrayElement (env, parameterTypes, i);

		// Access the internal class handle (Class) for the parameter type:
		Object_Handle_Struct *pclazz_h = (Object_Handle_Struct *)pclazz;
		Class *pcl = (Class *)pclazz_h->java_reference;

		GetClassSignature (pcl, asig);
        strcat(sig, asig);
    }

    strcat(sig, ")");

    return sig;

} // ParameterTypesToMethodSignature

//should we care about buffer overflow and return len?
void GetClassSignature (Class* cl, char *sig)
{
	assert (cl);

    const char* name = cl->name->bytes;

    if (name[0] == '[') {
		sprintf (sig, "%s",name);
	}
    else if (cl->is_primitive) {
		sig[0] = PrimitiveNameToSignature (name);
		sig[1] = '\0';
    }
    else {
		sprintf (sig, "L%s;", name);
	}
} // GetClassSignature

//get class name from jclass object, by far used in
// java_net_PlainSocketImpl_common.h to differentiate Socket/DiagramSocket
const char* GetJClassName(jclass clazz)
{
	Object_Handle h = (Object_Handle)clazz;

    Class *clss = (Class *)h->java_reference;
	return clss->name->bytes;
}

/*
const char* ClassSignatureToName (const char* sig)
{
	assert (sig);
    const char* sig_ptr = sig;
 
	char* name = (char *) malloc (256);
	char* name_ptr = name;

    if (*sig_ptr++ == 'L') {   // if signature for reference type:
		while (*sig_ptr != ';') {
			*name_ptr++ = *sig_ptr++;
		}
		*name_ptr = '\0';

		return name;         // return class name
    } else {                 
		return sig;          // return original signature otherwise
	}
} // ClassSignatureToName
*/

//should we care about buffer overflow and return len?
void ClassSignatureToName (const char* sig, char *name) 
{
	assert (sig);
    const char* sig_ptr = sig;
 
    if (*sig_ptr++ == 'L') {   // if signature for reference type:
		while (*sig_ptr != ';') {
			*name++ = *sig_ptr++;
		}
		*name = '\0';
    } else {                 
		strcpy(name, sig);
	}
} // ClassSignatureToName

/*
char* PrimitiveSignatureToName (const char sig)
{
	char* classname = (char*)malloc(10);

	switch (sig) {
	case 'Z':
		//classname = "boolean";
		strcpy(classname, "boolean");
		break;
	case 'B':
		//classname = "byte";
		strcpy(classname, "byte");
		break;
	case 'C':
		//classname = "char";
		strcpy(classname, "char");
		break;
	case 'S':
		//classname = "short";
		strcpy(classname, "short");
		break;
	case 'I':	
		//classname = "int";
		strcpy(classname, "int");
		break;
	case 'J':
		//classname = "long";
		strcpy(classname, "long");
		break;
	case 'F':
		//classname = "float";
		strcpy(classname, "float");
		break;
	case 'D':
		//classname = "double";
		strcpy(classname, "double");
		break;
	case 'V':
		//classname = "void";
		strcpy(classname, "void");
		break;
	default:
		assert(0);
		break;
	}

	return classname;
} // PrimitiveSignatureToName
*/

//should we care about buffer overflow and return len?
void PrimitiveSignatureToName (const char sig, char *classname)
{
	switch (sig) {
	case 'Z':
		//classname = "boolean";
		strcpy(classname, "boolean");
		break;
	case 'B':
		//classname = "byte";
		strcpy(classname, "byte");
		break;
	case 'C':
		//classname = "char";
		strcpy(classname, "char");
		break;
	case 'S':
		//classname = "short";
		strcpy(classname, "short");
		break;
	case 'I':	
		//classname = "int";
		strcpy(classname, "int");
		break;
	case 'J':
		//classname = "long";
		strcpy(classname, "long");
		break;
	case 'F':
		//classname = "float";
		strcpy(classname, "float");
		break;
	case 'D':
		//classname = "double";
		strcpy(classname, "double");
		break;
	case 'V':
		//classname = "void";
		strcpy(classname, "void");
		break;
	default:
		assert(0);
		break;
	}
} // PrimitiveSignatureToName


//
// Given its name, lookup for a field in the class hierarchy:
//
Field* LookupField (Class *clss, const char *name)
{
    String *field_name =
        ORP_Global_State::loader_env->string_pool.lookup(name);

    Field *f = 0;
    for(; clss && !f; clss = clss->super_class) {
		if(f = LookupDeclaredField(clss, name))
			return f;
	}

	return NULL;
} // LookupField

Method* LookupMethod (Class *clss, const char *mname, const char *msig)
{
    String *method_name = ORP_Global_State::loader_env->string_pool.lookup(mname);
   
    Method* m = 0;
	Class *oclss = clss;
    for (; clss; clss = clss->super_class) {        // for each superclass
		if(m = LookupDeclaredMethod(clss, mname, msig))
			return m;
    }

	Class_Superinterface *intfs = oclss->superinterfaces;
	for(int i = 0; i < oclss->n_superinterfaces; i++)
		if(m = LookupMethod(intfs[i].clss, mname, msig))
			return m;

    return (Method*)0;                                // method not found
} // LookupMethod

char PrimitiveNameToSignature (const char* name)
{
    char sig;

    switch (name[0]) {
    case 'b':
		if (name[1] == 'o') sig = 'Z';
		else sig = 'B';
		break;
    case 'c': sig = 'C'; break;
    case 'd': sig = 'D'; break;
    case 'f': sig = 'F'; break;
    case 's': sig = 'S'; break;
    case 'i': sig = 'I'; break;
    case 'l': sig = 'J'; break;
	case 'v': sig = 'V'; break;
    }
    return sig;
} // PrimitiveNameToSignature

/*
const char* SignatureToName (const char* sig)
{
	assert (sig);
	const char* classname;

	if (sig[0] == 'L') {                      // declared type
		classname = ClassSignatureToName (sig);
	} else if (sig[0] == '[') {                // array type
		classname = (char*)malloc(256);
		strcpy((char*)classname, sig);
	} else {                                   // primitive type
		classname = PrimitiveSignatureToName (sig[0]);
	}

	return classname;
} // SignatureToName
*/

//should we care about buffer overflow and return len?
void SignatureToName (const char* sig, char *name)
{
	assert (sig);

	if (sig[0] == 'L') {                      // declared type
		ClassSignatureToName (sig, name);
	} else if (sig[0] == '[') {                // array type
		strcpy(name, sig);
	} else {                                   // primitive type
		PrimitiveSignatureToName (sig[0], name);
	}
} // SignatureToName

jclass SignatureToClass (JNIEnv* env, const char* sig)
{
	assert (sig);
    char classname[512];
	//const char* classname = SignatureToName (sig);
    SignatureToName(sig, classname);
	jclass clazz = FindClass (env, classname);
	//free((void*)classname);
	return clazz;
} // SignatureToClass

//We'd better use this routine
jclass SignatureToClass (JNIEnv* env, const char* sig, Class_Loader *class_loader)
{
	assert (sig);
    char classname[512];
    SignatureToName(sig, classname);
	//const char* classname = SignatureToName (sig);
	jclass clazz = FindClassWithClassLoader(classname, class_loader);
	//free((void*)classname);
	return clazz;
} // SignatureToClass


//////////////////////// add stuff VVVVVVVVVVVv

char* JavaStringToCharArray (JNIEnv *env, jstring jstr, jint* len)
{
	jint jstr_count;
	jint jstr_offset;
	jcharArray jstr_char_array;

	get_string_specifics(env, jstr, &jstr_count, &jstr_offset, &jstr_char_array);
		
	assert(jstr_count >= 0);
	assert(jstr_char_array);

	jboolean isCopy;
    jchar *jstr_jchar_array = GetCharArrayElements(env, jstr_char_array, &isCopy);
	jstr_jchar_array += jstr_offset; //skip the offset

    char*  cstr = (char *)malloc (jstr_count + 1);
    cstr[jstr_count] = 0;
    for (int ii = 0; ii < jstr_count; ii++) {
        cstr[ii] = ((char*)jstr_jchar_array)[ii + ii]; // 16 bit characters!
    }
	env->ReleaseCharArrayElements(jstr_char_array, jstr_jchar_array, JNI_ABORT);
	if (len) {
		*len = jstr_count;
	}
	return cstr;
} // JavaStringToCharArray


char IsWrapperClass (const char* name)
{
	char _sig;
	const char* c_ptr = (name);
	char* type = "java/lang/";

	while (*type++ == *c_ptr++);

	type--;
	c_ptr--;

	if (*type != '\0') _sig = '\0';
	else {
		switch (*c_ptr) {
		case 'B':
			if (*(c_ptr+1) == 'o') {
				_sig = 'Z';
				type = "Boolean";
			}
			else {
				_sig = 'B';
				type = "Byte";
			}
			break;
		case 'C':
			_sig = 'C';
			type = "Character";
			break;
		case 'D':
			_sig = 'D';
			type = "Double";
			break;
		case 'F':
			_sig = 'F';
			type = "Float";
			break;
		case 'I':
      _sig = 'I';
      type = "Integer";
      break;
    case 'L':
      _sig = 'J';
      type = "Long";
      break;
    case 'S':
      _sig = 'S';
      type = "Short";
    }

    while ((*c_ptr == *type) && (*c_ptr != '\0')) {
      c_ptr++;
      type++;
    }
    if ((*c_ptr != '\0') || (*type != '\0'))
      _sig = '\0';
  }

  return _sig;
} // IsWrapperClass


//
// Given its name, lookup for a field in the given class
//
Field* LookupDeclaredField (Class *clss, const char *name)
{
    String *field_name = ORP_Global_State::loader_env->string_pool.lookup(name);

	assert (clss);

	for (unsigned i =0; i < clss->n_fields; i++) {
		// toss if (strcmp (clss->fields[i].get_name()->bytes, name) == 0) {
        if (clss->fields[i].get_name() == field_name) {
			return &clss->fields[i];
		}
	}

	return NULL;
} // LookupDeclaredField


void VerifyArray (JNIEnv* env, jarray array)
{
	if (!array) {
		ThrowNew_Quick (env, "java/lang/NullPointerException", 0);
		return;
	}

	jclass aclazz = GetObjectClass (env, array);

	// Acquire handle to internal class handle (Class):
    Object_Handle_Struct *aclazz_h = (Object_Handle_Struct *)aclazz;
    Class *clss = (Class *)aclazz_h->java_reference;

	if (!clss->is_array) {
		ThrowNew_Quick (env, "java/lang/IllegalArgumentException", 0);
		return;
	}
} // VerifyArray


void VerifyArrayAndBounds (JNIEnv* env, jarray array, jint index)
{
	VerifyArray (env, array);

	// Check if any exceptions have been thrown:
	jthrowable exception = ExceptionOccurred (env);
	if (exception) {
		// Return without clearing exception; the exception is "rethrown".
		return;   
	}

	jint length = GetArrayLength (env, array);
	if ((index < 0) || (index >= length)) {
		ThrowNew_Quick (env, "java/lang/ArrayIndexOutOfBoundsException", 0);
		return;
	}
} // VerifyArrayAndBounds


char GetComponentSignature (JNIEnv *env, jarray array)
{
	jclass aclazz = GetObjectClass (env, array);

	// Acquire handle to internal class handle (Class):
    Object_Handle_Struct *aclazz_h = (Object_Handle_Struct *)aclazz;
    Class *clss = (Class *)aclazz_h->java_reference;

	return clss->name->bytes[1]; // return component first character
}


Method* LookupDeclaredMethod (Class *clss, const char *mname, const char *msig)
{
    String *method_name = ORP_Global_State::loader_env->string_pool.lookup(mname);
   
	int len = strlen (msig);
    Method* m = 0;
	for (unsigned i =0; i < clss->n_methods; i++) {   // for each method
		m = &clss->methods[i];
		const char* sig = m->get_descriptor();
		if (m->get_name() == method_name     &&      // if names and signatures
			strncmp (msig, sig, len) == 0) {         // (excluding return types) match,   
			return m;                                // return method
		}
	}

    return (Method*)0;                                   // method not found
} // LookupDeclaredMethod


Field* GetAndVerifyField (JNIEnv *env, jobject field, jobject obj)
{
    Field *f = GetField (env, field);
	if (!f) return (Field*)0;

	// If field is not static, check if the provided object is of the 
	// right type, i.e., is an instance of the class declaring the
	// field.
	if (!f->is_static()) {
		if (!obj) {
			ThrowNew_Quick (env, "java/lang/NullPointerException", 0);
			return (Field*)0;
		}
		jclass oclazz = GetObjectClass (env, obj);
		if (!IsInstanceOf(env, obj, oclazz)) {
			ThrowNew_Quick (env, "java/lang/IllegalArgumentException", 0);
			return (Field*)0;
		}
	}

#if 0 // whether a field is accessible depends on the context in which it is accessed
      // assume if we got this far, it does not matter if it is declared pubic or not
      // visibility depends on the context as opposed to the declaration
      // perhaps there needs to be a check somewhere else,  it probably not
      // appropriate here
	if (!f->is_public()) {
		// Access the field's internal class handle (Class):
	    jclass clazz = GetFieldClass (env, field);
		if (!clazz) return (Field*)0;

		Object_Handle_Struct *clazz_h = (Object_Handle_Struct *)clazz;
		Class *fcl = (Class *)clazz_h->java_reference;
		ThrowNew_Quick (env, "java/lang/IllegalAccessException", fcl->name->bytes);
	}
#endif // if 0

	return f;
} // GetAndVerifyField


jchar* CToUnicode (JNIEnv *env, const char* cchars, jsize len)
{
	assert (cchars);
	if (len == 0) {
		return (jchar*)0;
	}

	jchar* jchars = (jchar *)malloc(sizeof(jchar) * len);
	char* j_ptr = (char*)jchars;
	const char* c_ptr = cchars;

    for (int ii = 0; ii < len; ii++) {
        *j_ptr++ = *c_ptr++;
		*j_ptr++ = 0;             // 16 bit characters!
    }

	return jchars;
} // CToUnicode


char* UnicodeToC (JNIEnv *env, const jchar* jchars, jsize len)
{
	if (len == 0) {
		return (char*)0;
	}

    char*  cstr = (char *)malloc (len + 1);
	cstr[len] = 0;

	const char* j_ptr = (char*)jchars;
	char* c_ptr = cstr;

    for (int ii = 0; ii < len; ii++) {
		*j_ptr++;           // 16 bit characters!
        *c_ptr++ = *j_ptr++;
    }

	return cstr;
} // UnicodeToC


jboolean PrimitiveIsAssignableFrom (const char sig1, const char sig2)
{
	switch (sig2) {
	case 'Z':
		if (sig1 != 'Z') return (jboolean)0;
		break;
	case 'B':
		if ((sig1 != 'B') && (sig1 != 'S') && (sig1 != 'I') &&
			(sig1 != 'J') && (sig1 != 'F') && (sig1 != 'D')) {
			return (jboolean)0;
		}
		break;
	case 'C':
		if ((sig1 != 'C') && (sig1 != 'I') && (sig1 != 'J') &&
			(sig1 != 'F') && (sig1 != 'D')) {
			return (jboolean)0;
		}
		break;
	case 'S':
		if ((sig1 != 'S') && (sig1 != 'I') && (sig1 != 'J') &&
			(sig1 != 'F') && (sig1 != 'D')) {
			return (jboolean)0;
		}
		break;
	case 'I':
		if ((sig1 != 'I') && (sig1 != 'J') && (sig1 != 'F') &&
			(sig1 != 'D')) {
			return (jboolean)0;
		}
		break;
	case 'J':
		if ((sig1 != 'J') && (sig1 != 'F') && (sig1 != 'D')) {
			return (jboolean)0;
		}
		break;
	case 'F':
		if ((sig1 != 'F') && (sig1 != 'D')) {
			return (jboolean)0;
		}
		break;
	case 'D':
		if (sig1 != 'D') return (jboolean)0;
		break;
	case 'L':
	case '[':
		return (jboolean)0;
		break;
	default:
		assert (0);
	}	

	return (jboolean)1;
}


jobject WrapPrimitiveValue (JNIEnv *env, jvalue value, char sig)
{
	jvalue args[1];
	jmethodID method_id;
	jobject retobj;
	jclass clazz;

	switch (sig) {
	case 'Z':
		args[0].z = value.z;
		
		// Allocate java/lang/Boolean object:
		clazz = FindClass (env, "java/lang/Boolean");	
		method_id = GetMethodID (env, clazz, "<init>",	"(Z)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'B': 
		args[0].b = value.b;
				
		// Allocate java/lang/Byte object
		clazz = FindClass (env, "java/lang/Byte");	
		method_id = GetMethodID (env, clazz, "<init>",	"(B)V");
			
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'C':
		args[0].c = value.c;
		
		// Allocate java/lang/Character object:
		clazz = FindClass (env, "java/lang/Character");	
		method_id = GetMethodID (env, clazz, "<init>",	"(C)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'S':
		args[0].s = value.s;
		
		// Allocate java/lang/Short object:
		clazz = FindClass (env, "java/lang/Short");	
		method_id = GetMethodID (env, clazz, "<init>",	"(S)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'I':
		args[0].i = value.i;
		
		// Allocate java/lang/Integer object:
		clazz = FindClass (env, "java/lang/Integer");	
		method_id = GetMethodID (env, clazz, "<init>",	"(I)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'J':
		args[0].j = value.j;
		
		// Allocate java/lang/Long object:
		clazz = FindClass (env, "java/lang/Long");	
		method_id = GetMethodID (env, clazz, "<init>",	"(J)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'F':
		args[0].f = value.f;
		
		// Allocate java/lang/Float object:
		clazz = FindClass (env, "java/lang/Float");	
		method_id = GetMethodID (env, clazz, "<init>",	"(F)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'D':
		args[0].d = value.d;
		
		// Allocate java/lang/Double object:
		clazz = FindClass (env, "java/lang/Double");	
		method_id = GetMethodID (env, clazz, "<init>",	"(D)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	default:
		assert (0);
	}

	return retobj;
} // WrapPrimitiveValue

//transform J_Value type to jobject
jobject WrapPrimitiveValueJ (JNIEnv *env, J_Value value, char sig)
{
	jvalue args[1];
	jmethodID method_id;
	jobject retobj;
	jclass clazz;

	switch (sig) {
	case 'Z':
		args[0].z = value.z;
		
		// Allocate java/lang/Boolean object:
		clazz = FindClass (env, "java/lang/Boolean");	
		method_id = GetMethodID (env, clazz, "<init>",	"(Z)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'B': 
		args[0].b = value.b;
				
		// Allocate java/lang/Byte object
		clazz = FindClass (env, "java/lang/Byte");	
		method_id = GetMethodID (env, clazz, "<init>",	"(B)V");
			
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'C':
		args[0].c = value.c;
		
		// Allocate java/lang/Character object:
		clazz = FindClass (env, "java/lang/Character");	
		method_id = GetMethodID (env, clazz, "<init>",	"(C)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'S':
		args[0].s = value.s;
		
		// Allocate java/lang/Short object:
		clazz = FindClass (env, "java/lang/Short");	
		method_id = GetMethodID (env, clazz, "<init>",	"(S)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'I':
		args[0].i = value.i;
		
		// Allocate java/lang/Integer object:
		clazz = FindClass (env, "java/lang/Integer");	
		method_id = GetMethodID (env, clazz, "<init>",	"(I)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'J':
		args[0].j = value.j;
		
		// Allocate java/lang/Long object:
		clazz = FindClass (env, "java/lang/Long");	
		method_id = GetMethodID (env, clazz, "<init>",	"(J)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'F':
		args[0].f = value.f;
		
		// Allocate java/lang/Float object:
		clazz = FindClass (env, "java/lang/Float");	
		method_id = GetMethodID (env, clazz, "<init>",	"(F)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'D':
		args[0].d = value.d;
		
		// Allocate java/lang/Double object:
		clazz = FindClass (env, "java/lang/Double");	
		method_id = GetMethodID (env, clazz, "<init>",	"(D)V");
		
		// Construct new Java object and initialize it with the field value: 
		retobj = NewObjectA (env, clazz, method_id, args);
		break;
	case 'V': //add the handling of 'V'
		retobj = NULL; 
		break;
	default:
		assert (0);
	}

	return retobj;
} // WrapPrimitiveValueJ

jvalue UnwrapWrapperObject (JNIEnv *env, jobject wobj, char sig)
{
	jvalue value;
	jfieldID value_id;

	switch (sig) { // Value argument signature
	case 'Z':
		// Get the fielID of the value field of a Boolean object:
		value_id = GetFieldID_Quick(env, "java/lang/Boolean", "value", "Z");
		value.z = GetBooleanField (env, wobj, value_id);
		break;
	case 'B':
		// Get the fielID of the value field of a Byte object:
		value_id = GetFieldID_Quick(env, "java/lang/Byte", "value", "B");
		value.b = GetByteField (env, wobj, value_id);
		break;	
	case 'C':
		// Get the fielID of the value field of a Character object:
		value_id = GetFieldID_Quick(env, "java/lang/Character", "value", "C");
		value.c = GetCharField (env, wobj, value_id);
		break;
	case 'S':
		// Get the fielID of the value field of a Short object:
		value_id = GetFieldID_Quick(env, "java/lang/Short", "value", "S");
		value.s = GetShortField (env, wobj, value_id);
		break;
	case 'I':
		// Get the fielID of the value field of a Integer object:
		value_id = GetFieldID_Quick(env, "java/lang/Integer", "value", "I");
		value.i = GetIntField (env, wobj, value_id);
		break;
	case 'J':
		// Get the fielID of the value field of a Long object:
		value_id = GetFieldID_Quick(env, "java/lang/Long", "value", "J");
		value.j = GetLongField (env, wobj, value_id);
		break;	
	case 'F':
		// Get the fielID of the value field of a Float object:
		value_id = GetFieldID_Quick(env, "java/lang/Float", "value", "F");
		value.f = GetFloatField (env, wobj, value_id);
		break;
	case 'D':
		// Get the fielID of the value field of a Double object:
		value_id = GetFieldID_Quick(env, "java/lang/Double", "value", "D");
		value.d = GetDoubleField (env, wobj, value_id);
		break;
	default:
		assert (0);
	}

	return value;
} // UnwrapWrapperObject

//transform jobject into J_Value
J_Value UnwrapWrapperObjectJ (JNIEnv *env, jobject wobj, char sig)
{
	J_Value value;
	jfieldID value_id;

	switch (sig) { // Value argument signature
	case 'Z':
		// Get the fielID of the value field of a Boolean object:
		value_id = GetFieldID_Quick(env, "java/lang/Boolean", "value", "Z");
		value.z = GetBooleanField (env, wobj, value_id);
		break;
	case 'B':
		// Get the fielID of the value field of a Byte object:
		value_id = GetFieldID_Quick(env, "java/lang/Byte", "value", "B");
		value.b = GetByteField (env, wobj, value_id);
		break;	
	case 'C':
		// Get the fielID of the value field of a Character object:
		value_id = GetFieldID_Quick(env, "java/lang/Character", "value", "C");
		value.c = GetCharField (env, wobj, value_id);
		break;
	case 'S':
		// Get the fielID of the value field of a Short object:
		value_id = GetFieldID_Quick(env, "java/lang/Short", "value", "S");
		value.s = GetShortField (env, wobj, value_id);
		break;
	case 'I':
		// Get the fielID of the value field of a Integer object:
		value_id = GetFieldID_Quick(env, "java/lang/Integer", "value", "I");
		value.i = GetIntField (env, wobj, value_id);
		break;
	case 'J':
		// Get the fielID of the value field of a Long object:
		value_id = GetFieldID_Quick(env, "java/lang/Long", "value", "J");
		value.j = GetLongField (env, wobj, value_id);
		break;	
	case 'F':
		// Get the fielID of the value field of a Float object:
		value_id = GetFieldID_Quick(env, "java/lang/Float", "value", "F");
		value.f = GetFloatField (env, wobj, value_id);
		break;
	case 'D':
		// Get the fielID of the value field of a Double object:
		value_id = GetFieldID_Quick(env, "java/lang/Double", "value", "D");
		value.d = GetDoubleField (env, wobj, value_id);
		break;
	default:
		assert (0);
	}

	return value;
} // UnwrapWrapperObjectA


jclass GetClassComponentType (JNIEnv* env, jclass clazz) 
{
	// Acquire handle to internal class handle (Class):
    Object_Handle_Struct *clazz_h = (Object_Handle_Struct *)clazz;
    Class *clss = (Class *)clazz_h->java_reference;

	// Return null if this class does not represent an array type:
	if (!clss->is_array) return (jclass)0;

	const char* componentName = clss->name->bytes + 1;
	jclass compclazz = SignatureToClass (env, componentName);

	// The code above does not depend on the existence of the 
	// array_element_class field in the internal class Class. If it is known that
	// this field will be maintained, and/or performance becomes an issue,
	// the line below can be used instead:
	// 
	// jclass compclazz = FindClass (env, SignatureToName (clss->array_element_class->name->bytes));

    return compclazz;
} // GetClassComponentType



//Checks if the object is null.

jboolean IsNullRef(jobject jobj)
{
    Object_Handle h = (Object_Handle)jobj;

    orp_disable_gc();       //---------------------------------v

    void *cl = h->java_reference;
	
	jboolean ret = (cl == NULL) ? true : false;
	cl = 0;

    orp_enable_gc();        //---------------------------------^

	return ret;
}

jclass FindClassWithClassLoader(const char *name, Class_Loader *loader)
{
	String *str = ORP_Global_State::loader_env->string_pool.lookup(name);
	Loader_Exception exc;
	orp_disable_gc();
	Class *c = class_load_verify_prepare_by_loader(
										ORP_Global_State::loader_env,
										str,
										loader,
										&exc);
	orp_enable_gc();
	assert(c && !get_current_thread_exception());
	if(!strcmp(name, "javax/ejb/EJBHome"))
		int i = 0;
	jclass clazz = orp_create_local_object_handle();
	((Object_Handle)clazz)->java_reference = (Java_java_lang_Object*)c;

	return clazz;
}

jclass FindClassWithContextClassLoader(const char *name)
{
	void *ip = get_last_j2n_saved_ip();
	JIT_Specific_Info *jit_info = methods.find(ip);
	assert(jit_info);
	Class *caller = jit_info->get_method()->get_class();
	/*
	char *clname = (char*)caller->name->bytes;
	if(!strcmp(clname, "java/lang/Class") ||
		strstr(clname, "java/lang/reflect/") == clname){
		return FindClassWithContextClassLoader(name, 2);
	}
	*/	
	return FindClassWithClassLoader(name, caller->class_loader);
}

jclass FindClassWithContextClassLoader(const char *name, int depth)
{
	Class **class_info = (Class **) malloc (sizeof(Class *) * depth);
    assert(class_info);

    unsigned stack_depth = orp_get_stack_depth_from_native_new(class_info, 0, depth);

	if(stack_depth <= 0){
		return FindClassWithContextClassLoader(name);
	}

	Class_Loader *class_loader = class_info[depth-1]->class_loader;

	free(class_info);

	return FindClassWithClassLoader(name, class_loader);
}
