// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/finalize.cpp,v 1.3 2001/12/07 00:16:00 xli18 Exp $
//


#include "platform.h"
#include <stdio.h>
#include <assert.h>

#include "object_layout.h"
#include "orp_types.h"
#include "Class.h"
#include "orp_utils.h"
#include "environment.h"
#include "ini.h"
#include "exceptions.h"
#include "nogc.h"
#include "jit_runtime_support.h"
#include "orp_synch.h"

#include "finalize.h"

#ifdef USE_GC_DLL
__declspec(dllexport) 
#endif
int running_finalizers_deferred = 0;


//
// This code holds the logic that deals with finalization as well as weak/soft/phantom 
// references. Finalization runs arbitrary code that can include synchronization logic.
// If finalizers were run when the garbage collector has stopped the world then deadlock
// becomes a problem.
//
// There are two different times that finalizers can be run. The first is to have a
// seperate thread that receives the objects that need to be finalized and executes them
// sometime after the GC has completed. This is probable what was intended by the 
// designers of the finalization and weak/soft/phantom reference features of the language. :) 
//
// The second approach that can be taken is to run the finalization code immediately 
// after the gc is complete prior to resuming the thread that caused the gc to happen. 
// This is allowed by the language design. This is what we do.
//
// We chose this mechanism for performance reasons. The problem with a design 
// that assumes that there is a active finalization thread is that the system 
// now has two threads running. If the application is a single threaded application 
// then the system can optimize the synchronization logic and we do just that. 
// Currently (1999) a lot of applications are single threaded and any 
// synchronization optimizations that are possible makes a difference.
//

class Objects_To_Finalize
{
    Java_java_lang_Object **objects;
    unsigned capacity;
    volatile unsigned num_objects;

    void reallocate(unsigned new_capacity);
public:
    Objects_To_Finalize();
    void add_object(Java_java_lang_Object *p_obj);
    void enumerate_for_gc();
    void run_finalizers();

    void notify_pending_reference_queues();

}; //Objects_To_Finalize



Objects_To_Finalize::Objects_To_Finalize()
{
    objects     = 0;
    capacity    = 0;
    num_objects = 0;
    reallocate(128);
} //Objects_To_Finalize::Objects_To_Finalize



void Objects_To_Finalize::reallocate(unsigned new_capacity)
{
    Java_java_lang_Object **new_table =
        (Java_java_lang_Object **)gc_malloc_fixed_data_C(new_capacity * sizeof(Java_java_lang_Object *));
    assert(new_table);
    assert(num_objects <= capacity);
    memcpy(new_table, objects, num_objects * sizeof(Java_java_lang_Object *));
    if(objects)
        free(objects);
    objects = new_table;
    capacity = new_capacity;
} //Objects_To_Finalize::reallocate



void Objects_To_Finalize::add_object(Java_java_lang_Object *p_obj)
{
    if(num_objects >= capacity) {
        reallocate(2 * capacity);
    }
    assert(num_objects < capacity);
    objects[num_objects++] = p_obj;
} //Objects_To_Finalize::add_object



void Objects_To_Finalize::enumerate_for_gc()
{
    for(unsigned i = 0; i < num_objects; i++) {
        orp_enumerate_root_reference(0, (void **)&(objects[i]));
    }
} //Objects_To_Finalize::enumerate_for_gc



void Objects_To_Finalize::run_finalizers()
{
    while(num_objects > 0) {
        // We reload the value of num_objects in every iteration.
        // This is required, because the execution of a finalizer may
        // cause GC which in turn may add new objects to the finalize list.
        unsigned idx = num_objects - 1;
        Class *clss = objects[idx]->vt->clss;
        Method *finalize_meth =
            class_lookup_method_recursive(clss, ORP_Global_State::loader_env->Finalize_Signature);
        assert(finalize_meth);
        J_Value arg0;
        arg0.r = (void *)objects[idx];
        num_objects--;
        orp_execute_java_method_array(finalize_meth, 0, &arg0);
        if(get_current_thread_exception()) {
#ifdef _DEBUG
            // (mjc 980902) This message shouldn't be really printed, but I put
            // it in to make sure that we do the right thing in this case.
            // Remove when we hit this path and test out approach.
            orp_cout << "Uncaught exception while running a finalizer of class "
                     << clss->name->bytes
                     << endl;
#endif
            clear_current_thread_exception();
        }
    }
} //Objects_To_Finalize::run_finalizers

void Objects_To_Finalize::notify_pending_reference_queues()
{
#ifdef GC_TRACE_WEAK_REF
        orp_cout << "Notifying pending ReferenceQueues " << endl;
#endif

    while(num_objects > 0) {
        // We reload the value of num_objects in every iteration.
        // This is required, because the execution of other threads may
        // cause GC which in turn may add new objects to the list.
        unsigned idx = num_objects - 1;

#ifdef GC_TRACE_WEAK_REF
        orp_cout << "Notifying ReferenceQueue " << objects[idx] << endl;
#endif

        orp_monitor_enter (objects[idx]);

        Class *jlo = ORP_Global_State::loader_env->JavaLangObject_Class;
        Method *java_lang_object_notify = object_lookup_method(jlo, "notify", "()V");
        orp_execute_java_method(java_lang_object_notify, 0, objects[idx]);

        orp_monitor_exit (objects[idx]);

        num_objects--; 

        if(get_current_thread_exception()) {
#ifdef _DEBUG
            // (mjc 980902) This message shouldn't be really printed, but I put
            // it in to make sure that we do the right thing in this case.
            // Remove when we hit this path and test out approach.
            orp_cout << "Uncaught exception while notifying reference queues. "
                     << endl;
#endif
            clear_current_thread_exception();
        }
    }
} //Objects_To_Finalize::notify_reference_queues


static Objects_To_Finalize objects_to_finalize;

void orp_finalize_object(Java_java_lang_Object *p_obj)
{

    objects_to_finalize.add_object(p_obj);

} //orp_finalize_object



void orp_run_pending_finalizers()
{
    bool was_gc_enabled = orp_disable_gc();

    objects_to_finalize.run_finalizers();

    if (was_gc_enabled)
        orp_enable_gc();

} //orp_run_pending_finalizers



void orp_enumerate_objects_to_be_finalized()
{
    objects_to_finalize.enumerate_for_gc();
} //orp_enumerate_objects_to_be_finalized


// -- Code to deal with Reference Queues that need to be notified.

static Objects_To_Finalize reference_queues_to_notify;

void orp_notify_reference_queue(Java_java_lang_Object *p_obj)
{

#ifdef GC_TRACE_WEAK_REF
        orp_cout << "In orp_notify_reference_queue " << p_obj << endl;
#endif

    reference_queues_to_notify.add_object(p_obj);
} //orp_finalize_object

void orp_notify_pending_reference_queues()
{ 
    reference_queues_to_notify.notify_pending_reference_queues();
} //orp_notify_pending_reference_queues

void orp_enumerate_reference_queues_to_be_notified()
{
    reference_queues_to_notify.enumerate_for_gc();
} //orp_enumerate_objects_to_be_finalized
