// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_net_PlainSocketImpl.cpp,v 1.8 2001/12/12 06:30:03 xli18 Exp $
//



#include "platform.h"
#include <assert.h>
#include <errno.h>

#ifdef ORP_NT 
#include <winsock2.h>
#endif

#ifdef ORP_POSIX
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h> 
#include <sys/ioctl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <string.h>
#endif

#include <jni.h>
#include <gnu_classpath_jni_utils.h>


#include "java_net_PlainSocketImpl_common.h"
#include "java_net_PlainSocketImpl.h"

#ifdef OBJECT_LOCK_V2
#include "platform_utils_olv2.h"  //used to replace strerror(error) with socket_strerror(error)
#else
#include "platform_utils.h"
#endif

#ifdef ORP_POSIX
extern int WSAGetLastError ();
#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#define SOCKET int
#endif

/****
testSocketReadability: timeout facility for ServerSocket.accept or Socket.read
jenv: 
sock: SOCKET handle
jsock: Java Socket object handle, only useful in Linux
*****/
BOOL testSocketReadability(JNIEnv *jenv, SOCKET sock, jobject jsock)
{
	int ret, timeout; 
	int size = sizeof(timeout);
#ifdef ORP_POSIX
	//XXX get timeout value from jsock
	timeout = 0;
#else
	ret = getsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, (char *)&timeout, &size);
#endif
	if (ret != SOCKET_ERROR && timeout > 0) {
		struct timeval tv;
		tv.tv_sec = timeout / 1000;
		tv.tv_usec = (timeout % 1000)*1000L;
		fd_set rfds;
		FD_ZERO(&rfds);
		FD_SET(sock, &rfds);
		ret = select(sock+1, &rfds, NULL, NULL, &tv);
		if(ret < 0){
			int  error = WSAGetLastError();
#ifdef _DEBUG
			printf("java_net_PlainSocketImpl_select(): %s\n", socket_strerror(error));
#endif
			throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
			return FALSE;
		}else
			if(ret == 0){ //timeout
#ifdef _DEBUG
				printf("java_net_PlainSocketImpl: Socket select timeout!\n");
#endif
				throw_exception_from_jni(jenv, "java/io/InterruptedIOException", "Timeout on ServerSocket.accept() or Socket.read()");
				return FALSE;
			}
	}
	return TRUE;
}

/*
 * Class:     java_net_PlainSocketImpl
 * Method:    accept
 * Signature: (Ljava/net/SocketImpl;)V
 */

JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_accept
  (JNIEnv *jenv, jobject jobj, jobject psock)
{
	struct sockaddr socketStruct; 

#ifdef ORP_POSIX
    socklen_t nameLen = sizeof(socketStruct);
#else
    int nameLen = sizeof(socketStruct);
#endif
	memset(&socketStruct, 0, nameLen);

	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);
	
	if (sock < 0) { 
#ifdef _DEBUG
		printf("Try to accept on a closed/invalid socket.\n");
#endif
		throw_exception_from_jni(jenv, "java/io/IOException", "Socket already closed or invalid.");
		return;
	}

	if(!testSocketReadability(jenv, sock, jobj))
		return;
	
  	SOCKET remote_socket = accept(sock, &socketStruct, &nameLen);

	if (remote_socket == INVALID_SOCKET) { 
		int error = WSAGetLastError ();
#ifdef _DEBUG
        printf("java_net_PlainSocketImpl_accept(): %s\n", socket_strerror(error));
#endif
        throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
        return;
    }

    // DO NOT LINGER....

	struct linger linger_struct; 
	memset(&linger_struct, 0, sizeof(linger_struct));
	linger_struct.l_onoff = 0;      //  Do not linger.

#ifdef ORP_NT
	int len = sizeof(linger_struct);
#else
	socklen_t len = sizeof(linger_struct);
#endif
	uint32 status = setsockopt(sock, SOL_SOCKET, SO_LINGER, (const char *) &linger_struct, len);

	if (status == SOCKET_ERROR) {
       	int error = WSAGetLastError ();    
#if _DEBUG
       	printf("java_net_PlainSocketImpl_accept(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
       	return;
    }
	
	// Set the native_fd field of psock to remote_socket.
	set_socket_native_fd(jenv, clazz, psock, remote_socket);

	// Create a new FileDescriptor object with "sock" and point "fd" to it.
	//XXX:jobject fd_object = create_new_FD(jenv, sock);
	jobject fd_object = create_new_FD(jenv, remote_socket);
	set_socket_fd(jenv, clazz, psock, fd_object);

	struct sockaddr_in *inaddr = (struct sockaddr_in*)&socketStruct;
	//Set localport to the localport of ServerSocket
	set_socket_localport(jenv, clazz, psock, get_socket_localport(jenv, clazz, jobj));
	//Set port to the port of remote socket 
	//XXX:set_socket_port(jenv, clazz, psock, inaddr->sin_port);
	set_socket_port(jenv, clazz, psock, ntohs(inaddr->sin_port));
	
	jobject inaobj = create_inet_addr(jenv, inaddr->sin_addr.s_addr);
	
	jfieldID address_field = jenv->GetFieldID(clazz, "address", "Ljava/net/InetAddress;");
	//DEBUG: jobject aobj = jenv->GetObjectField(psock, address_field);
	jenv->SetObjectField(psock, address_field, inaobj);

} // Java_java_net_PlainSocketImpl_accept



/*
 * Class:     java_net_PlainSocketImpl
 * Method:    bind
 * Signature: (Ljava/net/InetAddress;I)V
 */
JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_bind
  (JNIEnv *jenv, jobject jobj, jobject INET_address, jint localPortNumber)
{
	bind_socket(jenv, jobj, INET_address, localPortNumber, 1);

} // Java_java_net_PlainSocketImpl_bind


  
/*
 * Class:     java_net_PlainSocketImpl
 * Method:    close
 * Signature: ()V
 */


JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_close
  (JNIEnv *jenv, jobject jobj)
{
	close_socket(jenv, jobj);
} // Java_java_net_PlainSocketImpl_close


/*
 * Class:     java_net_PlainSocketImpl
 * Method:    connect
 * Signature: (Ljava/net/InetAddress;I)V
 */

JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_connect
  (JNIEnv *jenv, jobject jobj, jobject address, jint portNumber)
{
	if (portNumber == -1)
		portNumber = 0;		// Default value.

	// Get the 32-bit value representing the IP address to connect to.
	uint32 net_address = get_net_address(jenv, address);
	
	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);

	if (sock < 0) { 
#ifdef _DEBUG
		printf("Try to connect on a closed/invalid socket.\n");
#endif
		throw_exception_from_jni(jenv, "java/io/IOException", "Socket already closed or invalid.");
		return;
	}

	// Store the port number and IP address representation.
    struct sockaddr socketStruct;
	memset(&socketStruct, 0, sizeof(socketStruct));
	socketStruct.sa_family = AF_INET;
    uint16 *p16 = (uint16 *)&(socketStruct.sa_data[0]);
	*p16 = htons((unsigned short)portNumber);
	uint32 *p32 = (uint32 *)&(socketStruct.sa_data[2]);
    *p32 = net_address;
  
	// Set the required socket options.
        
	// DO NOT LINGER....

	struct linger linger_struct; 
	memset(&linger_struct, 0, sizeof(linger_struct));
	linger_struct.l_onoff = 0;      //  Do not linger.

#ifdef ORP_NT
	int len = sizeof(linger_struct);
#else
	socklen_t len = sizeof(linger_struct);
#endif
	uint32 status = setsockopt(sock, SOL_SOCKET, SO_LINGER, (const char *) &linger_struct, len);

	if (status == INVALID_SOCKET) {
       	int error = WSAGetLastError (); 
#ifdef _DEBUG
       	printf("java_net_PlainSocketImpl_connect(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
       	return;
    }

	// Try to connect now.
	if (connect(sock, &socketStruct, sizeof(socketStruct)) == SOCKET_ERROR) {
		// Error
	   	int error = WSAGetLastError ();
#ifdef _DEBUG
		printf("java_net_PlainSocketImpl_connect(): %s\n", socket_strerror(error));
#endif
	   	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
	   	return;
	}

	// Retrieve the local name of the socket next.
#ifdef ORP_NT
	int struct_size = sizeof(socketStruct);
#else 
	socklen_t struct_size;
#endif
	if (getsockname(sock, &socketStruct, &struct_size) == SOCKET_ERROR) {
		//Error
	   	int error = WSAGetLastError ();
#ifdef _DEBUG
	   	printf("java_net_PlainSocketImpl_connect(): %s\n", socket_strerror(error));
#endif
	   	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
	   	return;
	}

	// Create a new FileDescriptor object and point "sock" to it.
	jobject fd_object = create_new_FD(jenv, 0);
	set_socket_fd(jenv, clazz, jobj, fd_object);

	// Set the localport value correctly
	set_socket_localport(jenv, clazz, jobj, ntohs(*p16));

	// Call getpeername() to retrieve the name of the peer to which the socket is now connected.
	if (getpeername(sock, &socketStruct, &struct_size) == SOCKET_ERROR) { 
		//Error
	  	int error = WSAGetLastError ();
#ifdef _DEBUG
	   	printf("java_net_PlainSocketImpl_connect(): %s\n", socket_strerror(error));
#endif
	   	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
	   	return;
	}
	// Create a new InetAddress object using the return value from getpeername().
	//XXX: jobject inet_addr = create_inet_addr(jenv, ntohl(*p32));
	jobject inet_addr = create_inet_addr(jenv, *p32); 
	assert(inet_addr);
	jfieldID address_id = jenv->GetFieldID(clazz, "address", "Ljava/net/InetAddress;");
	assert(address_id);
	jenv->SetObjectField(jobj, address_id, inet_addr);

	// Now, set the "port' value of the socket object.
	//XXX: set_socket_port(jenv, clazz, jobj, ntohs(*p16));
	set_socket_port(jenv, clazz, jobj, portNumber);

} // Java_java_net_PlainSocketImpl_connect


/*
 * Class:     java_net_PlainSocketImpl
 * Method:    create
 * Signature: (Z)V
 */

JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_create
  (JNIEnv *jenv, jobject jobj, jboolean is_stream)
{
	create_socket(jenv, jobj, is_stream);
} // Java_java_net_PlainSocketImpl_create



/*
 * Class:     java_net_PlainSocketImpl
 * Method:    listen
 * Signature: (I)V
 */

JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_listen
  (JNIEnv *jenv, jobject jobj, jint queue_length)
{
	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);

	if (sock == INVALID_SOCKET) { 
#ifdef _DEBUG
		printf("Try to listen on a closed/invalid socket.\n");
#endif
		throw_exception_from_jni(jenv, "java/io/IOException", "Socket already closed or invalid.");
		return;
	}

	// Set the queue length passed in.
	uint32 status = listen(sock, queue_length);

	if (status == SOCKET_ERROR) {
       	int error = WSAGetLastError ();
#ifdef _DEBUG
       	printf("java_net_PlainSocketImpl_socketListen(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
       	return;
    }
} // Java_java_net_PlainSocketImpl_listen


  
  
/*
 * Class:     java_net_PlainSocketImpl
 * Method:    read
 * Signature: ([BII)I
 */

JNIEXPORT jint JNICALL Java_java_net_PlainSocketImpl_read
  (JNIEnv *jenv, jobject jobj, jbyteArray recvBuf, jint offset, jint length)
{
	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);

	if (sock == INVALID_SOCKET) { 
#ifdef _DEBUG
		printf("Try to read on a closed/invalid socket.\n");
#endif
		throw_exception_from_jni(jenv, "java/io/IOException", "Socket already closed or invalid.");
		return -1;
	}
	// Extract the actual byte array to be read into.
	jboolean is_copy;
	jbyte *bytes = jenv->GetByteArrayElements(recvBuf, &is_copy);
   	assert(bytes);

	if(!testSocketReadability(jenv, sock, jobj))
		return -1;

	int ret;
	// Read in the data using "recv/recvfrom()" starting at the given offset
	// t = recvfrom(sock, (char *) bytes + offset, length, 0, 0, 0);
	ret = recv(sock, (char *) bytes + offset, length, 0);
#if 0
	printf("Want to read %d bytes: ************\n", length);
	printf("read: %s\n", bytes + offset);
	printf("read: ***************************\n\n\n");
#endif
	
	if (ret == SOCKET_ERROR) {
		int error = WSAGetLastError ();
#ifdef _DEBUG
       	printf("java_net_PlainSocketImpl_read(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
       	return -1;
	}
	jenv->ReleaseByteArrayElements(recvBuf, bytes, 0);
	return ret;

} // Java_java_net_PlainSocketImpl_read




/*
 * Class:     java_net_PlainSocketImpl
 * Method:    write
 * Signature: ([BII)V
 */

JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_write
  (JNIEnv *jenv, jobject jobj, jbyteArray sendBuf, jint offset, jint length)
{
	assert(offset >= 0);
	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);

	if (sock == SOCKET_ERROR) { 
#ifdef _DEBUG
		printf("Try to write on a closed/invalid socket.\n");
#endif
		throw_exception_from_jni(jenv, "java/io/IOException", "Socket already closed or invalid.");
		return;
	}
	// Extract the actual byte array to send from.
	jboolean is_copy;
	jbyte *bytes = jenv->GetByteArrayElements(sendBuf, &is_copy);
   	assert(bytes);

#if 0
	printf("write: ***************************\n");
	printf("write: %s\n", bytes + offset);
	printf("write: ***************************\n\n\n");
#endif

	if (send(sock, (const char *) bytes + offset, length, 0) == SOCKET_ERROR) { 
		int error = WSAGetLastError ();
#ifdef _DEBUG
       	printf("java_net_PlainSocketImpl_write(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
  		return;
	}
	jenv->ReleaseByteArrayElements(sendBuf, bytes, JNI_ABORT);
	return;
} // Java_java_net_PlainSocketImpl_write



/*
 * Class:     java_net_PlainSocketImpl
 * Method:    setOption
 * Signature: (ILjava/lang/Object;)V
 */


JNIEXPORT void JNICALL Java_java_net_PlainSocketImpl_setOption
  (JNIEnv *jenv, jobject jobj, jint option_type, jobject value)
{
	socketSetOption(jenv, jobj, option_type, value);
} // Java_java_net_PlainSocketImpl_setOption



/*
 * Class:     java_net_PlainSocketImpl
 * Method:    getOption
 * Signature: (I)Ljava/lang/Object;
 */

 JNIEXPORT jobject JNICALL Java_java_net_PlainSocketImpl_getOption
  (JNIEnv *jenv, jobject jobj, jint option_type)
{
	return socketGetOption(jenv, jobj, option_type);
} // Java_java_net_PlainSocketImpl_getOption


/*
 * Class:     java_net_PlainSocketImpl
 * Method:    available
 * Signature: (V)I
 */
JNIEXPORT jint JNICALL Java_java_net_PlainSocketImpl_available
	(JNIEnv *jenv, jobject jobj)
{
	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);
#ifdef ORP_POSIX
	jint a;
	if(ioctl(sock, FIONREAD, &a))
		return 0;
	return a;
#else
	unsigned long a;
	if(ioctlsocket(sock, FIONREAD, &a))
		return 0;
	return (jint)a;
#endif
}