#include "platform.h"
#include "orp_utils.h"

#include "jni.h"
#include "native_utils.h"
#include "jni_utils.h"
#include "jni_direct.h"
#include "gnu_classpath_jni_utils.h"

#include "java_lang_PlatformProcess.h"

#ifdef ORP_POSIX
#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#endif

inline HANDLE getProcessHandle(JNIEnv *jenv, jobject process){
	jclass clazz = GetObjectClass(jenv, process);
	jfieldID handle_field = GetFieldID(jenv, clazz, "native_handle", "I");
	assert(handle_field);
	jint handle = GetIntField(jenv, process, handle_field);
	return (HANDLE)handle;
}

inline void setProcessExitValue(JNIEnv *jenv, jobject process, int exit_value){
    jclass clazz = GetObjectClass(jenv, process);
    
    jfieldID value_field = GetFieldID(jenv, clazz, "exit_value", "I");
	assert(value_field);
	SetIntField(jenv, process, value_field, exit_value);
	
	jfieldID gotvalue_field = GetFieldID(jenv, clazz, "got_exit_value", "Z");
    assert(gotvalue_field);
    SetBooleanField(jenv, process, gotvalue_field, TRUE);
}

inline jboolean getProcessExitValue(JNIEnv *jenv, jobject process, int* p_exit_value){
    jclass clazz = GetObjectClass(jenv, process);
    
    jfieldID gotvalue_field = GetFieldID(jenv, clazz, "got_exit_value", "Z");
    assert(gotvalue_field);
    jboolean gotvalue = GetBooleanField(jenv, process, gotvalue_field);
    if(gotvalue){
        jfieldID value_field = GetFieldID(jenv, clazz, "exit_value", "I");
	    assert(value_field);
	    *p_exit_value = GetIntField(jenv, process, value_field);
    }
    return gotvalue;
}

#ifdef ORP_POSIX
void reaper(int sig) 
{ 
  int status;

  /* We should comment out the following statement to prevent exit status from being collected
     while (wait3(&status, WNOHANG, 0) > 0); 
  */
}
#endif

/*
 * Class:     java_lang_PlatformProcess
 * Method:    spawn
 * Signature: ([Ljava/lang/String;[Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_java_lang_PlatformProcess_spawn
  (JNIEnv *jenv, jobject pthis, jobjectArray cmd, jobjectArray env, jobjectArray fds)
{
	int i;
	char** cmds = NULL;
	char** envs = NULL;
	if(!cmd){
		throw_exception_from_jni(jenv, "java/lang/IllegalArgumentException",
			"No command line specified");
		return 0;
	}
#ifndef ORP_POSIX
	ExpandableMemBlock mb; //for command line
	ExpandableMemBlock mb1; //for environment block
#endif
	jsize len1 = GetArrayLength(jenv, cmd);
	if(len1 > 0){
		cmds = new char*[len1 + 1];
		for(i = 0; i < len1; i++){
			jint slen;
			jstring acmd = GetObjectArrayElement(jenv, cmd, i);
			cmds[i] = JavaStringToCharArray(jenv, acmd, &slen);
#ifndef ORP_POSIX
			if(i > 0){
                //use AppendBlock is more efficient than AppendFormatBlock
 				//mb.AppendFormatBlock(" %s", cmds[i]);
                mb.AppendBlock(" ");
                mb.AppendBlock(cmds[i]);
            }
#endif
			DeleteLocalRef(jenv, acmd);
		}
	}
	cmds[len1] = NULL;
	jsize len2 = 0;
	if(env){
		len2 = GetArrayLength(jenv, env);
		if(len2 > 0){
			envs = new char*[len2];
			for(i = 0; i < len2; i++){
				jint slen;
				jstring aenv = GetObjectArrayElement(jenv, env, i);
				envs[i] = JavaStringToCharArray(jenv, aenv, &slen);
#ifndef ORP_POSIX
				mb1.AppendBlock(envs[i], strlen(envs[i]) + 1); //including '\0'
#endif
				DeleteLocalRef(jenv, aenv);
			}
		}
		envs[len2] = NULL;
	}

	HANDLE hProcess;
    
    //we should unquote cmds[0]
	char *p = cmds[0];
	if(p[0] == '\"' || p[0] == '\''){
		p++;
		p[strlen(p) -1 ] = '\0';
	}
	
#ifdef ORP_POSIX
    /* 
      We should set SIGCHLD handler to prevent exit status from being collected,
      but should we recover the old handler???
    */
    signal(SIGCHLD, reaper); 

	hProcess/*pid*/ = fork();
	if (hProcess == -1){
	    throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException",
				"Can't fork the process");
		return 0;
	}
	if (hProcess == 0) {
	    
	    int ret = execve(p/*cmds[0]*/, cmds, envs);
		if (ret == -1){
		    char buf[256];
		    sprintf(buf, "Errors when spawning the process, cause:%s", strerror(errno));
			throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException", buf);
			return 0;
		}
	}
#else //ORP_NT
    BOOL succeed;
	
	PROCESS_INFORMATION pi;
	STARTUPINFO si;
	char buf[_MAX_PATH];
	GetCurrentDirectory(_MAX_PATH, buf);
	
	memset(&pi, 0, sizeof(PROCESS_INFORMATION));
	memset(&si, 0, sizeof(STARTUPINFO));
	si.cb = sizeof(STARTUPINFO);
	si.wShowWindow = SW_HIDE;

	if(fds){
	/* COMPLETE ME in future
		si.dwFlags != STARTF_USESTDHANDLES;
		si.hStdError = ;
		si.hStdInput = ;
		si.hStdOutput = ;
	*/
	}

	succeed =
	CreateProcess(
   	  p/*cmds[0]*/,// pointer to name of executable module
	  (char*)mb.toString(),  // pointer to command line string
	  NULL,  // process security attributes
	  NULL,   // thread security attributes
	  TRUE,  // handle inheritance flag
	  REALTIME_PRIORITY_CLASS, // creation flags
	  envs?(void*)mb1.toString():NULL,  // pointer to new environment block
	  buf,   // pointer to current directory name
	  &si,  // pointer to STARTUPINFO
	  &pi  // pointer to PROCESS_INFORMATION
	);
	
	if(!succeed){
	    char buf[256];
	    DWORD errCode = GetLastError();
	    char* message;
	    FormatMessage(  FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, 
                        NULL, 
    					errCode,
    					MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_US),
    					(LPTSTR)&message,   
    					0,
    					NULL);
		sprintf(buf, "Errors when spawning the process, cause:%s", message);
		throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException", buf);
		return 0;
	}

   	hProcess = pi.hProcess;

#endif //ORP_NT

	for(i = 0; i < len1; i++)
		free((void*)cmds[i]);
	delete[] cmds;
	
	if(envs){
		for(i = 0; i < len2; i++)
			free((void*)envs[i]);
		delete[] envs;
	}

	return (jint)hProcess;
}

/*
 * Class:     java_lang_PlatformProcess
 * Method:    waitFor
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_java_lang_PlatformProcess_waitFor
  (JNIEnv *jenv, jobject pthis)
{
	DWORD exitCode = -1;
	BOOL succeed = TRUE;
	HANDLE hProcess = getProcessHandle(jenv, pthis);
#ifdef ORP_POSIX
    int status;
    
    if (waitpid(hProcess/*pid*/, &status, 0) == -1) {
        succeed = FALSE;
        //Handle errno
        throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException",
			"Can't spawn the process");
		return -1;
    }
    if(WIFEXITED(status))
        exitCode = WEXITSTATUS(status);
#else
	while(succeed = GetExitCodeProcess(hProcess, &exitCode)){
		if(exitCode == STILL_ACTIVE)
			Sleep(50);
		else
			break;
	}
#endif
	if(!succeed)
		return -1;
	
	setProcessExitValue(jenv, pthis, exitCode);
	
	return exitCode;
}

/*
 * Class:     java_lang_PlatformProcess
 * Method:    exitValue
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_java_lang_PlatformProcess_exitValue
  (JNIEnv *jenv, jobject pthis)
{
	DWORD exitCode = -1;
	HANDLE hProcess = getProcessHandle(jenv, pthis);
	
	// Check whether exit value has been got thru previous calls of exitValue or waitFor
	if (getProcessExitValue(jenv, pthis, (int*)&exitCode))
        return exitCode;
    
#ifdef ORP_POSIX
    int status = 0; 
    int i;
    if( (i = waitpid(hProcess/*pid*/, &status, WNOHANG)) < 0) {
        throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException",
			"Can't get exit status of the process");
		return -1;
    }
    if( (i == 0) || (waitpid(hProcess/*pid*/, &status, WNOHANG) > 0) ){
        throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException",
			"The process is still alive");
		return -1;
    }
    if(WIFEXITED(status))
        exitCode = WEXITSTATUS(status);
    
#else
	BOOL succeed = GetExitCodeProcess(hProcess, &exitCode);
	if(!succeed || exitCode == STILL_ACTIVE){
		throw_exception_from_jni(jenv, "java/lang/IllegalThreadStateException",
			"The process is still alive");
		return -1;
	}
#endif
    
    setProcessExitValue(jenv, pthis, exitCode);
    
	return exitCode;
}

/*
 * Class:     java_lang_PlatformProcess
 * Method:    destroy
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_java_lang_PlatformProcess_destroy
  (JNIEnv *jenv, jobject pthis)
{
	HANDLE hProcess = getProcessHandle(jenv, pthis);
#ifdef ORP_POSIX
    kill(hProcess/*pid*/, SIGKILL);
#else
	TerminateProcess(hProcess, 1);
#endif
}
