/**************************************************************
*   
*   Creation Date: <97/07/03 13:43:50 samuel>
*   Time-stamp: <2001/03/11 21:01:23 samuel>
*   
*	<promif.c>
*	
*	OF Device Tree
*
*	TODO: The parsing of device tree files is primitive.
*	Also, all properties are ignored (we need properties)
*
*
*   Copyright (C) 1997-01 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "mol_config.h"
#include <stdarg.h>
#include <obstack.h>

#include "wrapper.h"
#include "promif.h"
#include "debugger.h"
#include "res_manager.h"
#include "booter.h"
#include "mol_assert.h"

#define obstack_chunk_alloc malloc
#define obstack_chunk_free free


#define INDENT_VALUE	4

typedef struct 
{
	int			inited;		/* 1 if in use */
	struct obstack		stack;		/* storage */

	mol_device_node_t	*root;
	mol_device_node_t	*allnext;
} of_tree;

/* properties to drop on export */
static char *drop_properties[] = {
	"nvramrc",
	/* make sure no MacOS created properties comes through (BootX...) */
	"driver-ist",
	"driver-pointer",
	"driver-ref",
	"driver-descriptor",
	"driver-ptr",
	"did",	/* ? */
	NULL
};

static char *drop_nodes[] = {
	/* "options", "chosen", "packages", "aliases", "openprom", "sd", "st", */
	NULL
};

/* 
 * hw_tree (hardware tree - describes the physical machine)
 * em_tree (emulation tree - describes the hardware to be emulated)
 */
static of_tree tree_space[2], *hw_tree, *em_tree;

static ulong s_next_phandle=1;	/* node identifier */

static mol_device_node_t	*copy_node( mol_device_node_t *kernel_node );
static void 			export_node( mol_device_node_t *node, FILE *file, int indent );

static int    			import_oftree( of_tree *tree, char *filename );
static mol_device_node_t *	import_node( of_tree *tree, char *filename, mol_device_node_t *par );

static mol_device_node_t 	*read_node( of_tree *tree, char **rbuf, mol_device_node_t *parent );
static mol_device_node_t 	*build_node( of_tree *tree, char *buf );

static p_property_t 		*find_property( mol_device_node_t *dn, const char *name );
static mol_device_node_t 	*create_node( char *path, char *name );

static char			*balance( char *buf );
static int			findkey( char *buf, char *key, char **start, char **end );
static int			numargs( char *start, char *end );

static int 			indprint(FILE *file, int ind, const char *fmt,... );

static int 			cmd_ofexport( int, char**);	/* export OF tree */
static void			build_full_names( void );

typedef char *format_proc( char *str, struct obstack *stack, int *retlen, int refcon );

typedef struct 
{
	char 		*key;
	format_proc	*proc;
	int		refcon;
} format_entry;

static format_proc hex_format;
static format_proc string_format;

/* the trailing > MUST be present (used to find the end of the word) */
static format_entry format_table[]  =
{
	{ "<byte>",	hex_format, 1  },
	{ "<half>",	hex_format, 2  },
	{ "<word>",	hex_format, 4  },
	{ "<str>",	string_format, 0 },
	{ NULL, NULL   }
};

/**************************************************************
*  promif_init
*    
**************************************************************/

void 
promif_init( void ) 
{
	memset( &tree_space, 0, sizeof(tree_space) );
	hw_tree = &tree_space[0];
	em_tree = &tree_space[1];

	s_next_phandle = 1;		/* node identifier */

	obstack_init( &hw_tree->stack );
	obstack_init( &em_tree->stack );
	hw_tree->inited = 1;
	em_tree->inited = 1;

	/* setup hw_tree (copy from kernel) */
	hw_tree->root = copy_node( (mol_device_node_t*)1 );

	/* load em_tree (emulation tree from file) */
	printm("OF device tree: %s\n", gPE.oftree_filename);
	if( import_oftree( em_tree, gPE.oftree_filename ) ){
		printm("---> Import of OF device tree failed\n");
		exit(1);
	}
	build_full_names();

	/* cmds */
	add_cmd( "ofexport", "ofexport filename\nDump hardware OF Device tree to file\n",-1, cmd_ofexport );
	add_cmd( "ofexport_em", "ofexport filename\nDump loaded OF Device tree to file\n",-1, cmd_ofexport );
}


/**************************************************************
*  promif_cleanup
*    
**************************************************************/

void 
promif_cleanup( void ) 
{
	if( tree_space[0].inited )
		obstack_free( &tree_space[0].stack, NULL );
	if( tree_space[1].inited )
		obstack_free( &tree_space[1].stack, NULL );
	tree_space[0].inited = tree_space[1].inited = 0;
}


/**************************************************************
*  copy_node
*    Copy device node from kernel (we are building hw_tree).
**************************************************************/

static mol_device_node_t *
copy_node( mol_device_node_t *kernel_node ) 
{
	size_t 			size;
	mol_device_node_t	*node, *sib;

	if( !kernel_node )
		return 0;

	/* check if root */
	if( kernel_node == (mol_device_node_t*)1 )
		kernel_node = 0;

	/* copy node */
	if( (size = _prom_get_node_size( kernel_node )) < 0 )
		printm("Error while obtaining OF device node size!\n");

	node = obstack_alloc( &hw_tree->stack, size );
	memset( node, 0, size );

	if( _prom_copy_node( node, kernel_node, size ) )
		printm("Error while copying OF device node!\n");

	node->parent = NULL;
	node->child = copy_node( node->child );

	/* The phandle value (node) is unfortunately not set when booted with BootX. */
	if( !node->node )
		node->node = (p_phandle_t)s_next_phandle++;

	if( node->child ) {
		node->child->parent = node;
		for( sib = node->child->sibling; sib; sib=sib->sibling )
			sib->parent = node;
	}
	node->sibling = copy_node( node->sibling );
	node->allnext = hw_tree->allnext;
	hw_tree->allnext = node;

	return node;
}


/**************************************************************
*  Return a linked list of the device_nodes with a given type
*    
**************************************************************/

static mol_device_node_t *prom_find_type_t( of_tree *tree, char *type );

mol_device_node_t *
prom_find_type( char *type )
{
	return prom_find_type_t( em_tree, type );
}

mol_device_node_t *
prom_find_type_hw( char *type )
{
	return prom_find_type_t( hw_tree, type );
}

static mol_device_node_t *
prom_find_type_t( of_tree *tree, char *type )
{
        mol_device_node_t *head, **prevp, *np;

        prevp = &head;
        for (np = tree->allnext; np != 0; np = np->allnext) {
                if (np->type != 0 && strcasecmp(np->type, type) == 0) {
                        *prevp = np;
                        prevp = &np->next;
                }
        }
        *prevp = 0;
        return head;
}


/************************************************************************/
/* Find device node by path 						*/
/************************************************************************/

mol_device_node_t *
prom_find_dev_by_path( char *path )
{
	char *s, *str, *orgstr = strdup( path );
	mol_device_node_t *dn, *dn2;

	str = orgstr;
	dn = prom_get_root();
	while( (s=strsep( &str, "/" )) != NULL && dn != NULL ){
		if( !strlen(s) )
			continue;

		dn2 = dn;
		for ( dn=dn->child ; dn ; dn = dn->sibling)
			if( dn->name != 0 && strcasecmp(dn->name, s) == 0 )
				break;
		if( dn )
			continue;

		/* support @N addressing using unit_string */
		if( !(s=strchr(s, '@')) )
			continue;
		
		for( dn=dn2->child; dn ; dn = dn->sibling ){
			if( !dn->unit_string || strcmp( dn->unit_string, s+1) )
				continue;
			break;
		}
	}
 	free( orgstr );
	return dn;
}

unsigned char *
prom_get_prop_by_path( char *path, int *retlen )
{
	mol_device_node_t *dn;
	char *p, *cpy = strdup(path);
	char *ret=NULL;

	if( retlen )
		*retlen=0;
	if( (p = strrchr( cpy, '/' )) != NULL ){
		*p=0;
		if( (dn = prom_find_dev_by_path( cpy )) != NULL )
			ret = prom_get_property( dn, p+1, retlen );
	}
	free( cpy );
	return ret;
}

ulong
prom_dn_to_phandle( mol_device_node_t *dn ) 
{
	return dn != NULL ? (ulong)dn->node : 0;
}

mol_device_node_t *
prom_phandle_to_dn( ulong phandle )
{
	mol_device_node_t *dn;
	
	for( dn=em_tree->allnext; dn ; dn = dn->allnext )
		if( (ulong)dn->node == phandle )
			break;
	return dn;
}


/**************************************************************
*  Return a linked list of the device_nodes with a given name
*    
**************************************************************/

static mol_device_node_t *prom_find_devices_t( of_tree *tr, char *name);

mol_device_node_t *
prom_find_devices( char *name)
{
	return prom_find_devices_t( em_tree, name );
}
mol_device_node_t *
prom_find_devices_hw( char *name)
{
	return prom_find_devices_t( hw_tree, name );
}

static mol_device_node_t *
prom_find_devices_t( of_tree *tr, char *name)
{
        mol_device_node_t *head, **prevp, *np;

        prevp = &head;
        for (np = tr->allnext; np != 0; np = np->allnext) {
                if (np->name != 0 && strcasecmp(np->name, name) == 0) {
                        *prevp = np;
                        prevp = &np->next;
                }
        }
        *prevp = 0;
        return head;
}

int 
is_ancestor( mol_device_node_t *ancestor, mol_device_node_t *dn )
{
	for( ; dn; dn=dn->parent )
		if( dn == ancestor )
			return 1;
	return 0;
}


/**************************************************************
*  prom_get_root
*    
**************************************************************/

mol_device_node_t *
prom_get_root( void ) 
{
	return em_tree->root;
}

mol_device_node_t *
prom_get_root_hw( void ) 
{
	return hw_tree->root;
}

/**************************************************************
*  prom_get_property (NULL name to obtain first property)
*    
**************************************************************/

static p_property_t *
find_property( mol_device_node_t *dn, const char *name )
{
	p_property_t *pp;
	if( !dn )
		return NULL;
	for( pp = dn->properties; pp; pp=pp->next )
		if( !strcmp(pp->name, name) )
			return pp;
	return NULL;
}

unsigned char *
prom_get_property(mol_device_node_t *np, const char *name, int *lenp)
{
	p_property_t *pp;
	
	if( lenp != NULL )
		*lenp=0;
	if( (pp = find_property( np, name )) != NULL ){
		if (lenp != NULL)
			*lenp = pp->length;
		return pp->value ? (char*)pp->value : "";
	}
	return NULL;
}

/* NULL to obtain first property */
char *
prom_next_property( mol_device_node_t *dn, const char *prev_name )
{
	p_property_t *pp = dn->properties;
	
	if( !prev_name || !strlen( prev_name ) )
		return pp ? (char*) pp->name : NULL;
	for( ; pp != 0 ; pp=pp->next ){
		if( !strcmp(pp->name, prev_name ) )
			return pp->next ? (char*) pp->next->name : NULL;
	}
	return NULL;
}


/************************************************************************/
/*	node creation							*/
/************************************************************************/

static void
_build_full_name( mol_device_node_t *dn )
{
	struct obstack *st = &em_tree->stack;

	if( dn->parent )
		_build_full_name( dn->parent );
	obstack_1grow( st, '/');
	/* Root node has the name 'device-tree' (not to be included) */
	if( dn->parent )
		obstack_grow( st, dn->name, strlen(dn->name) );
	if( dn->unit_string ) {
		obstack_1grow( st, '@');
		obstack_grow( st, dn->unit_string, strlen(dn->unit_string) );
	}
}
	
static void
build_full_name( mol_device_node_t *dn )
{
	if( dn->full_name )
		return;
	_build_full_name( dn );
	obstack_1grow( &em_tree->stack, 0);
	dn->full_name = obstack_finish( &em_tree->stack );
}

static void
build_full_names( void )
{
	mol_device_node_t *dn = em_tree->allnext;
	for( ; dn; dn=dn->allnext )
		build_full_name(dn);
}

/* Modifies path! */
static mol_device_node_t *
create_node( char *path, char *name )
{
	char *p;
	mol_device_node_t *par;

	par = prom_find_dev_by_path( path );
	if( !par ) {
		while( (p=strrchr(path, '/')) && !*(p+1) )
			*p = 0;
		if( !p )
			goto out;
		*p = 0;
		par = create_node( path, p+1 );
		if( !par )
			goto out;
	}
	if( !name )
		return par;

	return prom_add_node( par, name );
 out:
	printm("Unexpected error in create_node\n");
	return NULL;
}

mol_device_node_t *
prom_create_node( char *path )
{
	mol_device_node_t *dn;
	char *str = strdup( path );
	dn = create_node( str, NULL );
	free( str );

	return dn;
}

mol_device_node_t *
prom_add_node( mol_device_node_t *par, char *name )
{
	mol_device_node_t *dn;
	char *p;

	name = strdup( name );
	
	dn = obstack_alloc( &em_tree->stack, sizeof( mol_device_node_t) );
	memset( dn, 0, sizeof(mol_device_node_t) );

	dn->type = "";

	if( (p = strchr( name, '@' )) != NULL )
		*p = 0;
	dn->name = obstack_copy0( &em_tree->stack, name, strlen(name) );
	dn->node = (p_phandle_t)s_next_phandle++;
	dn->parent = par;
	dn->sibling = par->child;
	par->child = dn;
	dn->allnext = par->allnext;
	par->allnext = dn;

	prom_add_property( dn, "name", name, strlen(name)+1 );

	free(name);
	build_full_names();

	return dn;
}


mol_device_node_t *
prom_import_node( mol_device_node_t *par, char *filename )
{
	mol_device_node_t *dn;

	dn = import_node( em_tree, filename, par);
	build_full_names();
	return dn;
}

int
prom_delete_node( mol_device_node_t *dn )
{
	mol_device_node_t **dd;

	if( !dn )
		return 0;
	assert(dn->parent);

	while( dn->child )
		prom_delete_node( dn->child );

	/* Unlink us from parent */
	for( dd = &dn->parent->child; *dd && *dd != dn; dd = &(**dd).sibling )
		;
	assert( *dd );
	*dd = (**dd).sibling;

	/* ...and from allnext list */
	for( dd=&em_tree->allnext ; *dd && *dd != dn ; dd = &(**dd).allnext )
		;
	assert( *dd );
	*dd = (**dd).allnext;

	return 0;
}


/* dynamically modify the device tree (dn must belong to the emulated tree!)  */
void
prom_add_property( mol_device_node_t *dn, const char *name, char *data, int len )
{
	p_property_t 	*pp;
	char 		*ptr = NULL;
	static int 	wasted=0;

	pp = find_property( dn, name );
	if( pp ) {
		if( pp->length >= len )
			ptr = pp->value;
		else
			wasted += pp->length;			
	} else {
		pp = obstack_alloc( &em_tree->stack, sizeof( p_property_t ));
		pp->name = obstack_copy0( &em_tree->stack, name, strlen(name) );
		pp->next = dn->properties;
		dn->properties = pp;
	}
	if( !ptr )
		ptr = obstack_alloc( &em_tree->stack, len );
	pp->value = ptr;
	pp->length = len;
	memmove( pp->value, data, len );

	if( wasted >= 0x200 ){
		printm("Warning %d bytes wasted in promif\n", wasted);
		wasted=0;
	}
}

/************************************************************************/
/*	PARSE / EXPORT OFTREE						*/
/************************************************************************/

static int 
cmd_ofexport( int argc, char **argv )
{
	FILE *file;
	mol_device_node_t *root;
	
	if( argc!=2 )
		return 1;

	if( !strcmp(argv[0], "ofexport_em") )
		root = prom_get_root();
	else
		root = prom_get_root_hw();

	file = fopen( argv[1], "w" );
	if( !file ){
		printm("Could not create file '%s'\n", argv[1] );
		return 0;
	}
	indprint(file, 0, "# *************************\n");	
	indprint(file, 0, "#  Exported OF Device Tree \n");
	indprint(file, 0, "# *************************\n\n");
	export_node( root, file, 0 );
	fclose( file );
	return 0;
}

static void 
export_node( mol_device_node_t *node, FILE *file, int ind )
{
	p_property_t *pr;
	char **ns;
	int i;

	if( !node )
		return;

	/* drop uninteresting nodes */
	for( ns = &drop_nodes[0]; *ns; ns++ )
		if( !strcmp(node->name, *ns) ) {
			export_node( node->sibling, file, ind );
			return;
		}
	
	indprint(file, ind, "#\n");
	indprint(file, ind, "# **************** NODE %s ***************\n",node->name);
	if( node->full_name )
		indprint( file, ind, "# %s\n", node->full_name );
	indprint(file, ind, "#\n");

	indprint(file, ind, "{\n");
	ind += INDENT_VALUE;

	if( node->name )
		indprint( file, ind, "(name %s )\n", node->name );
	if( node->type && strlen( node->type ) )
		indprint( file, ind, "(type %s )\n", node->type );
	if( node->unit_string && strlen( node->unit_string ) )
		indprint( file, ind, "(unit_string %s )\n", node->unit_string );
	else if( node->full_name ){
		char *s = strrchr( node->full_name, '@' );
		if( s && !strchr( s,'/') && strlen(s+1))
			indprint( file, ind, "(unit_string %s )\n", s+1 );
	}
	if( (ulong)node->node > PHANDLE_BASE )
		indprint( file, ind, "(mol_phandle 0x%08X )\n",(ulong)node->node );
	
	/* regs */
	if( node->n_addrs ) {
		indprint( file, ind, "(addrs\n", node->n_addrs );
		for( i=0; i<node->n_addrs; i++ )
			indprint( file, ind+INDENT_VALUE, "(0x%08X 0x%08X 0x%08X )\n", 
				  node->addrs[i].space, node->addrs[i].address, node->addrs[i].size );
		indprint( file, ind, ")\n");
	}

	/* intrs */
	if( node->n_intrs ) {
		indprint( file, ind, "(intrs\n", node->n_intrs );
		for(i=0; i<node->n_intrs; i++ )
			indprint( file, ind+INDENT_VALUE, "(%d %d )\n", 
				  node->intrs[i].line, node->intrs[i].sense );
		indprint( file, ind, " )\n");
	}

	/* properties */
	for( pr=node->properties; pr; pr=pr->next ) {
		int do_continue=0;
		char **dp;

		/* drop uninteresting properties */
		for( dp = &drop_properties[0]; *dp; dp++ )
			if( pr->name && !strcmp( pr->name, *dp )) {
				do_continue = 1;
				break;
			}
		if( do_continue )
			continue;
		/* Skip all AAPL properties - probably a side effect of BootX */
		if( pr->name && !strncmp( pr->name, "AAPL", 4 ) )
			continue;

		/* detect likely strings/string arrays */
		for(i=0; i<pr->length ; i++ ) {
			char ch = pr->value[i];
			if( !ch && i && !pr->value[i-1] )
				break;
			if( ch && !isgraph(ch) && !isspace(ch) )
				break;
		}
		if( i==pr->length && i && !pr->value[i-1] ) {
			int is_long = pr->length > 38;
			indprint( file, ind, "(property %-18s <str>  \t", pr->name );
			if( is_long ) {
				fprintf( file, "\n");
				ind += INDENT_VALUE;
				indprint( file, ind, "" );
			}
			for( i=0; i<pr->length; ) {
				fprintf( file, "\"%s\"", &pr->value[i] );
				i += strlen( &pr->value[i] ) + 1;
				if( i != pr->length ) {
					fprintf( file, ",%c", is_long? '\n' : ' ' );
					if( is_long )
						indprint( file, ind, "" );
				}
			}
			fprintf( file, " )\n");
			if( is_long )
				ind -= INDENT_VALUE;
			continue;
		}

		/* print long words */
		if( pr->length == 4 ){
			indprint( file, ind, "(property %-18s <word> \t0x%08lX )\n",
				  pr->name, *(ulong*)pr->value );
			continue;
		}
		if( !(pr->length % 4) ) {
			int nw = 4;
			if( !(pr->length % (4*5)) )
				nw=5;
			else if( !(pr->length % (4*6)) )
				nw = 6;
			else if( (pr->length % (4*4)) )
				nw=2;
			indprint( file, ind, "(property %-18s <word>\n", pr->name );
			ind += INDENT_VALUE;
			indprint(file, ind, "" );
			for(i=0; i<pr->length; i+=4 ){
				fprintf(file, "0x%08lx ",*(ulong*)&pr->value[i] );
				if( (i/4)%nw == nw-1 && i != pr->length -4 ) {
					fprintf(file,"\n");
					indprint(file, ind, "");
				}
			}
			ind -= INDENT_VALUE;
			fprintf( file, " )\n" );
			continue;
		}

		/* assume byte */
		indprint( file, ind, "(property %s <byte> \n", pr->name );
		ind += INDENT_VALUE;
		indprint(file, ind, "");
		for(i=0; i<pr->length; i++ ) {
			fprintf(file, "0x%02X ",pr->value[i]);
			if( i%12 == 11 && i != pr->length -1 ) {
				fprintf(file, "\n");
				indprint(file, ind, "");
			}
		}
		ind -= INDENT_VALUE;
		fprintf(file, " )\n");
	}

	if( node->child ) {
		indprint(file, ind, "\n");
		export_node( node->child, file, ind );
	}
	ind -= INDENT_VALUE;
	indprint( file, ind, "}\n" );

	export_node( node->sibling, file, ind );
}


static int 
indprint( FILE *file, int ind, const char *fmt,... ) 
{
        va_list args;
	int ret, i;

	va_start( args, fmt );

	if( ind<0 )
		ind = 0;
	for(i=0; i<ind; i++ )
		fprintf( file, " " );
	
	ret = vfprintf( file, fmt, args ); 	
	va_end( args );
	return ret;
}

static mol_device_node_t *
import_node( of_tree *tree, char *filename, mol_device_node_t *par )
{
	mol_device_node_t *node;
	char *buf, *rbuf;
	int fd;
	off_t size;
	
	if( (fd = open( filename, O_RDONLY )) == -1 ) {
		printm("Failed opening '%s'\n", filename );
		return NULL;
	}
	size = lseek( fd, 0, SEEK_END );
	lseek( fd, 0, SEEK_SET );

	if( (buf = malloc( size+1 )) == NULL ) {
		close(fd);
		return NULL;
	}
	
	read( fd, (void*)buf, size );
	buf[size]=0;
	
	rbuf = buf;
	node = read_node( tree, &rbuf, NULL );

	free(buf);
	close( fd );

	if( par && node ) {
		mol_device_node_t *last_sib;
		for( last_sib = node; last_sib->sibling; last_sib = last_sib->sibling )
			;
		last_sib->sibling = par->child;
		par->child = node;
		node->parent = par;
	}
     	return node;
}

static int
import_oftree( of_tree *tree, char *filename )
{
	tree->allnext = NULL;
	tree->root = import_node( tree, filename, NULL );
	return tree->root ? 0:1;
}


/*
 * The format of a node is as follows:
 *   { 
 *	...node definitions...
 *	{ child node 1 }
 *	{ child node 2 }
 *	...
 *   }
 */
static mol_device_node_t *
read_node( of_tree *tree, char **rbuf, mol_device_node_t *parent ) 
{
	char	*buf, *child, *start, *end;
	mol_device_node_t **next_sib;
	mol_device_node_t *retsib=NULL;

	buf = *rbuf;
	next_sib = &retsib;

	for(;;) {
		mol_device_node_t	*dn;

		if( (start = strstr( buf, "{" )) == NULL )
			break;
		if( (end = strstr( buf, "}" )) == NULL ) {
			printm("Parse error: Missing '{'\n");
			break;
		}

		/* end this hierarchy level if '}' precede '{' */
		if( start > end ) {
			buf = end+1;
			break;
		}

		/* replace next '}' and any child marker '{' with 0 */
		*end = 0;
		if( (child = strstr( start+1, "{")) != NULL)
			*child = 0;

		dn = build_node( tree, start+1 );

		dn->parent = parent;
		*end++ = '}';

		/* read children */
		if( child ){
			*child = '{';
			end = child;
			dn->child = read_node( tree, &end, dn );  /* child */
		}

		dn->allnext = tree->allnext;
		tree->allnext = dn;

		/* prepare for next sibling */
		*next_sib = dn;
		next_sib = &dn->sibling;		/* sibling */
		buf = end;
	}
	*rbuf = buf;
	return retsib;
}


/*
 * Build node. Buf contains an the description of this node (without the
 * enclosing brackets and without any child nodes).
 */
static mol_device_node_t *
build_node( of_tree *tree, char *buf )
{
	mol_device_node_t 	*dn;
	p_address_range_t	addrs[5];
	p_interrupt_info_t	intrs[5];
	int	i;	
	char	tmpbuf[100], tmpbuf2[40];
	char	*start, *end;
	p_property_t **prop_ptr;

	dn = obstack_alloc( &tree->stack, sizeof(mol_device_node_t) );
	memset( dn, 0, sizeof(mol_device_node_t) );

	/* name, unit_string, type */
	tmpbuf[0] = 0;
	if( findkey( buf, "name", &start, NULL ))
		sscanf( start, "%99s", tmpbuf );
	dn->name = obstack_copy( &tree->stack, tmpbuf, strlen(tmpbuf)+1 );

	tmpbuf[0] = 0;
	if( findkey( buf, "unit_string", &start, NULL )) {
		sscanf( start, "%99s", tmpbuf );
		dn->unit_string = obstack_copy( &tree->stack, tmpbuf, strlen(tmpbuf)+1 );
	}

	tmpbuf[0] = 0;
	if( findkey( buf, "type", &start, NULL ))
		sscanf( start, "%99s", tmpbuf );
	dn->type = obstack_copy( &tree->stack, tmpbuf, strlen(tmpbuf)+1 );

	/* phandle field -- used internally by MOL as node identifier */
	if( findkey( buf, "mol_phandle", &start, NULL )) {
		sscanf( start, "%x", (int*)&dn->node );
		if( (ulong)dn->node < PHANDLE_BASE ) {
			printm("-----> molrc: mol_phandle field 0x%lX < 0x%x\n", 
			       (ulong)dn->node, PHANDLE_BASE );
			dn->node = 0;
		}
	}
	if( !dn->node )
		dn->node = (p_phandle_t)s_next_phandle++;
	
	/* addrs */
	dn->n_addrs = 0;
	if( findkey( buf, "addrs", &start, &end )) {
		int n = numargs( start, end );
		for(i=0; i<n && i<5; i++ ){
			if( (start = strchr( start, '(')) == NULL )
				break;
			start++;
			dn->n_addrs++;
			sscanf( start, "%X %X %X", 
				&addrs[i].space, &addrs[i].address, &addrs[i].size );
		}
		dn->addrs = obstack_copy( &tree->stack, addrs, 
					  sizeof(p_address_range_t)*dn->n_addrs );
	}

	/* intrs */
	dn->n_intrs = 0;
	if( findkey( buf, "intrs", &start, &end )) {
		int n = numargs( start, end );
		for(i=0; i<n && i<5; i++ ){
			if( (start = strchr( start, '(')) == NULL )
				break;
			start++;
			dn->n_intrs++;
			sscanf( start, "%d %d", 
				&intrs[i].line, &intrs[i].sense );
		}
		dn->intrs = obstack_copy( &tree->stack, intrs,
					  sizeof(p_interrupt_info_t)*dn->n_intrs );
	}

	/* properties */
	prop_ptr = &dn->properties;
	while( findkey(buf,"property", &start, &end )){
		p_property_t *pr;
		format_entry *fe;
		
		/* prepare for next property */
		buf = end;

		/* read name and data format */
		tmpbuf[0] = tmpbuf2[0]=0;
		sscanf( start, "%99s %39s", tmpbuf, tmpbuf2 );

		/* allocate and insert property */
		pr = obstack_alloc( &tree->stack,sizeof( p_property_t ) );
		memset(pr, 0, sizeof(p_property_t) );
		*prop_ptr = pr;
		prop_ptr = &pr->next;
		
		pr->name = obstack_copy( &tree->stack, tmpbuf, strlen(tmpbuf)+1 );

		for( fe = format_table; fe->key ; fe++ ){
			if( strcmp( tmpbuf2, fe->key ) )
				continue;
			start = strchr( start, '>' ) + 1;
			*end = 0;
			pr->value = (*fe->proc)( start, &tree->stack, &pr->length, fe->refcon );
			*end = ')';
			break;
		}
		if( !fe->key )
			printm("WARNING: Data format unimplemented\n");
	}

	return dn;
}

static char *
hex_format( char *str, struct obstack *stack, int *retlen, int refcon )
{
	int add, len=0, v;
	char *str2;
	
	if( (str2 = strchr( str, '(' )) != NULL )	/* compatibility */
		str = str2+1;

	while( sscanf(str, "%x%n\n", &v, &add ) > 0 ) {
		str += add;
		if( refcon == 1 )
			v = v << 24;
		if( refcon == 2 )
			v = v << 16;
		obstack_grow( stack, &v, refcon );
		len += refcon;
	}
	*retlen = len;
	return len ? obstack_finish( stack ) : NULL;
}

static char *
string_format( char *str, struct obstack *stack, int *retlen, int refcon )
{
	char 	ch;
	int 	on=0, len=0, add;

	while( sscanf(str,"%c%n\n", &ch, &add ) > 0 ) {
		str+=add;
		if( ch == '"' ) {
			on=!on;
			continue;
		}
		if( on ) {
 			obstack_1grow( stack, ch );
			len++;
		} else if( ch == ',' ){
			obstack_1grow( stack, 0 );
			len++;
		}
	}
	obstack_1grow( stack, 0 );
	len++;
	
	*retlen = len;
	return obstack_finish( stack );
}

/*
 * Find key expression, that is an expression of the form (name xxxxx).
 * Returns 1 if successful. Sets start to first data, and end to the closing
 * parenthesis.
 */
static int 
findkey( char *buf, char *key, char **start, char **end ) 
{
	char *p, *p2;
	int comment = 1;
	char *savestart = buf;

	for( p=buf; p; p=balance(p) ) {
		if( (p = strchr( p, '(' )) == NULL )
			break;

		comment = 0;
		for( p2 = p-1; p2 >= savestart; p2-- ){
			if( isblank(*p2) )
				continue;
			if( *p2 == '\n' || *p2 == '\r' )
				break;
			/* commented line */
			comment++;
			break;
		}
		if( comment )
			continue;

		p2 = p+1;
		while( isspace(*p2) )
			p2++;
     		if( strncmp( p2, key, strlen(key) ) == 0 ) {
			if( start )
				*start = p2 + strlen(key);
			p = balance( p );
			if( end )
				*end = p;
			return p ? 1 : 0;
		}
	}
	return 0;
}


/* Find second parenthesis of (..(...)) expression.
 * Assumes buf[0] == '('. Returns pointer to ')'.
 */
static char *
balance( char *buf ) 
{
	int 	count=1;
	char 	*p;
	
	if( !buf || buf[0] !='(' || !buf[0] )
		return NULL;
	p=buf;
	while( count && (p=strpbrk( p+1, "()")) != NULL ) {
		if( *p == '(' )
			count++;
		else
			count--;
	}
	return p;
}

static int
numargs( char *start, char *end )
{
	int count=0;
	
	for( ;; ) {
		start = strchr( start, '(');
		if( !start || start>end )
			break;
		count++;
		start = balance(start);
	}
	return count;
}

