/* 
 *   Creation Date: <2000/12/09 16:16:40 samuel>
 *   Time-stamp: <2001/09/30 16:40:32 samuel>
 *   
 *	<keycodes.c>
 *	
 *	Handles different keyboard layouts and user customization
 *   
 *   Copyright (C) 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   Based upon initial work by Ian Reinhart Geiser <geiseri@yahoo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include <sys/param.h>
#include "keycodes.h"
#include "debugger.h"
#include "res_manager.h"
#include "driver_mgr.h"
#include "mol_assert.h"

int _uses_linux_keycodes;

#define USE_LINUX_KEYCODES_RES		"use_linux_keycodes"
#define SHOW_KEY_TRANS_RES		"show_key_trans"

static char *key_res_names[ NUM_KEYCODE_TABLES ] = {
	"remap_key",
	"remap_xkey"
};

static char *file_res_names[ NUM_KEYCODE_TABLES ] = {
	"kbd_file",
	"xkbd_file"
};

/* 108 is the "eject CD" button */

static const unsigned char linux_to_adb_keycodes[128] = {
/* 0 */	 -1,  53,  18,  19,  20,  21,  23,  22,  26,  28,
	 25,  29,  27,  24,  51,  48,  12,  13,  14,  15,
/*20*/	 17,  16,  32,  34,  31,  35,  33,  30,  36,  54,
 	  0,   1,   2,   3,   5,   4,  38,  40,  37,  41,
/*40*/	 39,  10,  56,  42,   6,   7,   8,   9,  11,  45,
	 46,  43,  47,  44, 123,  67,  58,  49,  57, 122,
/*60*/  120,  99, 118,  96,  97,  98, 100, 101, 109,  71,
	107,  89,  91,  92,  78,  86,  87,  88,  69,  83,
/*80*/	 84,  85,  82,  65,  -1, 105,  50, 103, 111,  -1,
	 -1,  -1,  -1,  -1,  -1,  -1,  76,  54,  75, 105,
/*100*/	 58,  -1, 115,  62, 116,  59,  60, 119,  61, 121,
	114, 117,  -1,  -1,  -1, 107,  -1,  81,  -1, 113,
/*120*/	 -1,  -1,  -1,  -1,  -1,  55,  55,  -1
};

typedef struct {
	char	*table;
	int 	max;		/* num keycodes in table */
	int	min;		/* lowest keycode in use */
} keytable_t;

static keytable_t ktab[ NUM_KEYCODE_TABLES ];

static int initialized;
static int verbose_key_trans;

static void 	parse_kbd_files( int ktype, char *res_name );
static void 	parse_kbd_resources( int ktype, char *res_name );
extern int 	keycodes_init( void );
extern void 	keycodes_cleanup( void );

driver_interface_t keycodes_driver = {
    "keycodes", keycodes_init, keycodes_cleanup
};


int
keycodes_init( void )
{
	FILE *f;
	
	_uses_linux_keycodes = 1;
	verbose_key_trans = get_bool_res( SHOW_KEY_TRANS_RES ) == 1;
	
	if( !get_bool_res( USE_LINUX_KEYCODES_RES ) ) {
		_uses_linux_keycodes = 0;
	} else if( (f = fopen("/proc/sys/dev/mac_hid/keyboard_sends_linux_keycodes", "r")) ) {
		if( fgetc(f) == '0' )
			_uses_linux_keycodes = 0;
		fclose(f);
	} else {
		printm("Could not open /proc/sys/dev/mac_hid/keyboard_sends_linux_keycodes\n");
	}
	printm( "Using %s keycodes\n", _uses_linux_keycodes ? "Linux" : "ADB");
	initialized = 1;
	return 1;
}

void
keycodes_cleanup( void )
{
	int i;
	for(i=0; i<NUM_KEYCODE_TABLES; i++ )
		if( ktab[i].table )
			free( ktab[i].table );
}

int
register_key_table( int ktype, int min, int max )
{
	keytable_t *kt = &ktab[ktype];
	char *p;
	int i, size;
	
	assert( initialized && !kt->table && min <= max );

	size = max-min+1;
	p = kt->table = malloc( size );
	kt->max = max;
	kt->min = min;

	memset( p, 0xff, size );
	for( i=0; i<size; i++, p++ ) {
		if( _uses_linux_keycodes && i<sizeof(linux_to_adb_keycodes) )
			*p = linux_to_adb_keycodes[i];
		if( !_uses_linux_keycodes && i<128 )
			*p = i;
	}
	return 0;
}

int 
user_kbd_customize( int ktype ) 
{
	parse_kbd_files( ktype, file_res_names[ktype] );
	parse_kbd_resources( ktype, key_res_names[ktype] );
	return 0;
}


/************************************************************************/
/*	F U N C T I O N S						*/
/************************************************************************/

int
keycode_to_adb( int ktype, int keycode, int is_keypress )
{
	keytable_t *kt = &ktab[ktype];
	int ret;

	assert(kt->table);
	if( keycode > kt->max || keycode < kt->min ) {
		printm("Keycode %d out of range\n", keycode );
		keycode = kt->min;
	}

	ret = kt->table[ keycode - kt->min ];
	if( ret == 0xff ){
		printm("Keycode %d has no corresponding ADB keycode (table %d)\n", keycode, ktype );
		return 0x43 | (is_keypress? 0 : 0x80);
	}
	if( verbose_key_trans && is_keypress )
		printm("keycode %3d (0x%02x) --> ADB-code %3d (0x%02x)\n", keycode,keycode,ret,ret);

	return (ret & 0x7f) | (is_keypress? 0 : 0x80);
}

void
set_keycode( int ktype, int keycode, int adbcode )
{
	keytable_t *kt = &ktab[ktype];

	assert(kt->table);

	if( keycode > kt->max || keycode < kt->min ) {
		printm("keycode %d (table %d) out of range (%d...%d)\n", keycode, ktype, kt->min, kt->max );
		return;
	}
	if( adbcode >= 128 || adbcode < 0 ) {
		printm("Adbcode %d out of range (max=127)\n", adbcode);
		return;
	}
	kt->table[keycode-kt->min] = adbcode;
}

static void
parse_kbd_files( int ktype, char *res_name )
{
	char *name;
	unsigned int a,b;
        int n, r, ind=0;
	FILE *f;

	while( (name=get_str_res_ind( res_name, ind++, 0 )) ) {
		if( !(f = fopen( name, "r" ) )) {
			printm("Could not open '%s'\n", name );
			continue;
		}
		n=0;
		while( (r=fscanf(f, " %d : %d ", &a, &b )) != EOF) {
			if( r != 2 )
				r=fscanf(f, " 0x%x : 0x%x ", &a, &b );
			if( r != 2 ) {
				printm("Parse error in '%s' (%d pairs read)\n", name, n );
				break;
			}
			n++;
			set_keycode( ktype, a, b );
			/* printm("KEYREMAP: %d --> %d\n", a, b ); */
		}
		fclose(f);
	}
}

static void
parse_kbd_resources( int ktype, char *res_name )
{
	unsigned int a,b, ind=0;
	char *s1,*s2;
	int err;

	while( (s1=get_str_res_ind( res_name, ind, 0 )) ) {
		err = !(s2=get_str_res_ind( res_name, ind++, 1 ));

		if( err || !(sscanf(s1,"0x%x", &a) || sscanf(s1,"%d",&a)) 
		    || !(sscanf(s2,"0x%x",&b) || sscanf(s2,"%d", &b)) )
			err++;

		if( err ) {
			printm("Bad keyremap resource.\n");
			printm("Correct syntax: 'keyremap: old_keycode new_keycode'\n");
		} else {
			set_keycode( ktype, a, b );
			printm("KEYREMAP: %d --> %d\n", a, b );
		}
	}
}
