/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.system.handler;

import org.apache.avalon.excalibur.system.RoleManager;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.parameters.Parameterizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.avalon.excalibur.pool.ObjectFactory;
import org.apache.avalon.excalibur.logger.LoggerManager;

/**
 * Factory for Avalon components.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:paul@luminas.co.uk">Paul Russell</a>
 * @version CVS $Revision: 1.1 $ $Date: 2002/01/28 20:54:01 $
 * @since 4.0
 */
public class ComponentFactory
    implements ObjectFactory, ThreadSafe
{
    /** The class which this <code>ComponentFactory</code>
     * should create.
     */
    private Class                   m_componentClass;

    /** The Context for the component
     */
    private Context                 m_context;

    /** The component manager for this component.
     */
    private ComponentManager        m_componentManager;

    /** The configuration for this component.
     */
    private Configuration           m_configuration;

    /** The RoleManager for child ComponentSelectors
     */
    private RoleManager             m_roles;

    /** The LogKitManager for child ComponentSelectors
     */
    private LoggerManager           m_logManager;

    /** The logger for the ComponentFactory
     */
    private Logger                  m_logger;

    /**
     * Construct a new component factory for the specified component.
     *
     * @param componentClass the class to instantiate (must have a default constructor).
     * @param configuration the <code>Configuration</code> object to pass to new instances.
     * @param componentManager the component manager to pass to <code>Composable</code>s.
     * @param context the <code>Context</code> to pass to <code>Contexutalizable</code>s.
     * @param roles the <code>RoleManager</code> to pass to <code>DefaultComponentSelector</code>s.
     */
    public ComponentFactory( final Class componentClass,
                             final Configuration configuration,
                             final ComponentManager componentManager,
                             final Context context,
                             final RoleManager roles,
                             final LoggerManager logkit )
    {
        m_componentClass = componentClass;
        m_configuration = configuration;
        m_componentManager = componentManager;
        m_context = context;
        m_roles = roles;
        m_logManager = logkit;
        m_logger = m_logManager.getLoggerForCategory("system.factory");
    }

    public Object newInstance()
        throws Exception
    {
        final Object component = m_componentClass.newInstance();

        if (m_logger.isDebugEnabled())
        {
            m_logger.debug( "ComponentFactory creating new instance of " +
                               m_componentClass.getName() + "." );
        }

        if( component instanceof LogEnabled )
        {
            final String logger = m_configuration.getAttribute( "logger", null );
            if( null == logger )
            {
                m_logger.debug( "no logger attribute available, using standard logger" );
                ((LogEnabled)component).enableLogging( m_logManager.getDefaultLogger() );
            }
            else
            {
                m_logger.debug( "logger attribute is " + logger );
                ((LogEnabled)component).enableLogging( m_logManager.getLoggerForCategory( logger ) );
            }
        }

        if( component instanceof Contextualizable )
        {
            ((Contextualizable)component).contextualize( m_context );
        }

        if( component instanceof Composable )
        {
            ((Composable)component).compose( m_componentManager );
        }

        if( component instanceof Configurable )
        {
            ((Configurable)component).configure( m_configuration );
        }

        if( component instanceof Parameterizable )
        {
            ((Parameterizable)component).
                parameterize( Parameters.fromConfiguration( m_configuration ) );
        }

        if( component instanceof Initializable )
        {
            ((Initializable)component).initialize();
        }

        if( component instanceof Startable )
        {
            ((Startable)component).start();
        }

        return component;
    }

    public final Class getCreatedClass()
    {
        return m_componentClass;
    }

    public final void decommission( final Object component )
        throws Exception
    {
        if (m_logger.isDebugEnabled())
        {
            m_logger.debug( "ComponentFactory decommissioning instance of " +
                               m_componentClass.getName() + "." );
        }

        if( component instanceof Startable )
        {
            ((Startable)component).stop();
        }

        if( component instanceof Disposable )
        {
            ((Disposable)component).dispose();
        }
    }
}
