/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.lang;

import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;

/**
 * The ThreadContext defines a set of data that is associated
 * with a particular thread. In particular this is useful as a
 * location to centralize management of ThreadLocal-type variables.
 * As such the type of data contained in <code>ThreadContext</code> 
 * is usually data such as ContextClassLoader, Transaction ID, 
 * User ID/Subject, etc. Note that these ThreadLocal variables may 
 * not actually be implemented using the <code>ThreadLocal</code> 
 * class but need to model such behaviour.
 *
 * <p>These variables are managed by the ThreadContext. However as it
 * is not possible to inject code to be executed at the start of a 
 * thread, new threads may not be activated and the variables may not
 * be set appropriately. In such cases it is recomended that the developer
 * use <code>InheritableThreadLocal</code> as the underlying representation
 * of the variable. By doing this the <code>InheritableThreadLocal</code>
 * will maintain the appropriate state in newly created Thread.</p>
 *
 * <p>The policy chosend to manage such state is pluggable by the user. 
 * It is expected developers will provide a policy object that will manage 
 * thread local variables. For instance an application server may choose to 
 * keep the name of the application, the Subject it is running as and perhaps 
 * other state in such variables.</p>
 * 
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 */
public final class ThreadContext
{
    /**
     * Permission used to guard setThreadContext method.
     */
    private final static RuntimePermission      c_setThreadContext =
        new RuntimePermission( "ThreadContext.setThreadContext" );

    private final static InheritableThreadLocal c_context    = new InheritableThreadLocal();

    ///Accessor object used to provide policy with access to variables in context
    private final ThreadContextAccessor  m_accessor = new InnerThreadContextAccessor();

    ///The policy for ThreadContext
    private final ThreadContextPolicy    m_policy;

    ///Basic map to store variable mappings placed in context.
    private final HashMap                m_map;

    /**
     * Retrieve the ThreadContext associated with the current thread.
     *
     * @return the ThreadContext associated with the current thread.
     */
    public static ThreadContext getThreadContext()
    {
        return (ThreadContext)c_context.get();
    }

    /**
     * Set the ThreadContext associated with the current thread.
     * This code will also call <code>deactivate()</code> on the old 
     * <code>ThreadContext</code> if present and <code>activate()</code>
     * on new <code>ThreadContext</code> (if not null). 
     *
     * @param threadContext the new ThreadContext
     * @exception SecurityException if the caller does not have permission to set thread pool
     */
    public static void setThreadContext( final ThreadContext threadContext )
        throws SecurityException
    {
        final SecurityManager securityManager = System.getSecurityManager();
        if( null != securityManager )
        {
            securityManager.checkPermission( c_setThreadContext );
        }

        final ThreadContext oldThreadContext = (ThreadContext)c_context.get();
        if( null != oldThreadContext ) oldThreadContext.deactivate();

        c_context.set( threadContext );
        if( null != threadContext ) 
        {
            threadContext.activate();
        }
    }

    /**
     * Constructor that places values specified in <code>Map</code> 
     * into <code>ThreadContext</code>.
     *
     * @param map the Map
     */
    public ThreadContext( final ThreadContextPolicy policy, final Map map )
    {
        if( null == policy )
        {
            throw new NullPointerException( "policy property is null" );
        }

        if( null == map )
        {
            throw new NullPointerException( "map property is null" );
        }

        m_policy = policy;
        m_map = new HashMap();

        final Iterator keys = map.keySet().iterator();
        while( keys.hasNext() )
        {
            final Object key = keys.next();
            final Object value = map.get( key );
            final String keyString = key.toString();
            
            m_policy.verifyKeyValue( keyString, value );
            m_map.put( keyString, value );                
        }
    }

    /**
     * Utility method to call activate on policy object.
     */
    private void activate()
    {
        m_policy.activate( m_accessor );
    }

    /**
     * Utility method to call deactivate on policy object.
     */
    private void deactivate()
    {
        m_policy.deactivate( m_accessor );
    }

    /**
     * Inner class to offer accessor interface to policy object.
     */
    private class InnerThreadContextAccessor implements ThreadContextAccessor
    {
        public boolean containsKey( final String key )
        {
            if( null == m_map ) return false;
            else return m_map.containsKey( key );
        }

        public Object get( final String key )
        {
            if( null == m_map ) return null;
            else return m_map.get( key );
        }
    }
}
