/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.i18n.util;

/**
@author <a href="mailto:neeme@one.lv">Neeme Praks</a>
@version $Id: ResourceCheck.java,v 1.4 2001/12/11 09:53:35 jefft Exp $
*/

import java.util.*;
import java.io.*;

// DOM
import org.jdom.*;
import org.jdom.input.*;

import org.apache.log.Hierarchy;
import org.apache.log.Logger;

public class ResourceCheck
{
    protected static Logger logger = Hierarchy.getDefaultHierarchy()
                                            .getLoggerFor("XMLResourceBundle");

    private HashMap defaultPaths = new HashMap();
    private HashMap comparePaths = new HashMap();
    private Vector defaultPathOrder = new Vector();
    private Vector comparePathOrder = new Vector();

    public static void main(String[] args) throws Exception
    {
        if (args.length >= 3)
            main( args[0], args[1], args[2]);
        else
        {
            StringBuffer usage = new StringBuffer(20);
            usage.append("XMLResourceBundle resource checker.\n");
            usage.append("Usage example:\n");
            usage.append("\t arguments \"\\resources en lv\" will take ");
                usage.append("\"\\resources\" path \n\t as the root path for all resource ");
                usage.append("files and will compare \n\t resources.xml in \\resources\\en ");
                usage.append("to resources.xml \n\t in \\resources\\lv.\n");
            System.out.print(usage.toString());
        }
    }

    public static void main(String resPath, String baseLang, String compareLang) throws Exception
    {
        String[] resourceNames = getResourceFilenames(resPath, baseLang);
        SAXBuilder builder = new SAXBuilder(false);
        if (logger.isInfoEnabled()) logger.info("Comparing " + baseLang + " to " + compareLang + ":");
        for (int i = 0; i < resourceNames.length; i++)
        {
            String resourceName = resourceNames[i];
            if (logger.isInfoEnabled()) logger.info("Bundle: " + resourceName);
            String baseName = getResourceFilename(resPath, resourceName, baseLang);
            String compareName = getResourceFilename(resPath, resourceName, compareLang);

            ResourceCheck rc = new ResourceCheck(resourceName);
            Document doc = builder.build( new File(baseName) );
            rc.readBaseElement("", doc.getRootElement(), false);

            File compareFile = new File(compareName);
            Element rootElement = null;
            if (compareFile.exists())
            {
                doc = builder.build( compareFile );
                rootElement = doc.getRootElement();
            }
            rc.readCompareElement("", rootElement, false);

            rc.printMissingKeys();
            rc.printExtraKeys();
        }
    }

    private static String[] getResourceFilenames(String rootPath, String language)
    {
        LinkedList listResult = getResourceFilenames(rootPath, language, "", ".xml");
        String[] result = new String[listResult.size()];
        listResult.toArray(result);
        return result;
    }

    private static LinkedList getResourceFilenames(String rootPath, String language, String relativePath, String extension)
    {
        LinkedList result;
        File file = new File(rootPath + File.separatorChar + language + File.separatorChar + relativePath);
        if (file.isDirectory())
        {
            File[] files = file.listFiles();
            result = new LinkedList();
            for (int i = 0; i < files.length; i++)
            {
                result.addAll( getResourceFilenames(rootPath, language, relativePath + File.separatorChar + files[i].getName(), extension) );
            }
        }
        else
        {
            result = new LinkedList();
            String fileName = relativePath;
            if (fileName.endsWith(extension))
            {
                fileName = fileName.substring(0, fileName.length() - extension.length());
                if (logger.isDebugEnabled()) logger.debug("Found bundle: " + fileName);
                result.add(fileName);
            }
        }

        return result;
    }

    private static String getResourceFilename(String rootPath, String resourceName, String language)
    {
        return rootPath + File.separator + language + File.separator + resourceName + ".xml";
    }

    private String bundleName = "";
    public ResourceCheck(String bundleName)
    {
        this.bundleName = bundleName;
    }

    private void readBaseElement(String parentPath, Element e, boolean addAttributes)
    {
        String path = getPath(parentPath, e, addAttributes);
        List list = e.getChildren();
        if (list.size() == 0)
        {
            defaultPaths.put( path, new Boolean(false) );
            defaultPathOrder.add(path);
        }
        else
        {
            for (int i = 0; i < list.size(); i++)
                this.readBaseElement(path, (Element) list.get(i), true);
        }
    }

    private void readCompareElement(String parentPath, Element e, boolean addAttributes)
    {
        if (e == null)
        {
            logger.error(this.bundleName + ", bundle not found!");
            return;
        }
        String path = getPath(parentPath, e, addAttributes);
        List list = e.getChildren();
        if (list.size() == 0)
        {
            Boolean value = (Boolean) defaultPaths.get(path);
            comparePathOrder.add(path);
            if (value != null)
            {
                defaultPaths.put( path, new Boolean(true) );
                comparePaths.put( path, new Boolean(true) );
            }
            else
            {
                comparePaths.put( path, new Boolean(false) );
            }
        }
        else
        {
            for (int i = 0; i < list.size(); i++)
                readCompareElement(path, (Element) list.get(i), true);
        }
    }

    private static String getPath(String parentPath, Element e, boolean addAttributes)
    {
        StringBuffer sb = new StringBuffer(parentPath);
        List list;
        sb.append('/');
        sb.append( e.getName() );

        list = e.getAttributes();
        if (list.size() > 0 && addAttributes)
        {
            sb.append("[");
            for (int i = 0; i < list.size(); i++)
            {
                Attribute a = (Attribute) list.get(i);
                sb.append("@").append(a.getName()).append("='").append(a.getValue()).append("'");
                if (i < list.size() - 1)
                    sb.append(" and ");
            }
            sb.append("]");
        }

        return sb.toString();
    }

    private void printMissingKeys()
    {
        Enumeration keys = defaultPathOrder.elements();

        String keyName, key;
        Boolean value;

        while (keys.hasMoreElements())
        {
            keyName = (String) keys.nextElement();
            value = (Boolean) defaultPaths.get( keyName );
            if (!value.booleanValue()) logger.error(this.bundleName + ", add: " + keyName);
        }
    }

    private void printExtraKeys()
    {
        Enumeration keys = comparePathOrder.elements();
        String keyName, key;
        Boolean value;

        while (keys.hasMoreElements())
        {
            keyName = (String) keys.nextElement();
            value = (Boolean) comparePaths.get( keyName );
            if (!value.booleanValue()) logger.warn(this.bundleName + ", del: " + keyName);
        }
    }
}
