/*
 * Copyright  2001-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.tools.ant.taskdefs;

import java.io.*;

import junit.framework.AssertionFailedError;
import org.apache.tools.ant.BuildFileTest;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.util.JavaEnvUtils;

/**
 */
public class FixCrLfTest extends BuildFileTest {

    public FixCrLfTest(String name) {
        super(name);
    }

    public void setUp() {
        configureProject("src/etc/testcases/taskdefs/fixcrlf/build.xml");
    }

    public void tearDown() {
        executeTarget("cleanup");
    }

    public void test1() throws IOException {
        executeTarget("test1");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk1.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk1.java"));
    }

    public void test2() throws IOException {
        executeTarget("test2");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk2.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk2.java"));
    }

    public void test3() throws IOException {
        executeTarget("test3");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk3.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk3.java"));
    }

    public void test4() throws IOException {
        executeTarget("test4");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk4.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk4.java"));
    }

    public void test5() throws IOException {
        executeTarget("test5");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk5.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk5.java"));
    }

    public void test6() throws IOException {
        executeTarget("test6");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk6.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk6.java"));
    }

    public void test7() throws IOException {
        executeTarget("test7");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk7.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk7.java"));
    }

    public void test8() throws IOException {
        executeTarget("test8");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk8.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk8.java"));
    }

    public void test9() throws IOException {
        executeTarget("test9");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Junk9.java"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk9.java"));
    }

    public void testMacLines() throws IOException {
        executeTarget("testMacLines");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/Mac2Unix"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/Mac2Unix"));
    }

    public void testNoOverwrite() throws IOException {
        executeTarget("test1");
        File result =
            new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk1.java");
        long modTime = result.lastModified();

        /*
         * Sleep for some time to make sure a newer file would get a
         * more recent timestamp according to the file system's
         * granularity (should be > 2s to account for Windows FAT).
         */
        try {
            Thread.currentThread().sleep(5000);
        } catch (InterruptedException ie) {
            fail(ie.getMessage());
        } // end of try-catch

        /*
         * make sure we get a new Project instance or the target won't get run
         * a second time.
         */
        configureProject("src/etc/testcases/taskdefs/fixcrlf/build.xml");

        executeTarget("test1");
        result =
            new File("src/etc/testcases/taskdefs/fixcrlf/result/Junk1.java");
        assertEquals(modTime, result.lastModified());
    }

    public void testEncoding() throws IOException {
        if (JavaEnvUtils.isJavaVersion(JavaEnvUtils.JAVA_1_1)) {
            // UTF16 is not supported in JDK 1.1
            return;
        }
        executeTarget("testEncoding");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/input.lf.utf16"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/input.crlf.utf16"));
    }

    public void testLongLines() throws IOException {
        executeTarget("testLongLines");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/longlines.lf"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/longlines.crlf"));
    }

    public void testCrCrLfSequenceUnix() throws IOException {
        executeTarget("testCrCrLfSequence-unix");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/crcrlf.unix"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/crcrlf"));
    }

    public void testCrCrLfSequenceDos() throws IOException {
        executeTarget("testCrCrLfSequence-dos");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/crcrlf.dos"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/crcrlf"));
    }

    public void testCrCrLfSequenceMac() throws IOException {
        executeTarget("testCrCrLfSequence-mac");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/crcrlf.mac"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/crcrlf"));
    }

    public void testFixlastDos() throws IOException {
        executeTarget("testFixlastDos");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/fixlast.dos"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/fixlastfalse.lf"));
    }

    public void testFixlastFalseMac() throws IOException {
        executeTarget("testFixlastFalseMac");
        assertEqualContent(new File("src/etc/testcases/taskdefs/fixcrlf/expected/fixlastfalse.mac"),
                           new File("src/etc/testcases/taskdefs/fixcrlf/result/fixlastfalse.lf"));
    }

    /**
     * Bugzilla Report 20840
     *
     * Will fail with an exception if the parent directories do not
     * get created.
     */
    public void testCreateParentDirs() {
        executeTarget("createParentDirs");
    }

    public void assertEqualContent(File expect, File result)
        throws AssertionFailedError, IOException {
        if (!result.exists()) {
            fail("Expected file "+result+" doesn\'t exist");
        }

        InputStream inExpect = null;
        InputStream inResult = null;
        try {
            inExpect = new BufferedInputStream(new FileInputStream(expect));
            inResult = new BufferedInputStream(new FileInputStream(result));

            int expectedByte = inExpect.read();
            while (expectedByte != -1) {
                assertEquals(expectedByte, inResult.read());
                expectedByte = inExpect.read();
            }
            assertEquals("End of file", -1, inResult.read());
        } finally {
            if (inResult != null) {
                inResult.close();
            }
            if (inExpect != null) {
                inExpect.close();
            }
        }
    }

}
