#include <stdio.h>
#include <stdlib.h>
#include <jack/jack.h>
#include <jack/transport.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include "specimen.h"
#include "driver.h"
#include "patch.h"
#include "mixer.h"
#include "sync.h"
#include "lfo.h"
#include "gui/gui.h"

/* prototypes */
static int start ( );
static int stop ( );

/* file-global variables */
static GtkWidget*      config_frame;
static jack_port_t*    lport;
static jack_port_t*    rport;
static jack_client_t*  client;
static float*          buffer;
static int             rate = 44100;
static int             periodsize = 2048;
static int             running = 0;
static pthread_mutex_t running_mutex = PTHREAD_MUTEX_INITIALIZER;

/* working together to stop CTS */
typedef jack_default_audio_sample_t jack_sample_t;

static int process (jack_nframes_t frames, void* arg)
{
     int i;
     jack_sample_t* l = (jack_sample_t*) jack_port_get_buffer (lport, frames);
     jack_sample_t* r = (jack_sample_t*) jack_port_get_buffer (rport, frames);
     jack_position_t pos;

     /* transport state info */
     static jack_transport_state_t last_state = JackTransportStopped;
     jack_transport_state_t new_state;

     /* transport tempo info */
     static float last_tempo = -1;
     float new_tempo;
     
     /* behold: the jack_transport sync code */
     if (((new_state = jack_transport_query (client, &pos)) == JackTransportRolling)
	 && (pos.valid & JackPositionBBT))
     {
	  new_tempo = pos.beats_per_minute;

	  if ((last_state == JackTransportStopped)
	      || (last_state == JackTransportStarting))
	  {
	       debug ("got transport start\n");
	       sync_start_jack (new_tempo);
	  }
	  else if (new_tempo != last_tempo)
	  {
	       debug ("got tempo change\n");
	       sync_start_jack (new_tempo);
	  }

	  last_tempo = new_tempo;
     }
     last_state = new_state;
     
     mixer_mixdown (buffer, frames);
     for (i = 0; i < frames; i++)
     {
	  l[i] = buffer[i * 2];
	  r[i] = buffer[i * 2 + 1];
     }

     return 0;
}

static int sample_rate_change (jack_nframes_t r, void* arg)
{
     rate = r;

     driver_set_samplerate (rate);
     return 0;
}

static int buffer_size_change (jack_nframes_t b, void* arg)
{
     float* new;
     float* old;

     if ((new = malloc (sizeof (float) * b * 2)) == NULL)
     {
	  errmsg ("Failed to change buffer size\n");
	  stop ( );
     }

     old = buffer;
     buffer = new;
     if (old != NULL)
	  free (old);

     periodsize = b;

     /* let the rest of the world know the good news */
     driver_set_buffersize (b);
     return 0;
}

static void shutdown (void* arg)
{
     pthread_mutex_lock (&running_mutex);
     running = 0;
     pthread_mutex_unlock (&running_mutex);
     return;
}

static void restart ( )
{
     stop ( );
     start ( );
}

static void cb_sync (GtkToggleButton* button, gpointer data)
{
     if (gtk_toggle_button_get_active (button))
	  sync_set_method (SYNC_METHOD_JACK);
     else
	  sync_set_method (SYNC_METHOD_MIDI);
}

static void init ( )
{
     GtkWidget* hbox;
     GtkWidget* vbox;
     GtkWidget* button;

     config_frame = gtk_frame_new ("JACK");

     vbox = gtk_vbox_new (FALSE, GUI_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (vbox), GUI_SPACING);
     gtk_container_add (GTK_CONTAINER (config_frame), vbox);
     gtk_widget_show (vbox);
     
     hbox = gtk_hbox_new (FALSE, GUI_SPACING);
     gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show (hbox);

     button = gtk_button_new_with_label ("Reconnect");
     gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
     g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (restart),
		       NULL);
     gtk_widget_show (button);

     button = gtk_button_new_with_label ("Disconnect");
     gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
     g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (stop),
		       NULL);
     gtk_widget_show (button);

     button = gtk_check_button_new_with_label
	  ("Sync to JACK Transport instead of MIDI");
     gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
     g_signal_connect (G_OBJECT (button), "toggled", G_CALLBACK (cb_sync),
		       NULL);
     gtk_widget_show (button);

     /* make sure that we reset the sync method appropriately after
      * the user switches to alsa */
     g_signal_connect_swapped (G_OBJECT (config_frame), "show", G_CALLBACK (cb_sync),
			       (gpointer) button);
}

static int start ( )
{
     const char** ports;

     debug ("Initializing Jack Driver...\n");
     pthread_mutex_lock (&running_mutex);
     running = 0;
     if ((client = jack_client_new ("specimen")) == 0)
     {
	  errmsg ("Failed to create new jack client.  Is jackd running?\n");
	  pthread_mutex_unlock (&running_mutex);
	  return -1;
     }

     jack_set_process_callback (client, process, 0);
     jack_on_shutdown (client, shutdown, 0);

     lport =
	  jack_port_register (client, "out_left", JACK_DEFAULT_AUDIO_TYPE,
			      JackPortIsOutput, 0);
     rport =
	  jack_port_register (client, "out_right", JACK_DEFAULT_AUDIO_TYPE,
			      JackPortIsOutput, 0);

     rate = jack_get_sample_rate (client);
     driver_set_samplerate (rate);
     jack_set_sample_rate_callback (client, sample_rate_change, 0);

     periodsize = jack_get_buffer_size (client);
     driver_set_buffersize (periodsize);
     jack_set_buffer_size_callback (client, buffer_size_change, 0);
     if ((buffer = malloc (sizeof (float) * periodsize * 2)) == NULL)
     {
	  errmsg ("Failed to allocate space for buffer\n");
	  jack_client_close (client);
	  pthread_mutex_unlock (&running_mutex);
	  return -1;
     }

     mixer_flush();
     if (jack_activate (client) != 0)
     {
	  errmsg ("Failed to activate client\n");
	  jack_client_close (client);
	  pthread_mutex_unlock (&running_mutex);
	  return -1;
     }

     ports = jack_get_ports (client, NULL, NULL,
			  JackPortIsInput | JackPortIsPhysical);

     if (ports[0] != NULL)
     {
	  if (jack_connect (client, jack_port_name (lport), ports[0]) != 0)
	       errmsg ("Cannot connect left output port\n");
	  if (ports[1] != NULL)
	  {
	       if (jack_connect (client, jack_port_name (rport), ports[1]) !=
		   0)
		    errmsg ("Cannot connect right output port\n");
	  }
	  else
	  {
	       errmsg ("Cannot connect right output port\n");
	  }
	  free (ports);
     }
     else
     {
	  errmsg ("Cannot connect output ports\n");
     }

     debug ("Initialization complete\n");
     running = 1;
     pthread_mutex_unlock (&running_mutex);
     return 0;
}

static int stop ( )
{
     pthread_mutex_lock (&running_mutex);
     if (running)
     {
	  debug ("Shutting down...\n");
	  jack_deactivate (client);
	  jack_client_close (client);
	  if (buffer != NULL)
	       free (buffer);
	  debug ("Shutdown complete\n");
     }
     else
     {
	  debug ("Not running, so not shutting down\n");
     }

     running = 0;
     pthread_mutex_unlock (&running_mutex);
     return 0;
}

static int getrate ( )
{
     return rate;
}

static int getperiodsize ( )
{
     return periodsize;
}

static const char* getname ( )
{
     return "JACK";
}

static GtkWidget* getwidget ( )
{
     return config_frame;
}

Driver jack_driver = {
     init,
     start,
     stop,
     getrate,
     getperiodsize,
     getwidget,
     getname
};
