// sound.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "application.h"
#include "controller.h"
#include "converter.h"
#include "interpolater.h"
#include "sound.h"
#include "soundheader.h"
#include "request.h"
#include "smpte.h"
#include "sndconfig.h"
#include "typeconvert.h"
#include "format.h"
#ifdef NO_TEMPLATES
#include <InterViews/action.h>        /* sgi native CC cant handle templates */
    declareActionCallback(Sound)
    implementActionCallback(Sound)
#else
#include "actioncallback.h"
#endif

// storage for public "class methods" -- macros in sndconfig.h

int Sound::default_SampleRate = SOUND_DEFAULT_RATE;

DataType Sound::default_DataType = SOUND_DEFAULT_TYPE;

char* Sound::default_FileSuffix = SOUND_SUFFIX;

int Sound::default_Raw_ChannelCount = 1;

int Sound::default_Raw_SampleRate = SOUND_DEFAULT_RATE;

DataType Sound::default_Raw_DataType = SOUND_DEFAULT_TYPE;

// class methods

int
Sound::defaultSampleRate() { return default_SampleRate; }

DataType
Sound::defaultDataType() { return default_DataType; }

boolean
Sound::setDefaultSampleRate(int rate) {
    if (rate > 0) {
        default_SampleRate = rate;
        return true;
    }
    return false;
}

boolean
Sound::setDefaultDataType(DataType type) {
    default_DataType = type;
    return true;
}

int
Sound::defaultRawChannelCount() { return default_Raw_ChannelCount; }

int
Sound::defaultRawSampleRate() { return default_Raw_SampleRate; }

DataType
Sound::defaultRawDataType() { return default_Raw_DataType; }

boolean
Sound::setDefaultRawChannelCount(int chans) {
    if (chans > 0) {
        default_Raw_ChannelCount = chans;
        return true;
    }
    return false;
}

boolean
Sound::setDefaultRawSampleRate(int rate) {
    if (rate > 0) {
        default_Raw_SampleRate = rate;
        return true;
    }
    return false;
}

boolean
Sound::setDefaultRawDataType(DataType type) {
    default_Raw_DataType = type;
    return true;
}

const char*
Sound::defaultFileSuffix() { return default_FileSuffix; }

// ctors, etc.

Sound::Sound(double duration, int srate, int nchans, DataType type)
    : Data(type, iround(duration*srate), nchans), sr(srate) {
}

Sound::Sound(int len, int srate, int nchans, DataType type)
    : Data(type, len, nchans), sr(srate) {
}

// these protected constructors are used only with Sound::clone()

Sound::Sound(const Sound *s, const Range &selection)
    : Data(s, selection), sr(s->sr) {
}

Sound::Sound(const Sound *s, const Range &selection, const Range &chans)
    : Data(s, selection, chans), sr(s->sr) {
}

Sound::~Sound() {
}

Data *
Sound::newData() {
	Sound *s = new Sound(this);
	if (s->length() != length()) {
		s->unref();
		s = nil;
	}
	return s;
}

Data *
Sound::newData(int length) {
	Sound *s = new Sound(this, length);
	if (s->length() != length) {
		s->unref();
		s = nil;
	}
	return s;
}

Data *
Sound::clone(const Range &selection) { return new Sound(this, selection); }

Data *
Sound::clone(const Range &selection, const Range &chans) {
    return new Sound(this, selection, chans);
}

void
Sound::copyFrom(const Data *src) {
    // if copying data with different max sample values
    if(valueLimit() != src->valueLimit() && dataType() < FloatData) {
        boolean wasSetTo = deferRescan(true);
        if(wasSetTo == true)
            ((Sound *)src)->checkValues();    // scan if checking was turned off
        boolean sourceIsFloat = src->dataType() >= FloatData;
        double scalingFactor = valueLimit() /
            (sourceIsFloat ? src->maxValue() : src->valueLimit());
        Application::inform("Rescaling for copy...");
        deferRescan(wasSetTo);
        copyRescaledFrom(src, scalingFactor, sourceIsFloat);
    }
    else
        Super::copyFrom(src);
}

double
Sound::duration() const {
    return length()/double(sr);
}

const char *
Sound::fileSuffix() const {
    return dataType() >= FloatData ? SOUND_FLOAT_SUFFIX : defaultFileSuffix();
}

double
Sound::peakAmp() {
    return maxValue();
}

int
Sound::changeSRate(int rate, boolean interp) {
    int status = false;
    if(rate != sr && rate > 0) {
        status = true;
        int oldrate = sr;
        sr = rate;
        boolean was = deferRescan(true);
        if(interp) {    // copy sound and then transpose it back into original
	    // scale gains down if necessary to avoid out-of-range values
	    double gain = 1.0;
	    if (oldrate < rate) {
	        double fractionOfPeak = peakAmp() / valueLimit();
	        if (fractionOfPeak > 0.99)
	            gain = 0.99 / fractionOfPeak;
	    }
            Data* source = copyOf();
            source->ref();
            Interpolater interp(source, this, double(oldrate)/sr, true, gain);
            if((status = interp.ready()) == true)
                interp.apply();
            Resource::unref(source);
            deferRescan(was);    // to get rescan in Notify()
        }
        if(status) Notify();
        deferRescan(was);        // if !interp, reset here
    }
    return status;
}

// base class method is protected -- made public in Sound class

int
Sound::changeDataType(DataType newType, bool rescaleFirst) {
    return Super::changeDataType(newType, rescaleFirst);
}

// protected methods

Header *
Sound::createHeader(DataFile *file, boolean reading) {
    SoundHeader *hdr = SoundHeader::create(
        file,
        dataType(),
        sRate(),
        nChans(),
        reading ? 0.0 : maxValue(),    // only extract peak if writing
        reading
    );
    if(!reading) configureHeader(hdr);
    return hdr;
}

void
Sound::readFromHeader(Header *h) {
    Super::readFromHeader(h);
    SoundHeader *header = (SoundHeader *) h;
    sr = header->sampleRate();
    double peak = header->peakAmp();
    if(peak != 0.0)
        setMaxValue(peak);
    else
        checkValues();
}

void
Sound::checkValues() {
    double max = scanForMaxValue();
    setMaxValue((max == 0.0) ? 1.0 : max);    // recheck peak amp
}

Range
Sound::limits(int, boolean) const {
    double peak = maxValue();
    return Range(-peak, peak);    // same for all chans
}

// special protected method for use by the Converters

const char *
Sound::getDataPointer() const {
    return rep->getAddressOfContiguousData();
}

int
Sound::play(Converter *converter, ProgressAction *progress) {
    Sound *playMe = nil;
    int status = true;
    if(converter->hasPlayableFormat(this))
        playMe = this;
    else
	{
        Application::inform("Converting to playable format...");
		DataType bestType = converter->bestPlayableType();
        playMe = new Sound(duration(), sRate(), nChans(), bestType);
        playMe->ref();
        playMe->copyFrom(this);
        status = (playMe->length() > 0);    // until Exceptions enabled
    }
    status = status && playMe->doPlay(converter, progress);
    if(playMe != this)
        Resource::unref(playMe);
    return status;
}

int
Sound::doPlay(Converter *converter, ProgressAction *progress) {
    if(converter->configure(this, Converter::Play)) {
        return converter->start(progress, nil);
    }
    return false;
}

int
Sound::record(Converter *converter, ProgressAction *progress) {
    return doRecord(converter, progress);
}

int
Sound::doRecord(Converter *converter, ProgressAction *progress) {
    if(converter->configure(this, Converter::Record)) {
        return converter->start(
			progress,
#ifdef NO_TEMPLATES
            new Sound_ActionCallback(this, &Sound::Notify)
#else
            new ActionCallback<Sound>(this, &Sound::Notify)
#endif
        );
    }
    return false;
}

void
Sound::rescale() {
    boolean wasSetTo = deferRescan(false);    // we need to rescan here
    if(wasSetTo == true)
        checkValues();    // check first if checking was turned off
    // rather than rescanning again after rescaling, defer check and
    // simply set the peak to valueLimit()
    double oldpeak = maxValue();
    if(oldpeak != 0.0) {
        deferRescan(true);
        setMaxValue(valueLimit());
        scaleSelf(valueLimit()/oldpeak);
        deferRescan(wasSetTo);
    }
}

Range
Sound::frameRange(RangeUnit units) const {
    return (units == FrameTimeUnit) ? Range(0.0, duration())
    : (units == FrameSmpteUnit) ? Range(0.0, SMPTE::frames(duration()) - 1)
    : (units == FrameCDUnit) ? Range(0.0, (length() - 1) / 588.0)
    : Super::frameRange(units);
}

const char * 
Sound::frameRangeLabel(RangeUnit units) const {
    return (units == FrameTimeUnit) ? "    Time" :
        (units == FrameSmpteUnit) ? "SMPTE Frames" :
        (units == FrameUnit) ? "Sample Number" :
        (units == FrameCDUnit) ? "CDDA Frames" :
        Super::frameRangeLabel(units);
}

void
Sound::information(Controller *controller) {
    char str[128];
    AlertRequest request("Soundfile Information:");
    request.setBell(false);
    request.appendLabel("------------");
    request.appendLabel("Filename: ", controller->fileName());
    request.appendLabel("Sample Rate: ", toString(str, sRate()));
    request.appendLabel("Duration: ", tFormat(duration(), 0.01));
    request.appendLabel("File Size (Mb): ",
        toString(str, sizeInBytes()/1000000.0));
    request.appendLabel("NSamps: ", toString(str, length()));
    request.appendLabel("NChans: ", toString(str, nChans()));
    int typeIndex = powerOfTwoToLinearEnum(dataType());
    static char* labels[] = {
        "unknown",
        "8-bit linear", "8-bit unsigned", "mu law",
        "16-bit linear", "32-bit linear", "floating point",
        "double-precision float", "unknown"
    };
    request.appendLabel("Sample Format: ", labels[typeIndex]);
    request.appendLabel("Peak Amplitude: ", toString(str, maxValue()));
    controller->handleRequest(request);
}

#ifdef sgi

#include "aiff_header.h"
#include "datafile.h"

// these kludges are needed since AIFF sounds have special i/o routines

int
Sound::read(DataFile *f, Header* header) {
    BUG("Sound::read()");
    if(header == nil)
        header = createHeader(f, true);
    if(((SoundHeader *) header)->headerType() != SoundHeader::Aifc)
        return Super::read(f, header);    // not AIFC file
    header->ref();
    int status = false;
    if(status = header->read(f)) {        // supposed to be =, not ==
        if(header->dataSize() == 0)
            Application::alert("Warning:  file contains 0 frames.");
        Resource::unref(rep);
        // since compressed sounds cannot use file size in bytes as an indicator
        // of the # of frames to read, just use readSize/dataSize as fraction.
        double fractionToRead = double(f->readSize()) / f->dataSize();
        AIFFSoundHeader* ah = (AIFFSoundHeader *) header;
        int len = iround(ah->frameCount() * fractionToRead);
        rep = DataRep::create(header->dataType(), max(len, 8), header->nChans());
        rep->ref();
        char* data =  (char *) getDataPointer();
        long samps = ah->readSound(&data, long(len));
        if(samps > 0L) {    // not checked against len since it uses internal frame count
            readFromHeader(header);
            boolean was = deferRescan(true);
            Notify();                    // update views but dont rescan first
            deferRescan(was);
            modified(false);
        }
        else status = false;
    }
    Resource::unref(header);
    return status;
}

int
Sound::write(DataFile* f) {
    BUG("Sound::write(f)");
    Header* header = createHeader(f, false);
    if(((SoundHeader *) header)->headerType() != SoundHeader::Aifc)
        return write(f, header);    // not AIFC file
    header->ref();
    AIFFSoundHeader* aheader = (AIFFSoundHeader *) header;
    int status = false;
    if((status = aheader->write(f)) == true) {
        char* data =  (char *) getDataPointer();
        status = (aheader->writeSound(data, long(length())) == length());
    }
    Resource::unref(aheader);
    modified(modified() && (status != true));
    return status;
}

#endif /* sgi */

