// controller.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/event.h>
#include <InterViews/world.h>
#include <X11/keysym.h>
#include "localdefs.h"
#include "application.h"
#include "datafile.h"
#include "controller.h"
#include "converter.h"
#include "cmdstate.h"
#include "channelview.h"
#include "frameview.h"
#include "editor.h"
#include "range.h"
#include "data.h"
#include "datawindow.h"
#include "dataview.h"
#include "dialogbox.h"
#include "dialog_ctor.h"
#include "filename.h"
#include "optionsetter.h"
#include "request.h"
#include "progressaction.h"
#include "progressdialog.h"
#include "version.h"

#undef DEBUG_SELECTION

// global member initialization

// to avoid g++ bug
static ViewInfo Default_Controller_ViewInfo;

Controller*			Controller::_sHead = nil;
int					Controller::_sNumControllers = 0;
int					Controller::_sIsQuitting = false;
int					Controller::_sAvailableSources = 0;
ViewInfo&			Controller::DefaultViewInfo = Default_Controller_ViewInfo;

// this is the static public meta-constructor for use at startup time
// returns nil if file could not be opened and read

Controller *
Controller::create(DataFile* file) {
	Controller* controller = new Controller(file->name());
	if(!controller->editor()->readFile(file) || !controller->initialize()) {
		delete controller;
		controller = nil;
	}
	return controller;
}

// this is the protected ctor called by the above function
// note that it does not call initialize() on its own

Controller::Controller(const char *filename) {
	earlyInit();
	_editor = DataEditor::create(this, filename);
	setFileName(filename);
}

// this is used for the display of newly created data objects

Controller::Controller(Data *d, const char *name, ViewInfo& info) {
	earlyInit();
	_editor = DataEditor::create(this, d);
	setFileName(
		(name != nil) ? name : FileName::tmpName(d->fileSuffix())
	);
	initialize(info);
}

// this init is done in all cases, regardless of final init outcome

void
Controller::earlyInit() {
	_prev = _next = nil;
	_window = nil;
	_view = nil;
	_converterProgressAction = nil;
	_progressDialog = nil;
	_converting = false;
	_editDelegate = nil;
	addToList();
}

boolean
Controller::initialize(ViewInfo& info) {
	if(_editor->model() != nil) {	// DataEditor successfully initialized
		_converterProgressAction = new ProgressCallback<Controller>(
			this, &Controller::playbackProgress
		);
		_converterProgressAction->ref();
		ref();	// to make sure no one else deletes this
		createAndAttachView(info);
		externalSourceNotify(false);	// XXX bool ignored
		return true;
	}
	return false;
}

Controller::~Controller() {
	delete _editor;
	Resource::unref(_converterProgressAction);
	Resource::unref_deferred(_progressDialog);
	removeFromList();
	if(_sNumControllers == 0) {		// if we are closing last controller
		Converter::destroyInstance();
		DialogConstructor::destroyInstance();
		if(world())
			world()->quit();
		else
			exit(0);
	}
	detachView();
}

void
Controller::addToList() {
	_prev = _next = this;
	if(!_sHead)
		_sHead = this;
	else {
		_prev = _sHead->_prev;
		_next = _sHead;
		_sHead->_prev->_next = this;
		_sHead->_prev = this;
	}
	_sNumControllers++;
}

void
Controller::removeFromList() {
	_next->_prev = _prev;
	_prev->_next = _next;
	if (this == _sHead)
		_sHead = (_prev != this) ? _prev : nil;
	if (Application::isGlobalController(this))
		Application::setGlobalController(_sHead);
	_sNumControllers--;
}

DataView*
Controller::newView(ViewInfo& info) {
	Data* data = model();
	DataView *d = nil;
	if (data->displayAsFrames()) {
		ViewInfo frameinfo(ChannelSpecialUnit);
		d = new FrameView(this, frameinfo);
	}
	else {
		const char *a = nil;
		Range frames(-1, -1);
		if (!info.frameRange.isNegative())		// if specific range requested
			frames = info.frameRange;
		Range channels(-1, -1);
		if (!info.channelRange.isNegative())	// if specific range requested
			channels = info.channelRange;
		else if((a = Application::getGlobalResource(
				data->channelDisplayAttribute())) != nil) {
			int totalChans;
			if((totalChans = atoi(a)) > 0)
				channels.set(0, totalChans-1);
		}
		a = Application::getGlobalResource(
			data->horizontalScaleModeAttribute());
		RangeUnit units = (a != nil && (strcmp(a, "Time") == 0)) ?
			FrameTimeUnit : FrameUnit;
		ViewInfo channelinfo(units, frames, channels);
		d = new ChannelView(this, channelinfo);
	}
	return d;
}

void
Controller::createAndAttachView(ViewInfo& info) {
	_view = newView(info);
	_window = new DataWindow(this, _view);
	_editDelegate = _view->getEditDelegate();
	_editDelegate->ref();
	_editDelegate->update(model()->frameRange());
}

void
Controller::detachView() {
	_view = nil;
	Resource::unref(_editDelegate);
	Resource::unref_deferred(_window);
	_window = nil;
}

ProgressAction *
Controller::createProgressAction(const char *message) {
	if (_progressDialog) {
		if (_progressDialog->isMapped())
			_progressDialog->disappear();
		Resource::unref_deferred(_progressDialog);
		_progressDialog = nil;
	}
	_progressDialog = new ProgressDialog(message, _window);
	_progressDialog->ref();
	return new ProgressCallback<Controller>(this,
											&Controller::modifyProgress);
}

int
Controller::modifyProgress(double percent) {		// For modifier progress
	int retval = 1;
	if (_progressDialog) {
		_progressDialog->appear();	// no-op if already visible
		retval = _progressDialog->update(percent);
	}
	return retval;
}

// this method is called, via the playbackProgressAction object, during the
// play and record loops to display the progress.

int 
Controller::playbackProgress(double fractionDone)
{
	const Range frames = _editor->currentRegion();
	int location = frames.intMin() + int(fractionDone * frames.size());
	const Range chansInView = _view->getChannelRange();
	_converting = true;
	showInsertPoint(location, chansInView, true);
	_converting = false;
	return _window->checkForControlKeyEvent();
}

Data*
Controller::model() const { return _editor->model(); }

// called when attempting to quit application

boolean
Controller::closeChain() {
	register Controller *c, *cnext = nil;
	for(c = this->_next; ; c = cnext) {
		cnext = c->_next;
		if(!c->close())			// close and decrement no.
			return false;		// user chose to stop closing
		if(c == this)
			break;			// until back to current
	}
	return true;
}

const char*
Controller::windowName() const {
	return FileName::stripPath(fileName());
}

void
Controller::setFileName(const char *nm) {
	if(_filename != nm) {
		_filename = nm;
		if(_window)
			_window->changeName(windowName());
	}
}

void
Controller::setEditRegion(const Range &region, int chan) {
	_editor->setEditRegion(region, chan);
	_editDelegate->update(_editor->currentRegion());
}

void
Controller::setInsertPoint(int point, int chan) {
	Range currentRegion;
	if (!_converting) {
		_editor->setInsertPoint(point, chan);
		currentRegion = _editor->currentRegion();
	}
	else {
		currentRegion.set(point, point);
	}
	_editDelegate->update(currentRegion);
}

void
Controller::setCommandState(unsigned int state, bool set) {
	_editor->commandState()->Set(state, set);
}

// all subview events are handed back to the Controller for redistribution
// via this method.  Only events with the control key down are of interest.

boolean
Controller::handleEvent(Event& e) {
	if(e.eventType == KeyEvent && e.control)
		return handleKeyEvent(e);
	else {
		Application::setGlobalController(this);
		_view->Handle(e);
		return true;
	}
}

#ifndef IV_IS_PATCHED

// this code is taken from the modified and correct IV 3.2 version of
// Event::keysym()

#include <IV-X11/Xutil.h>
#include <IV-X11/xevent.h>

static unsigned long
getKeySymFrom(Event& e) {
    KeySym k = NoSymbol;
	XEvent& xe = e.rep()->xevent_;
	if (xe.type == KeyPress) {
		char buf[10];
		XLookupString(&xe.xkey, buf, sizeof(buf), &k, nil);
	}
	return k;
}
#endif

boolean
Controller::handleKeyEvent(Event& e) {
#ifndef IV_IS_PATCHED
	// this is a temporary hack until the IV source is officially patched
	return keyCommand(getKeySymFrom(e));
#else
	return keyCommand(e.keysym());
#endif
}

// examine Controller chain for ones with a selected region and return the
// selection (with the most recent timestamp) to current Controller

Data *
Controller::findSelection() {
	Data *sel = nil;
	long currentStamp = 0;
	for(Controller* c = _next; c != this; c = c->_next) {
		long tstamp = 0;
		Data* s = c->getSelection(tstamp);
		if(s != nil) {
			if(tstamp > currentStamp) {
				Resource::unref(sel);	// free older selection
				sel = s;
				currentStamp = tstamp;
			}
			else {
				Resource::unref(s);		// free this selection
				s = nil;
			}
		}
	}
	return sel;	// not referenced here since never used in this Ctlr
}

Data *
Controller::getSelection(long &timestamp) {
	return _editor->getSelection(timestamp);
}

// Notify all other controllers that a source selection
// is available here.  If our own editor still has access to the scratch
// buffer or an external source, we set our own state to show that.

void
Controller::notifyOfSelection(bool selected) {
	_sAvailableSources += (selected) ? 1 : -1;

	setCommandState(Source_Selected,
					_sAvailableSources - _editor->selectionMade() > 0 || 
						_editor->copyBuffer() != nil);

#ifdef DEBUG_SELECTION
	printf("Controller::notifyOfSelection(%p, %d): avail sources = %d\n", this, selected, _sAvailableSources);
#endif
	bool sourceAvailable = (_sAvailableSources > 0);
	for(Controller* c = _next; c != this; c = c->_next) {
		c->externalSourceNotify(sourceAvailable);
	}
}

void
Controller::busy(boolean b) {
	for(Controller *c = _next; ; c = c->_next) {
		if(c->_window)
			c->_window->busy(b);
		if(c == this)
			break;
	}
}

// Notifies this controller whether there is an external selection available for
//	use as a source.  We check to make sure the available source count does
//  not include our own selection.

void
Controller::externalSourceNotify(bool isAvailable) {
	bool sourceSelected = (_sAvailableSources - _editor->selectionMade() > 0);
#ifdef DEBUG_SELECTION
	printf("Controller::externalSourceNotify(%p, %d) called\n", this, isAvailable);
	printf("\tSource_Selected = %d\n", sourceSelected);
#endif
	setCommandState(Source_Selected, sourceSelected);
}

int
Controller::handleRequest(Request& request) {
	int response = Cancel;
	DialogBox *dialog = nil;
	if(_window != nil) _window->makeVisible();
	dialog = DialogConstructor::getInstance()->createDialog(_window, request);
	while((response = dialog->display()) != Cancel) {	
		if(request.checkValues())
			break;		// vals ok; go on
	}
	Resource::unref(dialog);
	return response;
}

void
Controller::showInsertPoint(int point, const Range &chans, boolean scroll) {
	_view->setInsertPoint(point, chans, scroll);
}

void
Controller::showEditRegion(const Range& region, const Range& chans, 
		boolean scroll) {
	_view->setEditRegion(region, chans, scroll);
}

void
Controller::resetScaleTimes() {
	_view->resetHorizontalScale();
	_editDelegate->update(_editor->currentRegion());
}

void
Controller::unselectView() {
	_editDelegate->update(Range(0,0));	// sets display to zeros
	_view->unselect();
}

World *
Controller::world() {
	return (_window != nil) ? _window->getWorld() : nil;
}

void
Controller::display(World *world) {
	if(_window)
		_window->display(world);
	Application::setGlobalController(this);
}

// sym is checked first by controller, then by view, then by editor
// subclass, and lastly by editor base class.  Checking stops when matched.

boolean
Controller::keyCommand(unsigned long sym) {
	boolean interested = true;
	Application::setGlobalController(this);
	busy(true);
	switch (sym) {
	case XK_N:
		newViewOfSelection();
		break;
	case XK_copyright:
		viewAsChannels();
		break;
	case XK_ordfeminine:
		viewAsFrames();
		break;
	case XK_z:
		zoomToSelection();
		break;
	case XK_Z:
		zoomToFull();
		break;
	case XK_W:
		close();
		return false;
	case XK_quoteright:
		changeName();
		break;
	case XK_yen:	// another unused keysym
		showVersion();
		break;
	case XK_F12:
		showManual();
		break;
	case XK_F11:
		showHomePage();
		break;
	case XK_Q:
		quit();
		return false;
	case XK_section:
		Converter::useNative(true);
		break;
	case XK_diaeresis:
		Converter::useNative(false);
		break;
	case XK_threesuperior:
		Converter::destroyInstance();	// functions as reset
		break;
	case XK_slash:
		Data::deferRescan(false);
		break;
	case XK_backslash:
		Data::deferRescan(true);
		break;
	case XK_macron:
		setProgramOptions();
		break;
	case XK_acute:
		{
		FileOptionSetter fos;
		_editor->applyModifier(fos);
		}
		break;
	case XK_mu:
		{
		MemoryOptionSetter mos;
		_editor->applyModifier(mos);
		}
		break;
	case XK_cedilla:
		{
		SoundPlaybackOptionSetter spos;
		if (_editor->applyModifier(spos)) {
			Converter::destroyInstance();	// functions as reset
		}
		}
		break;
	case XK_onesuperior:
		if (confirm("Re-read .mxvrc file from disk?"))
			read_mxvrc();
		break;
	case XK_masculine:
		if (confirm("Write .mxvrc file to disk?"))
			write_mxvrc();
		break;
	default:
		if((interested = _view->keyCommand(sym)) != true)
			interested = _editor->keyCommand(sym);
		break;
	}
	busy(false);
	if(interested)
		Application::inform();
	return interested;
}

// The next several methods are those called by menu commands

void
Controller::newViewOfSelection() {
	Range chansInView = _view->getChannelRange();
	// display lesser of visible range or selected range
	if(_editor->nchans() < chansInView.size())
		chansInView = _editor->currentChannels();
	ViewInfo info(
		_view->horizRangeUnits(),
		_editor->currentRegion(),		// frame range to show
		chansInView						// channel range to show
	);
	Controller *newCtlr = new Controller(model(), fileName(), info);
	newCtlr->display(world());	
}

void
Controller::viewAsFrames() {
	viewAs(AsFrames);
}

void
Controller::viewAsChannels() {
	viewAs(AsChannels);
}

void
Controller::viewAs(ViewType type) {
	boolean flag = (type == AsFrames);
	if(model()->displayAsFrames() != flag) {
		if(model()->displayAsFrames(flag)) {
			_window->unDisplay();
			detachView();
			createAndAttachView();
			display(world());
		}
	}
}

void
Controller::zoomToSelection() {
	_view->setVisibleFrameRange(_editor->currentRegion());
}

void
Controller::zoomToFull() {
	_view->setVisibleFrameRange(model()->frameRange());
}

boolean
Controller::close() {
	boolean status = true;
	Response resp = _editor->closeFile();
	switch(resp) {
	case Yes:	// no previous warning from DataEditor::closeFile()
		if(_sNumControllers == 1 && !_sIsQuitting &&
			!confirm("Closing this window will quit mxv.",
				 "Continue?"))
			goto Stay;
		else
			goto Quit;
		break; /*NOTREACHED*/
	case No:	// warning given in DataEditor::closeFile, OK
Quit:
		busy(false);
		_window->unDisplay();
		Application::inform();
		delete this;
		break;
	case NullResponse:
	case Cancel:	// warning given, command cancelled
Stay:
		busy(false);
		_sIsQuitting = false;
		status = false;
		break;
	}
	return status;
}

void
Controller::showVersion() {
	alert(MXV_version_string, MXV_copyright_string);
}

void
Controller::showManual() {
	const char *browser = Application::getGlobalResource("WebBrowser");
	const char *manualURL = Application::getGlobalResource("ManualURL");
	if (!browser || !manualURL) {
		alert("You need to set the X resources \"MiXViews*WebBrowser\"",
			  "and \"MiXViews*ManualURL\" to your web browser and the URL for",
			  "the MiXViews manual page html, respectively");
		return;
	}
    char cmdstring[256];
	sprintf(cmdstring, "%s %s &", browser, manualURL);
	system(cmdstring);
}

void
Controller::showHomePage() {
	const char *browser = Application::getGlobalResource("WebBrowser");
	if (!browser) {
		alert("You need to set the X resource \"MiXViews*WebBrowser\"",
			  "to the name of your web browser");
		return;
	}
    char cmdstring[256];
	sprintf(cmdstring, "%s %s &", 
			browser, "http://www.create.ucsb.edu/~doug/htmls/MiXViews.html");
	system(cmdstring);
}

void
Controller::quit() {
	if(confirm("Please confirm quit.")) {
		_sIsQuitting = true;
		busy(false);
		closeChain();
	}
	else {
		busy(false);
		_sIsQuitting = false;
	}
}

// this method will be moved into the Editor class

void
Controller::changeName() {
	Request request("Change File Name:");
	String newName = fileName();
	request.appendValue("New Name:", &newName);
	if(handleRequest(request) == Yes) {
		String fullName = _editor->defaultDir();
		if(!FileName::isFullPathName(newName)) {
			fullName += "/";
			fullName += newName;
		}
		else
			fullName = newName;
		if(!DataFile::exists(fullName) || Application::confirm(
				"A file with that name already exists",
				FileName::isFullPathName(newName) ?
					"at the path you have specified."
					: "in the default directory for this file type.",
				"Use this file name anyway?"))
			setFileName(newName); // dont save name with added path
	}
}

void
Controller::setProgramOptions() {
	GlobalOptionSetter options;
	_editor->applyModifier(options);
}

// The remainder of these methods handle messages that need to get back to
// the user, whether they are errors or just queries for information

void
Controller::alert(const char *msg1, const char *msg2,
		const char *msg3, const char* button) {
	AlertRequest message(msg1, msg2, msg3, button);
	handleRequest(message);
}

boolean
Controller::confirm(const char *msg1, const char *msg2,
		const char *msg3, Response r, const char* btn1, const char* btn2) {
	ConfirmRequest message(msg1, msg2, msg3, r, btn1, btn2);
	return (handleRequest(message) == Yes);
}

Response
Controller::choice(const char *msg1, const char *msg2, const char *msg3,
		Response r, const char* btn1, const char* btn2, const char* btn3) {
	ChoiceRequest message(msg1, msg2, msg3, r, btn1, btn2, btn3);
	return (Response) handleRequest(message);
}
