// aiff_header.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <assert.h>
#include <stdarg.h>
#include "localdefs.h"
#include "application.h"
#include "aiff_header.h"
#include "comment.h"
#include "datafile.h"
#include "typeconvert.h"

// this file contains two completely separate implementations of the
// AIFFSoundHeader class -- one for SGI machines and one for all others

#ifdef sgi

AIFFSoundHeader::AIFFSoundHeader(DataType format, int rate, int chans,
    double peak, AFtype compression) : SoundHeader(format, rate, chans, peak, 0),
        audioFileHandle(nil), compression_type(compression) {
    header_type = Aifc;
    AFseterrorhandler((AFerrfunc) &AIFFSoundHeader::errorHandler);    
}

AIFFSoundHeader::~AIFFSoundHeader() {
    if(audioFileHandle) AFclosefile(audioFileHandle);
}

AFtype
AIFFSoundHeader::frameCount() {
        return (audioFileHandle) ? AFgetframecnt(audioFileHandle, AF_DEFAULT_TRACK) : 0;
}

int
AIFFSoundHeader::readInfo(DataFile* file) {
    BUG("AIFFSoundHeader::readInfo()");
    int fd = file->fdesc();
    AFtype fileType;
    if ((fileType = AFidentifyfd(fd)) < 0) {
	if (!Application::globalResourceIsTrue("ReadRawFiles"))
	    Application::alert("SoundHeader::readInfo: Sound file is not an AIFF-C or AIFF file.");
    }
    else if((audioFileHandle = AFopenfd(fd, "r", AF_NULL_FILESETUP))
            != AF_NULL_FILEHANDLE) {
        AFtype sampType, bitsPerSamp;
        if (fileType == AF_FILE_MPEG1BITSTREAM)
            header_type = Mpeg1;
        AFgetsampfmt(audioFileHandle, AF_DEFAULT_TRACK, &sampType, &bitsPerSamp);
        compression_type = AFgetcompression(audioFileHandle, AF_DEFAULT_TRACK);
        data_type = (bitsPerSamp <= 8) ? SignedCharData
            : (bitsPerSamp <= 16) ? ShortData :
            (bitsPerSamp <= 32) ? IntData : NoData;
        nchans = int(AFgetchannels(audioFileHandle, AF_DEFAULT_TRACK));
        samprate = int(AFgetrate(audioFileHandle, AF_DEFAULT_TRACK));
        AFtype totalFrames = AFgetframecnt(audioFileHandle, AF_DEFAULT_TRACK);
        off_t totalSize = totalFrames * nchans * sampleSize();
        if(compression_type == AF_COMPRESSION_NONE && file->dataSize() < totalSize) {
            Application::alert("Warning:",
                "File size does not match header frame count.");
            totalSize = file->dataSize();
        }
        setDataSize(totalSize);
        setDataOffset(file->dataSize() - dataSize());
        return checkHeader();
    }
    return false;
}

int
AIFFSoundHeader::readComment(DataFile*) {
    AFtype* miscIDs = nil;
    AFtype numIDs = AFgetmiscids(audioFileHandle, miscIDs);    // get size
    miscIDs = new AFtype[numIDs];
    AFgetmiscids(audioFileHandle, miscIDs);                    // get actual IDs
    for(int idx=0; idx < int(numIDs); idx++) {
        AFtype id = miscIDs[idx];
        AFtype chunkType = AFgetmisctype(audioFileHandle, id);
        // look for either annotation or author chunk, whichever comes first
        if(chunkType == AF_MISC_AIFF_ANNO || chunkType == AF_MISC_AIFF_AUTH) {
            AFtype chunkSize = AFgetmiscsize(audioFileHandle, id);
            char* buf = new char[chunkSize + 1];
            if(AFreadmisc(audioFileHandle, id, (void *) buf, chunkSize)
                    == chunkSize) {
                if(buf != nil)
                    setComment(buf);
            }
            delete [] buf;
            break;
        }
    }
    delete [] miscIDs;
    return true;
}

int
AIFFSoundHeader::seekInFile(DataFile*, int byteOffset) {
    AFtype frameOffset = byteOffset / (sampleSize() * nChans());
    return (AFseekframe(audioFileHandle, AF_DEFAULT_TRACK, frameOffset) >= 0);
}

int
AIFFSoundHeader::writeInfo(DataFile* file) {
    BUG("AIFFSoundHeader::writeInfo()");
    DataType type = dataType();
    if (!isValid(type)) {
	Application::alert("AIFF-C files do not support this sample format");
	return false;
    }
    AFfilesetup fileSetup = AFnewfilesetup();
    AFinitfilefmt(fileSetup, AF_FILE_AIFFC);
    AFinitchannels(fileSetup, AF_DEFAULT_TRACK,  AFtype(nChans()));
    AFinitrate(fileSetup, AF_DEFAULT_TRACK, double(sampleRate()));
    AFtype sampBits = (type == SignedCharData) ? 8 :
                    (type == MuLawData || type == ShortData) ? 16 : 32;
    AFinitcompression(fileSetup, AF_DEFAULT_TRACK,
        (type == MuLawData) ? AF_COMPRESSION_G711_ULAW : AF_COMPRESSION_NONE);
    AFinitsampfmt(fileSetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, sampBits);
    // if comment is available, initialize for its addition to the file
    Comment* com = getComment();
    AFtype commLen = (com == nil) ? 0 : com->len() + 1;
    if(commLen > 1) {
        AFtype idlist[1];
        idlist[0] = 314159;
        AFinitmiscids(fileSetup, idlist, 1L);
        AFinitmisctype(fileSetup, idlist[0], AF_MISC_AIFF_ANNO);
        AFinitmiscsize(fileSetup, idlist[0], commLen);
    }
    // this will truncate file if it already exists
    int status = true;
    if((audioFileHandle = AFopenfd(file->fdesc(), "w", fileSetup)) == NULL) {
        Application::alert("SoundHeader::writeInfo: error in AFopenfd().");
        status = false;
    }
    AFfreefilesetup(fileSetup);
    return status;
}

int
AIFFSoundHeader::writeComment(DataFile *) {
    Comment* com = getComment();
    AFtype commLen = (com == nil) ? 0 : com->len() + 1;
    int status = true;
    if(commLen > 1) {
        if(AFwritemisc(audioFileHandle, 314159, (void *) ((char *) *com), commLen) != commLen)
            status = false;
    }
    return status;
}

// these return number of frames written

AFtype
AIFFSoundHeader::readSound(char** buffer, AFtype frames) {
    BUG("AIFFSoundHeader::readSound()");
    AFtype framesRead = 0;
    if(audioFileHandle != nil) {
        AFtype totalFrames = frameCount();
        AFtype readFrames = frames <= totalFrames ? frames : totalFrames;
        framesRead = AFreadframes(audioFileHandle, AF_DEFAULT_TRACK,
            *buffer, readFrames);
    }
    else
        Application::alert("AIFFSoundHeader::readSound:  null file handle");
    return framesRead;
};

AFtype
AIFFSoundHeader::writeSound(char* buffer, AFtype frames) {
    BUG("AIFFSoundHeader::writeSound()");
    AFtype framesWritten = 0;
    if(audioFileHandle != nil) {
        framesWritten = AFwriteframes(audioFileHandle, AF_DEFAULT_TRACK,
            buffer, frames);
        if(framesWritten > 0 && AFsyncfile(audioFileHandle) < 0) {
            Application::alert("AIFFSoundHeader::writeSound:  AFsyncfile error.");
            framesWritten = 0;
        }
    }
    else
        Application::alert("AIFFSoundHeader::readSound: null file pointer.");
    return framesWritten;
};

int
AIFFSoundHeader::checkHeader() {
    int status = Super::checkHeader();
    if(status)
        if(compression_type == AF_COMPRESSION_UNKNOWN) {
            Application::alert("checkHeader: unknown compression format.");
            status = false;
        }
    return status;
}

boolean
AIFFSoundHeader::isValid(DataType t) {
    return (t == SignedCharData || t == MuLawData
        || t == ShortData || t == IntData);
}

void
AIFFSoundHeader::errorHandler(long arg1, const char* fmt, ...) {
    char string[256];
    va_list args;
    va_start(args, fmt);
    vsprintf(string, fmt, args);
    va_end(args);
#ifdef debug
    fprintf(stderr, "Error handler called: \"%s\"\n", string);
    fflush(stderr);
#endif
    Application::alert(string);
}

#else    /* sgi */

#include "valuesetter.h"

AIFFSoundHeader::AIFFSoundHeader(DataType format, int rate,
        int chans, double peak, AFtype compression)
        : SoundHeader(format, rate, chans, peak, FORM),
          compression_type(compression) {
    header_type = Aifc;
}

AIFFSoundHeader::~AIFFSoundHeader() {
}

int
AIFFSoundHeader::readInfo(DataFile* file) {
    file->seek(0);
    int status = false;
    char *errString = nil;
    long formOffset = 0;
    long blockSize = 0;
    int nsamps = 0;
    FormChunk formChunk;
    if (formChunk.read(file)) {
        if (!formChunk.isValid()) {
            Application::alert("readInfo:", "Header form type is not AIFF or AIF-C.");
            goto Error;
        }
        boolean isAIFC = formChunk.isAIFC();
        off_t totalSize = formChunk.size() + 8;
        StandardChunk chunk;
        long chunkSize = 0;
        do {            // search for COMM chunk
            status = file->skip(chunkSize + odd(chunkSize)).good()
                && chunk.read(file);
            chunkSize = chunk.size();
        } while(status && !chunk.isA("COMM"));
        if(!status)    {    // reached EOF
            Application::alert("readInfo:", "Missing common data chunk.");
            goto Error;
        }
        else {
            CommonData commonData(isAIFC);
            if(!(status = commonData.read(file)))
                Application::error("readInfo:", "Error while reading common data chunk.");
            else
	        status = parseCommonData(commonData, &nsamps);
	    if (!status)
                goto Error;
        }
        chunkSize = 0;
        do {            // search for SSND chunk
            status = file->skip(chunkSize + odd(chunkSize)).good()
                && chunk.read(file);
            chunkSize = chunk.size();
        } while(status && !chunk.isA("SSND"));
        if(!status)    {    // reached EOF
            Application::alert("readInfo:", "Missing sound data chunk.");
            goto Error;
        }
        else {
            setDataSize(chunkSize - 8);    // minus size of two longs
            SoundData soundData;
            if((status = soundData.read(file)) == true) {
                formOffset = soundData.offset;
                blockSize = soundData.blockSize;
            }
        }
        if (blockSize != 0)    {
            Application::alert("readInfo:", "Sound data chunk blocksize != 0.");
            status = false;
        }
        else {
	    if (compressionType() == AF_COMPRESSION_NONE) {
        	char msg[80];
        	off_t computedSize = nsamps * nchans * sampleSize();
        	if(dataSize() != computedSize) {
                    sprintf(msg, "SSND chunksize (%d) != size from frame count (%d).",
                	dataSize(), computedSize);
                    Application::alert("readInfo:", msg);
                    setDataSize(computedSize);
        	}
        	else if(totalSize != file->dataSize()) {
                    sprintf(msg, "Actual file size (%d) != header file size (%ld).",
                	file->dataSize(), totalSize);
                    Application::alert("readInfo:", msg);
        	}
	    }
            file->skip(formOffset);        // if any -- usually 0
            setDataOffset(int(file->tell()));
            status = true;
        }
    }
    else
        Application::error("readInfo:", "read error while reading FORM chunk.");
    Error: ;
    return status && checkHeader();
}

boolean
AIFFSoundHeader::parseCommonData(const AIFFSoundHeader::CommonData& commonData,
                                 int *nsamps) {
    boolean status = true;
    nchans = commonData.chans;
    compression_type = commonData.compression();
    if (compression_type == AF_COMPRESSION_NONE) {
	switch (commonData.sampleSize) {
	case 8:
            data_type = SignedCharData;
            break;
	case 16:
            data_type = ShortData;
            break;
	case 32:
            data_type = IntData;
            break;
	default:
            Application::alert("readInfo:",
	                       "unable to read files with this sample width");
            data_type = NoData;    // for now, nothing else allowed
	    status = false;
            break;
	};
    }
    else if (compression_type == AF_COMPRESSION_G711_ULAW) {
        data_type = MuLawData;
    }
    else if (compression_type == AIFC_FORMAT_FLOAT) {
        data_type = FloatData;
    }
    else if (compression_type == AIFC_FORMAT_DOUBLE) {
        data_type = DoubleData;
    }
    else {
        Application::alert("readInfo:",
	                   "unable to read files with compression tag:",
			   commonData.compressionName);
        data_type = NoData;	// no other compressions supported
	status = false;
    }
    *nsamps = commonData.frames;
    samprate = int(float(commonData.rate));
    return status;
}

int
AIFFSoundHeader::readComment(DataFile* file) {
    return true;
}

int
AIFFSoundHeader::writeInfo(DataFile* file) {
    static const long headerLength = 4 + 12 + 8 + 38 + 8 + 8;
    file->seek(0);
    FormChunk formChunk(dataSize() + headerLength);
    int status = formChunk.write(file);
    FormatChunk formatChunk;
    FormatVersion formatVersion;
    status = status && formatChunk.write(file) && formatVersion.write(file);
    CommonChunk commonChunk;
    status = status && commonChunk.write(file);
    if(status) {
        CommonData common(
            true,                    // for now, write out AIF-C standard
            nChans(),
            dataSize() / (nChans() * sampleSize()),
            short(sampleSize() << 3),
            double(sampleRate())
        );
        status = common.write(file);
        SoundChunk soundChunk(dataSize());
        SoundData soundData;
        status = status && soundChunk.write(file) && soundData.write(file);
    }
    if(status == false)
        Application::error("AIFFSoundHeader::writeInfo:");
    return status;
}

int
AIFFSoundHeader::writeComment(DataFile* file) {
    return true;
}

int
AIFFSoundHeader::checkHeader() {
    if(compression_type != AF_COMPRESSION_NONE
       && compression_type != AIFC_FORMAT_FLOAT
       && compression_type != AIFC_FORMAT_DOUBLE
       && compression_type != AF_COMPRESSION_G711_ULAW) {
        Application::alert("This version can only handle uncompressed, float, double, and ulaw formats");
        return false;
    }
    return Super::checkHeader();
}

boolean
AIFFSoundHeader::isValid(DataType t) {
    return (t == SignedCharData || t == ShortData || t == IntData || t == MuLawData
		|| t == FloatData || t == DoubleData);
}

//********

// in readAndSet(), we add one to the read length if it is even, because the
// initial length byte would make the entire length (and that of the Common
// chunk) odd, and we must read the pad to get the correct offset.

// in writeTo() we do the opposite

class ValueSetter<CompressionName> : public ValueSetterBase {
public:
    ValueSetter(CompressionName *name) : myName(name) {}
    redefined int readAndSet(DataFile *f) {
        char *nmPtr = (char *) myName;
        return (f->read(nmPtr, sizeof(char)).good()
	        && f->read(nmPtr + 1,
		           myName->length() + even(myName->length())).good());
    }
    redefined int writeTo(DataFile *f) {
        const char *nmPtr = (const char *) myName;
	// Add one to include initial length byte.
        boolean good = f->write(nmPtr, myName->length() + 1).good();
	if (good && even(myName->length())) {
	    char pad = 0;
	    good = f->write(&pad, 1).good();
	}
	return good;
    }
private:
    CompressionName *myName;
};

ValueSetterBase**
AIFFSoundHeader::FormChunk::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&formType);
    return setterList;
}

int
AIFFSoundHeader::FormChunk::read(DataFile* file) {
    return StandardChunk::doRead(file, StandardChunk::valueSetters())
        && doRead(file, valueSetters());
}

int
AIFFSoundHeader::FormChunk::write(DataFile* file) {
    return doWrite(file, StandardChunk::valueSetters())
        && doWrite(file, valueSetters());
}

boolean
AIFFSoundHeader::FormChunk::isValid() {
    return isA("FORM")
        && (formType == "AIFF" || formType == "AIFC") && size() > 0;
}

ValueSetterBase**
AIFFSoundHeader::FormatVersion::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&timeStamp);
    return setterList;
}

ValueSetterBase**
AIFFSoundHeader::CommonData::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&chans);
    delete setterList[1];
    setterList[1] = newValueSetter(&frames);
    delete setterList[2];
    setterList[2] = newValueSetter(&sampleSize);
    delete setterList[3];
    setterList[3] = newValueSetter(&rate);
    delete setterList[4]; setterList[4] = nil;
    delete setterList[5]; setterList[5] = nil;
    if(isAIFC) {
        setterList[4] = newValueSetter(&compressionType);
        setterList[5] = newValueSetter(&compressionName);
    }
    return setterList;
}

ValueSetterBase**
AIFFSoundHeader::SoundData::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&offset);
    delete setterList[1];
    setterList[1] = newValueSetter(&blockSize);
    return setterList;
}

#endif    /* sgi */

// the rest are the same for both platforms

boolean
AIFFSoundHeader::isMagic() {
    return (magic() == FORM || magic() == MROF || magic() == MPEG1_MAGIC);
}

boolean
AIFFSoundHeader::magicIsSwapped() { return magic() == MROF; }

void
AIFFSoundHeader::reset() {
    Super::reset();
    compression_type = AF_COMPRESSION_NONE;
}
