
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>

#include "typedefs.h"
#include "utils.h"
#include "config.h"

static FILE *handle = NULL;

static char buffer[strsize],tmp[strsize];

/* open configuration file */
bool cfg_open (const char *filename)
{
   return ((handle = fopen (filename,"r")) != NULL);
}

/* extract boolean value from configuration file */
bool cfg_get_bool (bool *var,const char *str)
{
   bool found = false;
   regmatch_t regs[3];
   fseek (handle,0L,SEEK_SET);
   while (!feof (handle) && !found)
	 {
		if (readline (handle,buffer) != NULL) return (false);
		trim (buffer);
		if (strlen (buffer) && !ereg_match ("^#",buffer))
		  {
			 if (ereg ("^([[:alnum:]_]+)[ \t]*=[ \t]*([Tt][Rr][Uu][Ee]|[Ff][Aa][Ll][Ss][Ee])[ \t]*$",buffer,regs,3))
			   {
				  ereg_extract (tmp,buffer,regs[1]);
				  if (!strcmp (str,tmp))
					{
					   found = true;
					   ereg_extract (tmp,buffer,regs[2]);
					   *var = !strcasecmp (tmp,"true");
					}
			   }
		  }
	 }
   return (found);
}

/* extract integer value from configuration file */
bool cfg_get_int (int *var,const char *str)
{
   bool found = false;
   regmatch_t regs[5];
   fseek (handle,0L,SEEK_SET);
   while (!feof (handle) && !found)
	 {
		if (readline (handle,buffer) != NULL) return (false);
		trim (buffer);
		if (strlen (buffer) && !ereg_match ("^#",buffer))
		  {
			 if (ereg ("^([[:alnum:]_]+)[ \t]*=[ \t]*(((0x)?[[:xdigit:]]+)|[[:digit:]]+)[ \t]*$",buffer,regs,5))
			   {
				  ereg_extract (tmp,buffer,regs[1]);
				  if (!strcmp (str,tmp))
					{
					   found = true;
					   ereg_extract (tmp,buffer,regs[2]);
					   *var = strtol (tmp,(char **) NULL,0);
					}
			   }
		  }
	 }
   return (found);
}

/* extract string from configuration file */
bool cfg_get_string (char *var,const char *str)
{
   bool found = false;
   regmatch_t regs[3];
   fseek (handle,0L,SEEK_SET);
   while (!feof (handle) && !found)
	 {
		if (readline (handle,buffer) != NULL) return (false);
		trim (buffer);
		if (strlen (buffer) && !ereg_match ("^#",buffer))
		  {
			 if (ereg ("^([[:alnum:]_]+)[ \t]*=[ \t]*\"(.*)\"[ \t]*$",buffer,regs,3))
			   {
				  ereg_extract (tmp,buffer,regs[1]);
				  if (!strcmp (str,tmp))
					{
					   found = true;
					   ereg_extract (var,buffer,regs[2]);
					}
			   }
		  }
	 }
   return (found);
}

/* extract specified type from configuration file */
bool cfg_get (int type,void *var,const char *str)
{
   switch (type)
	 {
	  case CFG_TYPE_BOOL:
		return (cfg_get_bool ((bool *) var,str));
	  case CFG_TYPE_INT:
		return (cfg_get_int ((int *) var,str));
	  case CFG_TYPE_STRING:
		return (cfg_get_string ((char *) var,str));
	  default:
		return (false);
	 }
}

static const char *get_whitespace (const char *str)
{
   while (*str == ' ' || *str == '\t') str++;
   return (str);
}

static const char *get_int (int *i,const char *str)
{
   char *endptr;
   *i = strtol (str,&endptr,0);
   return (endptr);
}

static bool extract_array (int *array,int *n,const char *str)
{
   const char *tmp1 = str,*tmp2;
   int i;
   bool another;
   *n = 0;
   do
	 {
		another = false;
		tmp2 = get_whitespace (tmp1);
		tmp1 = get_int (&i,tmp2);
		/* no integer found? */
		if (tmp1 == tmp2) return (false);
		tmp2 = get_whitespace (tmp1);
		if (*tmp2 == ',')
		  {
			 /* we expect another integer */
			 another = true;
			 tmp2++;
		  }
		/* if we're not at the end, it must be a mistake */
		else if (*tmp2 != '\0') return (false);
		/* make sure we can take another element */
		if (*n == arraysize) return (false);
		array[*n] = i;
		(*n)++;
		tmp1 = tmp2;
	 }
   while (*tmp1 != '\0');
   return (!another);
}

/* extract an array of integers from configuration file */
bool cfg_get_int_array (int *array,int *n,const char *str)
{
   bool found = false;
   regmatch_t regs[3];
   fseek (handle,0L,SEEK_SET);
   while (!feof (handle) && !found)
	 {
		if (readline (handle,buffer) != NULL) return (false);
		trim (buffer);
		if (strlen (buffer) && !ereg_match ("^#",buffer))
		  {
			 if (ereg ("^([[:alnum:]_]+)[ \t]*=[ \t]*(.+)[ \t]*$",buffer,regs,3))
			   {
				  ereg_extract (tmp,buffer,regs[1]);
				  if (!strcmp (str,tmp))
					{
					   ereg_extract (tmp,buffer,regs[2]);
					   if (extract_array (array,n,tmp)) found = true;
					}
			   }
		  }
	 }
   return (found);
}

/* close configuration file */
void cfg_close ()
{
   if (handle != NULL) fclose (handle);
}

