/* vim:tabstop=4:expandtab:shiftwidth=4
 * 
 * Idesk -- XIcon.cpp
 *
 * Copyright (c) 2002, Chris (nikon) (nikon@sc.rr.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *      Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *      
 *      Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *      
 *      Neither the name of the <ORGANIZATION> nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 * (See the included file COPYING / BSD )
 */

#include "XIcon.h"
#include "XShadowImage.h"
#include "SimpleSvgImage.h"
#include "XPngImage.h"

XIcon::XIcon(AbstractContainer * cont, AbstractConfig * con,
              AbstractIconConfig * iConfig) : AbstractIcon(cont, con, iConfig)
{
    DesktopIconConfig * dIcon = dynamic_cast<DesktopIconConfig *>(iconConfig);
    
    iconFilename = dIcon->getIconFilename();
    commandArray = dIcon->getCommandArray();
    x = dIcon->getX();
    y = dIcon->getY();
    lX = lY = 0;

    dragged = dragging = false;

    if (dIcon->isSvg())
        image = new SimpleSvgImage(cont, this, config, iconConfig);
    else if (dIcon->isPng())
        image = new XPngImage(cont, this, config, iconConfig);
    //else if (dIcon->isXcf())
        //
    else
    {
        cout << "Unknown file format: \n";
        // implement way to skip icon and not segfault
    }

    
    XIdeskImage * xImage = dynamic_cast<XIdeskImage *>(image);

    xImage->configure();
    xImage->createWindow();
    xImage->lowerWindow();
    
    if (iconConfig->getCaption() == "" || iconConfig->getCaption() == " ")
        captionOn = false;
    else
    {
        captionOn = true;
        caption = new XCaption(cont, this, config, iconConfig);
        XCaption * xCaption = dynamic_cast<XCaption *>(caption);
        xCaption->createWindow();
        xCaption->lowerWindow();
    }
}

XIcon::~XIcon()
{
    delete image;
    if (captionOn)
        delete caption;
}

void XIcon::lowerIcon()
{
    XIdeskImage * xImage = dynamic_cast<XIdeskImage *>(image);
    xImage->lowerWindow();

    if (captionOn)
    {
        XCaption * xCaption = dynamic_cast<XCaption *>(caption);
        xCaption->lowerWindow();
    }
}

Window * XIcon::getImageWindow()
{
    return dynamic_cast<XIdeskImage *>(image)->getWindow();
}

Window * XIcon::getCaptionWindow()
{
    if (captionOn)
        return dynamic_cast<XCaption *>(caption)->getWindow();
    else
        return NULL;
}

AbstractImage * XIcon::getImage()
{
    return image;
}

AbstractCaption * XIcon::getCaption()
{
    if (captionOn)
        return caption;
    else
        return NULL;
}

void XIcon::dragButtonPress(XEvent ev)
{
    lowerIcon();
    lX = ev.xbutton.x_root;
    lY = ev.xbutton.y_root;
    dragging = True;
}

void XIcon::dragMotionNotify(XEvent ev)
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);
    XIdeskImage * xImage = dynamic_cast<XIdeskImage *>(image);

    dragged = True;

    x  -= lX - ev.xmotion.x_root;
    y  -= lY - ev.xmotion.y_root;
    lX  = ev.xmotion.x_root;
    lY  = ev.xmotion.y_root;
    
    fixPosition(x, y);
    xImage->moveWindow(x, y);
    xImage->refreshIcon();
    if (captionOn)
    {
        XCaption * xCaption = dynamic_cast<XCaption *>(caption);
        xCaption->moveWindow(captionXShift(), captionYShift());
        xCaption->draw();
    }
    
}

void XIcon::dragButtonRelease(XEvent ev)
{
    if( dragged )
    {
        XDesktopContainer * xContainer =
            dynamic_cast<XDesktopContainer *>(container);
        XIdeskImage * xImage = dynamic_cast<XIdeskImage *>(image);
        DesktopConfig * dConfig = dynamic_cast<DesktopConfig *>(config);

        if (dConfig->getSnapState())
        {
            findSnapPosition(x, y);

            fixPosition(x, y);

            xImage->moveWindow(x, y);
            if (captionOn)
            {
                XCaption * xCaption = dynamic_cast<XCaption *>(caption);
                xCaption->moveWindow(captionXShift(), captionYShift());
            }
        }
        draw();
        xContainer->saveIcon(this);
    }
    dragging=False;
    dragged=False;
}

void XIcon::findSnapPosition(int &xCord, int &yCord)
{   
    DesktopConfig * dConfig =
            dynamic_cast<DesktopConfig *>(config);
    XDesktopContainer * xContainer =
            dynamic_cast<XDesktopContainer *>(container);

    int xMid = xCord + image->getWidth()/2;
    int yMid = yCord + image->getHeight()/2;

    xCord = xMid - xMid % (dConfig->getSnapWidth());
    yCord = yMid - yMid % (dConfig->getSnapHeight());

    xCord += dConfig->getSnapWidth()/2 - image->getWidth()/2;
    yCord += dConfig->getSnapHeight()/2 - image->getHeight()/2;

}


bool XIcon::setImage(AbstractImage * img)
{
    bool returnBool = false;

    if (img)
    {
        image = img;
        returnBool = true;
    }
    
    return returnBool;
}

bool XIcon::setCaption(AbstractCaption * cap)
{
    bool returnBool = false;

    if (cap && captionOn)
    {
        caption = cap;
        returnBool = true;
    }
    
    return returnBool;
}


void XIcon::draw()
{
    image->draw();
    if (captionOn)
    {
        XCaption * xCaption = dynamic_cast<XCaption *>(caption);
        xCaption->draw();
    }
}

void XIcon::show()
{
    moveImageWindow();
    mapImageWindow();
    //don't initially map caption for the hover effect
    initMapCaptionWindow();
}

void XIcon::drawText()
{
    caption->draw();
}

void XIcon::updateText()
{
    if (captionOn)
        caption->updateText();
}

void XIcon::mouseOverEffect()
{
    image->mouseOverEffect();
    if (captionOn)
        caption->mouseOverEffect();
}

void XIcon::mouseOffEffect()
{
    image->mouseOffEffect();
    if (captionOn)
        caption->mouseOffEffect();
}

void XIcon::save()
{
    DesktopIconConfig * dIcon = dynamic_cast<DesktopIconConfig *>(iconConfig);

    dIcon->saveIcon(x, y);
}

void XIcon::fixPosition(int &xCord, int &yCord)
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);
    
    if( xCord < 0 ) xCord = 0;
    if( yCord < 0 ) yCord = 0;

    if( xCord + image->getWidth() > xContainer->widthOfScreen() )
        xCord = xContainer->widthOfScreen() - image->getWidth();

    if( yCord + image->getHeight() > xContainer->heightOfScreen() )
        yCord = xContainer->heightOfScreen() - image->getHeight();
}

int XIcon::captionXShift()
{
    if (captionOn)
        return x + ( image->getWidth() - caption->getFontWidth() )/2;
    else
        return 0;
}

int XIcon::captionYShift()
{
    if (captionOn)
        return y + image->getHeight() + 4;
    else
        return 0;
}

int XIcon::getFontHeight()
{
    if (captionOn)
        return caption->getFontHeight();
    else
        return 0;
}

int XDesktopContainer::heightOfScreen()
{
    return HeightOfScreen(DefaultScreenOfDisplay(display));
}

void XIcon::moveImageWindow()
{
    XIdeskImage * xImage = dynamic_cast<XIdeskImage *>(image);

    xImage->moveWindow(x, y);
}

void XIcon::mapImageWindow()
{
    XIdeskImage * xImage = dynamic_cast<XIdeskImage *>(image);

    xImage->mapWindow();

}

void XIcon::mapCaptionWindow()
{
    if (captionOn)
    {
        XCaption * xCaption = dynamic_cast<XCaption *>(caption);

        xCaption->mapWindow();
    }
}

void XIcon::initMapCaptionWindow()
{
    if (captionOn)
    {
        XCaption * xCaption = dynamic_cast<XCaption *>(caption);

        xCaption->initMapWindow();
    }
}

