module LowLevel
  ( openHatFile		-- :: CString -> CString -> IO ()
  , getBridgeValue	-- :: IO FileNode
  , getErrorLoc		-- :: IO FileNode
  , getErrorMessage	-- :: IO CString
  , hatVersionNumber	-- :: String

  , FileNode(..)
  , nil			-- :: FileNode
  , unevaluated		-- :: FileNode
  , entered		-- :: FileNode
  , interrupted		-- :: FileNode
  , lambda		-- :: FileNode

  , NodeType(..)
  , nodeType		-- :: FileNode -> NodeType
  , SimpleNodeType(..)
  , simpleNodeType	-- :: FileNode -> SimpleNodeType

  , getParentNode	-- :: FileNode -> FileNode
  , getResult		-- :: FileNode -> Bool -> FileNode
  , peekResult		-- :: FileNode -> FileNode

  , getValue		-- :: FileNode -> String
  , getValueMod		-- :: FileNode -> String
  , getFixity		-- :: FileNode -> Int
  , isLiteral		-- :: FileNode -> Bool
  , isConstructor	-- :: FileNode -> Bool
  , isConstrFields	-- :: FileNode -> Bool
  , isLambda		-- :: FileNode -> Bool

  , getAtom		-- :: FileNode -> String
  , getAtomMod		-- :: FileNode -> String
  , getAtomFixity	-- :: FileNode -> Int

  , getSubExprs		-- :: FileNode -> [FileNode]
  , getFieldLabels	-- :: FileNode -> [String]

  , getSrcRef		-- :: FileNode -> FileNode
  , srcRefFile		-- :: FileNode -> String
  , srcRefLine		-- :: FileNode -> Int
  , srcRefCol		-- :: FileNode -> Int

  , getDefnRef		-- :: FileNode -> FileNode
  , defnFile		-- :: FileNode -> String
  , defnLine		-- :: FileNode -> Int
  , defnCol		-- :: FileNode -> Int

  , peekTrace		-- :: FileNode -> FileNode
  ) where

import FFIExtensions

-- Reference into the .hat file
newtype FileNode = FileNode {int::Int}
  deriving (Eq)
--deriving (Eq,Show)
nil         = FileNode 0
unevaluated = FileNode 1
entered     = FileNode 2
interrupted = FileNode 3
lambda      = FileNode 4

-- There are 25 basic types of node, encoded in the lower 5 bits of the tag
-- They fall into four broad classes: module info, srcpos, expressions, atoms.
data NodeType
  = Module
  | SrcPos
  | ExpApp
  | ExpValueApp
  | ExpChar
  | ExpInt
  | ExpInteger
  | ExpRat
  | ExpRational
  | ExpFloat
  | ExpDouble
  | ExpValueUse
  | ExpConstUse
  | ExpConstDef
  | ExpGuard
  | ExpCase
  | ExpIf
  | ExpFieldUpdate
  | ExpProjection
  | ExpHidden
  | ExpForward
  | ExpDoStmt
  | AtomVariable
  | AtomConstructor
  | AtomAbstract
  deriving (Eq)

instance Enum NodeType where
  toEnum  0 = Module
  toEnum  1 = SrcPos
  toEnum  2 = ExpApp
  toEnum  3 = ExpValueApp
  toEnum  4 = ExpChar
  toEnum  5 = ExpInt
  toEnum  6 = ExpInteger
  toEnum  7 = ExpRat
  toEnum  8 = ExpRational
  toEnum  9 = ExpFloat
  toEnum 10 = ExpDouble
  toEnum 11 = ExpValueUse
  toEnum 12 = ExpConstUse
  toEnum 13 = ExpConstDef
  toEnum 14 = ExpGuard
  toEnum 15 = ExpCase
  toEnum 16 = ExpIf
  toEnum 17 = ExpFieldUpdate
  toEnum 18 = ExpProjection
  toEnum 19 = ExpHidden
  toEnum 20 = ExpForward
  toEnum 21 = ExpDoStmt
  toEnum 26 = AtomVariable
  toEnum 27 = AtomConstructor
  toEnum 28 = AtomAbstract

-- For most purposes, we don't care about the exact node type, and a
-- simplified division of nodes into kinds is useful.
data SimpleNodeType
  = NodeModule		-- Module
  | NodeSrcPos		-- SrcPos
  | NodeApplication	-- ExpApp, ExpValueApp
  | NodeBasicValue	-- ExpChar, ExpInt, ..., ExpDouble
  | NodeIdentifier	-- ExpValueUse
  | NodeCAF		-- ExpConstUse, ExpConstDef
  | NodeConditional	-- ExpGuard, ExpCase, ExpIf
  | NodeSugar		-- ExpFieldUpdate, ExpDoStmt
  | NodeSpecial		-- ExpProjection, ExpHidden, ExpForward
  | NodeAtom		-- AtomVariable, AtomConstructor, AtomAbstract
  deriving (Eq)


-- For opening files, and and collecting values from the bridge file.
foreign import ccall openHatFile    :: CString -> CString -> IO ()
foreign import ccall getBridgeValue :: IO FileNode
foreign import ccall getErrorLoc    :: IO FileNode
foreign import ccall errorMessage   :: IO CString
foreign import ccall versionNumber  :: IO CString
getErrorMessage  :: IO String
getErrorMessage = do msg <- errorMessage
                     peekCString msg
hatVersionNumber :: String
hatVersionNumber = unsafePerformIO $ do num <- versionNumber
                                        peekCString num


-- Find out what node type we have a reference to.
nodeType :: FileNode -> NodeType
nodeType n = toEnum (getNodeType n)
foreign import ccall getNodeType   :: FileNode -> Int

-- Give a simple node type to a reference.
simpleNodeType :: FileNode -> SimpleNodeType
simpleNodeType n =
    case nodeType n of
      Module -> NodeModule
      SrcPos -> NodeSrcPos
      ExpApp      -> NodeApplication
      ExpValueApp -> NodeApplication
      ExpValueUse -> NodeIdentifier
      ExpConstUse -> NodeCAF
      ExpConstDef -> NodeCAF
      ExpFieldUpdate -> NodeSugar
      ExpDoStmt      -> NodeSugar
      ExpGuard -> NodeConditional
      ExpCase  -> NodeConditional
      ExpIf    -> NodeConditional
      ExpProjection -> NodeSpecial
      ExpHidden     -> NodeSpecial
      ExpForward    -> NodeSpecial
      AtomVariable    -> NodeAtom
      AtomConstructor -> NodeAtom
      AtomAbstract    -> NodeAtom
      _  -> NodeBasicValue

-- For any node type, get its parent.  If it doesn't have one, the
-- zero node is returned.
foreign import ccall parentNode :: FileNode -> FileNode
getParentNode :: FileNode -> FileNode
getParentNode n = peekTrace (parentNode n)

-- For any node, return its result pointer.  Only an application, CAF,
-- case/if/guard, field update, or hidden node actually has one - in all
-- other cases we get back 0.  The Boolean denotes whether to stop at
-- the first Hidden node.
foreign import ccall getResult     :: FileNode -> Bool -> FileNode

-- Because getResult stops one link in the chain *before* an Unevaluated,
-- Entered, Interrupted, or Lambda node, peekResult looks just one step
-- down the result chain.
foreign import ccall peekResult    :: FileNode -> FileNode

-- For nodes of value kind, we get back a string representation of the value
-- (Integer, Double etc) or name (identifier, constructor, etc),
-- and its fixity.  The predicate isLiteral reports True for values of basic
-- types like Int, Char, Double etc, and isConstructor identifies Constrs.
-- The predicate isLambda can only be used on an ExpValueUse node.
foreign import ccall getNm         :: FileNode -> CString
foreign import ccall getNmMod      :: FileNode -> CString
foreign import ccall getFixity     :: FileNode -> Int
foreign import ccall isLiteral     :: FileNode -> Bool
foreign import ccall isConstructor :: FileNode -> Bool
foreign import ccall isConstrFields:: FileNode -> Bool
foreign import ccall isLambda      :: FileNode -> Bool
getValue :: FileNode -> String
getValue n = unsafePerformIO (peekCString (getNm n))
getValueMod :: FileNode -> String
getValueMod n = unsafePerformIO (peekCString (getNmMod n))

data Atom;
foreign import ccall readAtomAt    :: FileNode -> Ptr Atom
foreign import ccall identName     :: Ptr Atom -> CString
foreign import ccall identModName  :: Ptr Atom -> CString
foreign import ccall identFixity   :: Ptr Atom -> Int
getAtom :: FileNode -> String
getAtom n = unsafePerformIO (peekCString (identName (readAtomAt n)))
getAtomMod :: FileNode -> String
getAtomMod n = unsafePerformIO (peekCString (identModName (readAtomAt n)))
getAtomFixity :: FileNode -> Int
getAtomFixity n = identFixity (readAtomAt n)

-- For Trace nodes excluding kind TNm, get any arguments.
getSubExprs :: FileNode -> [FileNode]
getSubExprs n = let arity = getExpArity n
                in (map (getExpArg n) [0..arity])
foreign import ccall getExpArity   :: FileNode -> Int
foreign import ccall getExpArg     :: FileNode -> Int -> FileNode

-- For a ExpFieldUpdate node only, get the list of updated labels.
getFieldLabels :: FileNode -> [String]
getFieldLabels n = let arity = getExpArity n
                   in map (getAtom . getFieldLabel n) [0..(arity-1)]
foreign import ccall getFieldLabel :: FileNode -> Int -> FileNode


-- For any node type, get its source reference.  If it doesn't have one,
-- we get a null pointer (0) back.
foreign import ccall getSrcRef     :: FileNode -> FileNode

-- For a variable, constructor, or application, get the Atom node that
-- contains its definition information.  If it isn't a variable,
-- constructor, or application, we get a null pointer (0) back.
foreign import ccall getDefnRef    :: FileNode -> FileNode

-- Only for a SrcPos node, get the module name, line or column.
-- If the node is null, return dummy values ("", 0, 0).
foreign import ccall getSrcRefFile :: FileNode -> CString
foreign import ccall srcRefLine    :: FileNode -> Int
foreign import ccall srcRefCol     :: FileNode -> Int
srcRefFile :: FileNode -> String
srcRefFile n = unsafePerformIO (peekCString (getSrcRefFile n))

-- Only for a Atom node, get the definition module name, line or column.
-- If the node is null, return dummy values ("", 0, 0).
foreign import ccall getDefnFile   :: FileNode -> CString
foreign import ccall defnLine      :: FileNode -> Int
foreign import ccall defnCol       :: FileNode -> Int
defnFile :: FileNode -> String
defnFile n = unsafePerformIO (peekCString (getDefnFile n))



-- Look past any SATs, indirections, or hidden nodes, to `real' trace.
foreign import ccall peekTrace     :: FileNode -> FileNode

