//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Neumann.hpp,v 1.2 2003/06/10 18:26:22 delpinux Exp $

// this class allows to define Neumann Boundary Conditions

#ifndef _NEUMANN_HPP_
#define _NEUMANN_HPP_

#include <TinyVector.hpp>
#include <TinyMatrix.hpp>

#include <PDECondition.hpp>

#include <UserFunction.hpp>
/*!
  \class Neumann

   This class sets Neumann boundary conditions
   \f$ A \nabla u\cdot n = g \f$ on \f$ \gamma \f$.

   where \f$ A \f$ is the matrix associated to the operator
   \f$ \nabla\cdot A\nabla \f$ and \f$ n \f$ the normal to 
   \f$ \gamma \$f. So \f$ g \f$, the given data is the \b
   conormal derivative of \f$ u \f$.

   \author Stphane Del Pino
 */
class Neumann
  : public PDECondition
{
private:
  //! The function to impose as a Neumann boundary condition.
  ReferenceCounting<UserFunction> __g;

public:

  inline const real_t g(const TinyVector<3,real_t>& X) const
  {
    return (*__g)(X);
  }

  inline ReferenceCounting<UserFunction> gReference() const
  {
    return __g;
  }

  const std::string typeName() const
  {
    return std::string("Neumann");
  }

  //! Constructs a Neumann boundary condition.
  Neumann(ReferenceCounting<UserFunction> uf,
	  const size_t unknownNumber)
    : PDECondition(PDECondition::neumann, unknownNumber),
      __g(uf)
  {
    ;
  }

  //! Copy a Neumann boundary condition.
  Neumann(const Neumann& N)
    : PDECondition(N),
      __g(N.__g)
  {
    ;
  }

  //! Destrutor.
  ~Neumann()
  {
    ;
  }
};


#endif // _NEUMANN_HPP_

