//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Transform.hpp,v 1.2 2003/05/04 18:09:01 delpinux Exp $

// This class is the base class for all transformations such as
// Rotations, Translations, ...

#ifndef TRANSFORM
#define TRANSFORM

#include <string>

#include <Vertex.hpp>
#include <TransType.h>

/*!
  \class Transform

  This class is the base class for transformations that can be applied to
  Virtual Reality objects.

  \author Stphane Del Pino

  \todo
  \li try to move the enume TransType inside the class.
  \li make virtual function instead of switch in \p TypeName (then remove
  TransType?)
  \li change the name to Transformation?
  \li check if the use of Default constructor is needed in children classes.
 */
class Transform
{
protected:
  //! The type of the Transformation.
  TransType __type;
public:

  //! Read-only access to the type of the transformation.
  inline const TransType& type() const;

  //! Returns the type name of the Transform in a \p std::string.
  inline std::string typeName() const;

  //! Writes the Transform in a \p std::string.
  virtual std::string povWrite() const = 0;

  //! Applies the Transform to the point \a p.
  virtual TinyVector<3>& operator()(TinyVector<3>& p) const = 0;

  //! Applies the inverse Transform to the point \a p.
  virtual TinyVector<3>& inverse(TinyVector<3>& p) const = 0;

  //! Virtual destructor.
  virtual ~Transform()
  {
    ;
  }
};

//! Read-only access to the type of the transformation.
const TransType& Transform::type() const
{
  return __type;
}

//! Returns the type name of the Transform in a \p std::string.
std::string Transform::typeName() const
{
  std::string name;
  switch (__type) {
  case translation: {
    name = "translation";
    break;
  }
  case rotation: {
    name = "rotation";
    break;
  }
  case scale: {
    name = "scale";
    break;
  }
  case matrix: {
    name = "matrix";
    break;
  }
  default: {
    fferr(2) << __FILE__ << ':' << __LINE__
	     << ": Unknown Transformation\n";
    std::exit(1);
  }
  }
  return name;
}

#endif // TRANSFORM

