//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Plane.hpp,v 1.3 2004/02/08 18:44:19 delpinux Exp $


// This class provides a toolkit to manipulate Planes. It inherits from the
// Shape class.

#ifndef PLANE_HPP
#define PLANE_HPP

#include <Shape.hpp>
#include <Index.hpp>

class Cube;
class Cylinder;
class Cone;

/**
 * @file   Plane.hpp
 * @author Stephane Del Pino
 * @date   Sat Feb  7 21:30:37 2004
 * 
 * @brief This is the class which defines a Virtual Reality Plane.
 * 
 */

class Plane
  : public Shape
{
private:
  //! Two faces centers of the plane:
  TinyVector<3, real_t> __normal;

  //! Distance to the origine
  real_t __distance;

  //! vector unit.
  TinyVector<3, real_t> __unitNormal;

  //! Normal projection of the origine.
  TinyVector<3, real_t> __origine;

  //! returns true if the point \a p is inside the shape.
  inline bool inShape(const TinyVector<3, real_t>& p) const;

  //! Prints the Plane informations using the stream \a s.
  std::ostream& put(std::ostream& s) const;

public:

  //! Copies the Plane \a C.
  const Plane& operator = (const Plane& C);

  //! Builds a Plane based on vertices \a a and \a b
  Plane(const Vertex& normal, const real_t& distance);

  /** 
   * Constructs the plane corresponding to a given face number of a
   * cube
   * 
   * @param c the given cube
   * @param faceNumber the face number
   * 
   */
  Plane(const Cube& c, const size_t& faceNumber);

  /** 
   * Constructs the plane corresponding to a given face number of a
   * cylinder
   * 
   * @param c the given cylinder
   * @param faceNumber the face number
   * 
   */
  Plane(const Cylinder& c, const size_t& faceNumber);

  /** 
   * Constructs the plane corresponding to a given face number of a
   * cone
   * 
   * @param c the given cone
   * @param faceNumber the face number
   * 
   */
  Plane(const Cone& c, const size_t& faceNumber);

  //! Default constructor.
  Plane();

  //! Copy constructor.
  Plane(const Plane& P);

  //! Destructor.
  ~Plane()
  {
    ;
  }
};

//! Returns true if the point \a p is inside the Plane.
inline bool Plane::inShape(const TinyVector<3, real_t>& p) const
{
  const real_t d = __unitNormal * (p-__origine) ;
  return (d < 0); // __unitNormal is exterior
}

#endif // PLANE_HPP

