//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: InfiniteCone.hpp,v 1.1 2004/02/08 18:44:19 delpinux Exp $

#ifndef INFINITE_CONE_HPP
#define INFINITE_CONE_HPP

#include <TinyVector.hpp>
#include <Shape.hpp>

#include <iostream>

class Cone;

class InfiniteCone
  : public Shape
{
private:
  //! Two faces centers of the cone:
  Vertex __center1;
  Vertex __center2; 

  TinyVector<3, real_t> __axisVector;
  real_t __height;

  //! The two radius of the cone:
  real_t __radius1;
  real_t __radius2;

  inline bool inShape(const TinyVector<3, real_t>& p) const;

  std::ostream& put(std::ostream& os) const
  {
    os << "internal infinite cone";
    return os;
  }

public:
  /** 
   * operator= overloading
   * 
   * @param i a given InfiniteCone
   * 
   * @return the modified current object
   */
  const InfiniteCone& operator=(const InfiniteCone& i);

  /** 
   * Extracts a from the cone \a c an InfiniteCone
   * 
   * @param c the given cone
   * 
   */
  InfiniteCone(const Cone& c);

  /** 
   * Copy constructor
   * 
   * @param c a given InfiniteCone
   * 
   */
  InfiniteCone(const InfiniteCone& c);

  /** 
   * 
   * Destructor
   * 
   */
  ~InfiniteCone()
  {
    ;
  }
};

inline bool
InfiniteCone::inShape(const TinyVector<3, real_t>& p) const
{
  TinyVector<3> c1p = p - __center1;
  const real_t alpha = __axisVector*c1p / __height;
  const real_t h2 = c1p*c1p - alpha*alpha;

  return (h2<std::pow(alpha/__height*__radius2 + (1-alpha/__height)*__radius1, 2));
}
#endif // INFINITE_CONE_HPP
