//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Cone.hpp,v 1.4 2004/02/08 18:44:19 delpinux Exp $


// This class provides a toolkit to manipulate Cones. It inherits from the
// Shape class.

#ifndef CONE_HPP
#define CONE_HPP

#include <Shape.hpp>
#include <Index.hpp>
/*!
  \class Cone
  This is the class which defines a Virtual Reality Cone.

  \author Stphane Del Pino.
*/
class Cone
  : public Shape
{
private:
  friend class InfiniteCone;
  friend class Plane;

  //! Two faces centers of the cone:
  Vertex __center1;
  Vertex __center2; 

  TinyVector<3, real_t> __axisVector;
  real_t __height;

  //! The two radius of the cone:
  real_t __radius1;
  real_t __radius2;

  //! returns true if the point \a p is inside the shape.
  inline bool inShape(const TinyVector<3, real_t>& p) const;

  //! Prints the Cone informations using the stream \a s.
  std::ostream& put(std::ostream& s) const;

public:

  //! Copies the Cone \a C.
  const Cone& operator = (const Cone& C);

  //! Builds a Cone based on vertices \a a and \a b
  Cone(const Vertex& a, const Vertex& b, const real_t r1, real_t r2);

  //! Default constructor.
  Cone();

  //! Copy constructor.
  Cone(const Cone& C);

  //! Destructor.
  ~Cone()
  {
    ;
  }
};

//! Returns true if the point \a p is inside the Cone.
inline bool Cone::inShape(const TinyVector<3, real_t>& p) const
{
  TinyVector<3> c1p = p - __center1;

  const real_t alpha = __axisVector*c1p / __height;

  if (alpha<0)
    return false;
  if (alpha>__height)
    return false;

  const real_t h2 = c1p*c1p - alpha*alpha;

  return (h2<std::pow(alpha/__height*__radius2 + (1-alpha/__height)*__radius1, 2));
}

#endif // CONE_HPP

