/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "enum.h"
#include "logging.h"

/**
 *	create_volume_button_activated - callback to create a compatibility volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Create button in the Create comaptibility
 *	volume dialog window is activated.
 */
int create_volume_button_activated(struct menu_item *item)
{
	int rc;
	struct selwin *selwin = item->user_data;

	rc = evms_create_compatibility_volume(get_selected_handle(selwin->clist));
	update_status(_("Create"), rc);

	if (rc == 0) {
		((struct dialog_window *)selwin)->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	filter_volume_create - a clist_filter_func that keeps we can use to create a compat volume
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be use to create a compatibility volume.
 */
int filter_volume_create(object_handle_t handle, void *user_data)
{
	return evms_can_create_compatibility_volume(handle);
}

/**
 *	create_evms_volume_button_activated - callback to create an EVMS volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Create button in the Create EVMS
 *	volume dialog window is activated.
 */
int create_evms_volume_button_activated(struct menu_item *item)
{
	int rc;
	char *name;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	name = dialog->user_data;
	rc = evms_create_volume(get_selected_handle(selwin->clist), name);
	update_status(_("Create"), rc);

	if (rc == 0) {
		dialog->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	filter_evms_volume_create - a clist_filter_func that keeps we can use to create a EVMS volume
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be use to create a native EVMS volume.
 */
int filter_evms_volume_create(object_handle_t handle, void *user_data)
{
	return evms_can_create_volume(handle);
}

/**
 *	on_volume_create_dialog_delete - callback invoked when dialog used to create a volume is deleted
 *	@dialog: the dialog window
 *
 *	This routine handles cleanup for the dialog used to create EVMS volumes.
 *	We free any string that may be associated with the dialog containing 
 *	a volume name.
 */
void on_volume_create_dialog_delete(struct dialog_window *dialog)
{
	g_free(dialog->user_data);
}

/**
 *	create_volume_create_dialog - create a dialog to handle volume creation
 *	@handle: object handle; non-zero when create initiated from context popup menu
 *	@activate_cb: the function to invoke when the Create button is activated
 *	@filter_func: the function to filter the objects used to create the volume with
 *	@select_cb: the function to invoke when an object in the clist is selected
 *
 *	This routine creates a dialog window for use by the routines that allow the
 *	creation of EVMS and compatibility type volumes.
 */
struct dialog_window *create_volume_create_dialog(object_handle_t handle,
						menuitem_activate_cb activate_cb,
						clist_filter_func filter_func,
						clist_select_item_cb select_cb)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(select_cb ? _("Create EVMS Volume") : _("Create Compatibility Volume"),
					NULL, NULL,
					_("_Create"),
					activate_cb,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Object"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_topmost_objects, filter_func,
				format_standard_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle, format_standard_item, NULL, NULL);

	if (select_cb != NULL) {
		set_clist_select_item_cb(selwin->clist, select_cb);
		set_dialog_delete_cb(dialog, (dialog_delete_cb)on_volume_create_dialog_delete);
	}

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	return dialog;
}

/**
 *	show_create_volume_dialog - display dialog to create a compatibity volume
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to create a compatibility
 *	volume from one or more available objects.
 **/
int show_create_volume_dialog(object_handle_t handle)
{
	struct dialog_window *dialog;

	dialog = create_volume_create_dialog(handle,
					(menuitem_activate_cb)create_volume_button_activated,
					filter_volume_create, NULL);
	process_modal_dialog(dialog);
	return 0;
}

/**
 *	get_volume_name - prompt the user for the name for an EVMS volume
 *	@old_name: if non-NULL, the previous volume name
 *
 *	This routine prompts the user for a name for an EVMS volume.
 */
char *get_volume_name(char *old_name)
{
	char *name;
	int starty, len;
	WINDOW *win, *border_win;
	PANEL *panel, *border_panel;

	name = g_malloc0(EVMS_VOLUME_NAME_SIZE + 1);

	panel = create_centered_popup_window(getmaxy(stdscr) - 8, getmaxx(stdscr) - 10, WHITE_BKGD);
	border_panel = (PANEL *)panel_userptr(panel);
	win = panel_window(panel);
	border_win = panel_window(border_panel);

	if (old_name != NULL) {
		char *prev_name;

		starty = getmaxy(win) / 2 - 2;
		prev_name = g_strdup_printf(_("Previous Name: %s"), old_name);
		print_centered(win, starty, prev_name);
		g_free(prev_name);
	}

	starty = getmaxy(win) / 2;
	len = strlen(_("Name:: "));

	print_centered(border_win, 0, _("Enter Volume Name"));
	mvwaddnstr(win, starty, 2, _("Name:: "), len);

	show_popup_window(panel);

	if (read_string(win, starty, len + 2, name, EVMS_VOLUME_NAME_SIZE) != OK) {
		g_free(name);
		name = NULL;
	}
	delete_popup_window(panel);

	return name;
}

/**
 *	object_for_evms_volume_selected - callback activated when object for EVMS volume selected
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is invoked when an object in the clist is selected for use as
 *	the topmost object for a volume. We prompt for a volume name as store the 
 *	name in the dialog->user_data replacing any previous name.
 */
int object_for_evms_volume_selected(struct clist *clist, struct clist_item *item)
{
	char *prev_name;
	struct dialog_window *dialog = clist->user_data;

	prev_name = dialog->user_data;
	dialog->user_data = get_volume_name(prev_name);
	if (dialog->user_data != NULL) {
		row_selected(clist, item);
		g_free(prev_name);
	}

	return (dialog->user_data != NULL ? 0 : ENODATA);
}

/**
 *	show_create_evms_volume_dialog - display dialog to create an EVMS volume
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to create a native
 *	EVMS volume from one or more available objects.
 **/
int show_create_evms_volume_dialog(object_handle_t handle)
{
	struct dialog_window *dialog;

	dialog = create_volume_create_dialog(handle,
					(menuitem_activate_cb)create_evms_volume_button_activated,
					filter_evms_volume_create,
					(clist_select_item_cb)object_for_evms_volume_selected);
	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_create_vol_menuitem_activated - compatibility volume creation initiated from context menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to create a compatibility volume from the object selected
 *	from the context popup menu if a view.
 */
int context_create_vol_menuitem_activated(struct menu_item *item)
{
	return show_create_volume_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_create_vol_menuitem_activated - compatibility volume creation initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Create->Compatibility Volume Actions pulldown
 *	menu item.
 */
int actions_create_vol_menuitem_activated(struct menu_item *item)
{
	return show_create_volume_dialog(0);
}

/**
 *	context_create_evms_vol_menuitem_activated - EVMS volume creation initiated from context menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to create a native EVMS volume from the object selected
 *	from the context popup menu if a view.
 */
int context_create_evms_vol_menuitem_activated(struct menu_item *item)
{
	return show_create_evms_volume_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_create_evms_vol_menuitem_activated - EVMS volume creation initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Create->EVMS Volume Actions pulldown
 *	menu item.
 */
int actions_create_evms_vol_menuitem_activated(struct menu_item *item)
{
	return show_create_evms_volume_dialog(0);
}
