/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "readable.h"
#include "logging.h"

/**
 *	delete_selection_window - cleanup resources used by the standard selection window
 *	@selwin: the selection window
 *
 *	This routine cleans up resources for a selection window before deleting the window.
 */
void delete_selection_window(struct selwin *selwin)
{
	delete_clist(selwin->clist);
}

/**
 *	show_selection_window - this displays and processes events for a selection window
 *	@selwin: the selection window
 *
 *	This routine displays the elements of a selection window.
 **/
void show_selection_window(struct selwin *selwin)
{
	draw_clist(selwin->clist);
}

/**
 *	process_selection_window_events - process events related to the selection window
 *	@selwin: the selection window
 *	@key: the input keyboard value
 *
 *	This routine is the event handler registered for the standard selection window.
 *	It basically handles the events specific to the selection window which are
 *	clist events.
 */
int process_selection_window_events(struct selwin *selwin, int key)
{
	return process_clist_events(selwin->clist, key);
}

/**
 *	mark_selection - simply marks a selected row
 *	@item: the clist item that was selected
 *
 *	This routine simply identifies that the item was selected by placing
 *	an 'X' for the first column text.
 */
inline void mark_selection(struct clist_item *item)
{
	*(item->row_text[0]) = 'X';
}

/**
 *	unmark_selection - simply unmarks a selected row
 *	@item: the clist item that was unselected
 *
 *	This routine simply removes any 'X' that may have been there had
 *	the item been selected.
 */
inline void unmark_selection(struct clist_item *item)
{
	*(item->row_text[0]) = ' ';
}

/**
 *	row_selected - the default callback that enables Next button when selection is made
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is the default callback for the row_selected event in a selection
 *	window dialog. It is invoked anytime that a row is selected. It ensures that
 *	the "Next" button is sensitive.
 */
int row_selected(struct clist *clist, struct clist_item *item)
{
	make_next_button_sensitive((struct dialog_window *)clist->user_data);
	mark_selection(item);
	return 0;
}

/**
 *	row_unselected - the default callback that disables Next button when last row unselected
 *	@clist: the clist for the row item unselected
 *	@item: the clist item that was unselected
 *
 *	This routine is the default callback for the row_unselected event in a selection
 *	window dialog. It is invoked anytime that a row is unselected. It ensures that
 *	the "Next" button is made insensitive if this was the last row unselected.
 */
int row_unselected(struct clist *clist, struct clist_item *item)
{
	if (g_list_length(clist->selections) == 0)
		make_next_button_insensitive((struct dialog_window *)clist->user_data);
	unmark_selection(item);
	return 0;
}

/**
 *	create_selection_window - create a standard selection window containing a clist
 *	@title: the title text for the dialog window
 *	@help_text: the text for the help window activated by the Help button
 *	@prompt: a description prompting the user what to do in the dialog
 *	@next_button_text: the label text for the Next button or NULL for default
 *	@next_button_callback: the callback for when the Next button menuitem is activated
 *	@prev_button_text: the label text for the Previous button or NULL for default
 *	@prev_button_callback: the optional callback for when the Previous button is activated
 *	@user_data: whatever info the user wants to keep along with this dialog
 *
 *	This routine allows the creation of a standard selection window that
 *	contains a clist subwindow to allow selecting one or more rows, a
 *	"button bar" with a Help, Cancel, Previous and Next button.
 */
struct selwin *create_selection_window(const char *title, const char *help_text,
					char *prompt, char *next_button_text,
					menuitem_activate_cb next_button_callback,
					char *prev_button_text,
					menuitem_activate_cb prev_button_callback,
					gpointer user_data)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = g_new0(struct selwin, 1);
	dialog = (struct dialog_window *)selwin;

	init_dialog_window(dialog, title, help_text,
				(dialog_event_handler)process_selection_window_events,
				(dialog_show_func)show_selection_window,
				(dialog_delete_func)delete_selection_window,
				(dialog_delete_cb)NULL,
				next_button_text,
				next_button_callback,
				prev_button_text,
				prev_button_callback,
				user_data);

	/*
	 * Create a three column clist for the window leaving enough space for the
	 * column titles, the button bar and the descriptive text. The caller can change
	 * the number of columns afterwards if the default of 3 is insufficient.
	 */
	selwin->clist = create_clist(dialog->win, 3, getmaxy(dialog->win) - 9, 
				getmaxx(dialog->win) - 2,
				getbegy(dialog->win) + 3,
				getbegx(dialog->win) + 1,
				selwin);

	set_clist_color_attrs(selwin->clist, WHITE_BKGD, STD_COLOR, MENU_COLOR);
	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)row_selected);
	set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)row_unselected);

	mvwhline(dialog->win, 2, 1, ACS_HLINE, getmaxx(dialog->win) - 2);

	if (prompt == NULL) {
		char *button_text;

		if (next_button_text == NULL)
			button_text = g_strdup(_("Next"));
		else 
			button_text = remove_accel_identifier(next_button_text);

		prompt = g_strdup_printf(_("Use the spacebar key to make selections and choose %s to continue"),
					button_text);
		print_centered(dialog->win, getmaxy(dialog->win) - 4, prompt);
		g_free(prompt);
		g_free(button_text);
	} else {
		print_centered(dialog->win, getmaxy(dialog->win) - 4, prompt);
	}

	/*
	 * Size the columns as if the first one was for the 'X' that indicates the selection
	 * has been made, the second for the description or name of the object and the last
	 * one the size of the object.
	 */
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.50),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(selwin->clist, 2, calc_clist_column_width(selwin->clist, 0.12),
				get_clist_column_end(selwin->clist, 1),
				CLIST_TEXT_JUSTIFY_RIGHT);

	return selwin;
}

/**
 *	format_standard_item - return column strings for a row in the standard selection clist
 *	@handle: the thing handle
 *	@not_used: extra info that we don't pay attention to
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a thing placed
 *	in the standard clist containing the size and name of an object. This is
 *	a clist_format_func type function called by clist_populate().
 */
int format_standard_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(" "));
		switch (object->type) {
		case DISK:
		case REGION:
		case SEGMENT:
		case EVMS_OBJECT:
			g_ptr_array_add(text, g_strdup(object->info.object.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
			break;
		case CONTAINER:
			g_ptr_array_add(text, g_strdup(object->info.container.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.container.size));
			break;
		case VOLUME:
			g_ptr_array_add(text, g_strdup(object->info.volume.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.volume.vol_size));
			break;
		case PLUGIN:
			g_ptr_array_add(text, g_strdup(object->info.plugin.long_name));
			break;
		default:
			log_warning("%s: Type %u was not processed.\n", __FUNCTION__, object->type);
			break;
		}
		evms_free(object);
	}
	return rc;
}

/**
 *	get_selected_handles - returns a list of the handles for the selected rows in a clist
 *	@clist: the selection list
 *
 *	This routine builds a singly linked list containing the engine handles
 *	associated with each selected row in a clist. The caller should issue
 *	a g_slist_free() on GSList when it is no longer needed.
 */
GSList *get_selected_handles(struct clist *clist)
{
	GList *element;
	GSList *handles = NULL;

	element = clist->selections;
	while (element != NULL) {
		struct clist_item *item;
	
		item = element->data;
		handles = g_slist_append(handles, item->user_data);
		element = g_list_next(element);
	}
	return handles;
}

/**
 *	get_selected_handle - get the handle for the selected row in a single select clist
 *	@clist: the selection list
 *
 *	This routine allows retrieving the handle associated with a row in clist
 *	in single select mode.
 */
engine_handle_t get_selected_handle(struct clist *clist)
{
	GSList *selection;
	engine_handle_t handle = 0;

	selection = get_selected_handles(clist);
	
	if (selection != NULL) {
		handle = GPOINTER_TO_UINT(selection->data);
		g_slist_free(selection);
	}
	return handle;
}

/**
 *	get_selected_data - get the user data for the selected row in a single select clist
 *	@clist: the selection list
 *
 *	This routine allows retrieving the user data associated with a row in clist
 *	in single select mode.
 */
void *get_selected_data(struct clist *clist)
{
	void *data = NULL;
	GSList *selection;

	selection = get_selected_handles(clist);
	
	if (selection != NULL) {
		data = selection->data;
		g_slist_free(selection);
	}
	return data;
}

/**
 *	create_list_dialog - create a dialog with a non-selectable clist
 *	@columns: the number of columns in the clist
 *	@user_data: whatever the caller wants to store in the various window elements
 *
 *	This routine creates a dialog window with a read-only (item selections are ignored)
 *	clist. This dialog is derived from the selwin dialog except we play some tricks
 *	with some of it's button labels (hide Cancel and Prev and call Next "OK" for example)
 *	and setup the select and unselect row handlers in the clist to basically reject
 *	these events.
 */
struct selwin *create_list_dialog(char *title, const char *help_text, char *prompt,
					int columns, void *user_data)
{
	struct selwin *list_dialog;
	struct dialog_window *dialog;

	list_dialog = create_selection_window(title, help_text, prompt, _("_OK"),
				(menuitem_activate_cb)close_window_button_activated,
				NULL, NULL, user_data);

	dialog = (struct dialog_window *)list_dialog;
	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_menu_item_visibility(dialog->prev_button, FALSE);
	set_menu_item_visibility(dialog->cancel_button, FALSE);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);

	set_clist_select_item_cb(list_dialog->clist, (clist_select_item_cb)disallow_select_events);
	set_clist_unselect_item_cb(list_dialog->clist, (clist_unselect_item_cb)disallow_select_events);

	return list_dialog;
}
