/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <frontend.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "create.h"
#include "enum.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "task.h"
#include "plugin.h"
#include "object.h"
#include "logging.h"

/**
 *	can_create_from_freespace_object - determine if freespace object can create something
 *	@handle: the handle of a freespace object to see if suitable for create
 *
 *	This routine determines if a create task on the plugin for
 *	the given freespace storage object could work.
 */
gboolean can_create_from_freespace_object(object_handle_t handle)
{
	gboolean result = FALSE;
	handle_object_info_t *object;
    
	if ((evms_get_info(handle, &object)) == 0) {
		if (object->info.object.data_type == FREE_SPACE_TYPE)
			result = is_task_acceptable_object(object->info.object.plugin,
								handle, EVMS_Task_Create);
		evms_free(object);
	}
	return result;
}

/**
 *	can_create_feature_object - determine if given object can be create a feature
 *	@handle: the handle of the storage object to check
 *
 *	This routine determines if the given object can be
 *	used to create any feature object.
 */
gboolean can_create_feature_object(object_handle_t handle)
{
	gboolean result = FALSE;
	handle_object_info_t *object;
    
	if ((evms_get_info(handle, &object)) == 0) {
		/*
		 * Check only topmost objects
		 */
		if (object->info.object.parent_objects == NULL ||
			object->info.object.parent_objects->count == 0) {
			handle_array_t *plugins;

			if (enum_feature_plugins(NULL, &plugins) == 0) {
				gint i;

				for (i = 0; i < plugins->count && result == FALSE; i++) {
					result = is_task_acceptable_object(plugins->handle[i],
								handle, EVMS_Task_Create);
				}
				g_free(plugins);
			}
		}
		evms_free(object);
	}
	return result;
}

/**
 *	create_feature_create_task_dialog - create task, set selected object and create options/objects dialog
 *	@handle: the acceptable object handle
 *	@feature: the feature plugin handle
 *	@list: the dialog list
 *
 *	This routine creates the EVMS_Task_Create task, sets the given object as the selected
 *	object and creates the option dialog to configure the feature if no other acceptable
 *	objects else creates the acceptable objects dialog.
 */
struct dialog_window *create_feature_create_task_dialog(object_handle_t handle, object_handle_t feature,
							struct dialog_list *list)
{
	int rc, min, max;
	task_handle_t task;
	GHashTable *hash_table;
	char *title, *verb, *help;
	struct dialog_window *dialog = NULL;
	task_requirements_t requirements;

	help = NULL;
	verb = _("Create");
	title = _("Create Feature");

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);

	rc = create_task(feature, EVMS_Task_Create, &requirements, &task, &min, &max);

	if (rc == 0) {
		g_hash_table_insert(hash_table, "task", GUINT_TO_POINTER(task));

		switch (requirements) {            
		case TASK_REQUIRES_OPTIONS_ONLY:
			dialog = create_task_options_dialog(task, title, verb, help, hash_table);
			break;
		case TASK_REQUIRES_OBJECTS_ONLY:
		case TASK_REQUIRES_OBJECTS_AND_OPTIONS:
			set_selected_object(task, handle);
			if ((min <= 1 && object_is_only_available(task, handle)) ||
				(min <= 1 && max == 1)) {
				if (requirements == TASK_REQUIRES_OBJECTS_ONLY)
					dialog = create_task_confirmation_dialog(list, task,
									title, verb, help);
				else
					dialog = create_task_options_dialog(task, title, verb,
									help, hash_table);
			} else {
				dialog = create_acceptable_objects_selection_dialog(task, title, verb, 
									requirements, help, hash_table);
				reload_acceptable_objects(((struct selwin *)dialog)->clist);
			}
			break;
		case TASK_REQUIRES_NO_ADDITIONAL_ARGS:
			dialog = create_task_confirmation_dialog(list, task, title, verb, help);
			break;
		default:
			log_error("%s: Unable to determine task requirements.\n", __FUNCTION__);
			break;
		}
	}
	if (dialog != NULL) {
		dialog->user_data = hash_table;

		g_hash_table_insert(hash_table, "verb", g_strdup(verb));
		g_hash_table_insert(hash_table, "title", g_strdup(title));

		set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_task_dialog);
	} else {
		g_hash_table_destroy(hash_table);
	}
	return dialog;
}

/**
 *	feature_selection_button_activated - callback invoke when the feature is selected for create
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a feature to create in conjunction with the acceptable object. We then create
 *	a task now that we know the feature and the object to set as selected.
 */
int feature_selection_button_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	struct selwin *selwin = item->user_data;
	object_handle_t saved_plugin, curr_plugin;
	struct dialog_window *dialog = item->user_data;

	hash_table = dialog->user_data;
	curr_plugin = get_selected_handle(selwin->clist);
	saved_plugin = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "plugin"));
	
	if (curr_plugin != saved_plugin) {
		GList *next_dialog;
		object_handle_t handle;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected feature has changed we need to delete any
		 * dialogs that we may have built for the create feature task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		handle = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "handle"));
		new_dialog = create_feature_create_task_dialog(handle, curr_plugin, dialog->list);

		if (new_dialog != NULL) {
			g_hash_table_insert(hash_table, "plugin", GUINT_TO_POINTER(curr_plugin));
			append_dialog_to_list(new_dialog, dialog->list);
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	filter_features_for_create - keep the plugins that find the handle acceptable for create
 *	@handle: the plugin handle
 *	@user_data: contains the object handle
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given feature can be added to a certain volume.
 */
int filter_features_for_create(object_handle_t handle, void *user_data)
{
	return !is_task_acceptable_object(handle, GPOINTER_TO_UINT(user_data), EVMS_Task_Create);
}

/**
 *	delete_feature_object_plugin_dialog_cb - callback invoked when feature plugin dialog is being deleted
 *	@dialog: the feature plugin selection dialog
 *
 *	This routine is invoked when the feature plugin dialog is being deleted. We take
 *	this opportunity to destroy the hash table associated with the dialog window.
 */
void delete_feature_object_plugin_dialog_cb(struct dialog_window *dialog)
{
	g_hash_table_destroy(dialog->user_data);
}

/**
 *	create_feature_from_object - given an object, select it when creating a feature object
 *	@handle: the supposedly acceptable object for one or more features
 *
 *	This routine handles the presentation of features that claim the given handle
 *	is an acceptable object for the creation of a feature object. We create the
 *	task on the selected feature plugin, set the object as selected and allow the
 *	user to select other object or simply display the options dialog if there aren't
 *	any others.
 */
void create_feature_from_object(object_handle_t handle)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "handle", GUINT_TO_POINTER(handle));

	selwin = create_selection_window(_("Create Feature - Plugin Selection"),
					NULL, NULL, NULL,
					(menuitem_activate_cb)feature_selection_button_activated,
					NULL, NULL, hash_table);

	dialog = (struct dialog_window *)selwin;
	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("Feature"));

	clist_populate(selwin->clist, enum_feature_plugins, filter_features_for_create,
			format_standard_item, NULL, GUINT_TO_POINTER(handle));

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);
	set_dialog_delete_cb(dialog, (dialog_delete_cb)delete_feature_object_plugin_dialog_cb);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);
}

/**
 *	create_object_from_freespace - create a new object from a freespace object
 *	@title: the window title for the dialog
 *	@freespace: the handle of the freespace object
 *
 *	This routine simply allows a fast path to creating a region or segment
 *	from a freespace object by starting a create task with the plugin for
 *	the freespace object and setting the freespace object as the selected
 *	object. This way we can go straight on to the option configuration dialog.
 */
void create_object_from_freespace(char *title, object_handle_t freespace)
{
	int rc;
	task_handle_t task;
	struct dialog_window *dialog = NULL;

	rc = evms_create_task(get_plugin_handle_for_object(freespace), EVMS_Task_Create, &task);
	if (rc == 0) {
		rc = set_selected_object(task, freespace);
		if (rc == 0) {
			GHashTable *hash_table;
			struct dialog_list dialogs;

			hash_table = g_hash_table_new(g_str_hash, g_str_equal);
			g_hash_table_insert(hash_table, "task", GUINT_TO_POINTER(task));
			g_hash_table_insert(hash_table, "verb", g_strdup(_("Create")));

			dialog = create_task_options_dialog(task, title,
					_("Create"), NULL, hash_table);

			set_menu_item_visibility(dialog->prev_button, FALSE);
			set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_task_dialog);

			init_dialog_list(&dialogs);
			append_dialog_to_list(dialog, &dialogs);
			process_dialog_list_events(&dialogs);
			delete_dialog_list(dialogs.list);
		} else {
			evms_destroy_task(task);
			show_message_dialog(_("Set Selected Object Error"),
					_("Received an error communicating freespace object to plugin: %s"),
					evms_strerror(rc));
			log_error("%s: evms_set_selected_objects() returned error code %d.\n", __FUNCTION__, rc);
		}
	} else {
		show_message_dialog(_("Error starting Create task"),
				_("Received the following error starting Create task: %s"),
				evms_strerror(rc));
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	create_feature_menuitem_activated - initiate the creation of a feature object
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow creating a new
 *	feature object. If the menu item contains an non-zero handle then it is expected to
 *	be a topmost object that can be pre-selected as an acceptable object.
 */
int create_feature_menuitem_activated(struct menu_item *item)
{
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (handle == 0) {
		process_plugin_task(_("Create Feature Object"), EVMS_FEATURE, FALSE,
					EVMS_Task_Create);
	} else {
		create_feature_from_object(handle);
	}

	return 0;
}

/**
 *	create_region_menuitem_activated - initiate the creation of a storage region
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow creating a new
 *	region. If the menu item contains an non-zero handle then it is expected to
 *	be a freespace region. In that case we don't start from a plugin selection window.
 *	Rather, we go straight to creating the task and setting the freespace region
 *	as the selected region in the hopes that we end up only with the options
 *	configuration dialog to deal with.
 */
int create_region_menuitem_activated(struct menu_item *item)
{
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (handle == 0) {
		process_plugin_task(_("Create Storage Region"), EVMS_REGION_MANAGER, FALSE,
					EVMS_Task_Create);
	} else {
		create_object_from_freespace(_("Create Storage Region"), handle);
	}

	return 0;
}

/**
 *	create_container_menuitem_activated - initiate the creation of a storage container
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow creating a new
 *	storage container.
 */
int create_container_menuitem_activated(struct menu_item *item)
{
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (handle == 0) {
		process_plugin_task(_("Create Storage Container"), 0, TRUE,
					EVMS_Task_Create_Container);
	}

	return 0;
}

/**
 *	create_segment_menuitem_activated - initiate the creation of a segment
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow creating a new
 *	segment. If the menu item contains an non-zero handle then it is expected to
 *	be a freespace segment. In that case we don't start from a plugin selection window.
 *	Rather, we go straight to creating the task and setting the freespace segment
 *	as the selected segment in the hopes that we end up only with the options
 *	configuration dialog to deal with.
 */
int create_segment_menuitem_activated(struct menu_item *item)
{
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (handle == 0) {
		process_plugin_task(_("Create Disk Segment"), EVMS_SEGMENT_MANAGER, FALSE,
					EVMS_Task_Create);
	} else {
		create_object_from_freespace(_("Create Disk Segment"), handle);
	}

	return 0;
}
