/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: commit.c
 */

#include <unistd.h>
#include <sys/types.h>
#include <pthread.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "commit.h"
#include "thing.h"
#include "progress.h"
#include "logging.h"
#include "pixmap.h"
#include "views.h"
#include "main.h"
#include "readable.h"
#include "message.h"

/*
 * File scope globals
 */
static gboolean commit_in_progress = FALSE;

/*
 * Structure containing information needed by commit thread.
 */
typedef struct evmsgui_commit_info_s {
	gint rc;
	gboolean exit_after_commit;
} evmsgui_commit_info_t;

gboolean is_commit_in_progress(void)
{
	return commit_in_progress;
}

/*
 *
 *   void commit_cleanup (evmsgui_commit_info_t *)
 *   
 *   Description:
 *      This routine handles cleanup after a commit
 *      changes has completed.
 * 
 *   Entry:
 *      commit_info - contains information about commit
 *
 *   Exit:
 *      Clears main progress bar and re-enables main window 
 *      menu bar sensitivity.
 *
 */
void commit_cleanup(evmsgui_commit_info_t * commit_info)
{
	reset_progress_bar(GTK_PROGRESS_BAR(get_progress_bar_id()));
	gtk_widget_set_sensitive(get_menu_bar_id(), TRUE);
	gtk_widget_set_sensitive(get_button_toolbar_id(), TRUE);

	g_free(commit_info);
}

/*
 *
 *   void process_post_commit (evmsgui_commit_info_t *)
 *   
 *   Description:
 *      This routine handles post commit processing.
 * 
 *   Entry:
 *      commit_info - contains information about commit
 *
 *   Exit:
 *      Displays results screen if necessary and calls
 *      cleanup routine. If exit_after_commit flag is
 *      set, it emits the signal that destroys the 
 *      main window.
 *
 */
void process_post_commit(evmsgui_commit_info_t * commit_info)
{
	if (commit_info->rc != SUCCESS) {
		if (commit_info->rc == E_CANCELED) {
			set_status_bar_message(_("The Save Changes operation was cancelled"));
		} else {
			display_results_window(commit_info->rc, _("Save Changes"),
					       _("An error was encountered saving changes."),
					       NULL, TRUE, get_main_window_id());
		}
	} else {
		set_status_bar_message(_("All changes were saved successfully."));

		/*
		 * If this is a commit and exit, present a popup to show commit
		 * succeeded. Connect the gtk_widget_destroy of the main window
		 * with the results window's close button to close the app when
		 * clicked.
		 */

		if (commit_info->exit_after_commit) {
			display_results_and_exit(commit_info->rc, _("Save Changes"),
						 _("All changes were saved successfully."),
						 NULL, FALSE, get_main_window_id());

		} else {
			refresh_main_window_views();
		}
	}

	commit_cleanup(commit_info);
}

/*
 *
 *   static gboolean monitor_commit_progress (evmsgui_commit_info_t *)
 *   
 *   Description:
 *      This routine is called periodically to update the
 *      main progress bar mainly for lengthy commit operations.
 *      If commit is no longer in progress it calls the
 *      function that processes the commit results and
 *      cleanup.
 * 
 *   Entry:
 *      commit_info - contains information about commit
 *
 *   Exit:
 *      Progress bar is updated by 5% if commit in progress
 *      or call post commit function if commit ended.
 *
 */
static gboolean monitor_commit_progress(evmsgui_commit_info_t * commit_info)
{
	gboolean keep_timeout = TRUE;
	static gfloat pos = 0;
	static gint orientation = 0;

	gdk_threads_enter();

	if (commit_in_progress) {
		/*
		 * During commit time, progress callbacks take precendence.
		 * Therefore, we can't update the progress indicator during
		 * this time.
		 */
		if (get_progress_callbacks_started() == 0)
			update_indeterminate_progress(GTK_PROGRESS_BAR(get_progress_bar_id()),
						      &pos, &orientation);
	} else {
		/*
		 * Commit is done. Check its return code and
		 * do any cleanup work then return FALSE to
		 * have this timeout callback deregistered.
		 */

		process_post_commit(commit_info);
		keep_timeout = FALSE;
	}

	gdk_threads_leave();

	return keep_timeout;
}

/*
 *
 *   void commit_status_callback (gchar *)
 *   
 *   Description:
 *      This routine is called by the engine during commit
 *      to provide status of the commit phases.
 * 
 *   Entry:
 *      status - a pointer to a string containing the commit status
 *
 *   Exit:
 *      
 *      or call post commit function if commit ended.
 *
 */
void commit_status_callback(gchar * status)
{
	if (status) {
		gboolean is_main_thread = is_main_event_loop_thread();

		if (!is_main_thread)
			gdk_threads_enter();

		gtk_idle_add_priority(GTK_PRIORITY_HIGH,
				      (GtkFunction) update_status_bar_text_from_main_thread,
				      g_strdup(status));

		if (!is_main_thread)
			gdk_threads_leave();
	}
}

/*
 *
 *   void *commit_thread (void *)
 *   
 *   Description:
 *      This routine invokes the commit API. Once it
 *      the API call is complete it updates the return
 *      code in the commit info structure it was given
 *      and the thread terminates.
 * 
 *   Entry:
 *      args - address of argument given at thread create
 *
 *   Exit:
 *      Commit API is invoked, return code is saved and
 *      commit_in_progress flag is reset to FALSE.
 *
 */
void *commit_thread(void *arg)
{
	evmsgui_commit_info_t *commit_info;

	commit_info = (evmsgui_commit_info_t *) arg;

	get_ui_callback_table()->status = commit_status_callback;
	commit_info->rc = evms_commit_changes();

	commit_in_progress = FALSE;

	return NULL;
}

/*
 *
 *   void initiate_commit_changes (gboolean)
 *   
 *   Description:
 *      This routine starts a thread that invokes the commit API.
 *      It also registers a timeout handler that increments the
 *      main window progress indicator and monitors for commit
 *      completion.
 * 
 *   Entry:
 *      exit_after_commit - flag that indicates whether after a
 *                          successful commit we exit the app
 *
 *   Exit:
 *      Thread that invokes commit is started and progress bar
 *      timeout handler is added.
 *
 */
void initiate_commit_changes(gboolean exit_after_commit)
{
	gint rc;
	guint handler_id;
	pthread_t tid;
	evmsgui_commit_info_t *info;

	commit_in_progress = TRUE;

	info = g_malloc(sizeof(evmsgui_commit_info_t));

	/*
	 * Lookup the main_progress_bar and set activity mode to allow
	 * a steady, rhythmic progress. Add the monitor_commit_progress
	 * callback to a main event loop timeout scheduler queue.  
	 * Set the main window menubar to be insensitive while commit
	 * is in progress. Finally, start the commit thread.
	 */

	set_status_bar_message(_("Saving changes..."));

	gtk_widget_set_sensitive(get_menu_bar_id(), FALSE);
	gtk_widget_set_sensitive(get_button_toolbar_id(), FALSE);

	info->exit_after_commit = exit_after_commit;

	gtk_progress_set_activity_mode(GTK_PROGRESS(get_progress_bar_id()), TRUE);
	handler_id = gtk_timeout_add(50, (GtkFunction) monitor_commit_progress, info);

	rc = pthread_create(&tid, NULL, commit_thread, info);

	if (rc != SUCCESS) {
		/*
		 * This stinks. I can't run commit on a separate thread.
		 * The best I can do is issue it here and pray it doesn't
		 * freeze the app for very long. BUGBUG: There should be
		 * a better way or should I just report the error and 
		 * not commit?
		 */

		log_error("%s: pthread_create() failed! Return code is %d.\n", __FUNCTION__, rc);

		commit_thread(info);
		gtk_timeout_remove(handler_id);
		process_post_commit(info);
	}
}

/*
 *
 *   void display_nothing_to_commit_popup (void)
 *   
 *   Description:
 *      This routine display a small popup that indicates
 *      that there are no changes available to be committed.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */
void display_nothing_to_commit_popup(void)
{
	display_popup_window(_("Save Changes"), _("There are no changes to save."));
}

/*
 *
 *   void on_commit_button_clicked (GtkButton *, gpointer *)
 *   
 *   Description:
 *      This routine calls the function that initiates the
 *      commit changes. It also dismisses the dialog 
 *      containing the button that caused this callback 
 *      to get invoked.
 * 
 *   Entry:
 *      button    - the commit button clicked
 *      user_data - contains either a gboolean which
 *                  determines whether an exit should
 *                  occur after a successful commit                  
 *
 *   Exit:
 *      Commit changes is invoked and the button's parent dialog
 *      is destroyed.
 *
 */
void on_commit_button_clicked(GtkButton * button, gpointer * user_data)
{
	gboolean exit_after_commit = GPOINTER_TO_INT(user_data);

	initiate_commit_changes(exit_after_commit);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 *
 *   void populate_clist_with_changed_objects (GtkCList *, change_record_array_t *)
 *
 *   Description:
 *      This routine populates the given changed objects clist with
 *      the name of the volumes, containers that have changed and
 *      why.
 * 
 *   Entry:
 *      changes - array containing changed object names and reasons
 *
 *   Exit:
 *     The clist is populated with information on object changed for commit
 *
 */
void populate_clist_with_changed_objects(GtkCList * clist, change_record_array_t * changes)
{
	gint i;
	gint row;
	gchar *text[MAX_CO_COLUMNS];

	for (i = 0; i < changes->count; i++) {
		text[CO_ICON_COLUMN] = "";
		text[CO_NAME_COLUMN] = changes->changes_pending[i].name;
		text[CO_REASON_COLUMN] =
		    changes_flag_to_string(changes->changes_pending[i].changes);

		row = clist_append_row(clist, text);

		if (row != -1)
			set_clist_row_pixmap(clist, row, changes->changes_pending[i].type);

		g_free(text[CO_REASON_COLUMN]);
	}
}

/*
 *
 *   void on_details_button_clicked (GtkButton *, gpointer *)
 *   
 *   Description:
 *      This routine displays a dialog listing the objects
 *      that indicate they have changes pending.
 * 
 *   Entry:
 *      details_button - the details button clicked
 *      user_data - not used
 *
 *   Exit:
 *      A dialog is created and displayed populate with a list
 *      of objects that report changes.
 *
 */
void on_details_button_clicked(GtkButton * details_button, gpointer * user_data)
{
	boolean changes_pending;
	change_record_array_t *changes = NULL;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *text_box_frame;
	GtkWidget *scrolledwindow;
	GtkWidget *clist;
	GtkWidget *icon_col_label;
	GtkWidget *object_col_label;
	GtkWidget *reason_col_label;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *button;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	vbox = gtk_vbox_new(FALSE, 0);
	text_box_frame = gtk_frame_new(NULL);
	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	clist = gtk_clist_new(MAX_CO_COLUMNS);
	icon_col_label = gtk_label_new("");
	object_col_label = gtk_label_new(_("Name"));
	reason_col_label = gtk_label_new(_("Reason"));
	hseparator = gtk_hseparator_new();
	hbuttonbox = gtk_hbutton_box_new();
	button = gtk_button_new();

	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	get_ok_pixmap(&pixmap, &mask);
	add_pixmap_label_to_button(button, pixmap, mask, _("_OK"), accel_group);

	gtk_widget_show(vbox);
	gtk_widget_show(text_box_frame);
	gtk_widget_show(scrolledwindow);
	gtk_widget_show(clist);
	gtk_widget_show(icon_col_label);
	gtk_widget_show(object_col_label);
	gtk_widget_show(reason_col_label);
	gtk_widget_show(hseparator);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(button);

	gtk_window_set_title(GTK_WINDOW(window), _("Objects with Pending Changes"));
	gtk_window_set_default_size(GTK_WINDOW(window), 475, -1);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), text_box_frame, TRUE, TRUE, 0);
	gtk_widget_set_usize(text_box_frame, 213, 300);
	gtk_container_set_border_width(GTK_CONTAINER(text_box_frame), 6);

	gtk_container_add(GTK_CONTAINER(text_box_frame), scrolledwindow);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(scrolledwindow), clist);
	gtk_clist_set_column_width(GTK_CLIST(clist), CO_ICON_COLUMN, 24);
	gtk_clist_set_column_width(GTK_CLIST(clist), CO_NAME_COLUMN, 151);
	gtk_clist_set_column_width(GTK_CLIST(clist), CO_REASON_COLUMN, 80);
	gtk_clist_column_titles_show(GTK_CLIST(clist));

	gtk_clist_set_column_widget(GTK_CLIST(clist), CO_ICON_COLUMN, icon_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), CO_NAME_COLUMN, object_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), CO_REASON_COLUMN, reason_col_label);

	gtk_box_pack_start(GTK_BOX(vbox), hseparator, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, FALSE, FALSE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 18);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 108, 46);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	if (evms_changes_pending(&changes_pending, &changes) == 0 && changes != NULL) {
		populate_clist_with_changed_objects(GTK_CLIST(clist), changes);
		evms_free(changes);
	}

	gtk_tooltips_set_tip(tooltips, button, _("Close this window"), NULL);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(window));

	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(on_readonly_clist_select_row), NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void display_commit_changes_popup (void)
 *   
 *   Description:
 *      This routine creates and displays a popup to 
 *      notify the user that changes are pending.
 *      The popup then waits for a response from the
 *      user on whether to commit changes or exit 
 *      and lose those changes.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a popup that asks the user whether they
 *      would like to commit changes, exiting without saving,
 *      or cancel exit and return to main window.
 *
 */
void display_commit_changes_popup(void)
{
	GtkWidget *commit_changes_window;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *question_label;
	GtkWidget *hbuttonbox;
	GtkWidget *commit_button;
	GtkWidget *exit_button;
	GtkWidget *cancel_button;
	GtkWidget *details_button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	commit_changes_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(commit_changes_window), _("Unsaved Changes Detected"));
	gtk_window_set_position(GTK_WINDOW(commit_changes_window), GTK_WIN_POS_CENTER);
	gtk_accel_group_attach(accel_group, GTK_OBJECT(commit_changes_window));

	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new(_("Save changes before quitting?"));
	question_label =
	    gtk_label_new(_
			  ("If you quit without saving, changes made since the last save operation will be discarded."));
	hbuttonbox = gtk_hbutton_box_new();
	commit_button = gtk_button_new();
	exit_button = gtk_button_new();
	cancel_button = gtk_button_new();
	details_button = gtk_button_new();

	get_dialog_pixmap(WARNING_PIXMAP, &gdk_pixmap, &mask);
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	get_commit_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(commit_button, gdk_pixmap, mask, _("_Save"), accel_group);

	get_exit_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(exit_button, gdk_pixmap, mask, _("_Quit without Saving"),
				   accel_group);

	get_cancel_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(cancel_button, gdk_pixmap, mask, _("_Cancel"), accel_group);

	get_search_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(details_button, gdk_pixmap, mask, _("_Details"), accel_group);

	gtk_container_add(GTK_CONTAINER(commit_changes_window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 5);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_misc_set_padding(GTK_MISC(label), 63, 10);

	gtk_box_pack_start(GTK_BOX(vbox), question_label, FALSE, FALSE, 0);
	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);
	gtk_misc_set_padding(GTK_MISC(question_label), 63, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 14);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 50);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), details_button);
	GTK_WIDGET_SET_FLAGS(details_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, details_button,
			     _("List objects that report changes pending"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), exit_button);
	GTK_WIDGET_SET_FLAGS(exit_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, exit_button,
			     _("Quit the application without saving changes"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), cancel_button);
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, cancel_button, _("Don't quit the application."), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), commit_button);
	GTK_WIDGET_SET_FLAGS(commit_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, commit_button, _("Save changes then quit the application"),
			     NULL);

	gtk_signal_connect(GTK_OBJECT(commit_button), "clicked", on_commit_button_clicked,
			   GINT_TO_POINTER(TRUE));
	gtk_signal_connect_object(GTK_OBJECT(exit_button), "clicked", gtk_widget_destroy,
				  GTK_OBJECT(get_main_window_id()));
	gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked", gtk_widget_destroy,
				  GTK_OBJECT(commit_changes_window));
	gtk_signal_connect(GTK_OBJECT(details_button), "clicked", on_details_button_clicked, NULL);

	gtk_widget_grab_default(commit_button);
	gtk_window_set_transient_for(GTK_WINDOW(commit_changes_window),
				     GTK_WINDOW(get_main_window_id()));

	gtk_widget_show_all(commit_changes_window);
	gdk_beep();
}

/*
 *
 *   void display_commit_changes_confirmation_dialog (void)
 *   
 *   Description:
 *      This routine creates and displays a dialog to 
 *      allow the user to confirm they want to commit
 *      changes or cancel and return to the main window.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a dialog that asks the user whether they
 *      would like to commit changes or cancel and return
 *      to the main window.
 *
 */
void display_commit_changes_confirmation_dialog(void)
{
	GtkWidget *commit_confirmation_window;
	GtkWidget *vbox;
	GtkWidget *question_label;
	GtkWidget *hbuttonbox;
	GtkWidget *commit_button;
	GtkWidget *cancel_button;
	GtkWidget *details_button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	get_dialog_pixmap(QUESTION_PIXMAP, &gdk_pixmap, &mask);

	commit_confirmation_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(commit_confirmation_window),
			     _("Save Changes Confirmation"));
	gtk_window_set_position(GTK_WINDOW(commit_confirmation_window), GTK_WIN_POS_CENTER);
	gtk_accel_group_attach(accel_group, GTK_OBJECT(commit_confirmation_window));

	vbox = gtk_vbox_new(FALSE, 0);
	question_label = gtk_label_new(_("Do you wish to save changes now or cancel?"));
	hbuttonbox = gtk_hbutton_box_new();
	commit_button = gtk_button_new();
	cancel_button = gtk_button_new();
	details_button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	get_commit_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(commit_button, gdk_pixmap, mask, _("_Save"), accel_group);

	get_cancel_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(cancel_button, gdk_pixmap, mask, _("_Cancel"), accel_group);

	get_search_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(details_button, gdk_pixmap, mask, _("_Details"), accel_group);

	gtk_container_add(GTK_CONTAINER(commit_confirmation_window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 5);

	gtk_box_pack_start(GTK_BOX(vbox), question_label, FALSE, FALSE, 0);
	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);
	gtk_misc_set_padding(GTK_MISC(question_label), 63, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 14);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 50);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), details_button);
	GTK_WIDGET_SET_FLAGS(details_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, details_button,
			     _("List objects that report changes pending"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), cancel_button);
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, cancel_button, _("Cancel save and dismiss this dialog"),
			     NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), commit_button);
	GTK_WIDGET_SET_FLAGS(commit_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, commit_button, _("Save changes"), NULL);

	gtk_signal_connect(GTK_OBJECT(commit_button), "clicked", on_commit_button_clicked,
			   GINT_TO_POINTER(FALSE));
	gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked", gtk_widget_destroy,
				  GTK_OBJECT(commit_confirmation_window));
	gtk_signal_connect(GTK_OBJECT(details_button), "clicked", on_details_button_clicked, NULL);

	gtk_widget_grab_default(commit_button);
	gtk_window_set_transient_for(GTK_WINDOW(commit_confirmation_window),
				     GTK_WINDOW(get_main_window_id()));

	gtk_widget_show_all(commit_confirmation_window);
}

/*
 *
 *   void on_commit_discard_button_clicked (GtkButton *, gpointer *)
 *   
 *   Description:
 *      This routine destroys the commit before reopen 
 *      dialog and issues a gtk_main_quit to exit the
 *      nested event loop.
 * 
 *   Entry:
 *      button    - the discard changes button clicked
 *      user_data - not used
 *
 *   Exit:
 *      The button's parent dialog is destroyed and the
 *      nested event loop is exited.
 *
 */
void on_commit_discard_button_clicked(GtkButton * button, gpointer * user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_main_quit();
}

/*
 *
 *   void display_save_before_reopen_confirmation (void)
 *   
 *   Description:
 *      This routine creates and displays a dialog to 
 *      allow the user to confirm they want to commit
 *      changes or cancel and return to the main window.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a dialog that asks the user whether they
 *      would like to commit changes or cancel and return
 *      to the main window.
 *
 */
void display_save_before_reopen_confirmation(gchar * node_name)
{
	gchar *prompt;
	GtkWidget *commit_confirmation_window;
	GtkWidget *vbox;
	GtkWidget *question_label;
	GtkWidget *hbuttonbox;
	GtkWidget *commit_button;
	GtkWidget *discard_button;
	GtkWidget *details_button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	prompt = g_strdup_printf(_("Save changes before switching to node %s?"), node_name);
	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	get_dialog_pixmap(QUESTION_PIXMAP, &gdk_pixmap, &mask);

	commit_confirmation_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(commit_confirmation_window),
			     _("Save Changes Confirmation"));
	gtk_window_set_position(GTK_WINDOW(commit_confirmation_window), GTK_WIN_POS_CENTER);
	gtk_accel_group_attach(accel_group, GTK_OBJECT(commit_confirmation_window));

	vbox = gtk_vbox_new(FALSE, 0);
	question_label = gtk_label_new(prompt);
	hbuttonbox = gtk_hbutton_box_new();
	commit_button = gtk_button_new();
	discard_button = gtk_button_new();
	details_button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	get_commit_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(commit_button, gdk_pixmap, mask, _("_Save"), accel_group);

	get_search_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(details_button, gdk_pixmap, mask, _("_Details"), accel_group);

	get_cancel_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(discard_button, gdk_pixmap, mask, _("_Continue without Saving"),
				   accel_group);

	gtk_container_add(GTK_CONTAINER(commit_confirmation_window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 5);

	gtk_box_pack_start(GTK_BOX(vbox), question_label, FALSE, FALSE, 0);
	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);
	gtk_misc_set_padding(GTK_MISC(question_label), 63, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 14);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 50);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), details_button);
	GTK_WIDGET_SET_FLAGS(details_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, details_button,
			     _("List objects that report changes pending"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), discard_button);
	GTK_WIDGET_SET_FLAGS(discard_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, discard_button,
			     _("Continue opening remote node without saving changes"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), commit_button);
	GTK_WIDGET_SET_FLAGS(commit_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, commit_button, _("Save changes"), NULL);

	gtk_signal_connect(GTK_OBJECT(commit_button), "clicked", on_commit_button_clicked,
			   GINT_TO_POINTER(TRUE));
	gtk_signal_connect(GTK_OBJECT(discard_button), "clicked", on_commit_discard_button_clicked,
			   NULL);
	gtk_signal_connect(GTK_OBJECT(details_button), "clicked", on_details_button_clicked, NULL);

	gtk_widget_grab_default(commit_button);
	gtk_window_set_transient_for(GTK_WINDOW(commit_confirmation_window),
				     GTK_WINDOW(get_main_window_id()));

	gtk_widget_show_all(commit_confirmation_window);
	g_free(prompt);
}
